/*  =======================================================================
 *
 *  "$Id: lc_timer.c,v 1.3 2005/02/15 21:02:02 cmayncvs Exp $"
 *
 *   This item is the property of GTECH Corporation, West Greenwich,
 *   Rhode Island, and contains confidential and trade secret information.
 *   It may not be transferred from the custody or control of GTECH except
 *   as authorized in writing by an officer of GTECH.  Neither this item
 *   nor the information it contains may be used, transferred, reproduced,
 *   published, or disclosed, in whole or in part, and directly or
 *   indirectly, except as expressly authorized by an officer of GTECH,
 *   pursuant to written agreement.
 *
 *   Copyright (c) 2002 GTECH Corporation.  All rights reserved.
 *
 *  ======================================================================= */

/** \file lc_timer.c
 *
 *  \brief Implement the Linux Comm timer functionality defined in lc_timer.h.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <signal.h>
#include <malloc.h>
#include <string.h>
#include <errno.h>

#include "lc_timer.h"

/*
 *   Uses the ITIMER_REAL interval timer and SIGALRM.  Timers requested
 *   by application program (with "lc_timer_start()") are put
 *   into a linked list sorted by absolute time of expiration.
 *   The first timer on the list uses the single 'itimer' provided
 *   by the system.
 *
 *
 *  NOTE:  this was originally written so that a "cookie" would
 *    be a pointer to an instance of internal_cookie_t.  It was
 *    discovered that this could lead to problems in cases where
 *    timers were started and canceled with no coordination between
 *    the activities.  (A cancelation could apply to a NEW timer
 *    that reused the same internal data structure.)
 *
 *    Accordingly, cookies are now unsigned longs, monotomically increasing
 *    from '1'.  Relative uniqueness assured only by the number of
 *    discrete values in a 32-bit unsigned integer.
 */

/**
 *  Structure put into a list sorted by absolute timer expire time.
 */
typedef struct _cookie_ {
        struct timeval                abs_expire;    /**< absolute expiry */
        unsigned long                 external_cookie_val; /**< user val  */
        lc_timer_user_func_t          handler;       /**< ptr to func.    */
        void                         *handler_data;  /**< handler data    */
        struct _cookie_              *next;          /**< list link       */
} internal_cookie_t;

/**
 *  the lc_timer_cookie_t values passed to/from applications are
 *    based on a monotonically increasing unsigned long.
 */
static unsigned long       user_cookie_val;


/**
 *  a list of all pending timers, sorted by absolute time of expiry.
 */
static internal_cookie_t  *timer_list;

/**
 *  use a free-list for internal_cookie_t structs.  (Help avoid
 *   heap fragmentation.)
 */
static internal_cookie_t  *free_list;

static internal_cookie_t *new_cookie(void);

/**
 *  We protect user-level code against incoming sigalrm.
 */
static sigset_t  sigalrm_set;   /* initialized ONCE, and re-used */

/**
 *  boolean.  make sure we perform per-process initialization
 */
static int       initialized;

/**
 * <sys/time.h> has a macro "timercmp(a, b, CMP)" where the CMP arg
 *  is an operator.  Since "timercmp(&now, &curr->abs_expire, -)" looks
 *  strange, we wrap the header file macro in one of our own.
 */
#define TIMERCMP(a,b)  timercmp((a), (b), -)

/*
 *  Prototype static functions.
 */
static void timer_init(void);    /* one-time (per-process) initialization */

static void insert_into_timer_list(const struct timeval *now,
                                   internal_cookie_t *new);
static void start_kernel_itimer(const struct timeval *now);
static internal_cookie_t  *new_cookie(void);
static void sigalrm_handler(int signo);
static void timediff(const struct timeval *p_smaller,
                     const struct timeval *p_bigger,
                     struct timeval       *diff);
/**
 * \brief catch SIGALRM, generated when itimer expires.
 * \param signo signal number (i.e., SIGALRM)
 *
 *  SIGALRM is invoked when itimer expires.  We always use "one shots", so
 *   at the time of this function, there is no further pending itimer.
 *    - call the function whose pointer is in the first node on the \n
 *      list of timers.
 *    - remove the node from the list (put on free list)
 *    - start a new itimer for the next node on the list.
 */
static void
sigalrm_handler(int signo)
{
        struct timeval  now;

        internal_cookie_t  *curr = timer_list;

        if (curr == NULL)   /* TBD:  log error */
                return;
        /*
         *  If we're short of the abs_expire time, restart
         */
        (void) gettimeofday(&now, NULL);
        if (TIMERCMP(&now, &curr->abs_expire) < 0) {
                start_kernel_itimer(&now);
                return;
        }
        timer_list = curr->next;   /* remove from timer_list */

        (*(curr->handler))(curr->handler_data);  /* call user function */

        curr->next = free_list;    /* put on free list */
        free_list = curr;

        if (timer_list != NULL) {  /* start itimer for next on list */
                (void) gettimeofday(&now, NULL); /* get time, again */
                start_kernel_itimer(&now);
        }
}

/**
 * \brief per-process initialization of library code.
 *
 *  One-time per-process initialization.  Arrange for the signal
 *   handler to be called on each SIGALRM.
 */
static void
timer_init(void)
{
        struct sigaction sa;

        if (initialized)
                return;
        sigemptyset(&sigalrm_set);
        sigaddset(&sigalrm_set, SIGALRM);
        memset(&sa, 0, sizeof(sa));
        sa.sa_handler = sigalrm_handler;
        (void) sigaction(SIGALRM, &sa, NULL);
        initialized = 1;
}


/**
 * \brief sort a new structure in to list of pending timers.
 * \param now the current time
 * \param new the new structure.
 *
 *  insert a new timer-cookie structure on the list sorted by absolute
 *    expiration time.  If new new struct becomes the first one on the list,
 *    then start (or re-start) the interval timer.
 *    Note:  SIGALRM is blocked when called.
 */
static void
insert_into_timer_list(const struct timeval *now, internal_cookie_t *new)
{
        internal_cookie_t *curr;
        internal_cookie_t *prev = NULL;

        for (curr = timer_list; curr; prev = curr, curr = curr->next) {
                if (TIMERCMP(&new->abs_expire, &curr->abs_expire) <= 0)
                        break;
        }
        /* 3 cases:  first on list,
                     between 'prev' and 'curr'
                     last on list (between 'prev' and 'curr', but 'curr' NULL)
        */
        if (prev == NULL) {               /* first on list */
                new->next = timer_list;
                timer_list = new;
                start_kernel_itimer(now);
        } else {                          /* between prev and curr       */
                prev->next = new;         /*  NOTE:  if curr is NULL,    */
                new->next  = curr;        /*    then new becomes last    */
        }                                 /*    (handles last two cases) */
}

/**
 * \brief subtract two struct timevals (either one may be larger)
 *   (we compare the times before using timersub macro from <sys/time.h>)
 */
static void
timediff(const struct timeval *p_smaller,
         const struct timeval *p_bigger,
         struct timeval *diff)
{
        if (TIMERCMP(p_smaller, p_bigger) > 0) {
                const struct timeval *tmp;
                tmp = p_smaller;
                p_smaller = p_bigger;
                p_bigger = tmp;
        }
        /* use macro from <sys/time.h> */
        timersub(p_bigger, p_smaller, diff);
}


/**
 * \brief start itimer for first struct on list (earliest expiring)
 * \param now the current time
 *
 *  set the itimer (real) value to the difference between the absolute
 *    time in the first cookie-struct on the timer_list, and now.
 *    For convenience, the caller has already obtained 'now'.
 *
 *    If the list is empty (i.e., there are no pending timers now),
 *    then the itimer is stopped with a zero-valued \c itimerval.
 *
 *    We always use "one shot" timers, i.e., we set it_interval to
 *    zero so that the itimer doesn't reset itself.
 */
static void
start_kernel_itimer(const struct timeval *now)
{
        internal_cookie_t *curr = timer_list;
        struct itimerval itv;

        memset(&itv, 0, sizeof(itv));
        if (timer_list == NULL) {
                /* list is empty --- stop any running itimer */
                setitimer(ITIMER_REAL, &itv, NULL);
                return;
        }
        /*
         *  If the requested expiration time has already passed, then
         *  just leave the it_value struct zero.  (It will be adjusted
         *  up to the minimum itimer value below.)
         */
        if (TIMERCMP(now, &curr->abs_expire) < 0) {
                timediff(now, &curr->abs_expire, &itv.it_value);
        }
        /*
         *  We "prevent" races by always making sure the it_value is
         *   at lease LC_MINIMUM_MSECS.  This degrades performance
         *   somewhat in cases where there are multiple timers set
         *   do go off at nearly identical times --- these get
         *   "spread out" to LC_MINIMUM_MSECS intervals instead of
         *   at their requested times.
         */
        if (itv.it_value.tv_sec == 0 &&
            itv.it_value.tv_usec < (LC_MINIMUM_MSECS * 1000)) {
                itv.it_value.tv_usec = (LC_MINIMUM_MSECS * 1000);
        }

        (void) setitimer(ITIMER_REAL, &itv, NULL);
}

/**
 * \brief allocate a internal_cookie_t structure.
 * \return pointer to structure, or NULL
 *
 *  allocate an instance of internal_cookie_t, either
 *      from the free list or with malloc().
 *
 *      TBD:  log an error if malloc fails.
 */
static internal_cookie_t *
new_cookie(void)
{
        internal_cookie_t *new = free_list;

        if (new == NULL)
                new = malloc(sizeof(internal_cookie_t));
        else
                free_list = new->next;
        if (new)
                new->next = NULL;
        return new;
}


/***************************************************************************
 ***************************************************************************
 *                        exported functions                               *
 ***************************************************************************/

/**
 * \brief start a Linux Comm Timer
 * \param msecs number of milleseconds
 * \param handler function to call when timer expires.
 * \param user_arg_data argument to handler function
 * \return an opaque "cookie" by which timer may be referenced, or 0 on error.
 *
 *   Implementation 1.  uses lists and NO kernel driver module.
 *
 *     Initialize a new 'cookie' structure and sort it into the
 *     \c timer_list.  (Which will start/restart the itimer if the
 *     new node has an earlier expiration time than any other timer.)
 */
lc_timer_cookie_t
lc_timer_start(unsigned long msecs,
               lc_timer_user_func_t handler,
               void *user_arg_data)
{
        int                secs;
        struct timeval     now;
        struct timeval     tv_new;  /* so we can use timeradd() */
        internal_cookie_t *new;
        sigset_t           old_sigset;
        unsigned long      user_cookie;

        if (!initialized)
                timer_init();

        if (handler == NULL)  /* disallow NULL function pointers */
                return LC_TIMER_COOKIE_FAIL;

        if (msecs < LC_MINIMUM_MSECS) {
                msecs = LC_MINIMUM_MSECS;
        }
        /*
         *  convert from msecs to secs/msecs
         */
        secs = msecs / 1000;
        msecs %= 1000;

        new = new_cookie();

        if (new == NULL)
                return LC_TIMER_COOKIE_FAIL;   /* TBD:  log error message */

        tv_new.tv_sec = secs;     /* put into struct timeval for timeradd() */
        tv_new.tv_usec = msecs * 1000;  /* msecs -->> usecs */

        new->handler = handler;
        new->handler_data = user_arg_data;

        /* block SIGALRM before doing anything with list or the time */
        (void) sigprocmask(SIG_BLOCK, &sigalrm_set, &old_sigset);
        /*
         *  "allocate" a new "pretty unique" value for the external cookie.
         */
        user_cookie = ++user_cookie_val;
        if (user_cookie == 0)  /* if rolled around to zero, skip to next */
                user_cookie = ++user_cookie_val;
        new->external_cookie_val = user_cookie;

        if (gettimeofday(&now, NULL)) {
                fprintf(stderr, "gettimeofday failed:  %s\n",
                        strerror(errno));
                exit(1);
        }
        timeradd(&now, &tv_new, &(new->abs_expire));
        insert_into_timer_list(&now, new);  /* may start new itimer */
        if (!sigismember(&old_sigset, SIGALRM))
                (void) sigprocmask(SIG_SETMASK, &old_sigset, NULL);
        /* SIGALRM (probably) now unblocked */
        return (lc_timer_cookie_t) user_cookie;
}

/**
 * \brief cancel a Linux Comm Timer
 * \param tcookie an opaque handle returned by lc_timer_start
 * \return 0 on success, 1 if the timer could not be cancelled.
 *
 *   Implementation 1.  uses lists and NO kernel driver module.
 *
 *  Find the node in \c timer_list matching the cookie, and
 *  remove it from the list.  If the matching node was first
 *  on the list, then use 'start_kernel_itimer()' to restart
 *  the itimer.
 */
int
lc_timer_cancel(lc_timer_cookie_t tcookie)
{
        internal_cookie_t *curr, *prev;
        struct timeval     now;
        int                rv = -1;  /* assume failure to find it */
        sigset_t           old_sigset;

        if (!initialized)
                timer_init();

        /* block SIGALRM before doing anything with list or the time */
        (void) sigprocmask(SIG_BLOCK, &sigalrm_set, &old_sigset);
        prev = NULL;
        for (curr = timer_list; curr; prev = curr, curr = curr->next) {
                if (curr->external_cookie_val == tcookie) {
                        if (prev == NULL) {
                                timer_list = curr->next;
                        } else {
                                prev->next = curr->next;
                        }
                        curr->next = free_list;
                        free_list = curr;
                        rv = 0;  /* return success */
                        break;
                }
        }
        /* restart itimer */
        (void) gettimeofday(&now, NULL);
        start_kernel_itimer(&now);
        if (!sigismember(&old_sigset, SIGALRM))
                (void) sigprocmask(SIG_SETMASK, &old_sigset, NULL);
        /* SIGALRM (probably) now unblocked */
        return rv;
}


/**
 * \brief block execution of timer-expiration functions (in user process)
 *
 *  Implementation 1:  uses lists, itimer, SIGALRM
 *
 *   block signal SIGALRM
 */
void
lc_timer_block(void)
{
        if (!initialized)
                timer_init();
        sigprocmask(SIG_BLOCK, &sigalrm_set, NULL);
}

/**
 * \brief unblock execution of timer-expiration functions (in user process)
 *
 *  Implementation 1:  uses lists, itimer, SIGALRM
 *
 *   unblock signal SIGALRM
 */
void
lc_timer_unblock(void)
{
        if (!initialized)
                timer_init();
        sigprocmask(SIG_UNBLOCK, &sigalrm_set, NULL);
}


/*
 *  End of "$Id: lc_timer.c,v 1.3 2005/02/15 21:02:02 cmayncvs Exp $"
 */


