/*  =======================================================================
 *
 *  "$Id: stress-test.c,v 1.1.1.1 2003/12/02 15:53:30 sbobcvs Exp $"
 *
 *   This item is the property of GTECH Corporation, West Greenwich,
 *   Rhode Island, and contains confidential and trade secret information.
 *   It may not be transferred from the custody or control of GTECH except
 *   as authorized in writing by an officer of GTECH.  Neither this item
 *   nor the information it contains may be used, transferred, reproduced,
 *   published, or disclosed, in whole or in part, and directly or
 *   indirectly, except as expressly authorized by an officer of GTECH,
 *   pursuant to written agreement.
 *
 *   Copyright (c) 2002 GTECH Corporation.  All rights reserved.
 *
 *  ======================================================================= */

/** \file stress-test.c
 *
 *   Run the "test" program N times at once, and each time one of them
 *   completes, start it again --- unless it failed.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/wait.h>
#include <string.h>
#include <errno.h>
#include <malloc.h>

/** \def MAX_TEST_PROCS
 *   maximum number of simultaneous processes running "test" to allow.
 */
#define MAX_TEST_PROCS 150

/**
 *  Structure describing each of the N processes outstanding.
 */
typedef struct {
        pid_t    pid;           /**< pid of child running "test" */
        int      status;        /**< exit status (if failed)     */
        char    *logfilename;   /**< name of log file            */
} test_process_t;

/** will use malloc() to allocate N instances.  (N given at runtime) */
test_process_t    *tp_arr;

/**
 *  N is the number of simultaneous instances of "test" to run.
 */
int     N;

/**
 *   count of number of processes running "test".  This is volatile
 *   as it's manipulated by a signal handler, whose execution cannot
 *   be forseen by the compiler.
 */
volatile int num_tests_running;

/**
 *   Boolean set TRUE if the tests are to stop.
 */
volatile int stop;

/**
 *   Exit code of a child if execve() fails.
 */
#define EXECVE_FAIL_STATUS  45

/** paranoid verification that shared lib works properly for many processes */
pid_t  main_pid;

/**
 *  Start a new instance of "test".  SIGCHLD is blocked.
 */
void
start_test_process(test_process_t *tp)
{
        extern char **environ;
        char         *argv[3];
        int           forktries = 10;
        int           newpid;
        struct sigaction sa;

        if (getpid() != main_pid) {
                fprintf(stderr, "start_test_process:  wrong PID\n");
                exit(1);
        }

        if (stop)
                return;   /* no more testing */

        if (tp->pid != 0) {
                fprintf(stderr, "start_test_process:  pid field non-zero\n");
                return;
        }
        argv[0] = "./test";
        argv[1] = tp->logfilename;
        argv[2] = NULL;
        newpid = -1;
        while (newpid < 0 && --forktries > 0) {
                newpid = fork();
                if (newpid < 0) {
                        sleep(1);   /* assume EAGAIN or EINTR */
                        continue;
                }
                if (newpid == 0) {
                        memset(&sa, 0, sizeof(sa));
                        sa.sa_handler = SIG_IGN;
                        (void) sigaction(SIGINT, &sa, NULL);
                        (void) sigaction(SIGCHLD, &sa, NULL);
                        (void) sigaction(SIGTERM, &sa, NULL);
                        fprintf(stderr, "%s %s\n", argv[0], argv[1]);
                        execve(argv[0], argv, environ);
                        exit(EXECVE_FAIL_STATUS);
                }
        }
        if (newpid < 0) {
                fprintf(stderr, "fork failed:  %s\n", strerror(errno));
                exit(1);
        }
        tp->pid = newpid;
        num_tests_running++;
}

/** \fn void finish_up(void)
 *
 *  Print some parting remarks and exit.  Note any non-zero 'status'
 *   values in the test_process_t array.
 */
void
finish_up(void)
{
        int   i;
        test_process_t *tp;

        fprintf(stderr, "all done\n");
        for (i = 0, tp = tp_arr; i < N; i++, tp++) {
                if (tp->status != 0) {
                        fprintf(stderr, "slot %d failed:  see %s\n",
                                i, tp->logfilename);
                }
        }
        exit(0);
}
        
        
/**
 *   handle SIGCHLD and other signals.
 *
 *   For SIGCHLD, see which process has terminated, and
 *    if it was an instance of "test", restart if it exited cleanly.
 *
 *   For other caught signals (SIGINT, and perhaps others), set "stop"
 *   which will cause no further "test" processes to start.
 */
void
signal_handler(int signo)
{
        int              i;
        pid_t            kidpid;
        int              status;
        test_process_t  *tp;
        static int       sigchld_num;

        if (signo != SIGCHLD) {
                stop = 1;
                fprintf(stderr, "signal %d:  num_tests_running %d\n",
                        signo, num_tests_running);
                return;
        }
        /****** SIGCHLD processing *********/
        ++sigchld_num;
        while ((kidpid = waitpid(-1, &status, WNOHANG)) > 0) {
                for (tp = tp_arr, i = 0; i < N; i++, tp++) {
                        if (kidpid == tp->pid) {
                                break;
                        }
                }
                if (i >= N) {      /* some other process */
                        fprintf(stderr, "%d some other process\n",sigchld_num);
                        continue;
                }
                if (!WIFEXITED(status)) {
                        fprintf(stderr, "%d child did not exit\n",sigchld_num);
                        continue;   /* it hasn't exited */
                }
                num_tests_running--;
                tp->pid = 0;
                if (WEXITSTATUS(status) == 0) {
                        fprintf(stderr, "%d test slot %d succeeded\n",
                                sigchld_num, i);
                } else {
                        tp->status = status;
                        if (WEXITSTATUS(status) == EXECVE_FAIL_STATUS) {
                                fprintf(stderr, "execve failed\n");
                        } else {
                                fprintf(stderr, "test process #%d failed.  "
                                        "see %s\n",i , tp->logfilename);
                        }
                }
                if (stop) {
                        if (num_tests_running <= 0) {
                                finish_up();
                                exit(0);
                        }
                } else if (WEXITSTATUS(status) == 0) {
                        start_test_process(tp);
                }
        }
}

/**
 *   temporarily block receipt of SIGCHLD
 */
void
block_sigchld(void)
{
        sigset_t set;

        sigemptyset(&set);
        sigaddset(&set, SIGCHLD);
        sigprocmask(SIG_BLOCK, &set, NULL);
}

/**
 *   unblock receipt of SIGCHLD
 */
void
unblock_sigchld(void)
{
        sigset_t set;

        sigemptyset(&set);
        sigaddset(&set, SIGCHLD);
        sigprocmask(SIG_UNBLOCK, &set, NULL);
}

/**
 *  Verify that "./test" is executable.  (This stress-test isn't too smart.
 *  Get the number of processes to maintain.  This can be either
 *      argv[1], or will be interactively obtained.
 *  Malloc the tp_arr[] array, and initialize it.
 *  Set up signal handlers for SIGCHLD, SIGINT, SIGTERM.
 *  Block SIGCHLD
 *     start all N processes.  (They will be restarted when they exit.)
 *  Unblock SIGCHLD
 *  Wait for the program to end.  (NOTE:  it is likely that actual
 *     program termination will take place in the signal handler.)
 */
int
main(int ac, char **av)
{
        char             buf[256];
        int              i;
        char            *p;
        test_process_t  *tp;
        struct sigaction sa;

        main_pid = getpid();

        if (access("./test", X_OK) != 0) {
                fprintf(stderr, "no executable named \"./test\"\n");
                exit(1);
        }

        if (ac > 2) {
            usage:
                fprintf(stderr, "usage:  %s [num-processes-to-run]\n", *av);
                exit(1);
        }
        if (ac == 2) {
                if (av[1][0] >= '0' && av[1][0] <= '9') {
                        N = strtol(av[1], NULL, 10);
                        if (N == 0 || N > MAX_TEST_PROCS) {
                                printf("invalid.  maximum is %d\n",
                                       MAX_TEST_PROCS);
                                goto usage;
                        }
                } else {
                        /* cmd arg isn't a number */
                        goto usage;
                }
        } else {
                N = 0;
                while (N == 0) {
                        printf("enter number of tests to keep running:  ");
                        if (fgets(buf, sizeof(buf), stdin) == NULL) {
                                printf("okay.  quitting\n");
                                return 1;
                        }
                        for (p = buf; *p; p++) {
                                if (*p == '\n') {
                                        *p = 0;
                                        break;
                                }
                                if (*p >= '0' && *p <= '9') {
                                        N = strtol(p, NULL, 10);
                                        break;
                                }
                        }
                        if (N == 0 || N > MAX_TEST_PROCS) {
                                if (N > MAX_TEST_PROCS)
                                        printf("maximum is %d\n",
                                               MAX_TEST_PROCS);
                                printf("invalid.  try again\n");
                                N = 0;
                        }
                }
        }
        if ((tp_arr = malloc( N * sizeof(test_process_t))) == NULL) {
            nomem:
                fprintf(stderr, "no memory\n");
                exit(1);
        }
        for (tp = tp_arr, i = 0; i < N; i++, tp++) {
                tp->pid = 0;
                tp->status = 0;
                sprintf(buf, "/tmp/timertest.%d", i);
                if ((tp->logfilename = strdup(buf)) == NULL)
                        goto nomem;
        }
        /* set up signals */
        memset(&sa, 0, sizeof(sa));
        sa.sa_handler = signal_handler;
        if (sigaction(SIGCHLD, &sa, NULL)) {
            sigaction_failed:
                fprintf(stderr, "sigaction failed:  %s\n", strerror(errno));
                exit(1);
        }
        /* for other signals, block SIGCHLD while handling */
        sigaddset(&sa.sa_mask, SIGCHLD);
        if (sigaction(SIGINT, &sa, NULL))
                goto sigaction_failed;
        if (sigaction(SIGTERM, &sa, NULL))
                goto sigaction_failed;
        
        /* fire off first set of N processes */
        block_sigchld();
        for (tp = tp_arr, i = 0; i < N; i++, tp++) {
                fprintf(stderr, "main calling start_test_process(%d)\n",i);
                start_test_process(tp);
        }
        unblock_sigchld();

        while (1) {
                if (stop == 0)
                        continue;
                if (num_tests_running > 0)
                        continue;
                break;
        }
        fprintf(stderr, "main finishing up\n");
        finish_up();
        return 0;
}
/*
 *  End of "$Id: stress-test.c,v 1.1.1.1 2003/12/02 15:53:30 sbobcvs Exp $"
 */
