/*  =======================================================================
 *
 *  "$Id: test.c,v 1.1.1.1 2003/12/02 15:53:30 sbobcvs Exp $"
 *
 *   This item is the property of GTECH Corporation, West Greenwich,
 *   Rhode Island, and contains confidential and trade secret information.
 *   It may not be transferred from the custody or control of GTECH except
 *   as authorized in writing by an officer of GTECH.  Neither this item
 *   nor the information it contains may be used, transferred, reproduced,
 *   published, or disclosed, in whole or in part, and directly or
 *   indirectly, except as expressly authorized by an officer of GTECH,
 *   pursuant to written agreement.
 *
 *   Copyright (c) 2002 GTECH Corporation.  All rights reserved.
 *
 *  ======================================================================= */

/** \file test.c
 *
 *  Simple program to test Linux Comm timers.  This program
 *    starts NUM_TIMERS timers for random intervals between 0 and
 *    10 seconds, and checks the actual elapsed time between
 *    start and expiration.  To test the 'cancel' function,
 *    at random times an unexpired timer will be randomly
 *    selected and canceled.
 *
 *    This program exits after all of the timers started have
 *    been received (via the function set up to be called upon
 *    expiration) or cancelled.  To catch a timer that gets lost,
 *    a watchdog child process is started that sleeps for 100
 *    seconds, and signals SIGUSR1 to us.  If we catch SIGUSR1,
 *    then an error message is printed.
 *
 *    A log file is written containing information about each
 *    timer expiration received.
 *
 *    Note:  this program can be run from "test.sh", which will
 *    run this program over and over and over in an 'xterm'. The
 *    "test.sh" script will prompt for the number of xterms in
 *    which to repeat this.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <errno.h>
#include <signal.h>

#include "lc_timer.h"

/**
 *   PID of child started to deliver SIGUSR1
 */
int watchdog_child_pid;

/** write data to a log file so that stress-test program is easier */
FILE *log;
/** count of calls to function called on timer expiration */
volatile int   handler_call_count;

/** Number of timers to start at program beginning. */
#define NUM_TIMERS 100

/**
 *  Data object describing a timer for this program.  A pointer
 *  to an instance will be passed to \c timer_handler when a
 *  timer expires.
 */
typedef struct {
        int            number;        /**< index into 'raw' array         */
        struct timeval starttime;     /**< absolute start time            */
        struct timeval interval;      /**< randomly chosen timer interval */
        struct timeval endtime;       /**< starttime + interval           */
        lc_timer_cookie_t cookie;     /**< timer cookie                   */
} test_t;

/** allocate \c NUM_TIMERS instances of \c test_t */
static test_t  raw[NUM_TIMERS];

/** Array of true/false/count for timers cancelled or expired. */
static volatile int timers_expired[NUM_TIMERS];

/** count of timers successfully cancelled. */
static int      num_cancellations;

/** count of timers whose cancellation failed. */
static int      num_failed_cancellations;

/**
 * <sys/time.h> has a macro "timercmp(a, b, CMP)" where the CMP arg
 *  is an operator.  Since "timercmp(&now, &curr->abs_expire, -)" looks
 *  strange, we wrap the header file macro in one of our own.
 */
#define TIMERCMP(a,b)  timercmp((a), (b), -)

/**
 *   buffer for printing the values of the cookies returned by
 *    lc_timer_start().  The finish_up() function will print this
 *    buffer to the log file.
 */
char    cookie_val_print_buf[32 * 4096];


/**
 * \brief finish up program execution
 *
 *  Function called to print final thoughts to stdout upon exiting.
 */
static void
finish_up(void)
{
        int i;
        double f_requested;
        test_t *t;

        if (watchdog_child_pid) {
                kill(watchdog_child_pid, 9);
                watchdog_child_pid = 0;
        }
        
        fprintf(log, "%s\n", cookie_val_print_buf);

        for (i = 0; i < NUM_TIMERS; i++) {
                if (timers_expired[i] != 1) {
                        t = &raw[i];
                        f_requested = t->interval.tv_sec;
                        f_requested += (double)(t->interval.tv_usec) /
                                1000000.0;
                        fprintf(log, "timers_expired[%d] is %d.  ",
                               i, timers_expired[i]);
                        fprintf(log, "  requested %.4f\n",
                               f_requested);
                }
        }
        fprintf(log, "cancelled timers %d, failed cancellations %d\n",
               num_cancellations, num_failed_cancellations);
}

/**
 * Handle SIGUSR1 sent from watchdog child process.
 */
static void
sigusr1_handler(int signo)
{
        fprintf(log, "SIGUSR1!!!\n");
        finish_up();
        exit(1);
}

/**
 *  Child process sleeps, then delivers SIGUSR1 to parent.
 */
static void
watchdog_child_function(void)
{
        sleep(100);
        kill(getppid(), SIGUSR1);
        exit(1);
}

/**
 *   Boolean predicate:  returns TRUE of all of the entries
 *     in the \c timers_expired array are non-zero.
 */
int
all_done(void)
{
        int i;

        for (i = 0; i < NUM_TIMERS; i++) {
                if (timers_expired[i] == 0)
                        return 0;
        }
        return 1;
}
/**
 * \brief subtract two struct timevals (either one may be larger)
 *   (we compare the times before using timersub macro from <sys/time.h>)
 */
static void
timediff(const struct timeval *p_smaller,
         const struct timeval *p_bigger,
         struct timeval *diff)
{
        if (TIMERCMP(p_smaller, p_bigger) > 0) {
                const struct timeval *tmp;
                tmp = p_smaller;
                p_smaller = p_bigger;
                p_bigger = tmp;
        }
        /* use macro from <sys/time.h> */
        timersub(p_bigger, p_smaller, diff);
}


/**
 *  Function called each time a timer expires.  Argument is a
 *  pointer to an instance of \c test_t.  Compute elapsed time
 *  and compare it with the requested timer duration.  There's
 *  some slop, so don't declare an error if the requested duration
 *  is over 1 second and the elapsed time is less than 50 msecs too
 *  short.  Increment the corresponding entry in \c timers_elapsed
 *  array.  If \c all_done() is TRUE, call \c finish_up() and exit.
 */
static void
timer_handler(void *arg)
{
        struct timeval now, elapsed;
        test_t *t = arg;

        if (gettimeofday(&now, NULL)) {
                fprintf(stderr, "gettimeofday failed:  %s\n",
                        strerror(errno));
                exit(1);
        }
        timediff(&t->starttime, &now, &elapsed);
        fprintf(log, "(%d) %d:  endtime %ld.%06ld actual %ld.%06ld, "
                "requested %ld.%06ld",
                ++handler_call_count,
                t->number,
                t->endtime.tv_sec, t->endtime.tv_usec,
                elapsed.tv_sec, elapsed.tv_usec,
                t->interval.tv_sec, t->interval.tv_usec);
        if (timers_expired[t->number] != 0) {
                fprintf(log, "timers_expired[%d] is %d\n",
                        t->number, timers_expired[t->number]);
        }
        (timers_expired[t->number])++;
        if (TIMERCMP(&now, &t->endtime) < 0) {
                struct timeval diff;
                timediff(&now, &t->endtime, &diff);
                fprintf(log, "  SHORT: diff %ld.%06ld, now %ld.%06ld",
                        diff.tv_sec, diff.tv_usec,
                        now.tv_sec, now.tv_usec);
        }
        fprintf(log, "\n");
        fflush(log);
        if (all_done()) {
                finish_up();
                exit(0);
        }
}

/** default name of log file.  May be changed if there's a command arg. */
char *logfilename = "/tmp/timertest.log";   /* default */

/**
 *  Seed the random number generator, set a handler for SIGUSR1,
 *   start the watchdog child process, and then start \c NUM_TIMERS
 *   timers.  Record the starting time for each.
 *
 *   Once the timers are started, loop calling pause(2) at each iteration.
 *   Each time pause(2) returns, BLOCK receipt of timer expirations and
 *   make a random decision on whether to attempt a timer cancellation,
 *   and if so, pick a timer at random.  Record whether cancellation was
 *   successful.  Check to see if \c all_done() is now TRUE.
 */
int main(int ac, char **av)
{
        int               i;
        test_t           *t = raw;
        lc_timer_cookie_t cookie;
        int               maxSecs = 0;
        struct sigaction  sa;
        struct itimerval  itv;
        int               msecs;
        char             *cookie_val_print_ptr = cookie_val_print_buf;
        char             *cookie_val_sentinel;

        cookie_val_sentinel = cookie_val_print_buf +
                sizeof(cookie_val_print_buf) - 128;

        if (ac > 1)
                logfilename = av[1];

        srandom(getpid());

        memset(&sa, 0, sizeof(sa));
        sa.sa_handler = sigusr1_handler;
        (void) sigaction(SIGUSR1, &sa, NULL);
        
        if ((log = fopen(logfilename, "w")) == NULL) {
                fprintf(stderr, "can't write %s:  %s\n",
                        logfilename, strerror(errno));
                if (watchdog_child_pid)
                        kill(watchdog_child_pid, 9);
                exit(1);
        }
        if ((watchdog_child_pid = fork()) == 0) {
                fclose(log);
                watchdog_child_function();
        }

        sprintf(cookie_val_print_ptr, "cookie values:  ");
        cookie_val_print_ptr += strlen(cookie_val_print_ptr);

        for (i = 0; i < NUM_TIMERS; i++, t++) {
                t->number = i;
                msecs = random() % (10 * 1000);
                t->interval.tv_sec = msecs / 1000;
                t->interval.tv_usec = (msecs % 1000) * 1000;
                if (t->interval.tv_sec > maxSecs)
                        maxSecs = t->interval.tv_sec;

                if (gettimeofday(&t->starttime, NULL)) {
                        fprintf(log, "gettimeofday failed:  %s\n",
                                strerror(errno));
                        if (watchdog_child_pid)
                                kill(watchdog_child_pid, 9);
                        exit(1);
                }
                timeradd(&t->starttime, &t->interval, &t->endtime);
                cookie = lc_timer_start(msecs, timer_handler, t);
                if (cookie == LC_TIMER_COOKIE_FAIL) {
                        fprintf(log, "bad cookie\n");
                        if (watchdog_child_pid)
                                kill(watchdog_child_pid, 9);
                        exit(1);
                }
                t->cookie = cookie;
                if (cookie_val_print_ptr < cookie_val_sentinel) {
                        sprintf(cookie_val_print_ptr, "%lu ", cookie);
                        cookie_val_print_ptr += strlen(cookie_val_print_ptr);
                }
        }
        fprintf(log, "all timers started:  max secs %d\n", maxSecs);
        while (!all_done()) {
                pause();
                lc_timer_block();
                if ((random() % 100) > 50) {
                        int  i;

                        i = random() % NUM_TIMERS;
                        if (timers_expired[i] == 0) {
                                if (lc_timer_cancel(raw[i].cookie) == 0) {
                                        (timers_expired[i])++;
                                        num_cancellations++;
                                } else {
                                        num_failed_cancellations++;
                                }
                        }
                }
                lc_timer_unblock();
        }
        fclose(log);
        finish_up();
        return 0;
}
/*
 *  End of "$Id: test.c,v 1.1.1.1 2003/12/02 15:53:30 sbobcvs Exp $"
 */
