/* ===[ $RCSfile: list.c,v $ ]===============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: list.c,v 1.4 2005/07/18 19:01:48 cmayncvs Exp $

 \brief Implements the basic list functions.

 The functions mimic the Lisp functions of the same name. More information on the standard
 definitions of Lisp functions can be found in the Common Lisp Specification available at
 http://www.lispworks.com/documentation/HyperSpec/Front/index.htm
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "list.h"
#include "gassert.h"

/** The cons cell structure is used to build lists. */
struct cell
{
    void *atom;             ///< Pointer to data.
    struct cell *next;      ///< Pointer to next cons cell.
};

/** List of free cons cells. */
static list free_list = 0;

/** List statistics structure. */
static struct _list_stats
{
    unsigned int num_alloc;     ///< Number allocated
    unsigned int num_freed;     ///< Number freed
    unsigned int num_dup_free;  ///< Number of duplicate frees.
} list_stats;

/**
 \brief Allocate a new cons cell.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return New cell.
 */
static list
alloc_cell (const char *file, int line)
{
    list t;

    if (free_list)
    {
        t = free_list;
        free_list = t->next;
    }
    else
    {
        libassert (t = malloc (sizeof (struct cell)));
    }
    list_stats.num_alloc++;
    return t;
}

/**
 \brief Check if a cons cell has already been freed.
 \return Non-zero if cell already freed.
 */
static int
already_freed (list t)
{
    list fl = free_list;

    while (fl)
    {
        if (fl == t)
            return 1;
        fl = fl->next;
    }
    return 0;
}

/**
 \brief Put a cons cell on the free list.
 \param t Cons cell to free.
 */
static void
free_cell (list t)
{
    if (t)
    {
        if (already_freed (t))
        {
            //printf ("***WARNING*** Cons cell %p already freed!\n", t);
            list_stats.num_dup_free++;
        }
        else
        {
            t->next = free_list;
            free_list = t;
            list_stats.num_freed++;
        }
    }
}

/**
 \brief Constructs a list.
 \param atom Pointer to the element to add.
 \param l List to add to.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return The new list.

 The new cons cell is added to the head of the list. If \a l is null a new list is created.

 \b Examples:
    - cons (A, (B C)) => (A B C)
    - cons ((A), (B C)) => ((A) B C)

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_cons.htm
 */
list
_cons (void *atom, list l, const char *file, int line)
{
    list t;

    t = alloc_cell (file, line);
    t->atom = atom;
    t->next = l;

    return t;
}

/**
 \brief Returns the first element in a list.
 \param l List.

 \b Examples:
    - car ((A B C)) => A
    - car (((A) B C)) => (A)

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_car_c.htm
 */
void *
car (list l)
{
    if (l)
        return l->atom;
    return 0;
}

/**
 \brief Returns the a list that is everything but the first element of list.
 \param l List.
 \return A list.

 \b Examples:
    - cdr ((A B C)) => (B C)
    - cdr ((A)) => ()

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_car_c.htm
 */
list
cdr (list l)
{
    if (!l)
        return 0;
    return l->next;
}

/**
 \brief Returns the cdr and frees the first cons cell in the list.
 \param l List.
 \return A list.
 */
list
cdrf (list l)
{
    list t = 0;

    if (l)
    {
        t = l->next;
        free_cell (l);
    }
    return t;
}

/**
 \brief Joins two lists into one.
 \param l List.
 \param a List.
 \return Joined list.

 \b Examples:
    - append ((A B), (C D)) => (A B C D)
    - append ((A B), ()) => (A B)

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_append.htm
 */
list
append (list l, list a)
{
    list t;

    if (!l)
        return a;

    t = l;
    while (t->next)
        t = t->next;
    t->next = a;

    return l;
}

/**
 \brief For each element a function is called.
 \param proc Function to call for each element.
 \param l List to iterate over.

 Call the supplied function once for each element in the list.
 The supplied function is called for it's side effect only.

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_mapc_.htm
 */
void
mapcar (void (*proc)(void *), list l)
{
    while (l)
    {
        proc (car (l));
        l = cdr (l);
    }
}

/**
 \brief Find an element in in a list.
 \param atom Element to find.
 \param l List to search.
 \param equal Function to use for comparing \a atom and \a l.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 Compares each element of the list with atom using the equal function until equal returns true.
 Member return list starting at the matched element. If the \a equal is null, then the default
 function is used. The default function does a string comparison.

 \b Examples:
    - member (B, (A B C), 0) => (B C)
    - member (D, (A B C), 0) => ()

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_mem_m.htm
 */
list
_member (void *atom, list l, int (*equal)(void *, void *), const char *file, int line)
{
    int eq (void *a, void *c)
    {
        libassert (a && c);
        return !strcmp (a, c);
    }

    if (!equal)
        equal = eq;

    while (l)
    {
        if (equal (atom, car (l)))
            return l;
        l = cdr (l);
    }
    return 0;
}

/**
 \brief Removes an element from a list.
 \param atom Element to remove.
 \param l List to remove from.
 \param equal Optional function used to compare elements.
 \param destroy_atom Optional function to be called with the element from the list before it is freed.
 \return The list without the matching element.

 Searches the list for the first matching element using the equal function and returns the list
 without the first matching element. The cons cell for the element is freed and if a
 \a destroy_atom function is supplied it is called with the matched element. The default \a equal
 function does a pointer comparison.

 \b Examples:
    - removef (B, (A B C), 0, 0) => (A C)
    - removef (A, (A B A C), 0 , 0) => (B A C)

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_rm_rm.htm
 */
list
removef (void *atom, list l, int (*equal)(void *, void*), void (*destroy_atom)(void *))
{
    list s = l;
    list p;

    int eq (void *a, void *c)
    {
        return a == c;
    }

    if (!equal)
        equal = eq;

    if (l)
    {
        if (equal (atom, car (l)))
        {
            if (destroy_atom)
                destroy_atom (car (l));
            return cdrf (l);
        }

        p = l;
        while ((l = cdr (l)))
        {
            if (equal (atom, car (l)))
            {
                if (destroy_atom)
                    destroy_atom (car (l));
                p->next = cdrf (l);
                return s;
            }
            p = l;
        }
    }
    return s;
}

/**
 \brief Returns the nth cdr of the list.
 \param n Index into list.
 \param l List to index into.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return A list.

 \b Examples:
    - nthcdr (0, (A B C)) => (A B C)
    - nthcdr (1, (A B C)) => (B C)

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_nthcdr.htm
 */
void *
_nthcdr (int n, list l, const char *file, int line)
{
    libassert (n >= 0);
    while (n--)
        l = cdr (l);
    return l;
}

/**
 \brief Returns the nth car of the list.
 \param n Index into list.
 \param l List to index into.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return A list.

 \b Examples:
    - nth (0, (A B C)) => A
    - nth (1, (A B C)) => B

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_nth.htm
 */
void *
_nth (int n, list l, const char *file, int line)
{
    return car (_nthcdr (n, l, file, line));
}

/**
 \brief Frees all the cons cells in a list.
 \param l List to free.
 */
void
destroy (void *l)
{
    while (l)
        l = cdrf (l);
}

/**
 \brief Constructs an association list.
 \param key Pointer to the key.
 \param datum Pointer to the data.
 \param alist List to add to.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return The alist.

 Contructs an association list of the following form ->
    ((key datum)(key datum)...)

 \b Examples:
    - acons (k1, d1, ()) => ((k1 d1))
    - acons (k1, d1, ((k2 d2))) => ((k1 d1)(k2 d2))

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_acons.htm
 */
list
_acons (void *key, void *datum, list alist, const char *file, int line)
{
    return _cons (_cons (key, _cons (datum, 0, file, line), file, line), alist, file, line);
}

/**
 \brief Finds the datum for the given key.
 \param key Key to find.
 \param alist Alist to search.
 \param equal Optional function for comparing keys.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Matching sub-list.

 Searches the association list for the first matching key using the equal function.
 assoc returns the list starting at the matched sublist.

 \b Examples:
    - assoc (k2, ((k1 d1)(k2 d2))) => (k2 d2)

 \b See: http://www.lispworks.com/documentation/HyperSpec/Body/f_assocc.htm
 */
list
_assoc (void *key, list alist, int (*equal)(void *, void *), const char *file, int line)
{
    int eq (void *a, void *c)
    {
        libassert (a && c);
        return !strcmp (a, c);
    }

    if (!equal)
        equal = eq;

    while (alist)
    {
        if (equal (key, car (car (alist))))
            return car (alist);
        alist = cdr (alist);
    }
    return 0;
}

/**
 \brief Frees all the cons cells in a list.
 \param alist Alist to free.
 \param destroy_kd Optional function to call for each key-dataum pair.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 *
 If the destroy_kd function is supplied, it is called
 for each key-datum pair in the list.
 */
void
_destroy_alist (list alist, void (*destroy_kd)(void *, void *), const char *file, int line)
{
    while (alist)
    {
        if (destroy_kd)
            destroy_kd (_nth (0, car (alist), file, line), _nth (1, car (alist), file, line));
        destroy (car (alist));
        alist = cdrf (alist);
    }
}

/**
 \brief Print list statistics to a file.
 \param f File to print to.
 */
void
list_print_stats (FILE *f)
{
    unsigned int i = 0;
    list t = free_list;

    fprintf (f, "---List Statistics---------------------------------------------\n");
    fprintf (f, "   Number of cons cells allocated          -> %d\n", list_stats.num_alloc);
    fprintf (f, "   Number of cons cells freed              -> %d\n", list_stats.num_freed);
    fprintf (f, "   Number of duplicate frees               -> %d\n", list_stats.num_dup_free);
    fprintf (f, "   Number of cons cells in use             -> %d\n", list_stats.num_alloc - list_stats.num_freed);
    while (t)
    {
        t = t->next;
        i++;
    }
    fprintf (f, "   Free list length                        -> %d\n\n", i);
}


#if 0

void
pp (void * a)
{
    printf ("%s ", (char *)a);
}


void
printl (void * l)
{
    if (l)
    {
        printf ("(");
        mapcar (pp, l);
        printf ("\b)\n");
    }
    else
    {
        printf ("()\n");
    }
}

void
printla (void * l)
{
    if (l)
    {
        printf ("(");
        mapcar (pp, l);
        printf ("\b)");
    }
    else
    {
        printf ("()");
    }
}


void
printa (void * l)
{
    if (l)
    {
        printf ("(");
        if (car (l))
            mapcar (printla, l);
        printf (")\n");
    }
    else
    {
        printf ("()\n");
    }
}

int
eq (void *a, void *s)
{
    return !strcmp (a, s);
}

int
main (void)
{
    list l = 0, ll = 0, t = 0, a = 0;

    list_print_stats (stdout);

    l = cons ("one", 0);
    printl (l);
    l = cons ("two", l);
    printl (l);
    l = cons ("three", l);
    printl (l);

    list_print_stats (stdout);

    t = l;
    while (t)
    {
        printf ("(%s)\n", (char *)car (t));
        t = cdr (t);
    }

    ll = append (ll, cons ("uno", 0));
    printl (ll);
    ll = append (ll, cons ("dos", 0));
    printl (ll);
    ll = append (ll, cons ("tres", 0));
    printl (ll);

    list_print_stats (stdout);

    t = ll;
    while (t)
    {
        printf ("(%s)\n", (char *)car (t));
        t = cdr (t);
    }

    l = append (ll, l);
    printl (l);

    t = l;
    while (t)
    {
        printf ("(%s)\n", (char *)car (t));
        t = cdr (t);
    }

    printf ("(member tres ");
    printl (member ("tres", l, eq));
    printf (")\n");

    printf ("(member four ");
    printl (member ("four", l, eq));
    printf (")\n");

    printl (l);
    destroy (l);
    l = 0;
    printl (l);

    list_print_stats (stdout);

    ll = cons ("one", cons ("two", 0));
    l = cons (ll, 0);
    ll = cons ("three", cons ("four", 0));
    l = append (l, cons (ll, 0));
    mapcar (printl, l);
    mapcar (destroy, l);
    destroy (l);

    list_print_stats (stdout);

    l = cons ("one", 0);
    printl (l);
    l = cons ("two", l);
    printl (l);
    l = cons ("three", l);
    printl (l);

    l = removef ("t", l, eq, destroy);
    printl (l);

    list_print_stats (stdout);

    a = acons ("key1", "datum1", 0);
    printa (a);
    a = acons ("key2", "datum2", a);
    printa (a);
    a = acons ("key3", "datum3", a);
    printa (a);

    printl (assoc ("key1", a, 0));
    printl (assoc ("key2", a, 0));
    printl (assoc ("key3", a, 0));
    printl (assoc ("key4", a, 0));

    destroy_alist (a, 0);

    list_print_stats (stdout);

    return 0;
}

#endif

/*
 * End of $Id: list.c,v 1.4 2005/07/18 19:01:48 cmayncvs Exp $
 */
