/* ===[ $RCSfile: mq.c,v $ ]==============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: mq.c,v 1.7 2005/07/18 19:01:48 cmayncvs Exp $

 \brief Implements the message queuing functions.

 \note Should mq_write free the message buffer passed to it??
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/sockios.h>
#include <netdb.h>
#include <unistd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "buf.h"
#include "list.h"
#include "stats.h"
#include "msg.h"
#include "mq.h"
#include "gassert.h"

#if !defined SIOCINQ
/** \brief Define this for ESPAD.
 *  \note This may or may not actually work: Please test first! */
#define SIOCINQ     FIONREAD
#endif

/** Local queue's udp socket. */
static int mq_sock = -1;

/** Local queue's name. */
static char *local_name;

/** Default ip and port for each queue. */
static const struct _mqs
{
    char *name;     ///< Queue name.
    char *ip;       ///< Ip address of queue in dotted decimal.
    int port;       ///< UDP Port of queue.
} mq_defaults[] =
{
    {"escp",            "127.0.0.1", 4089},
    {"devcomm",         "127.0.0.1", 4090},
    {"gtp",             "127.0.0.1", 4091},
    {"x25",             "127.0.0.1", 4092},
    {"x42",             "127.0.0.1", 4093},
    {"serial-if",       "127.0.0.1", 4094},
    {"udp-interface",   "127.0.0.1", 4095},
    {"esc",             "127.0.0.1", 4096},
    {"conman",          "127.0.0.1", 4097},
    {"stty",            "127.0.0.1", 4098},
    {"x42pp",           "127.0.0.1", 4099},
    {"pad",             "127.0.0.1", 4100},
    {"msecp",           "127.0.0.1", 4101},
    {"wireless",        "127.0.0.1", 4102},
    {"dole",            "127.0.0.1", 4103},
    {"ethmon",          "127.0.0.1", 4104}
};

/** Number of entries in the mq_defaults struct. */
static const int number_mqs = sizeof (mq_defaults) / sizeof (struct _mqs);

/** Alist with key = queue name, datum = sockaddr_in struct. */
static list mq_addrs = 0;

/** Message queue statistics. */
static struct _mq_stats
{
    unsigned int num_sent;  /**< Number of writes */
    unsigned int num_read;  /**< Number of reads */
} mq_stats;

/**
 \brief Get the sockaddr_in struct from the cache list.
 \param mq_name Queue name to look up.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return The associated sockaddr_in struct.
 */
static struct sockaddr_in *
get_address (char *mq_name, const char *file, int line)
{
    list t;

    libassert (t = _assoc (mq_name, mq_addrs, 0, file, line));
    return _nth (1, t, file, line);
}

/**
 \brief Attempts to resolve queue name.
 \param mq_name Queue to resolve.
 \param local Sockaddr_in struct to fill in.
 \return Zero if queue name resolved successfully.

 Does a DNS look up for the IP address and looks in /etc/services for the
 port number.
 */
static int
use_default (const char *mq_name, struct sockaddr_in *local)
{
    struct hostent *hip;
    struct servent *sp;

    return 1;

    if ((hip = gethostbyname (mq_name)) == 0)
        return 1;

    local->sin_addr.s_addr = *((unsigned long *)hip->h_addr_list[0]);

    if ((sp = getservbyname (mq_name, "udp")) == 0)
        return 1;
    local->sin_port = sp->s_port;

    return 0;
}


/**
 \brief Builds the queue address cache.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 First attempts to resolve the ip and port. If that fails the default address
 from mq_defaults is used.
 */
static void
make_address_list (const char *file, int line)
{
    int i;
    struct sockaddr_in *local;

    for (i = 0; i < number_mqs; i++)
    {
        libassert (local = malloc (sizeof (struct sockaddr_in)));
        local->sin_family = AF_INET;
        if (use_default (mq_defaults[i].name, local))
        {
            libassert (inet_aton (mq_defaults[i].ip, &(local->sin_addr)) !=0);
            local->sin_port = htons (mq_defaults[i].port);
        }
        mq_addrs = _acons ((char *)mq_defaults[i].name, local, mq_addrs, file, line);
    }
}

/**
 \brief Creates the local queue.
 \param mq_name Name of the local queue.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_mq_create (char *mq_name, const char *file, int line)
{
    struct sockaddr_in *local;
    int on = 1;

    local_name = mq_name;
    if (mq_addrs == 0)
        make_address_list (file, line);
    libassert ((mq_sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) != -1);
    libassert (setsockopt (mq_sock, SOL_SOCKET, SO_BSDCOMPAT, &on, sizeof (on)) != -1);
    local = get_address (mq_name, file, line);
    libassert (bind (mq_sock, (struct sockaddr *)local,
        sizeof (struct sockaddr_in)) != -1);
    _stats_init (file, line);
}

/**
 \brief Returns the file descriptor for the local queue.
 \return The queue's socket.

 This is so processes with multiple file descriptors can use select instead
 of threads.
 */
int
mq_get_socket (void)
{
    return mq_sock;
}

/**
 \brief Get local queue name.
 \return Local queue name.
 */
char *
mq_local_name (void)
{
    return local_name;
}

/**
 \brief Free resource used by the queue system.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_mq_destroy (const char *file, int line)
{
    void dkd (void *mq_name, void *local)
    {
        mq_name = 0;
        free (local);
    }

    _destroy_alist (mq_addrs, dkd, file, line);
    close (mq_sock);
}

/**
 \brief Put a message on a queue.
 \param mq_dest Name of destination queue.
 \param b Message to send.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 If \a mq_dest is null the message to put on the local queue.
 \warning The buffer \a b is freed.
 */
buf_t
_mq_write (char *mq_dest, buf_t b, const char *file, int line)
{
    int s;
    struct sockaddr_in *to;

    if (!mq_dest)
        mq_dest = local_name;

    if (mq_addrs == 0)
        make_address_list (file, line);

    if (mq_sock == -1)
        libassert ((s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) != -1);
    else
        s = mq_sock;

    to = get_address (mq_dest, file, line);
    libassert (WITH_RESTART(sendto (s, _buf_data (b, file, line), buf_length (b), MSG_NOSIGNAL,
        (struct sockaddr *)to, sizeof (struct sockaddr_in))) != -1);

    if (mq_sock == -1)
        close (s);

    _buf_free (b, file, line);
    mq_stats.num_sent++;
    return 0;
}

/**
 * \brief Broadcast a message to all queues.
 * \param b Message to broadcast.
 * \param file The file this call was made from.
 * \param line The line in the file where this call was made.
 * \return Null.
 */
buf_t
_mq_broadcast (buf_t b, const char *file, int line)
{
    int i;

    for (i = 0; i < number_mqs; i++)
        _mq_write (mq_defaults[i].name, _buf_dup (b, file, line), file, line);
    return _buf_free (b, file, line);
}

/**
 \brief Check if any messages are in the local queue.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Zero if no message are in the queue, non-zero otherwise.
 */
int
_mq_peek (const char *file, int line)
{
    int length = 0;

    libassert (ioctl (mq_sock, SIOCINQ, &length) != -1);
    return length;
}

/**
 \brief Print queue statistics.
 \param f File to print to.
 */
void
mq_print_stats (FILE *f)
{
    fprintf (f, "---Message Queue Statistics-----------------------"
        "-------------\n");
    fprintf (f, "   Number of writes                        -> %d\n",
        mq_stats.num_sent);
    fprintf (f, "   Number of reads                         -> %d\n\n",
        mq_stats.num_read);
}

/**
 \brief Send a sync-ack message.
 \param name Queue to send ack to.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
static void
send_sync_ack (char *name, const char *file, int line)
{
    _mq_write (name, _msg_create (name, local_name, "sync-ack", file, line), file, line);
}

/**
 \brief Get next message from queue.
 \param msec Number of milliseconds to wait for a new message.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Buffer read or zero on timeout.
 */
static buf_t
read_next_msg (unsigned int msec, const char *file, int line)
{
    int length;
    char temp [MAX_MESSAGE_LENGTH];
    struct timeval tv;
    fd_set rfds;
    int rt;

    if (msec)
    {
        for (;;)
        {
            tv.tv_sec = msec / 1000;
            tv.tv_usec = (msec % 1000) * 1000;
            FD_ZERO (&rfds);
            FD_SET (mq_sock, &rfds);

            rt = select (mq_sock + 1, &rfds, 0, 0, &tv);
            libassert ((rt >= 0) || (rt == -1 && errno == EINTR));
            if (rt == -1)
                continue;

            if (!FD_ISSET (mq_sock, &rfds))
                return 0;
            else
                break;
        }
    }

    libassert (WITH_RESTART(length = recvfrom (mq_sock, temp,
        MAX_MESSAGE_LENGTH, MSG_NOSIGNAL, 0, 0)) != -1);

    mq_stats.num_read++;
    return _buf_append (0, temp, length, file, line);
}

/**
 \brief Check if message is a sync message.
 \param b Message to check.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Non-zero if message is a sync or sync-ack.
 */
static int
is_a_sync_msg (buf_t b, const char *file, int line)
{
    if (_msg_is_type (b, "sync", file, line))
    {
        send_sync_ack (_msg_get_source (b, file, line), file, line);
        return 1;
    }
    else if (_msg_is_type (b, "sync-ack", file, line))
    {
        return 1;
    }
    return 0;
}

/**
 \brief Sync with another queue.
 \param name Queue to sync with.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_mq_sync_with (char *name, const char *file, int line)
{
    buf_t b;
    buf_q q = 0;

    for (;;)
    {
        _mq_write (name, _msg_create (name, local_name, "sync", file, line), file, line);
        if ((b = read_next_msg (500, file, line)) == 0)
            continue;
        if (is_a_sync_msg (b, file, line))
        {
            if (_msg_is_from (b, name, file, line))
            {
                _buf_free (b, file, line);
                break;
            }
            _buf_free (b, file, line);
        }
        else
        {
            _buf_queue (&q, b, file, line);
        }
    }
    while ((b = _buf_dequeue (&q, file, line)))
        _mq_write (local_name, b, file, line);
}

/**
 \brief Read next message from queue with auto-restarting on sync messages.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Buffer read or zero if sync or sync-ack read.
 */
buf_t
_mq_read_without_restart (const char *file, int line)
{
    buf_t b;

    b = read_next_msg (0, file, line);
    if (is_a_sync_msg (b, file, line))
        return _buf_free (b, file, line);
    return b;
}

/**
 \brief Reads next message from the queue.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return The message.
 */
buf_t
_mq_read (const char *file, int line)
{
    buf_t b;

    while ((b = _mq_read_without_restart (file, line)) == 0);
    return b;
}

/**
 * \brief Checks if the the name is a valid queue name.
 * \param name Name to check.
 * \return One if valid queue name or zero if not.
 */
int
mq_is_a_queue (const char *name)
{
    int i;

    for (i = 0; i < number_mqs; i++)
    {
        if (strcmp (name, mq_defaults[i].name) == 0)
            return 1;
    }
    return 0;
}



#if 0

int
main (int argc, char *argv[])
{
    buf_t b, sb;
    int len;
    char *t;
    int i;
    char *name;

    if (argc > 1)
        name = argv[1];
    else
        name = "gtp";

    mq_write ("gtp", msg_create ("dest", "source", "test"));

    mq_create (name);

    b = msg_create ("test-dest", "test-source", "test-type");
    b = msg_add_field (b, "test-field", "test-value", 11);
    b = msg_add_field (b, "test-field-two", "test-value-two", 15);
    b = msg_add_field (b, "test-field-three", "test-value-three", 0);
    b = msg_add_field (b, "test-field-four", "test-value-four", 16);

    printf ("(type %s)\n", msg_get_type (b));
    printf ("(dest %s)\n", msg_get_dest (b));
    printf ("(source %s)\n", msg_get_source (b));
    //printf ("(skip %p)\n", skip_ahead (b, buf_data (b), 4));

    t = msg_get_field (b, "test-field", &len);
    printf ("(field %s %d)\n", t, len);

    t = msg_get_field (b, "test-field-two", &len);
    printf ("(field-two %s %d)\n", t, len);

    t = msg_get_field (b, "test-field-three", &len);
    printf ("(field-three %s %d)\n", t, len);

    t = msg_get_field (b, "test-field-four", &len);
    printf ("(field-four %s %d)\n", t, len);

    printf ("(peek %d)\n", mq_peek ());

    mq_write (name, b);

    printf ("(peek %d)\n", mq_peek ());

    b = mq_read ();


    //mq_destroy ();


    buf_print (b);
    msg_print (b);

    buf_free (b);

    b = msg_create ("test-dest", "test-source", "test-type");
    t = msg_get_field (b, "test-field", &len);
    printf ("(field %s %d)\n", t, len);
    msg_print (b);

    for (i = 0; i < 255; i++)
    {
        sb = buf_dup (b);
        mq_write(name, sb);
    }
    printf ("Wrote %d\nRead ", i);
    for (i = 0; i < 255; i++)
    {
        buf_free (mq_read ());
        printf ("%d ", i+1);
    }
    printf ("\n");

    mq_broadcast (msg_create ("b-dest", "b-source", "b-test"));

    msg_print (mq_read ());

    /*if (argc > 2)
    {
        mq_sync_with (argv[2]);
        printf ("Synced with %s\n", argv[2]);
    }
    if (argc > 3)
    {
        mq_write (argv[2], msg_create (argv[2], name, "testing-sync"));
        mq_read_without_restart ();
    }
    else
    {
        mq_sync_with (argv[2]);
        msg_print (mq_read ());
        printf ("Again synced with %s\n", argv[2]);
    } */

    return 0;
}

#endif

/*
 * End of $Id: mq.c,v 1.7 2005/07/18 19:01:48 cmayncvs Exp $
 */

