/* ===[ $RCSfile: msg.c,v $ ]================================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: msg.c,v 1.4 2005/07/18 19:01:48 cmayncvs Exp $

 \brief Implements IMP message construction and parsing functions.

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "buf.h"
#include "list.h"
#include "mq.h"
#include "msg.h"
#include "gassert.h"
#include "gdebug.h"

static void
default_proc (buf_t, const char *file, int line);

/** List of function to call in msg_dispatch. */
static list dispatch_list = 0;
/** Default dispatch function. */
static void (*default_dispatch_proc)(buf_t, const char*, int) = default_proc;

/** Message statistics. */
static struct _msg_stats
{
    size_t max_msg_length;  ///< Max message length
} msg_stats;

/**
 \brief Adds message header to a buffer.
 \param source Source queue name.
 \param dest Destination queue name.
 \param type Message type.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return The message buffer.

 If \a source or \a dest is null the local queue name is used.
 */
buf_t
_msg_create (char *dest, char *source, char *type, const char *file, int line)
{
    buf_t msg = 0;

    if (!source)
        source = mq_local_name ();
    if (!dest)
        dest = mq_local_name ();

    msg = _buf_append (msg, type, strlen (type) + 1, file, line);
    msg = _buf_append (msg, dest, strlen (dest) + 1, file, line);
    msg = _buf_append (msg, source, strlen (source) + 1, file, line);
    return msg;
}

/**
 \brief Adds a message field to a buffer.
 \param msg Message to add to.
 \param field Message field name.
 \param data Field data.
 \param length Field data length.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return The message buffer.

 If \a length is zero the data is treated as a string.
 */
buf_t
_msg_add_field (buf_t msg, char *field, char *data, unsigned int length, const char *file, int line)
{
    char t[10];

    libassert (data);

    if (length == 0)
        length = strlen (data) + 1;

    snprintf (t, 10, "%d", length);

    msg = _buf_append (msg, field, strlen (field) + 1, file, line);
    msg = _buf_append (msg, t, strlen (t) + 1, file, line);
    msg = _buf_append (msg, data, length, file, line);
    msg_stats.max_msg_length = MAX (buf_length (msg), msg_stats.max_msg_length);
    return msg;
}

/**
 \brief Move ahead N strings in a message.
 \param b Message buffer.
 \param msg Pointer to current location in the buffer.
 \param n Number of strings to skip ahead.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Pointer to Nth string from \a msg.
 */
static char *
skip_ahead (buf_t b, char *msg, int n, const char *file, int line)
{
    char *t;

    t = msg;
    while (n--)
    {
        if ((t = memchr (t, '\0', buf_length (b) - (t - ((char *)_buf_data (b, file, line))))))
            t++;
        else
            return 0;
    }
    return t;
}

/**
 \brief Returns message type.
 \param msg Message buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Message type.
 */
char *
_msg_get_type (buf_t msg, const char *file, int line)
{
    libassert (msg);
    return skip_ahead (msg, _buf_data (msg, file, line), 0, file, line);
}

/**
 \brief Returns destination queue name.
 \param msg Message buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Destination queue name.
 */
char *
_msg_get_dest (buf_t msg, const char *file, int line)
{
    libassert (msg);
    return skip_ahead (msg, _buf_data (msg, file, line), 1, file, line);
}

/**
 \brief Returns source queue name.
 \param msg Message buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Source queue name.
 */
char *
_msg_get_source (buf_t msg, const char *file, int line)
{
    libassert (msg);
    return skip_ahead (msg, _buf_data (msg, file, line), 2, file, line);
}

/**
 \brief Returns a message field.
 \param msg Message buffer to parse.
 \param field Field name.
 \param length Returned field length.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Pointer to the start of the field data.

 \note It is valid to attempt to get fields that do not exist in the message.
 If the field does not exist \c null is returned and \a length is set to 0.
 */
char *
_msg_get_field (buf_t msg, char *field, unsigned int *length, const char *file, int line)
{
    char *name;
    char *len;
    char *data;

    libassert (msg);
    libassert (name = skip_ahead (msg, _buf_data (msg, file, line), 3, file, line));
    for (;;)
    {
        if ((len = skip_ahead (msg, name, 1, file, line)))
            *length = strtol (len, 0, 10);
        else
        {
            *length = 0;
            return 0;
        }
        data = skip_ahead (msg, len, 1, file, line);
        if (strcmp (name, field) == 0)
            return data;
        name = data + *length;
    }
}

/**
 \brief Gets the nth field of a message.
 \param b Message buffer.
 \param index Zero relative array reference.
 \param name Pointer to a pointer to return the name string.
 \param data Pointer to a pointer to retunr the data.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return The length of the data or zero if the index is beyond the bounds of the message.
 */
unsigned int
_msg_aref (buf_t b, unsigned int index, char **name, char **data, const char *file, int line)
{
    char *f, *l;
    unsigned int length;

    libassert (b && name && data);

    f = skip_ahead (b, _buf_data (b, file, line), 3, file, line);
    do
    {
        *name = f;
        if ((l = skip_ahead (b, f, 1, file, line)))
            length = strtol (l, 0, 10);
        else
            return 0;
        *data = skip_ahead (b, l, 1, file, line);
        f = *data + length;
    } while (index--);
    return length;
}

/** \def P Macro for print function. */
#define P f_debug_writemsg
/**
 \brief Pretty prints a message.
 \param b Message buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_msg_print (buf_t b, const char *file, int line)
{
    int i = 0;
    char *name, *data;
    unsigned int length;

    //#define P printf

    P ("IMP Message\n");
    P ("Type -> %s\n", _msg_get_type (b, file, line));
    P ("Source -> %s\n", _msg_get_source (b, file, line));
    P ("Destination -> %s\n", _msg_get_dest (b, file, line));

    while ((length = _msg_aref (b, i++, &name, &data, file, line)))
    {
        P ("Field  -> %s (%d)\n", name, length);
        print_hex_data (data, length);
    }
}
#undef P

/**
 \brief Compare message source with given name.
 \param msg Message.
 \param from From name.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Non-zero if same.
 */
int
_msg_is_from (buf_t msg, char *from, const char *file, int line)
{
    return !strcmp (from, _msg_get_source (msg, file, line));
}

/**
 \brief Compare message type with given type.
 \param msg Message.
 \param type Type name.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Non-zero if same.
 */
int
_msg_is_type (buf_t msg, char *type, const char *file, int line)
{
    return !strcmp (type, _msg_get_type (msg, file, line));
}

/**
 \brief Register a new procedure for a given message type.
 \param type Message type.
 \param proc Procedure to call.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_msg_dispatch_proc (char *type, void (*proc)(buf_t), const char *file, int line)
{
    libassert (type && proc);
    dispatch_list = _acons (type, proc, dispatch_list, file, line);
}

/**
 \brief Register a new default procedure.
 \param proc Default procedure to call.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_msg_default_dispatch_proc (void (*proc)(buf_t, const char*, int), const char *file, int line)
{
    libassert (proc);
    default_dispatch_proc = proc;
}

/**
 \brief Send message to a procedure.
 \param msg Message to send.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 The list of registered procedures is searched for the message type and the
 matching function is called with \a msg as its arguement. If the message
 type is not found the default procedure is called.

 This is function is designed to avoid large if/else or switches in the
 protocol implementations. Instead you would write code that looks like
 this.

 \code
    int main (void)
    {
        // Do initialization
        ...
        msg_dispatch_proc ("timeout", timeout_proc);
        msg_dispatch_proc ("transmit-req", tx_req_proc);
        ...
        for (;;)
            msg_dispatch (mq_read ());
    }

    void timeout_proc (buf_t msg)
    {
        to_flag++;
        buf_free (msg);
    }
 \endcode

 \note You must free the message buffer passed in called procedure.
 */
void
_msg_dispatch (buf_t msg, const char *file, int line)
{
    list t;
    void (*proc)(buf_t);

    if (msg)
    {
        if ((t = _assoc (_msg_get_type (msg, file, line), dispatch_list, 0, file, line)))
        {
            proc = _nth (1, t, file, line);
            proc (msg);
        }
        else
        {
            default_dispatch_proc (msg, file, line);
        }
        _buf_free (msg, file, line);
    }
}

/**
 \brief Default dispatch procedure.
 \param msg Message buffer.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.

 This function may be replaced with your own by calling msg_default_dispatch_proc.
 */
static void
default_proc (buf_t msg, const char *file, int line)
{
    //printf ("In default dispatch procedure "__func__":\n");
    _msg_print (msg, file, line);
}

/**
 \brief Prints message statistics to a file.
 \param f File to print to.
 */
void
msg_print_stats (FILE *f)
{
    list t = dispatch_list;

    fprintf (f, "---Message Statistics------------------------------------------\n");
    fprintf (f, "   Maximum message length                  -> %d\n", msg_stats.max_msg_length);
    fprintf (f, "   Message dispatch list -> (");
    while (t)
    {
        fprintf (f, "%s", (char *)car (car (t)));
        t = cdr (t);
        if (t)
            fprintf (f, " ");
    }
    fprintf (f, ")\n\n");
}



#if 0

void
test_proc (buf_t msg)
{
    printf ("In test procedure "__func__":\n");
    msg_print (msg);
}

int
main (void)
{
    buf_t b;
    int len;
    char *t;

    msg_dispatch_proc ("test-type", test_proc);

    b = msg_create ("test-dest", "test-source", "test-type");
    b = msg_add_field (b, "test-field", "test-value", 11);
    b = msg_add_field (b, "test-field-two", "test-value-two", 15);
    b = msg_add_field (b, "test-field-three", "test-value-three", 0);
    b = msg_add_field (b, "test-field-four", "test-value-four", 16);

    printf ("(type %s)\n", msg_get_type (b));
    printf ("(dest %s)\n", msg_get_dest (b));
    printf ("(source %s)\n", msg_get_source (b));
    //printf ("(skip %p)\n", skip_ahead (b, buf_data (b), 4));

    t = msg_get_field (b, "test-field", &len);
    printf ("(field %s %d)\n", t, len);

    t = msg_get_field (b, "test-field-two", &len);
    printf ("(field-two %s %d)\n", t, len);

    t = msg_get_field (b, "test-field-three", &len);
    printf ("(field-three %s %d)\n", t, len);

    t = msg_get_field (b, "test-field-four", &len);
    printf ("(field-four %s %d)\n", t, len);

    msg_dispatch (b);

    b = msg_create ("test-dest", "test-source", "test-type-default");
    t = msg_get_field (b, "test-field", &len);
    printf ("(field %s %d)\n", t, len);
    msg_print (b);

    msg_dispatch (b);

    msg_print_stats (stdout);

    return 0;
}

#endif

/*
 * End of $Id: msg.c,v 1.4 2005/07/18 19:01:48 cmayncvs Exp $
 */


