/* ===[ $RCSfile: stats.c,v $ ]==============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: stats.c,v 1.5 2005/02/14 02:00:22 cmayncvs Exp $

 \brief Implements a statistics monitoring function.

 A signal handler is installed for SIGUSR2. When ever the signal is
 recieved the stats are print to the file /tmp/linux-comm-stats.

 To use it look up the process id and do:

 kill -s SIGUSR2 pid
 cat /tmp/linux-comm-stats
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>
#include <ctype.h>
#include "list.h"
#include "buf.h"
#include "msg.h"
#include "mq.h"
#include "timer.h"
#include "fd.h"
#include "stats.h"
#include "gassert.h"

static char *
get_proc_name (long pid);

/** \brief The struct writer structure. */
static struct
{
    char fname [200];   /**< File name */
    char *buf;          /**< Buffer */
    int length;         /**< Length of buffer */
} writer;

/** List of user stat functions. */
static list proc_list = 0;

/**
 \brief Prints stats to a file.
 \param signo The signal number.
 */
static void
signal_handler (int signo)
{
    FILE *f;
    void caller (void *proc) {((void(*)(FILE *))proc) (f);}

    signo = 0;

    if (writer.fname[0])
    {
        assert (f = fopen (writer.fname, "w"));
        fwrite (writer.buf, 1, writer.length, f);
        fclose (f);
    }

    assert (f = fopen ("/tmp/linux-comm-stats", "w"));
    list_print_stats (f);
    buf_print_stats (f);
    msg_print_stats (f);
    mq_print_stats (f);
    timer_print_stats (f);
    fd_print_stats (f);
    mapcar (caller, proc_list);
    fclose (f);
}

/**
 \brief Initalizes the signal hander.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_stats_init (const char *file, int line)
{
    struct sigaction sa;

    memset (&sa, 0, sizeof (struct sigaction));
    sa.sa_handler = signal_handler;
    libassert (sigaction (SIGUSR2, &sa, 0) != -1);
}

/**
 \brief Register your own statistic function.
 \param proc Function to call.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 */
void
_stats_add_proc (void (*proc)(FILE *), const char *file, int line)
{
    libassert (proc);
    proc_list = _cons (proc, proc_list, file, line);
}

/**
 \brief Adds a struct writer.
 \param buf The buffer holding the data for the struct writer to write.
 \param length The number of bytes in the \a buf to write.
 \return Nothing.
 */
void
stats_add_struct_writer (void *buf, int length)
{
    snprintf (writer.fname, sizeof (writer.fname), "/tmp/%s-stat-struc", get_proc_name (getpid ()));
    writer.buf = buf;
    writer.length = length;
}

/**
 \brief Gets statistics, similar to Altura 'stats' shell script.
 \param process The process name to get statistics for.
 \param buf The buffer to hold the statistics data.
 \param count The maximum number of bytes to read.
 \param file The file this call was made from.
 \param line The line number this call was made from.
 \return The number of bytes read.
 */
int
_stats_get (char *process, void *buf, int count, const char *file, int line)
{
    char fname [200];
    int i;
    FILE *f = 0;
    const int ntries = 10;

    snprintf (fname, sizeof (fname), "/tmp/%s-stat-struc", process);
    unlink (fname);
    kill (_get_pid (process, file, line), SIGUSR2);

    for (i = 0; i < ntries; i++)
    {
        sleep (1);
        if ((f = fopen (fname, "r")))
            break;
    }
    if (i == ntries)
        return 0;

    i = fread (buf, 1, count, f);
    fclose (f);
    return i;
}

/**
 *  \brief Gets the process's name, based on the process's ID, \a pid.
 *  \param pid The process ID.
 *  \return The process name associated with the given \a pid.
 */
static char *
get_proc_name (long pid)
{

    char fn [100];
    char ln [100];
    static char nm [100];
    FILE *f;

    snprintf (fn, sizeof (fn) - 1, "/proc/%ld/status", pid);
    if ((f = fopen (fn, "r")) == 0)
        return "";

    if (fgets (ln, sizeof (ln) - 1, f) == 0)
    {
        fclose (f);
        return "";
    }
    fclose (f);
    sscanf (ln, "%*s %s", nm);
    return nm;
}

/**
 *  \brief Gets the process's ID, based on the process name, \a pname.
 *  \param pname The process name.
 *  \param file The file this call was made from.
 *  \param line The line number this call was made from.
 *  \return The process ID associated with the given \a pname.
 */
long
_get_pid (char *pname, const char *file, int line)
{
    DIR *dir;
    struct dirent *ent;
    long tpid, pid = 0;

    libassert (dir = opendir ("/proc"));
    while ((ent = readdir (dir)))
    {
        if (!isdigit (*ent->d_name))
            continue;

        tpid = strtol (ent->d_name, 0, 10);
        if (strcmp (get_proc_name (tpid), pname) == 0)
            pid = tpid;
    }
    closedir (dir);
    return pid;
}


#if 0
void
test_proc (FILE *f)
{
    fprintf (f, "(Test-proc-statistics)\n");
}

int
main (void)
{
    buf_t b;
    mq_create ("gtp");
    stats_add_proc (test_proc);
    msg_dispatch_proc ("test", test_proc);
    msg_dispatch_proc ("test-more", test_proc);

    for (;;)
    {
        b = msg_create ("dest", "source", "type");
        b = msg_add_field (b, "field", "data", 0);
        mq_write ("gtp", b);
        b = mq_read ();
        buf_free (b);
        timer_stop (timer_start (1000, 0));
    }

    return 0;
}

#endif

#if 0

int
main (void)
{
    long pid;

    pid = get_pid ("sshd");
    printf ("sshd pid = %d\n", pid);
    return 0;
}

#endif


/*
 * End of $Id: stats.c,v 1.5 2005/02/14 02:00:22 cmayncvs Exp $
 */


