/* ===[ $RCSfile: str.c,v $ ]================================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: str.c,v 1.4 2005/04/09 00:23:48 cmayncvs Exp $

 \brief Implements misc string functions.
 */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "list.h"
#include "str.h"

/* ======================================================================= */
/**
 *  \brief Recursively splits a string, based on \a sep, and builds a list.
 *  \param str The string to split
 *  \param sep The separator character, e.g. ';'
 *  \param dup A boolean flag to duplicate each split string or not.
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *  \return The list of split string pieces.
 */
/* ======================================================================= */
static list
_split (char *str, char *sep, int dup, const char *file, int line)
{
    list l = 0;
    char *s;

    while ((s = strsep (&str, sep)))
    {
        if (strlen(s))
        {
            if (dup)
                s = strdup (s);
            l = append (l, _cons (s, 0, file, line));
        }
    }

    return l;
}

/* ======================================================================= */
/**
 *  \brief Calls _split() to create a list of strings, split by \a sep.  It
 *      does not allocate duplicate strings.
 *  \param str The string to split
 *  \param sep The separator character, e.g. ';'
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *  \return The list of split string pieces.
 */
/* ======================================================================= */
list
_str_split (char *str, char *sep, const char *file, int line)
{
    return _split (str, sep, 0, file, line);
}


/* ======================================================================= */
/**
 *  \brief Calls _split() to create a list of strings, split by \a sep.  It
 *      allocates duplicate strings.
 *  \param str The string to split
 *  \param sep The separator character, e.g. ';'
 *  \param file The file this call was made from.
 *  \param line The line in the file where this call was made.
 *  \return The list of split string pieces.
 */
/* ======================================================================= */
list
_str_split_dup (char *str, char *sep, const char *file, int line)
{
    return _split (str, sep, 1, file, line);
}

/* ======================================================================= */
/**
 *  \brief Converts each character within the string \a s to upper-case.  The
 *      original string is not modified, but this function is not reentrant.
 *      It uses a global static buffer to store the string, and subsequent
 *      calls will overwrite the previous string.  If \a s is longer than 256
 *      characters, it will be truncated to that length.
 *  \param s The string to "upperfy"
 *  \return A pointer to the static buffer containing the "upperfied" string.
 */
/* ======================================================================= */
char *
str_upper (char *s)
{
    static char u[256];
    int i, l = strlen (s) > 255 ? 255 : strlen (s);

    for (i = 0; i < l; i++)
        u[i] = toupper (s[i]);
    u[i] ='\0';

    return u;
}


/* ======================================================================= */
/**
 *  \brief Chomps whitespace from the string \a s.  Only characters meeting
 *      isalnum()'s criteria will be copied to a new string.  The original
 *      string is not modified, but this function is not reentrant.  It uses a
 *      global static buffer to store the string, and subsequent calls will
 *      overwrite the previous string.  If \a s is longer than 256 characters,
 *      it will be truncated to that length.
 *  \param s The string to chomp.
 *  \return A pointer to the static buffer containing the "chomped" string.
 */
/* ======================================================================= */
char *
str_chomp (char *s)
{
    static char c[256];
    int i, j, l = strlen (s) > 255 ? 255 : strlen (s);

    for (i = 0, j = 0; i < l; i++)
    {
        if (isalnum (s[i]))
            c[j++] = s[i];
    }
    c[i] = '\0';

    return c;
}


/* ======================================================================= */
/**
 *  \brief Converts seconds to a string in format: DDd HHh MMn SSs.
 *  \param secs The seconds to convert.
 *  \return The string containing the formatted time.
 */
/* ======================================================================= */
char *sec2str(unsigned long secs)
{
    static char tbuf[256];
    unsigned int days, hours, mins;

    if ( secs < 60 )
        snprintf(tbuf, sizeof(tbuf), "%us", (int)secs);
    else if ( secs < 3600 )
    {
        mins = secs / 60;
        secs = secs % 60;
        snprintf(tbuf, sizeof(tbuf), "%um %us", mins, (int)secs);
    }
    else if ( secs < 86400 )
    {
        hours = secs / 3600;
        secs = secs % 3600;
        mins = secs / 60;
        secs = secs % 60;
        snprintf(tbuf, sizeof(tbuf), "%uh %um %us", hours, mins, (int)secs);
    }
    else
    {
        days = secs / 86400;
        secs = secs % 86400;
        hours = secs / 3600;
        secs = secs % 3600;
        mins = secs / 60;
        secs = secs % 60;
        snprintf(tbuf, sizeof(tbuf), "%ud %uh %um %us", days, hours, mins,
            (int)secs);
    }
    tbuf[sizeof(tbuf)-1] = '\0';
    return (tbuf);

} /* sec2str() */


/* ======================================================================= */
/**
 *  \brief Safely print text, accounting for possible non-printable char's.
 *  \param txt The text to print
 *  \param len The length of the data
 *  \return Nothing.
 */
/* ======================================================================= */
void print_text(const unsigned char *txt, unsigned int len)
{
    int i;

    for ( i = 0; i < len; i++ )
        if ( isprint(txt[i]) )
            putchar(txt[i]);
        else
            printf("\\x%02x", txt[i]);
} /* print_text() */


/* ======================================================================= */
/**
 *  \brief Safely print null-terminated text, accounting for
 *      possible non-printable char's.
 *  \param txt The text to print
 *  \return Nothing.
 */
/* ======================================================================= */
void print_textz(const unsigned char *txt)
{
    int i;

    for ( i = 0; txt[i]; i++ )
        if ( isprint(txt[i]) )
            putchar(txt[i]);
        else
            printf("\\x%02x", txt[i]);
} /* print_textz() */


#if 0
void
print (char *s)
{
    int i;

    for (i = 0; i < strlen(s); i++)
        printf ("%2.2X", s[i]);
    printf ("\n");
}

int
main (void)
{
    char *s;

    s = "  test  ";
    print (s);
    print (str_chomp (s));

    s = "\n\rtest.123  ";
    print (s);
    print (str_chomp (s));

    return 0;
}
#endif

#if 0

void
print (list l)
{
    void p (void *s)
    {
        printf ("%s ", (char *)s);
    }
    mapcar (p, l);
    printf ("\n");
}

int
main (void)
{
    list l;
    char *s;

    s = strdup ("t1;t2;t3");
    l = str_split (s, ";");
    print (l);

    s = strdup ("t1;t2;t3;");
    l = str_split (s, ";");
    print (l);

    s = strdup (";t1;t2;t3");
    l = str_split (s, ";");
    print (l);

    s = strdup ("t1;t2;t3");
    l = str_split (s, " ");
    print (l);

    s = strdup ("t1=t2;t3=t4");
    l = str_split (s, ";");
    print (l);

    s = strdup ("t1;;t2;;t3");
    l = str_split (s, ";");
    print (l);

    s = strdup ("  t1; t2; t3 ");
    l = str_split (s, "; ");
    print (l);

    ///////////
    printf ("\n");

    s = strdup ("t1;t2;t3");
    l = str_split_dup (s, ";");
    print (l);

    s = strdup ("t1;t2;t3;");
    l = str_split_dup (s, ";");
    print (l);

    s = strdup (";t1;t2;t3");
    l = str_split_dup (s, ";");
    print (l);

    s = strdup ("t1;t2;t3");
    l = str_split_dup (s, " ");
    print (l);

    s = strdup ("t1=t2;t3=t4");
    l = str_split_dup (s, ";");
    print (l);

    return 0;
}

#endif


/*
 * End of "$Id: str.c,v 1.4 2005/04/09 00:23:48 cmayncvs Exp $".
 */


