/* ===[ $RCSfile: timer.c,v $ ]===========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2002-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file

 $Id: timer.c,v 1.3 2005/07/18 19:01:48 cmayncvs Exp $

 \brief Implements the standard timer functions.

 \note Timer are one-shots.

 These functions use Bill's timer library.
 */

#include <stdio.h>
#include <stdlib.h>
#include "buf.h"
#include "mq.h"
#include "msg.h"
#include "timer.h"
#include "gassert.h"

/** Timer statistics. */
static struct _timer_stats
{
    unsigned int num_started;
    unsigned int num_stopped;
    unsigned int num_expired;
} timer_stats;

/**
 \brief Default timeout handler function.
 \param arg Id from timer_start.
 */
static void
timeout (void *arg)
{
    buf_t msg;
    unsigned long id = (unsigned long)arg;

    msg = msg_create (0, "timer", "timeout");
    msg = msg_add_field (msg, "id", (char *)&id, 4);
    mq_write (0, msg);
    timer_stats.num_expired++;
}

/**
 \brief Starts a new timer.
 \param msec Timeout in milliseconds.
 \param id Id included in timeout message.
 \param file The file this call was made from.
 \param line The line in the file where this call was made.
 \return Timer id.

 When a timer expires a timeout message is written to the local queue.
 */
timer
_timer_start (unsigned long msec, unsigned long id, const char *file, int line)
{
    timer t;

    libassert ((t = lc_timer_start (msec, timeout, (void *)id)) != LC_TIMER_COOKIE_FAIL);
    timer_stats.num_started++;
    return t;
}

/**
 \brief Stops a timer.
 \param t Timer id.
 \return Null timer.
 */
timer
timer_stop (timer t)
{
    if (t)
    {
        if (!lc_timer_cancel (t))
            timer_stats.num_stopped++;
    }
    return 0;
}

/**
 \brief Print timer statistics.
 \param f File to print to.
 */
void
timer_print_stats (FILE *f)
{
    fprintf (f, "---Timer Statistics--------------------------------------------\n");
    fprintf (f, "   Number started                          -> %d\n", timer_stats.num_started);
    fprintf (f, "   Number stopped                          -> %d\n", timer_stats.num_stopped);
    fprintf (f, "   Number expired                          -> %d\n\n", timer_stats.num_expired);
}



#if 0

int
main (void)
{
    timer t;
    buf_t b;
    int length;
    unsigned long *id;

    mq_create ("gtp");

    t = timer_start (2000, 0xAABBCCDD);

    b = mq_read ();

    timer_stop (t);

    msg_print (b);

    assert (msg_is_from (b, "timer"));
    assert (msg_is_type (b, "timeout"));

    id = (unsigned long *)msg_get_field (b, "id", &length);
    assert (length == 4);
    assert (*id == 0xAABBCCDD);

    t = timer_start (5000, 666);
    timer_stop (t);

    return 0;
}

#endif

/*
 * End of $Id: timer.c,v 1.3 2005/07/18 19:01:48 cmayncvs Exp $
 */


