/* ===[ $RCSfile: mescdef.c,v $ ]=========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mescdef.c,v 1.5 2005/03/23 22:11:36 cmayncvs Exp $"
 *
 *  \brief Utility to write initial MESC parameters.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include "debug.h"
#include "cfg/mesc_cfg.h"

/* ======= */
/* Defines */
/* ======= */
/** Maximum record name length */
#define MAXREC_FILESIZE     4
/** Maximum record file size */
#define MAXREC_RECSIZE      512

/* ========== */
/* Prototypes */
/* ========== */
static void cfg_le2be(mesc_cfg_t *cfg);
static int write_record(const char *record, const void *src, size_t size);
static int check_record(const char *record, size_t size);

/* ======================================================================= */
/**
 *  \brief Entry point to make defaults.
 *  \param argc Should be 1.  No args expected.
 *  \param argv - Not used.
 *
 *  \return 0, always.
 */
/* ======================================================================= */
int main(int argc, char *argv[])
{
    mesc_cfg_t cfg;

    memset(&cfg, 0, sizeof(mesc_cfg_t));
    cfg_le2be(&cfg);

    if ( write_record((char *)mesc_cfg_record, (char *)&cfg,
        sizeof(mesc_cfg_t)) == 0 )
        printf("\nRecord <%s.conf> written.\n", (char *)mesc_cfg_record);
    else
        printf("\nERROR writing <%s.conf>.\n", (char *)mesc_cfg_record);
    return (0);

} /* main() */


/* ======================================================================= */
/**
 *  \brief Converts the configuration structure to BIG-ENDIAN format.
 *  \param cfg The configuration record.
 *  \return Nothing.
 */
/* ======================================================================= */
static void cfg_le2be(mesc_cfg_t *cfg)
{
    cfg->dbg_level = htons(mesc_cfg_defaults.dbg_level);
    cfg->id_method = htons(mesc_cfg_defaults.id_method);
    memcpy(cfg->terminal_id, mesc_cfg_defaults.terminal_id,
        sizeof(mesc_cfg_defaults.terminal_id));
    cfg->terminal_idn = htonl(mesc_cfg_defaults.terminal_idn);
    cfg->site_id = htons(mesc_cfg_defaults.site_id);
    cfg->hw_type_id = htons(mesc_cfg_defaults.hw_type_id);
    cfg->unit_id = htons(mesc_cfg_defaults.unit_id);
    memcpy(cfg->rom_id, mesc_cfg_defaults.rom_id,
        sizeof(mesc_cfg_defaults.rom_id));
    memcpy(cfg->host1, mesc_cfg_defaults.host1,
        sizeof(mesc_cfg_defaults.host1));
    memcpy(cfg->host2, mesc_cfg_defaults.host2,
        sizeof(mesc_cfg_defaults.host2));
    cfg->base_port = htons(mesc_cfg_defaults.base_port);
    cfg->tmo_cnt = htonl(mesc_cfg_defaults.tmo_cnt);
    cfg->min_rsp_tmo = htonl(mesc_cfg_defaults.min_rsp_tmo);
    cfg->max_rsp_tmo = htonl(mesc_cfg_defaults.max_rsp_tmo);
    cfg->keepalive_tmo = htons(mesc_cfg_defaults.keepalive_tmo);
    cfg->drop0_is_special = htons(mesc_cfg_defaults.drop0_is_special);

} /* cfg_le2be() */


/* ======================================================================= */
/**
 *  \brief Writes and commits \a size bytes from \a src to record \a record.
 *  \param record The record name to write to.
 *  \param src The memory area holding the data to write.
 *  \param size How much data to write.
 *
 *  This function can be used to create site specific configuration parameters.
 *  which are included in a release.
 *  The defaults passed in are used to write to the file.
 *
 *  The configuration record can be found in the directory:
 *          /etc/gtech/config/default/record.conf
 *
 *  The configuration file list is also updated:
 *          /etc/config_files
 *
 *
 *  \return 0 on success; non-zero otherwise.
 */
/* ======================================================================= */
static int write_record(const char *record, const void *src, size_t size)
{
    int fp;
    char fname[MAXREC_FILESIZE + 6];    /* Plus 6 for ".conf\0" */

    if ( check_record(record, size) )
        return (-1);

    strcpy(fname, record);
    strcat(fname, ".conf");

    if ( (fp = open(fname, O_WRONLY | O_CREAT | O_TRUNC, S_IREAD | S_IWRITE)) == -1 )
    {
        printf("Can not open file <%s> for writing.\n", fname);
        return (-1);
    }
    if ( write(fp, src, size) != size )
    {
        printf("Write of record <%s> failed.\n", fname);
        close(fp);
        return (-1);
    }
    if ( close(fp) != 0 )
    {
        printf("Could not close the file <%s>.\n", fname);
        return (-1);
    }
    return (0);

} /* write_record() */


/* ======================================================================= */
/**
 *  \brief Verifies the record name complies with the rules we've established.
 *  \param record The record name to check.
 *  \param size The size of the record.
 *
 *  \return 0 if record is ok; non-zero otherwise.
 */
/* ======================================================================= */
static int check_record(const char *record, size_t size)
{
    if ( (record == NULL) || (size <= 0) )
    {
        printf("Bad record name or size (%d).\n", size);
        return (-1);
    }
    if ( strlen(record) > MAXREC_FILESIZE )
    {
        printf("Record name <%s> too long.  Max length is %d.\n",
            record, MAXREC_FILESIZE);
        return (-1);
    }
    if ( size > MAXREC_RECSIZE )
    {
      printf("Record size of %d is too big.  Max size is %d.\n",
        size, MAXREC_RECSIZE);
      return (-1);
    }
    return (0);
} /* check_record() */

/*
 * End of "$Id: mescdef.c,v 1.5 2005/03/23 22:11:36 cmayncvs Exp $".
 */


