/* ===[ $RCSfile: mesc.c,v $ ]============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc.c,v 1.7 2005/08/01 17:57:27 cmayncvs Exp $"
 *
 *  \brief Multi-Drop Enterprise Series Connect (MESC) module.
 *
 *  This code handles multi-drop ESC functions for an ES-PAD device.  The
 *  current device is the Cyclades TS100.  It is based on ESConnect v1.3.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include "libcom/buf.h"
#include "libcom/fd.h"
#include "mesc.h"
#include "mesc_cfg.h"
#include "mesc_dbg.h"
#include "mesc_tmr.h"
#include "mesc_art.h"
#include "mesc_udp.h"
#include "mesc_imp.h"
#include "mesc_stat.h"
#include "mesc_fsm.h"
#include "gassert.h"
#if !defined ARCH_ESPAD
#include "gdebug.h"
#endif /* !ARCH_ESPAD */

/** Specifies the MESC usage */
#define usage   "\nUsage: mesc <local_queue> <upper_queue> "    \
    "<lower_queue> [pad_queue] [-d dbglvl]\n"

/* =============== */
/* Local variables */
/* =============== */
/** MESC version - RCS will fill this in with the tag assigned */
static char _mesc_version[] = "$Name: ANBA $";
/** ROMID, gleaned from the version or set to a default if not tagged. */
static char _mesc_romid[5];

/* ========================= */
/* Local function prototypes */
/* ========================= */
static char *_mesc_get_version(void);
#if !defined ARCH_ESPAD
static void _mesc_debug_init(void);
#endif /* !ARCH_ESPAD */
static void _mesc_init(const char *local_q, const char *upper_q,
                       const char *lower_q, const char *pad_q, int level);
static void _mesc_delay(void);
static void _mesc(void);

/* ======================================================================= */
/**
 *  \brief MESC entry point.
 *  \param argc Should be 4, 5, or 7.  More than 7 are ignored.
 *  \param argv - argv[1]=local queue, argv[2]=upper queue,
 *      argv[3]=lower queue, optional: argv[4]=pad queue, argv[5:6]= -d level.
 *
 *  \return Never returns (intentionally at least).
 */
/* ======================================================================= */
int main(int argc, char *argv[])
{
    int level = -1;
    const char *pad_q = NULL;

    #if !defined ARCH_ESPAD
    _mesc_debug_init();
    #endif /* !ARCH_ESPAD */

    info("(%d) Version %s: Started.", getpid(), _mesc_get_version());

    /* Check usage */
    if ( argc < 4 )
    {
        warn(usage);
        exit(0);
    }
    assert(argv && argv[1] && argv[2] && argv[3]);

    /* Check for optional pad_queue & command-line override of debug level */
    if ( argc > 4 )
    {
        assert(argv[4]);
        if ( argc == 5 )            /* optional pad queue */
            pad_q = argv[4];
        else if ( argc == 6 )       /* optional -d dbglvl */
        {
            assert(argv[5]);
            if ( !strncmp(argv[4], "-d", 2) )
                level = strtol(argv[5], NULL, 0) & 0x07;
            else
            {
                warn(usage);
                exit(0);
            }
        }
        else if ( argc == 7 )       /* optional pad queue & -d dbglvl */
        {
            pad_q = argv[4];
            assert(argv[5] && argv[6]);
            if ( !strncmp(argv[5], "-d", 2) )
                level = strtol(argv[6], NULL, 0) & 0x07;
            else
            {
                warn(usage);
                exit(0);
            }
        }
        else                        /* invalid */
        {
            warn(usage);
            exit(0);
        }
    }

    /* Initialize everything */
    _mesc_init(argv[1], argv[2], argv[3], pad_q, level);

    /* Start MESC - should never return */
    _mesc();
    return (0);

} /* main() */


/* ======================================================================= */
/**
 *  \brief Gets MESC version.
 *  \return mesc version, as the ROMID gleaned from the RCS tag assigned.
 */
/* ======================================================================= */
char *mesc_get_version(void)
{
    return (_mesc_romid);
} /* mesc_get_version() */

                        /* =============== */
                        /* Local functions */
                        /* =============== */

/* ======================================================================= */
/**
 *  \brief Gets MESC version.
 *  \warning The tags can not have an occurence of space-dollar-sign " $".
 *  \return mesc version, as the tag assigned to it in RCS.
 */
/* ======================================================================= */
static char *_mesc_get_version(void)
{
    if ( strlen(&_mesc_version[7]) < 4 )
        strcpy(_mesc_romid, "AN00");
    else
    {
        strncpy(_mesc_romid, &_mesc_version[7], 4);
        _mesc_romid[4] = '\0';
    }
    return (_mesc_romid);

} /* _mesc_get_version() */


#if !defined ARCH_ESPAD
/* ======================================================================= */
/**
 *  \brief Redirects output from stdout to the circular debug log.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_debug_init(void)
{
    int debug_fd;

    /* Force kernel debug console_loglevel to lowest possible value */
    system("echo 0 > /proc/sys/kernel/printk");

    /*
     * If f_debug_init() returns -1, assume it's because we're not working
     * on actual target hardware, but a linux development box instead, with no
     * gboard installed.  In that case, don't redirect output.
     */
    if ( (debug_fd = f_debug_init()) != -1 )
    {
        dup2(debug_fd, STDOUT_FILENO);
        dup2(debug_fd, STDERR_FILENO);
    }
} /* _mesc_debug_init() */
#endif /* !ARCH_ESPAD */

/* ======================================================================= */
/**
 *  \brief Initializes everything MESC needs before it starts running,
 *      including config data, queue's, art, and state machine.
 *  \param local_q mesc queue name.
 *  \param upper_q upper queue name.
 *  \param lower_q lower queue name.
 *  \param pad_q Optional pad queue for the pad itself (primarily for resets)
 *  \param level An optional forced debug level to set mesc to.  This debug
 *      level is expected to be one from gdebug.h, but it will be mapped to
 *      one of the levels in debug.h.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_init(const char *local_q, const char *upper_q,
    const char *lower_q, const char *pad_q, int level)
{
    mesc_cfg_init();
    mesc_dbg_init(level);
    mesc_tmr_init();
    mesc_art_init();
    mesc_stat_init();
    mesc_imp_init(local_q, upper_q, lower_q, pad_q);
    mesc_udp_init();
    mesc_fsm_init();
    info("init(): Done.");

} /* _mesc_init() */


/* ======================================================================= */
/**
 *  \brief Waits a random amount of time to avoid terminal sychronization.
 *      (See "man rand" for why it's implemented the way it is.)
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_delay(void)
{
    int delay;
    struct timeval tv;
    struct timezone tz;

    /* Randomly seed the random number generator;
     * otherwise what's the point!? */
    assert(gettimeofday(&tv, &tz) == 0);
    srand((unsigned int)tv.tv_usec);

    /*
     * Wait some random amount of time (between 0-5000 msec's) before starting
     * so that if all es-pads are reset at the same time, they won't be
     * synchronized.
     */
    delay = (int)(5001.0 * rand() / (RAND_MAX + 1.0));
    info("Sleeping for %d.%d seconds ...", delay / 1000, delay % 1000);
    usleep(1000 * delay);
    info("... Ready!\n");

} /* _mesc_delay() */


/* ======================================================================= */
/**
 *  \brief MESC main function.
 *  \return Nothing (never returns).
 */
/* ======================================================================= */
static void _mesc(void)
{
    mesc_imp_send_status_msg(255, "Message=ESCIPInit");
    _mesc_delay();
    while ( 1 )
        fd_select();

} /* _mesc() */


/*
 * End of "$Id: mesc.c,v 1.7 2005/08/01 17:57:27 cmayncvs Exp $".
 */


