/* ===[ $RCSfile: mesc_art.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  $Id: mesc_art.c,v 1.4 2005/02/15 20:45:55 cmayncvs Exp $
 *
 *  \brief Implements the adaptive response timer functions.  Uses the
 *      TCP algorithm with a min and a max. See RFC 2988,
 *      http://www.ietf.org/rfc/rfc2988.txt?number=2988.
 */
/* ======================================================================= */
/* ============= */
/* Include files */
/* ============= */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "libcom/buf.h"
#include "mesc_dbg.h"
#include "mesc_cfg.h"
#include "mesc_tmr.h"
#include "mesc_art.h"
#include "gassert.h"

/* ======= */
/* Defines */
/* ======= */
#define ALPHA   (1/(double)8)   /**< Current response time weight. */
#define BETA    (1/(double)4)   /**< Current variance weight. */
#define K       ((double)4)     /**< Estimated mean variance multiplier. */

/** Typedef for art data */
typedef struct {
    double min;             /**< minimum response time, in milliseconds */
    double max;             /**< maximum response time, in milliseconds */
    double srtt;            /**< response time mean */
    double rttvar;          /**<  response time variance */
    int first_time_get;     /**< flag if 1st time get or not */
    int first_time_update;  /**< flag if 1st time updated or not */
    int mark_resp_tmo;      /**< flag if timer marked or not */
    int updated;            /**< flag if timer updated or not */
    double last_rto;        /**< the last response timeout value */
} _mesc_art_t;

/** Holds the ART data for all drops */
static _mesc_art_t _mesc_art[MESC_CFG_MAX_DROPS];

/* ======================================================================= */
/**
 *  \brief Initialize adaptive response timer for all drops.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_art_init(void)
{
    u_int8_t drop;
    u_int32_t min, max;

    min = mesc_cfg_get_min_rsp_tmo();
    max = mesc_cfg_get_max_rsp_tmo();

    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
    {
        _mesc_art[drop].min = (double)(min);
        _mesc_art[drop].max = (double)(max);
        _mesc_art[drop].first_time_get = 1;
        _mesc_art[drop].first_time_update = 1;
        _mesc_art[drop].updated = 1;
    }
    info("art_init(): Done (Min RTO: %u.%u sec, Max RTO: %u.%u sec).",
        min / 1000, min % 1000, max / 1000, max % 1000);

} /* mesc_art_init() */


/* ======================================================================= */
/**
 *  \brief Mark data accordingly for a response timeout.
 *  \param drop The drop address [range: 0 - (MESC_CFG_MAX_DROPS - 1)]
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_art_mark_resp_tmo(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);

    _mesc_art[drop].mark_resp_tmo = 1;
    _mesc_art[drop].updated = 1;
    mesc_tmr_reset_prev_resp_time(drop);

} /* mesc_art_mark_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Update estimated mean and variance. (a response was received)
 *  \param drop The drop address [range: 0 - (MESC_CFG_MAX_DROPS - 1)]
 *  \param rt The measured response time.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_art_update(u_int8_t drop, u_int32_t rt)
{
    double rto = (double)rt;

    assert(drop < MESC_CFG_MAX_DROPS);

    if ( rto < 1000.0 )
        rto = 1000.0;

    if ( _mesc_art[drop].first_time_update )
    {
        _mesc_art[drop].first_time_update = 0;
        _mesc_art[drop].srtt = rto;
        _mesc_art[drop].rttvar = _mesc_art[drop].srtt / 2;
    }
    else
    {
        _mesc_art[drop].rttvar = (1 - BETA) * _mesc_art[drop].rttvar +
            BETA * fabs(_mesc_art[drop].srtt - rto);
        _mesc_art[drop].srtt = (1 - ALPHA) * _mesc_art[drop].srtt + ALPHA * rto;
    }
    _mesc_art[drop].mark_resp_tmo = 0;
    _mesc_art[drop].updated = 1;

} /* mesc_art_update() */


/* ======================================================================= */
/**
 *  \brief Get the next timeout.
 *  \param drop The drop address [range: 0 - (MESC_CFG_MAX_DROPS - 1)]
 *  \return The timeout.
 */
/* ======================================================================= */
int16_t mesc_art_get_tmo(u_int8_t drop)
{
    double rto;

    assert(drop < MESC_CFG_MAX_DROPS);

    /* Handle the 1st timeout condition. */
    if ( _mesc_art[drop].first_time_get )
    {
        _mesc_art[drop].first_time_get = 0;
        _mesc_art[drop].last_rto =
            (_mesc_art[drop].max + _mesc_art[drop].min) / 2;
        info("Response timeout(%d) = %d.%d seconds.", drop,
            (u_int16_t)_mesc_art[drop].last_rto / 1000,
            (u_int16_t)_mesc_art[drop].last_rto % 1000);
        return ((int16_t)_mesc_art[drop].last_rto);
    }

    /* If the response timer hasn't been updated yet, return the last one */
    if ( _mesc_art[drop].updated == 0 )
    {
        info("Response timeout(%d) = %d.%d seconds.", drop,
            (u_int16_t)_mesc_art[drop].last_rto / 1000,
            (u_int16_t)_mesc_art[drop].last_rto % 1000);
        return ((int16_t)_mesc_art[drop].last_rto);
    }
    _mesc_art[drop].updated = 0;

    /* If a response timeout occurred last, return twice the last timeout as
     * we quickly back off, otherwise update the new value based on the
     * tcp algorithm using mean, variance of measured response. */
    if ( _mesc_art[drop].mark_resp_tmo )
        rto = 2 * _mesc_art[drop].last_rto;
    else
        rto = _mesc_art[drop].srtt + K * _mesc_art[drop].rttvar;

    /* Lastly, bounds check */
    if ( rto > _mesc_art[drop].max )
        rto = _mesc_art[drop].max;
    if ( rto < _mesc_art[drop].min )
        rto = _mesc_art[drop].min;
    info("(%d) Max, Min =  %d.%d seconds, %d.%d seconds.", drop,
        (u_int16_t)_mesc_art[drop].max / 1000,
        (u_int16_t)_mesc_art[drop].max % 1000,
        (u_int16_t)_mesc_art[drop].min / 1000,
        (u_int16_t)_mesc_art[drop].min % 1000);

    _mesc_art[drop].last_rto = rto;
    info("Response timeout(%d) = %d.%d seconds.", drop,
        (u_int16_t)rto / 1000, (u_int16_t)rto % 1000);
    return ((int16_t)rto);

} /* mesc_art_get_tmo() */


/* ======================================================================= */
/**
 *  \brief Returns some noise for the response timer as a percentage of the
 *      previous timeout value.  (20% was chosen) - IS THIS OK?.
 *      See man rand() for why it's implemented this way.
 *  \param drop The drop address whose last response timeout to use for noise
 *  \return The noise.
 */
/* ======================================================================= */
int mesc_art_get_noise(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return ((int)((1.0 + 0.2 * _mesc_art[drop].last_rto) * rand()
        / (RAND_MAX + 1.0)));
} /* mesc_art_get_noise() */


/* ======================================================================= */
/**
 *  \brief Sets the minimum response timeout for this drop to the tmo specified.
 *      Units for \a min are milliseconds.
 *  \param drop The drop address whose minimum response timeout to set.
 *  \param min The new minimum response timeout value.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_art_set_min_rsp_tmo(u_int8_t drop, u_int32_t min)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_art[drop].min = min;

} /* mesc_art_set_min_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Sets the maximum response timeout for this drop to the tmo specified.
 *      Units for \a max are milliseconds.
 *  \param drop The drop address whose maximum response timeout to set.
 *  \param max The new minimum response timeout value.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_art_set_max_rsp_tmo(u_int8_t drop, u_int32_t max)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_art[drop].max = max;

} /* mesc_art_set_max_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Returns the current minimum response timeout for this drop.
 *  \param drop The drop address whose minimum response timeout to get.
 *  \return The minimum response timeout, in milliseconds, for this drop.
 */
/* ======================================================================= */
u_int32_t mesc_art_get_min_rsp_tmo(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return ((u_int32_t)_mesc_art[drop].min);
} /* mesc_art_get_min_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Returns the current maximum response timeout for this drop.
 *  \param drop The drop address whose maximum response timeout to get.
 *  \return The maximum response timeout, in milliseconds, for this drop.
 */
/* ======================================================================= */
u_int32_t mesc_art_get_max_rsp_tmo(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return ((u_int32_t)_mesc_art[drop].max);
} /* mesc_art_get_max_rsp_tmo() */



#if MESC_ART_TEST
/* ======================================================================= */
/**
 *  \brief Test code for the MESC ART.
 *  \return Nothing.
 */
/* ======================================================================= */
int main (void)
{
    unsigned int rto[] = {
        2000, 1800, 2200, 2000, 5000, 5100, 3000, 2100,
        2000, 2000, 2000, 2000, 2000, 2000, 2000, 10000,
        11000, 17000, 23000, 23000, 12000, 12000, 12000, 12000,
        12000, 12000, 12000, 1200, 1200, 1200, 1200, 1200,
        1200, 0, 0, 1, 1000, 1000
    };
    int i;
    unsigned int r;

    mesc_art_init(1, 30);

    printf("Real  RTO   SRTT   RTTVAR   Last\n");

    for ( i = 0; i < sizeof (rto) / sizeof (unsigned int); i++ )
    {
        if ( (r = mesc_art_get_tmo(0)) < rto[i] )
        {
            mesc_art_mark_resp_tmo(0);
            printf("Response timeout\n");
        }
        else
            mesc_art_update(0, rto[i]);

        printf("%d  %d %f %f %f\n", rto[i], r, _mesc_art[drop].srtt,
            _mesc_art[drop].rttvar, _mesc_art[drop].last_rto);
    }
    return (0);
} /* main() */

#endif /* MESC_ART_TEST */


/*
 * End of $Id: mesc_art.c,v 1.4 2005/02/15 20:45:55 cmayncvs Exp $
 */


