/* ===[ $RCSfile: mesc_cfg.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc_cfg.c,v 1.14 2005/07/27 15:12:06 cmayncvs Exp $"
 *
 *  \brief MESC Run-Time Configuration routines.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "libcom/buf.h"
#include "libcom/list.h"
#include "libcom/str.h"
#include "libcom/c_config.h"
#include "mesc.h"
#include "mesc_dbg.h"
#include "cfg/mesc_cfg.h"
#include "mesc_cfg.h"
#include "escp.h"
#include "gassert.h"

#if !defined ARCH_ESPAD
#include <netdb.h>
#include "gpaths.h"
#endif /* !ARCH_ESPAD */

/* ======= */
/* Defines */
/* ======= */
#ifdef ARCH_ESPAD
/** Hardware ID prefix size is 19 for ESPAD (6 digits * 3 + 1) */
#define HW_ID_PREFIX_SIZE           19
/** The ES-PAD type ID (digits 4 & 5) of the Client name. (from Gtech doc) */
#define _MESC_CFG_HW_TYPE_ID        02
#else /* !ARCH_ESPAD */
/** Hardware ID prefix size is 25 for Altura/LVT (8 digits * 3 + 1) */
#define HW_ID_PREFIX_SIZE           25
/** Altura hardware id file */
#define ALTURA_HARDWARE_ID_FILE     "/proc/gtech/gboard/serialID"
/** LVT hardware id file */
#define LVT_HARDWARE_ID_FILE        "/proc/gtech/serialID"
#endif /* !ARCH_ESPAD */
/** The maximum number of Site IDs in our table */
#define MAX_SITE_IDS                57

/* =============== */
/* Local variables */
/* =============== */
/** MESC configuration parameter data */
static mesc_cfg_t _mesc_cfg;
/** MESC prefix data, common to all drops */
static struct {
    char client_name_prefix[64];    /**< "sss0-2aaa-aa" => append "bb-cc"  */
    char hw_id_prefix[HW_ID_PREFIX_SIZE];  /**< "00:11:22:33:44:55:" */
    char ip_addr[16];               /**< The Client's IP address */
    list proc_list;                 /**< Proc list for t0,t1,tc handling */
    cfg_t cfg;                      /**< Holds config info */
    u_int16_t secure;               /**< Flag for security enabled/not */
    u_int16_t have_rom_id;          /**< Flag to indicate we already have id */
    char rom_id[5];                 /**< The cached pad rom id */
} _mesc_cfg_misc;

/** The possible ascii values for the drop address */
static char *_ascii_drop[157] = { /* 157 possible drops */
    /* Normal addresses (0-29) */
    "_@", "_A", "_B", "_C", "_D", "_E", "_F", "_G", "_H",  "_I",
    "_J", "_K", "_L", "_M", "_N", "_O", "_P", "_Q", "_R",  "_S",
    "_T", "_U", "_V", "_W", "_X", "_Y", "_Z", "_[", "_\\", "_]",
    /* Extended addresses - % range (30-61) */
    "%@", "%A", "%B", "%C", "%D", "%E", "%F", "%G", "%H",  "%I",
    "%J", "%K", "%L", "%M", "%N", "%O", "%P", "%Q", "%R",  "%S",
    "%T", "%U", "%V", "%W", "%X", "%Y", "%Z", "%[", "%\\", "%]", "%>", "%?",
    /* Extended addresses - # range (62-93) */
    "#@", "#A", "#B", "#C", "#D", "#E", "#F", "#G", "#H",  "#I",
    "#J", "#K", "#L", "#M", "#N", "#O", "#P", "#Q", "#R",  "#S",
    "#T", "#U", "#V", "#W", "#X", "#Y", "#Z", "#[", "#\\", "#]", "#>", "#?",
    /* Extended addresses - $ range (94-125) */
    "$@", "$A", "$B", "$C", "$D", "$E", "$F", "$G", "$H",  "$I",
    "$J", "$K", "$L", "$M", "$N", "$O", "$P", "$Q", "$R",  "$S",
    "$T", "$U", "$V", "$W", "$X", "$Y", "$Z", "$[", "$\\", "$]", "$>", "$?",
    /* Extended addresses - ! range (126-156) */
    "!@", "!A", "!B", "!C", "!D", "!E", "!F", "!G", "!H",  "!I",
    "!J", "!K", "!L", "!M", "!N", "!O", "!P", "!Q", "!R",  "!S",
    "!T", "!U", "!V", "!W", "!X", "!Y", "!Z", "![", "!\\", "!]", "!>"
};

/** The Site ID table. */
static char *site_ids[MAX_SITE_IDS] = {
    "Testing",                              /* 0 */
    "US-Idaho (IDS)",                       /* 1 */
    /* "US-Washington (WAS)"    *** CONFLICT *** also assigned 1 */
    "US-Missouri (MOS)",                    /* 2 */
    "Switzerland (CHR)",                    /* 3 */
    /* "US-New Mexico (NMS)"    *** CONFLICT *** also assigned 3 */
    "US-District of Columbia (DCS)",        /* 4 */
    /* "US-New Jersey (NJS)"    *** CONFLICT *** also assigned 4 */
    /* "US-Texas (TXE)"         *** CONFLICT *** also assigned 4 */
    "Germany-Leipzig (DES)",                /* 5 */
    /* "US-Illinois (ILE)"      *** CONFLICT *** also assigned 5 */
    "US-Ohio (OHS)",                        /* 6 */
    "US-Wisconsin (WIS)",                   /* 7 */
    "US-Rhode Island (RIS)",                /* 8 */
    /* "Singapore (SGP)"        *** CONFLICT *** also assigned 8 */
    "Denmark (DKT)",                        /* 9 */
    "US-Massachusetts (MAS)",               /* 10 */
    "US-Oregon (ORS)",                      /* 11 */
    "US-Rhode Island (RIV)",                /* 12 */
    "US-Arizona (AZS)",                     /* 13 */
    "Sweden (SET)",                         /* 14 */
    "US-Kansas (KSS)",                      /* 15 */
    "US-Kentucky (KYS)",                    /* 16 */
    "New Zealand (NZC)",                    /* 17 */
    "Mexico (MXP)",                         /* 18 */
    "US-Michagan (MIS)",                    /* 19 */
    "Brazil (BRF)",                         /* 20 */
    "US-Texas (TXS)",                       /* 21 */
    "Australia (AUS)",                      /* 22 */
    "Israel (ILM)",                         /* 23 */
    "South Africa (ZAU)",                   /* 24 */
    "Australia (AUN)",                      /* 25 */
    "Australia (AUW)",                      /* 26 */
    "US-Illinois (ILE)",                    /* 27 */
    "France (FRJ)",                         /* 28 */
    "US-New York (NYS)",                    /* 29 */
    "Luxembourg (LUN)",                     /* 30 */
    "US-Louisiana (LAS)",                   /* 31 */
    "US-Minnesota (MNS)",                   /* 32 */
    "Canada-Alberta (ABG)",                 /* 33 */
    "Canada-Alberta (ABS)",                 /* 34 */
    "US-California (CAS)",                  /* 35 */
    "US-Georgia (GAS)",                     /* 36 */
    "US-Tennessee (TNA)",                   /* 37 */
    "Germany-Nordrhein-Westfalen (DEW)",    /* 38 */
    "Sri Lanka (LKM)",                      /* 39 */
    "US-Florida (FLA)",                     /* 40 */
    "Leeward Islands (LIL)",                /* 41 */
    "Sweden (SEV)",                         /* 42 */
    "Thailand (THA)",                       /* 43 */
    "Canada-British Columbia (BCS)",        /* 44 */
    "US-West Virginia (WVV)",               /* 45 */
    "US-New York (NYV)",                    /* 46 */
    "US-Delaware (DEV)",                    /* 47 */
    "US-Oregon (ORV)",                      /* 48 */
    "Germany-Thuringen (DET)",              /* 49 */
    "Canada-New Brunswick (NBV)",           /* 50 */
    "Greece (GRO)",                         /* 51 */
    "US-Louisiana Video (LAV)",             /* 52 */
    "US-Montana Video (MTV)",               /* 53 */
    "China-National (CNN)",                 /* 54 */
    "Czech Republic (CZS)",                 /* 55 */
    "Turks and Caicos Islands (TCV)"        /* 56 */
};

static u_int8_t _mesc_cfg_compute_csum(const char *str);
static void _mesc_cfg_set_client_name_prefix(void);
#if !defined ARCH_ESPAD
static int _mesc_cfg_get_rom_id_from_sysman(char *id);
static void _mesc_get_real_hw_id(void);
#endif /* !ARCH_ESPAD */
static void _mesc_cfg_set_ip_addr_and_hw_id_prefix(void);
static void _mesc_cfg_proc(char *name, void(*proc)(list));
static void _t0(list l);
static void _t1(list l);
static void _tc(list l);

/* ======================================================================= */
/**
 *  \brief Initializes the MESC run-time configuration parameters.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_cfg_init(void)
{
    memset(&_mesc_cfg_misc, 0, sizeof(_mesc_cfg_misc));

    /* Read the mesc parameters */
    c_config_read(mesc_cfg_record, &_mesc_cfg, &mesc_cfg_defaults,
        sizeof(mesc_cfg_t));

    /* Save the client name prefix (everything before the drop and csum */
    _mesc_cfg_set_client_name_prefix();

    /* Save the IP & hardware ID prefix (everything before the drop address) */
    _mesc_cfg_set_ip_addr_and_hw_id_prefix();

    /* Register our config processing functions */
    _mesc_cfg_misc.proc_list = 0;
    _mesc_cfg_proc("T0", _t0);
    _mesc_cfg_proc("T1", _t1);
    _mesc_cfg_proc("TC", _tc);

    info("cfg_init(): Done.");

} /* mesc_cfg_init() */


/* ======================================================================= */
/**
 *  \brief Get the debug level.
 *  \return The debug level.
 */
/* ======================================================================= */
u_int8_t mesc_cfg_get_dbg_level(void)
{
    return (_mesc_cfg.dbg_level);
} /* mesc_cfg_get_dbg_level() */


/* ======================================================================= */
/**
 *  \brief Get the string representation of the debug level.
 *  \param level The debug level.
 *  \return The string representation of the debug level (FW format).
 */
/* ======================================================================= */
char *mesc_cfg_level2str(int level)
{
    switch ( level & 0x07 )
    {
        case 7:  return ("ERR");
        case 6:  return ("WARN");
        case 5:  return ("INFO1");
        case 4:  return ("INFO2");
        case 3:  return ("INFO3");
        case 2:  return ("INFO4");
        case 1:  return ("ALL");
        default: return ("NONE");
    }

} /* mesc_cfg_level2str() */


/* ======================================================================= */
/**
 *  \brief Get the site ID.
 *  \return The site ID.
 */
/* ======================================================================= */
u_int16_t mesc_cfg_get_site_id(void)
{
    return (_mesc_cfg.site_id);
} /* mesc_cfg_get_site_id() */


/* ======================================================================= */
/**
 *  \brief Get the hardware type ID.
 *  \return The hardware type ID.
 */
/* ======================================================================= */
u_int16_t mesc_cfg_get_hw_type_id(void)
{
    return (_mesc_cfg.hw_type_id);
} /* mesc_cfg_get_hw_type_id() */


/* ======================================================================= */
/**
 *  \brief Get the string representation of the hardware type ID.
 *  \param id The hardware ID.
 *  \return The string representation of the hardware type ID.
 */
/* ======================================================================= */
char *mesc_cfg_hwid2str(u_int16_t id)
{
    if ( id == 2 )
        return ("ESPAD");
    return ("");
} /* mesc_cfg_hwid2str() */


/* ======================================================================= */
/**
 *  \brief Get the unit ID.
 *  \return The unit ID.
 */
/* ======================================================================= */
u_int16_t mesc_cfg_get_unit_id(void)
{
    return (_mesc_cfg.unit_id);
} /* mesc_cfg_get_unit_id() */

#ifdef ARCH_ESPAD
/* ======================================================================= */
/**
 *  \brief Get the rom ID.
 *  \return The rom ID.
 */
/* ======================================================================= */
char *mesc_cfg_get_rom_id(void)
{
    return (_mesc_cfg.rom_id);
} /* mesc_cfg_get_rom_id() */


/* ======================================================================= */
/**
 *  \brief Get the rom ID for the ESPAD itself.
 *  \return The rom ID.
 */
/* ======================================================================= */
char *mesc_cfg_get_pad_rom_id(void)
{
    /* Only go through the rigamoroll once */
    if ( _mesc_cfg_misc.have_rom_id )
        return (_mesc_cfg_misc.rom_id);
#if 0
    {
    int rc = -1;
    int fd = -1;
    /* Try getting it from the romid file.  If that fails, use a default. */
    if ( (fd = open("/etc/gtech/config/default/romid", O_RDONLY)) > 0 )
    {
        while ( ((rc = read(fd, _mesc_cfg_misc.rom_id, 4)) == -1) &&
            (errno == EINTR) )
            ;
        close(fd);
    }

    if ( (fd <= 0) || (rc <= 0) )
        strcpy(_mesc_cfg_misc.rom_id, "AN00");
    }
#else
    /* Rather than reading it from a file, just glean it from the tag.  This
     * will avoid issues when upgrading the zImage of an espad, w/out clearing
     * the script file or re-writing the romid file. */
    strncpy(_mesc_cfg_misc.rom_id, mesc_get_version(), 4);
#endif
    _mesc_cfg_misc.rom_id[4] = '\0'; /* be sure to null-terminate */
    _mesc_cfg_misc.have_rom_id = 1;
    info("ESPAD ROMID = %s\n", _mesc_cfg_misc.rom_id);
    return (_mesc_cfg_misc.rom_id);

} /* mesc_cfg_get_pad_rom_id() */

#else /* !ARCH_ESPAD */
/* ======================================================================= */
/**
 *  \brief Get the rom id from either system manager or file.
 *  \return The rom id.
 */
/* ======================================================================= */
char *mesc_cfg_get_rom_id(void)
{
    int rc;
    int fd;

    /* Only go through the rigamoroll once */
    if ( _mesc_cfg_misc.have_rom_id )
        return (_mesc_cfg.rom_id);
    memset(_mesc_cfg.rom_id, 0, sizeof(_mesc_cfg.rom_id));

    /* Try getting it from sysman first.  If that fails, use the romid file. */
    if ( _mesc_cfg_get_rom_id_from_sysman(_mesc_cfg.rom_id) <= 0 )
    {
        assert((fd = open(G_CONFIG_DEFAULT_DIR"/romid", O_RDONLY)) > 0);
        while ( ((rc = read(fd, _mesc_cfg.rom_id, 4)) == -1) &&
            (errno == EINTR) )
            ;
        close(fd);
        assert(rc > 0);
        _mesc_cfg.rom_id[4] = '\0'; /* be sure to null-terminate */
    }
    _mesc_cfg_misc.have_rom_id = 1;
    info("ROMID = %s\n", _mesc_cfg.rom_id);
    return (_mesc_cfg.rom_id);

} /* mesc_cfg_get_rom_id() */


/* ======================================================================= */
/**
 *  \brief Get the rom ID.
 *  \return The rom ID.
 */
/* ======================================================================= */
char *mesc_cfg_get_pad_rom_id(void)
{
    return (mesc_cfg_get_rom_id());
} /* mesc_cfg_get_pad_rom_id() */
#endif /* !ARCH_ESPAD */

/* ======================================================================= */
/**
 *  \brief Get the primary host address.
 *  \return The primary host address.
 */
/* ======================================================================= */
char *mesc_cfg_get_host1(void)
{
    return (_mesc_cfg.host1);
} /* mesc_cfg_get_host1() */


/* ======================================================================= */
/**
 *  \brief Get the secondary host address.
 *  \return The secondary host address.
 */
/* ======================================================================= */
char *mesc_cfg_get_host2(void)
{
    return (_mesc_cfg.host2);
} /* mesc_cfg_get_host2() */


/* ======================================================================= */
/**
 *  \brief Get the timeout count to rotate hosts.
 *  \return The timeout count to rotate hosts.
 */
/* ======================================================================= */
u_int32_t mesc_cfg_get_tmo_cnt(void)
{
    return (_mesc_cfg.tmo_cnt);
} /* mesc_cfg_get_tmo_cnt() */


/* ======================================================================= */
/**
 *  \brief Get the minimum response timeout value.
 *  \return The minimum response timeout value, in milliseconds.
 */
/* ======================================================================= */
u_int32_t mesc_cfg_get_min_rsp_tmo(void)
{
    return (1000 * _mesc_cfg.min_rsp_tmo);
} /* mesc_cfg_get_min_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Get the maximum response timeout value.
 *  \return The maximum response timeout value, in milliseconds.
 */
/* ======================================================================= */
u_int32_t mesc_cfg_get_max_rsp_tmo(void)
{
    return (1000 * _mesc_cfg.max_rsp_tmo);
} /* mesc_cfg_get_max_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Get the keep-alive timeout.
 *  \return The keep-alive timeout.
 */
/* ======================================================================= */
u_int32_t mesc_cfg_get_keepalive_tmo(void)
{
    return (_mesc_cfg.keepalive_tmo);
} /* mesc_cfg_get_keepalive_tmo() */


/* ======================================================================= */
/**
 *  \brief Returns the UDP Base Port.
 *  \return Base port
 */
/* ======================================================================= */
u_int16_t mesc_cfg_get_base_port(void)
{
    return (_mesc_cfg.base_port);
} /* mesc_cfg_get_base_port() */

    /* ------------------------------------------------------------- */
    /* These are not configurable, but the access method is the same */
    /* ------------------------------------------------------------- */

/* ======================================================================= */
/**
 *  \brief Sets the flag indicating security enabled or not.
 *  \param flag The flag indicating security enabled or not, as 1 or 0.
 *  \return nothing.
 */
/* ======================================================================= */
void mesc_cfg_set_secure(u_int16_t flag)
{
    _mesc_cfg_misc.secure = flag;
} /* mesc_cfg_set_secure() */


/* ======================================================================= */
/**
 *  \brief Returns whether ESCP security is enabled.
 *  \return 0: 0 Security disabled; 1:Security enabled.
 */
/* ======================================================================= */
u_int16_t mesc_cfg_is_secure(void)
{
    return (_mesc_cfg_misc.secure);
} /* mesc_cfg_is_secure() */


/* ======================================================================= */
/**
 *  \brief Get the port to use.
 *  \param drop The drop address requesting its port to use.
 *  \return The port.
 */
/* ======================================================================= */
u_int16_t mesc_cfg_get_port(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return ((mesc_cfg_get_base_port()) + drop);
} /* mesc_cfg_get_port() */


/* ======================================================================= */
/**
 *  \brief Get the maximum number of drops this ES-PAD supports.
 *  \return The maximum number of drops.
 */
/* ======================================================================= */
u_int16_t mesc_cfg_get_max_drops(void)
{
    return (MESC_CFG_MAX_DROPS);
} /* mesc_cfg_get_max_drops() */


/* ======================================================================= */
/**
 *  \brief Get the Client name.
 *      The client name has the following format:  sss0-2aaa-aabb-cc, where
 *      \li sss = The site id, range: 501-999 [default=501]
 *      \li 02 = ESPAD adapter hardware type ID (_MESC_CFG_HW_TYPE_ID)
 *      \li aaaaa = A unique enumerated espad ID for this site
 *      \li bb = The drop number, passed as an argument
 *      \li cc = A 2-digit checksum, generated via terminal nomenclature doc.
 *  \param drop The drop address requesting its client name
 *  \param name The client name.  Area must be large enough for 18 char's.
 *  \return The Client name, returned in \a name.
 */
/* ======================================================================= */
void mesc_cfg_get_client_name(u_int8_t drop, char *name)
{
    char str[64];
    char cc_str[3];
    int rc;
    int is_special = (drop == 0) && _mesc_cfg.drop0_is_special;

    assert((drop < MESC_CFG_MAX_DROPS) && name);

    /* Copy the prefix and fill in the drop address, if required */
#ifdef ARCH_ESPAD
    if ( is_special )
        rc = snprintf(str, sizeof(str), "%s-",
            _mesc_cfg_misc.client_name_prefix);
    else
        rc = snprintf(str, sizeof(str), "%s%02X-",
            _mesc_cfg_misc.client_name_prefix, drop);
#else /* !ARCH_ESPAD */
    if ( _mesc_cfg.id_method == MESC_CFG_ID_PAD_STYLE )
    {
        if ( is_special )
            rc = snprintf(str, sizeof(str), "%s-",
                _mesc_cfg_misc.client_name_prefix);
        else
            rc = snprintf(str, sizeof(str), "%s%02X-",
                _mesc_cfg_misc.client_name_prefix, drop);
    }
    else
    {
        if ( is_special )
            rc = snprintf(str, sizeof(str), "%s",
                _mesc_cfg_misc.client_name_prefix);
        else
            rc = snprintf(str, sizeof(str), "%s%02X",
                _mesc_cfg_misc.client_name_prefix, drop);
    }
#endif /* !ARCH_ESPAD */

    /* Be sure to null terminate */
    if ( rc < (sizeof(str) - 3) )   /* leave 2 spaces for checksum + Null */
        str[rc] = '\0';
    else
        str[sizeof(str) - 3] = '\0';

    /* Compute checksum and append to client name */
    snprintf(cc_str, sizeof(cc_str), "%02d", _mesc_cfg_compute_csum(str));
    cc_str[2] = '\0';
    strcat(str, cc_str);
    strcpy(name, str);

} /* mesc_cfg_get_client_name() */


/* ======================================================================= */
/**
 *  \brief Get the virtual hardware ID (ethernet address + drop address).
 *      The hardware ID has the following format:
 *          hh:hh:hh:hh:hh:hh:dd, where
 *      \li hh:hh:hh:hh:hh:hh is the ethernet (MAC) address
 *      \li dd is the drop address
 *  \param drop The drop address requesting its hardware ID.
 *  \param hw_id The hardware ID.  Area must be large enough for all 21 char's.
 *  \return The hardware ID, returned in \a hw_id.
 */
/* ======================================================================= */
void mesc_cfg_get_hw_id(u_int8_t drop, char *hw_id)
{
    assert((drop < MESC_CFG_MAX_DROPS) && hw_id);

    /* Copy the prefix and ensure it's null terminated */
    strncpy(hw_id, _mesc_cfg_misc.hw_id_prefix,
        sizeof(_mesc_cfg_misc.hw_id_prefix));
    hw_id[sizeof(_mesc_cfg_misc.hw_id_prefix) - 1] = '\0';      /* 2bsure */

    /* Handle the special case of drop 0 if not adding the drop address */
    if ( (drop == 0) && _mesc_cfg.drop0_is_special )
    {
        hw_id[strlen(_mesc_cfg_misc.hw_id_prefix) - 1] = '\0';  /* erase ':' */
        return;
    }

    /* Append the drop address */
    strncat(hw_id, mesc_cfg_drop2ascii(drop),
        sizeof(_mesc_cfg_misc.hw_id_prefix) + 2);
    hw_id[sizeof(_mesc_cfg_misc.hw_id_prefix) + 2 - 1] = '\0';  /* 2bsure */

} /* mesc_cfg_get_hw_id() */


/* ======================================================================= */
/**
 *  \brief Returns the client's IP address.
 *  \return The client's IP address as a string.
 */
/* ======================================================================= */
char *mesc_cfg_get_ip_addr(void)
{
    return (_mesc_cfg_misc.ip_addr);
} /* mesc_cfg_get_ip_addr() */


/* ======================================================================= */
/**
 *  \brief Processes a config string.
 *  \param data The data to process
 *  \param cfg The new parameters to use - flags indicates the valid ones.
 *
 *      Config parameters are parsed as a semicolon seperated list.  Each
 *      parameter name value is seperate by an equal sign.  Multiple value
 *      lists are comma seperated.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_cfg_process(const char *data, cfg_t *cfg)
{
    list nv;

    /* Function called for each name value pair. */
    void call_proc(void *param)
    {
        list p, t;

        /* Split name and value into a list. */
        p = str_split((char *)param, "=");

        /* Lookup name in function list & called function w/name value list. */
        if ( (t = assoc(str_upper(str_chomp((char *)car(p))), _mesc_cfg_misc.proc_list, 0)) )
            ((void(*)(list))nth(1, t))(p);
        else
            warn("Unknown Parameter: %s", (char *)car(p));
        destroy(p);
    } /* call_proc() */

    /* Clear the global config area */
    assert(data && cfg);
    memset(&_mesc_cfg_misc.cfg, 0, sizeof(cfg_t));

    /* Split string into name value pairs. */
    nv = str_split((char *)data, ";");

    /* For each name value pair call helper function. */
    mapcar(call_proc, nv);
    destroy(nv);

    /* Copy to user */
    memcpy(cfg, &_mesc_cfg_misc.cfg, sizeof(cfg_t));

} /* mesc_cfg_process() */


/* ======================================================================= */
/**
 *  \brief Converts a drop number to its ASCII representation.
 *  \param drop The drop number.  Valid range: 0-156.
 *  \return The ASCII representation of the drop.
 */
/* ======================================================================= */
char *mesc_cfg_drop2ascii(u_int8_t drop)
{
    if ( drop > 156 )
        return ("BAD!");
    return (_ascii_drop[drop]);
} /* mesc_cfg_drop2ascii() */


/* ======================================================================= */
/**
 *  \brief Converts a ROM ID to the terminal it represents.
 *  \param id The ROM ID, corresponding to a terminal type.
 *  \return A string containing the terminal type.
 */
/* ======================================================================= */
char *mesc_cfg_get_terminal_type(char *id)
{
    if ( id[0] == 'A' )                 /* Normal lottery terminal family */
    {
        switch ( id[1] )
        {
            case '3': return ("Tiffany Z80");
            case '4': return ("Tiffany Z80");
            case '5': return ("Tiffany 68000");
            case '6': return ("Spectra");
            case '7': return ("Spectra II");
            case '8': return ("Spiffany");
            case '9': return ("Isys-1");
            case 'A': return ("Spectra III");
            case 'B': return ("Isys-2");
            case 'C': return ("Comet");             /* a.k.a. EZ Express */
            case 'D': return ("Player Express");    /* a.k.a. Plex */
            case 'E': return ("Altura II");         /* 2nd gen VxWorks */
            case 'F': return ("Plex Bootloader");
            case 'G': return ("Isys Checkwriter");
            case 'H': return ("Player Express II"); /* cost-reduced */
            case 'I': return ("Plex Bootloader II");
            case 'J': return ("Isys-VA");
            case 'K': return ("Isys-UK");
            case 'L': return ("Altura 02");         /* cost-reduced Linux */
            case 'M': return ("Altura 03");         /* 2nd cost-reduced Linux */
            case 'N': return ("ESPAD TS1H");
            case 'O': return ("Kdraw PC");
            default: break;
        }
    }
    else if ( id[0] == 'T' )            /* GVT family */
    {
        if ( (id[1] >= '1') && (id[1] <= '9') )
            return ("GVT");
        if ( (id[1] >= 'A') && (id[1] <= 'E') )
            return ("GVT+");
    }
    else if ( !strncmp("SM", id, 2) )   /* Altura test ROM */
        return ("Altura");

    return ("Unknown");
} /* mesc_cfg_get_terminal_type() */


/* ======================================================================= */
/**
 *  \brief Converts a Site ID to the site it represents.
 *  \param id The Site ID, corresponding to a site (jurisdiction).
 *  \return A string containing the site/jurisdiction name.
 */
/* ======================================================================= */
char *mesc_cfg_siteid2str(u_int16_t id)
{
    /* Handle this special case first - due to error in site id assignment.
     * Other special cases will have to be added as needed, such as when one
     * of the conflicting site id's are encountered. */
    if ( id == 502 )
        return ("Singapore (SGP)");
    if ( id >= 500 )
        id -= 500;
    if ( id > (MAX_SITE_IDS - 1) )
        return ("");
    return (site_ids[id]);

} /* mesc_cfg_siteid2str() */

                        /* =============== */
                        /* Local functions */
                        /* =============== */

/* ======================================================================= */
/**
 *  \brief Compute the checksum of the string.
 *      From the terminal nomenclature document:
 *          3.4 Check Digit Calculation
 *          The check digits are caculated in an unusual way - all values
 *          are treated as their ASCII equivalent rather than their numeric
 *          equivalent. Also, the dashes are included in the calculation
 *          of the check digits.
 *
 *  \return The string checksum.
 */
/* ======================================================================= */
static u_int8_t _mesc_cfg_compute_csum(const char *str)
{
    u_int8_t result = 0;

    assert(str);
    for ( ; *str; str++ )
    {
        result = (result * 10 + *str) % 97;
        result = 97 - result;
    }
    return (result);

} /* _mesc_cfg_compute_csum() */

#ifdef ARCH_ESPAD
/* ======================================================================= */
/**
 *  \brief Sets the client name prefix as "sss0-2aaa-aa" ... where sss is the
 *      configured site ID, and aaaaa is the unique pad number.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_cfg_set_client_name_prefix(void)
{
    /* Add all components of the client name except drop and checksum */
    snprintf(_mesc_cfg_misc.client_name_prefix,
        sizeof(_mesc_cfg_misc.client_name_prefix), "%03d%01d-%01d%03d-%02d",
        _mesc_cfg.site_id,
        _MESC_CFG_HW_TYPE_ID / 10, _MESC_CFG_HW_TYPE_ID % 10,
        _mesc_cfg.unit_id / 100, _mesc_cfg.unit_id % 100);
    info("Client name prefix set to %s", _mesc_cfg_misc.client_name_prefix);

} /* _mesc_cfg_set_client_name_prefix() */

#else /* !ARCH_ESPAD */
/* ======================================================================= */
/**
 *  \brief Sets the client name prefix based on configured method
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_cfg_set_client_name_prefix(void)
{
    u_int32_t num;

    /* Add all components of the client name except drop and checksum */
    if ( _mesc_cfg.id_method == MESC_CFG_ID_RAW )
    {
        strncpy(_mesc_cfg_misc.client_name_prefix, _mesc_cfg.terminal_id,
            sizeof(_mesc_cfg_misc.client_name_prefix));
    }
    else if ( _mesc_cfg.id_method == MESC_CFG_ID_OBFUSCATE )
    {
        num = _mesc_cfg.terminal_idn + MESC_CFG_OBFUSCATE_NUM;
        snprintf(_mesc_cfg_misc.client_name_prefix,
            sizeof(_mesc_cfg_misc.client_name_prefix), "%03d%01d-%04d-%04d-",
            _mesc_cfg.site_id,
            (int)(num / 100000000L),
            (int)((num % 100000000L) / 10000),
            num % 10000);
    }
    else if ( _mesc_cfg.id_method == MESC_CFG_ID_PAD_STYLE )
    {
        snprintf(_mesc_cfg_misc.client_name_prefix,
            sizeof(_mesc_cfg_misc.client_name_prefix), "%03d%01d-%01d%03d-%02d",
            _mesc_cfg.site_id,
            _mesc_cfg.hw_type_id / 10, _mesc_cfg.hw_type_id % 10,
            _mesc_cfg.unit_id / 100, _mesc_cfg.unit_id % 100);
    }
    else /* what to do?  No prefix? */
        _mesc_cfg_misc.client_name_prefix[0] = '\0';

    _mesc_cfg_misc.client_name_prefix[
        sizeof(_mesc_cfg_misc.client_name_prefix) - 1] = '\0';
    info("Client name prefix set to %s", _mesc_cfg_misc.client_name_prefix);

} /* _mesc_cfg_set_client_name_prefix() */


/* ======================================================================= */
/**
 *  \brief Reads the ROM ID assigned to this terminal from the system manager.
 *  \return The number of romid bytes read (capped at 4) or -1 for failure
 */
/* ======================================================================= */
static int _mesc_cfg_get_rom_id_from_sysman(char *id)
{
  struct hostent *hostinfo;
  struct servent *serverinfo;
  int sock;
  struct sockaddr_in address;
  int rc;
  u_int8_t rqst[3];     /* See sysman document for message formats */
  u_int8_t resp[7];     /* Section 12.5 */

    /* Get the ip of the localhost */
    if ( (hostinfo = gethostbyname("localhost")) == NULL )
        return (-1);

    /* Server process 'sysman', port 4990 (see /etc/services) */
    if ( (serverinfo = getservbyname("sysman", "tcp")) == NULL )
        return (-1);

    if ( (sock = socket(PF_INET, SOCK_STREAM, 0)) == -1 )
        return (-1);

    address.sin_family = AF_INET;
    address.sin_port   = serverinfo->s_port;
    address.sin_addr   = *(struct in_addr *)*hostinfo->h_addr_list;
    rc = connect(sock,(struct sockaddr *)&address, sizeof(struct sockaddr_in));
    if ( rc == -1 )
        return (-1);

    /* Send romid request msg to sysman */
    rqst[0] = 0x00;  /* 2 bytes length, in BE format */
    rqst[1] = 0x01;
    rqst[2] = 0x05;  /* ROMID */
    if ( write(sock, rqst, 3) == -1 )
    {
        close(sock);
        return (-1);
    }

    /* Receive the response */
    memset(resp, 0, sizeof(resp));
    if ( read(sock, resp, sizeof(resp)) == -1 )
    {
        close(sock);
        return (-1);
    }
    close(sock);

    if ( resp[0] || (resp[1] > 4) ) /* cap 2-byte BE length at 4 */
        resp[1] = 4;
    strncpy(id, &resp[3], resp[1]);
    return (resp[1]);

} /* _mesc_cfg_get_rom_id_from_sysman() */


/* ======================================================================= */
/**
 *  \brief Retrieves the real hardware ID that uniquely identifies each
 *      terminal, and formats it into printable ASCII characters.
 *  \return nothing.
 */
/* ======================================================================= */
static void _mesc_get_real_hw_id(void)
{
    unsigned int hwid[8];
    FILE *f;

    memset(hwid, 0, sizeof(hwid));

    /* Get 8 digits of serial number */
    if ( (f = fopen(ALTURA_HARDWARE_ID_FILE, "r")) == NULL )
        assert((f = fopen(LVT_HARDWARE_ID_FILE, "r")) != NULL);
    fscanf(f, "%d %d %d %d %d %d %d %d",
        &hwid[0], &hwid[1], &hwid[2], &hwid[3],
        &hwid[4], &hwid[5], &hwid[6], &hwid[7]);
    fclose(f);

    snprintf(_mesc_cfg_misc.hw_id_prefix, sizeof(_mesc_cfg_misc.hw_id_prefix),
        "%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X:",
        hwid[0], hwid[1], hwid[2], hwid[3],
        hwid[4], hwid[5], hwid[6], hwid[7]);
    _mesc_cfg_misc.hw_id_prefix[sizeof(_mesc_cfg_misc.hw_id_prefix) - 1] = '\0';
} /* _mesc_get_real_hw_id() */
#endif /* !ARCH_ESPAD */

/* ======================================================================= */
/**
 *  \brief Sets the IP address and hardware ID prefix as the ethernet address
 *      of the pad.  Later, the drop address will be appended as the virtual
 *      hardware id of the specific drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_cfg_set_ip_addr_and_hw_id_prefix(void)
{
    int s;
    struct ifreq ifr;
    struct sockaddr_in *addr;
#ifdef ARCH_ESPAD
    u_int8_t *hwaddr;
#endif /* ARCH_ESPAD */

    /* Create a dummy socket. */
    assert((s = socket(AF_INET, SOCK_DGRAM, 0)) != -1);
    strcpy(ifr.ifr_name, "eth0");
    ifr.ifr_addr.sa_family = AF_INET;

    /* Get the IP address */
    while ( ioctl(s, SIOCGIFADDR, &ifr) == -1 )
        sleep(1);
    addr = (struct sockaddr_in *)&ifr.ifr_addr;
    snprintf(_mesc_cfg_misc.ip_addr, sizeof(_mesc_cfg_misc.ip_addr),
        "%s", inet_ntoa(addr->sin_addr));
    _mesc_cfg_misc.ip_addr[sizeof(_mesc_cfg_misc.ip_addr) - 1] = '\0';/*2bsure*/
    info("IP address is %s", _mesc_cfg_misc.ip_addr);

#ifdef ARCH_ESPAD
    /* Get the ethernet address for the interface & save as our hw_id prefix */
    assert(ioctl(s, SIOCGIFHWADDR, &ifr) != -1);
    hwaddr = (u_int8_t *)&ifr.ifr_addr;
    snprintf(_mesc_cfg_misc.hw_id_prefix, sizeof(_mesc_cfg_misc.hw_id_prefix),
        "%02X:%02X:%02X:%02X:%02X:%02X:",
        *(hwaddr + 2), *(hwaddr + 3), *(hwaddr + 4),
        *(hwaddr + 5), *(hwaddr + 6), *(hwaddr + 7));
    _mesc_cfg_misc.hw_id_prefix[sizeof(_mesc_cfg_misc.hw_id_prefix) - 1] = '\0';
    assert(close(s) != -1);
#else /* !ARCH_ESPAD */
    _mesc_get_real_hw_id();
#endif /* !ARCH_ESPAD */
    info("Hardware ID prefix set to %s", _mesc_cfg_misc.hw_id_prefix);

} /* _mesc_cfg_set_ip_addr_and_hw_id_prefix() */


/* ======================================================================= */
/**
 *  \brief Register a new procedure.
 *  \param name Procedure name.
 *  \param proc Function to register.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_cfg_proc(char *name,  void(*proc)(list))
{
    _mesc_cfg_misc.proc_list = acons(name, proc, _mesc_cfg_misc.proc_list);
} /* _mesc_cfg_proc() */


/* ======================================================================= */
/**
 *  \brief Sets the minimum response time.
 *  \param l Name value pair list.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _t0(list l)
{
    char *v = (char *)nth(1, l);

    if ( v )
    {
        _mesc_cfg_misc.cfg.flags |= MESC_CFG_T0;
        _mesc_cfg_misc.cfg.t0 = strtoul(v, 0, 10);
    }
} /* _t0() */


/* ======================================================================= */
/**
 *  \brief Sets the maximum response time.
 *  \param l Name value pair list.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _t1(list l)
{
    char *v = (char *)nth(1, l);

    if ( v )
    {
        _mesc_cfg_misc.cfg.flags |= MESC_CFG_T1;
        _mesc_cfg_misc.cfg.t1 = strtoul(v, 0, 10);
    }
} /* _t1() */


/* ======================================================================= */
/**
 *  \brief Sets the timeout count.
 *  \param l Name value pair list.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _tc(list l)
{
    char *v = (char *)nth(1, l);

    if ( v )
    {
        _mesc_cfg_misc.cfg.flags |= MESC_CFG_TC;
        _mesc_cfg_misc.cfg.tc = strtoul(v, 0, 10);
    }
} /* _tc() */


/*
 * End of "$Id: mesc_cfg.c,v 1.14 2005/07/27 15:12:06 cmayncvs Exp $".
 */


