/* ===[ $RCSfile: mesc_dbg.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc_dbg.c,v 1.2 2005/02/17 20:17:41 cmayncvs Exp $"
 *
 *  \brief Implements the debug functions.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include "libcom/buf.h"
#include "libcom/c_config.h"
#include "gdebug.h"
#include "gfault.h"
#include "mesc_dbg.h"
#include "mesc_cfg.h"

/* =============== */
/* Local variables */
/* =============== */
/** Debug level. */
u_int8_t mesc_dbg_level = PRINT_LEVEL;

static u_int8_t _mesc_dbg_fw2comm(u_int8_t level);

/* ======================================================================= */
/**
 *  \brief Initialize debug.
 *  \param level If >=0, the forced debug level.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_dbg_init(int level)
{
    /* Set the debug level, either a forced level, or our configured one. */
    if ( level >= 0 )
        mesc_dbg_set_level((u_int8_t)level);
    else
        mesc_dbg_set_level(mesc_cfg_get_dbg_level());
    info("dbg_init(): Done.");

} /* mesc_dbg_init() */


/* ======================================================================= */
/**
 *  \brief Set debug level in this module and the underlying shared libraries.
 *  \param level Debug level.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_dbg_set_level(u_int8_t level)
{
    mesc_dbg_level = _mesc_dbg_fw2comm(level);
    warn("Debug level changed to %s (%X).",
        mesc_dbg_level2str(mesc_dbg_level), mesc_dbg_level);

    f_fault_set_debug_level(level);
    c_config_set_debug_level(level);

} /* mesc_dbg_set_level() */


/* ======================================================================= */
/**
 *  \brief Get MESC debug level.
 *  \return The debug level (see debug.h for comm values).
 */
/* ======================================================================= */
u_int8_t mesc_dbg_get_level(void)
{
    return (mesc_dbg_level);
} /* mesc_dbg_get_level() */


/* ======================================================================= */
/**
 *  \brief Get current time.
 *  \return Time in milliseconds.
 */
/* ======================================================================= */
u_int32_t mesc_dbg_time(void)
{
    struct timeval t;

    gettimeofday (&t, 0);
    return ((u_int32_t)((t.tv_sec * 1000) + (t.tv_usec / 1000)));
} /* mesc_dbg_time() */


/* ======================================================================= */
/**
 *  \brief Print current time.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_dbg_time_stamp(void)
{
    info("Time is %u.", mesc_dbg_time());
} /* mesc_dbg_time_stamp() */


/* ======================================================================= */
/**
 *  \brief Print a buffer.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_dbg_print_buf(buf_t b)
{
    if ( mesc_dbg_level >= PRINT_LEVEL )
        buf_print(b);

} /* mesc_dbg_print_buf() */

                        /* --------------- */
                        /* Local functions */
                        /* --------------- */

/* ======================================================================= */
/**
 *  \brief Convert firmware debug level to comm debug level.
 *  \param level The firmware debug level (see debug.h for firmware values).
 *  \return The comm debug level (see debug.h for comm values).
 */
/* ======================================================================= */
static u_int8_t _mesc_dbg_fw2comm(u_int8_t level)
{
    static u_int8_t _fw2comm[8] = {
        [DBG_NONE] NONE_LEVEL,
        [DBG_ALL] PRINT_LEVEL,
        [DBG_INFO4] PRINT_LEVEL,
        [DBG_INFO3] TRACE_LEVEL,
        [DBG_INFO2] TRACE_LEVEL,
        [DBG_INFO1] INFO_LEVEL,
        [DBG_WARN] WARN_LEVEL,
        [DBG_ERR] WARN_LEVEL
    };
    return (_fw2comm[level & 0x07]);

} /* _mesc_dbg_fw2comm() */


/* ======================================================================= */
/**
 *  \brief Returns a human-readable text string of the debug level.
 *  \param level The debug level.
 *  \return The string.
 */
/* ======================================================================= */
char *mesc_dbg_level2str(u_int8_t level)
{
    switch ( level )
    {
        case NONE_LEVEL:    return ("NONE");
        case WARN_LEVEL:    return ("WARN");
        case INFO_LEVEL:    return ("INFO");
        case TRACE_LEVEL:   return ("TRACE");
        case PRINT_LEVEL:   return ("PRINT");
        default:            return ("UNKNOWN");
    }
} /* mesc_dbg_level2str() */


/*
 * End of $Id: mesc_dbg.c,v 1.2 2005/02/17 20:17:41 cmayncvs Exp $
 */


