/* ===[ $RCSfile: mesc_fsm.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc_fsm.c,v 1.13 2005/04/22 17:04:03 cmayncvs Exp $"
 *
 *  \brief Multi-drop ES Connect Finite State Machine (MESC FSM).
 *      See the mesc.txt file for details of the state machine.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include "libcom/buf.h"
#include "libcom/list.h"
#include "libcom/str.h"
#include "gassert.h"
#include "mesc_dbg.h"
#include "mesc_cfg.h"
#include "mesc_stat.h"
#include "mesc_art.h"
#include "mesc_imp.h"
#include "mesc_tmr.h"
#include "mesc_fsm.h"
#include "mesc_udp.h"
#include "mesc.h"

/* ======= */
/* Defines */
/* ======= */
/* ESCP defines */
#include "escp.h"

/* states: _MESC_FSM_STATE */
#define _INIT                           0       /**< INIT state */
#define _CS1                            1       /**< CS1 state */
#define _CS2                            2       /**< CS2 state */
#define _IDLE                           3       /**< IDLE state */
#define _PENDING                        4       /**< PENDING state */
#define _DISABLED                       5       /**< DISABLED state */
#define _NUM_STATES                     6       /**< The number of states */

/* Inputs */
#define _RESP_TMO                       0       /**< response timeout */
#define _KEEPALIVE_TMO                  1       /**< keepalive timeout */
#define _DISABLE_TMO                    2       /**< disable timeout */
#define _NUM_TMOS                       3       /**< The # of timers */

/* ESC Inputs  */
#define _CFG_RQRD                       0       /**< Config Required */
#define _CFG_STEP1_RESP                 1       /**< Config Step 1 */
#define _CFG_STEP2_RESP                 2       /**< Config Step 2 */
#define _KEEPALIVE_RESP                 3       /**< Keep-Alive */
#define _NUM_PROTO_INPUTS               4       /**< The # of proto inputs */

/** Down is the disabled state due to L2 down, as notified by "ethmon" */
#define _DOWN                           0
/** Offline is the disabled state due to being disabled from the host */
#define _OFFLINE                        1
/** Online is everything but disabled */
#define _ONLINE                         2
/** Online backup.  Not sure if we'll ever use this, but ... */
#define _ONLINE_BACKUP                  3

#define _DSAP_INVALID                   255     /**< Invalid DSAP */
#define _ERROR                          (-1)    /**< local error define */

/* X.42 Polling status */
#define _SLOW_POLLING                   0       /**< Drop is in slow poll */
#define _POLLING                        1       /**< Drop responding to polls */

/** \brief Typedef for Multi-drop ESCP finite state machine per drop data. */
typedef struct {
    int state;              /**< INIT, CS1, CS2, IDLE, PENDING, or DISABLED */
    int status;             /**< _ONLINE or _OFFLINE */
    int configured;         /**< Boolean flag configured or not */
    buf_q tx_q;             /**< Queue of requests */
    u_int8_t dsap;          /**< DCP Service Access Point */
    u_int8_t hsap;          /**< Host Service Access Point */
    u_int16_t nat_port;     /**< Host's idea of local port (NAT port) */
    u_int8_t seq_no;        /**< Host/client sequence number */
    u_int16_t disable_tmo;  /**< Disable tmo (0=enable, 0xFFFF=disable 4eva) */
    u_int16_t keepalive_tmo;/**< Keep-alive tmo (seconds) */
    u_int32_t tmo_cnt;      /**< The timeout counter value ("tc" of 1.7.4) */
    u_int32_t cur_tmo_cnt;  /**< The current timeout counter value */
    u_int8_t poll_status;   /**< The poll status: 0=slow, 1=normal polling */
} _mesc_fsm_t;

/** \brief Common physical link status for all drops */
static int _l2_status = _ONLINE;    /* assume ok until proven otherwise */

/** \brief Typedef for Multi-drop ESCP common ACN data. */
typedef struct {
    u_int8_t len;       /**< Application Configuration Name length */
    char data[256];     /**< ACN for multicast unso filter (if not empty) */
} _mesc_acn_t;

/** Typedef for option flags */
typedef struct {
    u_int8_t cntrl_opts;    /**< Indicates by bit position, which opts */
    u_int8_t mask;          /**< the mask */
    u_int16_t disable;      /**< the disable timeout (seconds) 0=enable */
    u_int16_t keepalive;    /**< keep-alive timeout (seconds) */
} _opt_t;

/** Host's idea of our IP address (NAT address) */
static char _mesc_fsm_nat_addr[64];
/** Holds the MESC common ACN data. */
static _mesc_acn_t _mesc_acn;
/** Holds the PAD-specific ACN data. */
static _mesc_acn_t _pad_acn;
/** Holds all the MESC finite state machine data. */
static _mesc_fsm_t _mesc_fsm[MESC_CFG_MAX_DROPS];
/** Array of request vectors. */
static void (*_mesc_fsm_rqst[_NUM_STATES])(u_int8_t, buf_t);
/** Array of timeout vectors. */
static void (*_mesc_fsm_tmo[_NUM_STATES][_NUM_TMOS])(u_int8_t);
/** Array of application unso vectors */
static void (*_mesc_fsm_app_unso[_NUM_STATES])(u_int8_t, u_int8_t *, int, int);
/** Array of unso config required vectors.  NOTE: There is no need for a
 * 2-dimensional array here because there's only one protocol unso, namely
 * Config Required.  If more protocol unso's are ever added, follow the
 * _mesc_fsm_proto_resp[][] format. */
static void (*_mesc_fsm_proto_unso[_NUM_STATES])(u_int8_t, u_int8_t *, int);
/** Array of application response vectors */
static void (*_mesc_fsm_app_resp[_NUM_STATES])(u_int8_t, u_int8_t *, int);
/** Array of protocol response vectors */
static void (*_mesc_fsm_proto_resp[_NUM_STATES][_NUM_PROTO_INPUTS])(u_int8_t, u_int8_t *, int);
/** Array of disable vectors */
static void (*_mesc_fsm_disable[_NUM_STATES])(u_int8_t, u_int16_t);
/** Array of TX Data Indicator vectors */
static void (*_mesc_fsm_tx_ind[_NUM_STATES])(u_int8_t);
/** Array of X.42 Drop Status vectors */
static void (*_mesc_fsm_x42_status[_NUM_STATES])(u_int8_t, const char *);
/** Array of Comm Status vectors */
static void (*_mesc_fsm_comm_status[_NUM_STATES])(u_int8_t, int);

/* Common functions */
static void _init_vectors(void);
static void _dispatch_tmo(u_int8_t drop, int input);
static void _dispatch_rqst(u_int8_t drop, buf_t b);
static void _dispatch_app_unso(u_int8_t drop, u_int8_t *data, int length, int dest);
static void _dispatch_proto_unso(u_int8_t drop, u_int8_t *data, int length);
static void _dispatch_proto_resp(u_int8_t drop, u_int8_t *data, int length);
static void _dispatch_app_resp(u_int8_t drop, u_int8_t *data, int length);
static void _dispatch_disable(u_int8_t drop, u_int16_t disable);
static void _dispatch_tx_ind(u_int8_t drop);
static void _dispatch_x42_drop_status(u_int8_t drop, const char *status);
static void _dispatch_comm_status(u_int8_t drop, int l2_status);

static void _set_state(u_int8_t drop, int state);
static void _set_status(u_int8_t drop, int status);
static char *_state2str(int state);
static char *_tmo2str(int input);
static char *_protoinput2str(int input);
static char *_status2str(int input);
static void _clear_queue(u_int8_t drop);
static void _disable(u_int8_t drop);
static void _l2_down(u_int8_t drop);
static void _send_app_rqst(u_int8_t drop);
static void _send_escp_rqst(u_int8_t drop, buf_t rqst, u_int8_t type);
static void _send_escp_cs1_rqst(u_int8_t drop);
static void _send_escp_cs2_rqst(u_int8_t drop);
static void _send_escp_keepalive_rqst(u_int8_t drop);
static buf_t _make_escp_cs1_rqst(u_int8_t drop);
static buf_t _make_escp_cs2_rqst(u_int8_t drop);
static buf_t _make_escp_keepalive_rqst(void);
static int _compare_target_name(int len, escp_unso_hdr_t *hdr, _mesc_acn_t *acn);
static int _parse_ucast_unso_header(u_int8_t drop, buf_t b, _opt_t *opt, const char *str);
static int _parse_mcast_unso_header(buf_t b, _opt_t *opt, int *dest);
static int _parse_resp_header(u_int8_t drop, buf_t b, _opt_t *opt);
static int _parse_options(u_int8_t type, u_int8_t *data, int len, _opt_t *opt);
static void _post_process_options(u_int8_t drop, _opt_t *opt);
static void _parse_cs1_comm_id(u_int8_t drop, u_int8_t *data, u_int8_t len);
static void _parse_cs1_config(u_int8_t drop, u_int8_t *data, u_int8_t len);
static u_int8_t _get_sap(buf_t b);
static void _set_sap(buf_t b, u_int8_t sap);
static void _send_imp_retry_msg(u_int8_t drop, const char *id);
static void _set_l2_status(int l2_status);
static int _get_l2_status(void);

/* These directly appear in the vector tables. */
/* Shared vectors */
static void _timer_do_nothing(u_int8_t drop);
static void _queue_rqst(u_int8_t drop, buf_t b);
static void _app_unso_do_nothing(u_int8_t drop, u_int8_t *data, int length, int dest);
static void _proto_unso_do_nothing(u_int8_t drop, u_int8_t *data, int length);
static void _resp_do_nothing(u_int8_t drop, u_int8_t *data, int length);
static void _disable_do_nothing(u_int8_t drop, u_int16_t data);
static void _tx_ind_do_nothing(u_int8_t drop);
static void _tx_ind(u_int8_t drop);
static void _app_unso(u_int8_t drop, u_int8_t *data, int length, int dest);
static void _x42_drop_status_do_nothing(u_int8_t drop, const char *status);
static void _cs_disable(u_int8_t drop, u_int16_t data);
static void _cs_x42_drop_status(u_int8_t drop, const char *status);
static void _cs_l2_status(u_int8_t drop, int l2_status);
/* INIT */
static void _init_rqst(u_int8_t drop, buf_t b);
static void _init_l2_status_do_nothing(u_int8_t, int l2_status);
/* CS1 */
static void _cs1_resp_tmo(u_int8_t drop);
static void _cs1_resp(u_int8_t drop, u_int8_t *data, int length);
/* CS2 */
static void _cs2_resp_tmo(u_int8_t drop);
static void _cs2_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length);
static void _cs2_resp(u_int8_t drop, u_int8_t *data, int length);
/* IDLE */
static void _idle_disable(u_int8_t drop, u_int16_t data);
static void _idle_keepalive_tmo(u_int8_t drop);
static void _idle_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length);
static void _idle_rqst(u_int8_t drop, buf_t b);
static void _idle_keepalive_resp(u_int8_t drop, u_int8_t *data, int length);
static void _idle_x42_drop_status(u_int8_t drop, const char *status);
static void _idle_l2_status(u_int8_t drop, int l2_status);
/* PENDING */
static void _pending_disable(u_int8_t drop, u_int16_t data);
static void _pending_resp_tmo(u_int8_t drop);
static void _pending_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length);
static void _pending_resp(u_int8_t drop, u_int8_t *data, int length);
static void _pending_l2_status(u_int8_t drop, int l2_status);
/* DISABLED */
static void _disabled_disable(u_int8_t drop, u_int16_t data);
static void _disabled_re_enable(u_int8_t drop);
static void _disabled_disable_tmo(u_int8_t drop);
static void _disabled_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length);
static void _disabled_rqst(u_int8_t drop, buf_t b);
static void _disabled_l2_status(u_int8_t drop, int l2_status);

/* ----------------------------------------------------------------------- */
/*                          SECTION: GLOBAL FUNCTIONS                      */
/* ----------------------------------------------------------------------- */

/* ======================================================================= */
/**
 *  \brief Initializes the MESC finite state machine.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_init(void)
{
    u_int8_t drop;

    /* FSM vectors */
    _init_vectors();

    /* Common state information */
    memset(&_mesc_acn, 0, sizeof(_mesc_acn_t));
    memset(&_pad_acn, 0, sizeof(_mesc_acn_t));

    /* Assume local IP address for now (i.e. assume no NAT) */
    snprintf(_mesc_fsm_nat_addr, sizeof(_mesc_fsm_nat_addr), "%s",
        mesc_cfg_get_ip_addr());
    _mesc_fsm_nat_addr[sizeof(_mesc_fsm_nat_addr)-1] = '\0';

    /* Drop-specific state information */
    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
    {
        /* FSM variables */
        _mesc_fsm[drop].state = _INIT;
        _mesc_fsm[drop].status = _ONLINE;
        _mesc_fsm[drop].configured = 0;
        _mesc_fsm[drop].tx_q = 0;
        _mesc_fsm[drop].dsap = _DSAP_INVALID;

        /* Request header:
         *   ART has previous response time,
         *   STAT has statistics option mask, client name, and
         *      # of requests, responses, and unsolicited's  */
        _mesc_fsm[drop].hsap = ESCP_SAP_UNKNOWN;
        _mesc_fsm[drop].nat_port = mesc_cfg_get_port(drop);
        _mesc_fsm[drop].seq_no = ESCP_SEQ_NO_UNKNOWN - 1;

        /* Response or Unso header */
        _mesc_fsm[drop].disable_tmo = ESCP_ENABLE;
        _mesc_fsm[drop].keepalive_tmo = mesc_cfg_get_keepalive_tmo();

        /* Config Step 1 Request/Response:
         *   STAT has client name
         *   CFG has ROM ID (a.k.a. device class)
         */

        _mesc_fsm[drop].tmo_cnt = mesc_cfg_get_tmo_cnt();   /* tc, sec 1.7.4 */
        _mesc_fsm[drop].cur_tmo_cnt = 0;
        _mesc_fsm[drop].poll_status = _POLLING;
    }

    info("fsm_init(): Done.");

} /* mesc_fsm_init() */


/* ======================================================================= */
/**
 *  \brief Print MESC state infomration for a drop to a file.
 *  \param f The file to write MESC state information to.
 *  \param drop The drop whose MESC state information to print.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_print(FILE *f, u_int8_t drop)
{
    char hw_id[32];

    assert(f && (drop < MESC_CFG_MAX_DROPS));
    mesc_cfg_get_hw_id(drop, hw_id);

    fprintf(f, "   State                             -> %s\n",
        _state2str(_mesc_fsm[drop].state));
    fprintf(f, "   Status                            -> %s\n",
        _status2str(_mesc_fsm[drop].status));
    fprintf(f, "   Configured                        -> %s\n",
        _mesc_fsm[drop].configured ? "Yes" : "No");
    fprintf(f, "   Queue'd requests                  -> %s\n",
        _mesc_fsm[drop].tx_q ? "Yes" : "No");
    fprintf(f, "   DSAP                              -> %d\n",
        _mesc_fsm[drop].dsap);
    fprintf(f, "   HSAP                              -> %d\n",
        _mesc_fsm[drop].hsap);
    fprintf(f, "   Sequence #                        -> %d\n",
        _mesc_fsm[drop].seq_no);
    fprintf(f, "   Disable Timeout                   -> %d %s\n",
        _mesc_fsm[drop].disable_tmo,
        (_mesc_fsm[drop].disable_tmo == 0) ? "(Enabled)" :
        (_mesc_fsm[drop].disable_tmo == 0xffff) ? "(Disabled \"forever\")" :
        "seconds");
    fprintf(f, "   Keepalive Timeout                 -> %s\n",
        sec2str(_mesc_fsm[drop].keepalive_tmo));
    fprintf(f, "   Host Timeout Count                -> %d\n",
        _mesc_fsm[drop].tmo_cnt);
    fprintf(f, "   Current Timeout Count             -> %d\n",
        _mesc_fsm[drop].cur_tmo_cnt);
    fprintf(f, "   Application Configuration Name    -> %s\n",
        (drop == MESC_PAD_DROP) ? _pad_acn.data : _mesc_acn.data);
    fprintf(f, "   ROM ID                            -> %s (%s)\n",
        (drop == MESC_PAD_DROP) ?
        mesc_cfg_get_pad_rom_id() : mesc_cfg_get_rom_id(),
        (drop == MESC_PAD_DROP) ?
        mesc_cfg_get_terminal_type(mesc_cfg_get_pad_rom_id()) :
        mesc_cfg_get_terminal_type(mesc_cfg_get_rom_id()));
    fprintf(f, "   Hardware ID                       -> %s\n",
        hw_id);
    fprintf(f, "   Primary Host                      -> %s\n",
        mesc_cfg_get_host1());
    fprintf(f, "   Secondary Host                    -> %s\n",
        mesc_cfg_get_host2());
    fprintf(f, "   Unicast Address:Port              -> %s:%d\n",
        mesc_cfg_get_ip_addr(), mesc_cfg_get_port(drop));
    fprintf(f, "   Unicast Address:Port (host view)  -> %s:%d\n",
        _mesc_fsm_nat_addr, _mesc_fsm[drop].nat_port);
} /* mesc_fsm_print() */


/* ======================================================================= */
/**
 *  \brief Process a transmit request from a drop
 *  \param drop The drop address.
 *  \param data The data to send
 *  \param len The number of data bytes
 *  \param sap The DCP service access point (sap)
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_process_tx_req(u_int8_t drop, char *data, int len, u_int8_t sap)
{
    buf_t b = NULL;

    if ( data && len )
    {
        b = buf_append(0, data, len);
        _set_sap(b, sap);
    }
    _dispatch_rqst(drop, b);

} /* mesc_fsm_process_tx_req() */


/* ======================================================================= */
/**
 *  \brief Process a response timeout.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_process_resp_tmo(u_int8_t drop)
{
    mesc_stat_update_resp_tmos(drop);
    _dispatch_tmo(drop, _RESP_TMO);
} /* mesc_fsm_process_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Process a keep-alive timeout.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_process_keepalive_tmo(u_int8_t drop)
{
    mesc_stat_update_keepalive_tmos(drop);
    _dispatch_tmo(drop, _KEEPALIVE_TMO);
} /* mesc_fsm_process_keepalive_tmo() */


/* ======================================================================= */
/**
 *  \brief Process a disable timeout.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_process_disable_tmo(u_int8_t drop)
{
    mesc_stat_update_disable_tmos(drop);
    _dispatch_tmo(drop, _DISABLE_TMO);
} /* mesc_fsm_process_disable_tmo() */


/* ======================================================================= */
/**
 *  \brief Process a unicast message from the host.
 *  \param drop The drop address.
 *  \param b The buffer.
 *  \return The buffer passed in.
 */
/* ======================================================================= */
buf_t mesc_fsm_process_unicast(u_int8_t drop, buf_t b)
{
    escp_unso_hdr_t *unso_hdr;
    escp_resp_hdr_t *resp_hdr;
    int len, remaining;
    _opt_t opt;
    char name[256];

    unso_hdr = (escp_unso_hdr_t *)buf_data(b);
    resp_hdr = (escp_resp_hdr_t *)buf_data(b);
    mesc_dbg_time_stamp();
    mesc_dbg_print_buf(b);

    /* Basic header checking */
    if ( (len = buf_length(b)) < 4 )
    {
        warn("Received message too small (%d)", len);
        return (b);
    }
    if ( unso_hdr->pid != ESCP_PROTOCOL_ID )
    {
        warn("Invalid protocol ID (%d)", unso_hdr->pid);
        return (b);
    }

    /* Pass the buffer down to the next layer */
    if ( ESCP_PACKET_TYPE(unso_hdr->type) == ESCP_PACKET_TYPE_UNSO )
    {
        mesc_cfg_get_client_name(drop, name);
        if ( (remaining = _parse_ucast_unso_header(drop, b, &opt, name)) > 0 )
        {
            if ( (ESCP_MESSAGE_TYPE(unso_hdr->type)) == ESCP_MESSAGE_TYPE_APP )
            {
                _post_process_options(drop, &opt);
                _dispatch_app_unso(drop,
                    (u_int8_t *)((u_int8_t *)unso_hdr + (len - remaining)), remaining, 0);
            }
            else /* ESCP_MESSAGE_TYPE_PROTO */
            {
                _post_process_options(drop, &opt);
                _dispatch_proto_unso(drop,
                    (u_int8_t *)((u_int8_t *)unso_hdr + (len - remaining)), remaining);
            }
        }
    }
    else if ( ESCP_PACKET_TYPE(resp_hdr->type) == ESCP_PACKET_TYPE_RESPONSE )
    {
        if ( (remaining = _parse_resp_header(drop, b, &opt)) > 0 )
        {
            if ( (ESCP_MESSAGE_TYPE(resp_hdr->type)) == ESCP_MESSAGE_TYPE_APP )
            {
                /* Post-process options after response delivery, in case we
                 * become disabled; otherwise if in pending state, app will not
                 * get its response.  If already in the disabled state, app
                 * will not be getting its response anyway. */
                _dispatch_app_resp(drop,
                    (u_int8_t *)((u_int8_t *)resp_hdr + (len - remaining)), remaining);
                _post_process_options(drop, &opt);
            }
            else /* ESCP_MESSAGE_TYPE_PROTO */
            {
                _post_process_options(drop, &opt);
                _dispatch_proto_resp(drop,
                    (u_int8_t *)((u_int8_t *)resp_hdr + (len - remaining)), remaining);
            }
        }
    }
    return (b);

} /* mesc_fsm_process_unicast() */


/* ======================================================================= */
/**
 *  \brief Process a multicast message from the host.
 *  \param b The message (better be an unso)
 *  \return The buffer passed in.
 */
/* ======================================================================= */
buf_t mesc_fsm_process_multicast(buf_t b)
{
    escp_unso_hdr_t *unso_hdr;
    u_int8_t drop;
    int len, remaining = 0;
    int dest = 0;
    _opt_t opt;

    unso_hdr = (escp_unso_hdr_t *)buf_data(b);
    mesc_dbg_time_stamp();
    mesc_dbg_print_buf(b);

    /* Basic header checking */
    if ( (len = buf_length(b)) < 4 )
    {
        warn("Received multicast message too small (%d)", len);
        return (b);
    }

    if ( ESCP_PACKET_TYPE(unso_hdr->type) == ESCP_PACKET_TYPE_UNSO )
    {
        if ( (remaining = _parse_mcast_unso_header(b, &opt, &dest)) > 0 )
        {
            if ( dest & MESC_COMMON_ACN )
            {
                for ( drop = 0; drop < MESC_PAD_DROP; drop++ )
                {
                    _post_process_options(drop, &opt);
                    if ( (ESCP_MESSAGE_TYPE(unso_hdr->type)) == ESCP_MESSAGE_TYPE_PROTO )
                        _dispatch_proto_unso(drop, (u_int8_t *)((u_int8_t *)unso_hdr + (len - remaining)), remaining);
                }
                if ( (ESCP_MESSAGE_TYPE(unso_hdr->type)) == ESCP_MESSAGE_TYPE_APP )
                    _dispatch_app_unso(255, (u_int8_t *)((u_int8_t *)unso_hdr + (len - remaining)), remaining, MESC_COMMON_ACN);
            }
            if ( dest & MESC_PAD_ACN )
            {
                _post_process_options(MESC_PAD_DROP, &opt);
                if ( (ESCP_MESSAGE_TYPE(unso_hdr->type)) == ESCP_MESSAGE_TYPE_PROTO )
                    _dispatch_proto_unso(MESC_PAD_DROP, (u_int8_t *)((u_int8_t *)unso_hdr + (len - remaining)), remaining);
                if ( (ESCP_MESSAGE_TYPE(unso_hdr->type)) == ESCP_MESSAGE_TYPE_APP )
                    _dispatch_app_unso(255, (u_int8_t *)((u_int8_t *)unso_hdr + (len - remaining)), remaining, MESC_PAD_ACN);
            }
        }
    }
    return (b);

} /* mesc_fsm_process_multicast() */


/* ======================================================================= */
/**
 *  \brief Process a tx-data-ind from a drop.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_process_tx_data_ind(u_int8_t drop)
{
    _dispatch_tx_ind(drop);

} /* mesc_fsm_process_tx_data_ind() */


/* ======================================================================= */
/**
 *  \brief Process an x42-drop-status message for a drop.
 *  \param drop The drop address.
 *  \param status The poll status of this drop.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_process_x42_drop_status(u_int8_t drop, const char *status)
{
    _dispatch_x42_drop_status(drop, status);
} /* mesc_fsm_process_x42_drop_status() */


/* ======================================================================= */
/**
 *  \brief Process a comm-status message.
 *  \param l2_status The comm L2 status, as either _DOWN or _ONLINE.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_fsm_process_comm_status(int l2_status)
{
    u_int8_t drop;

    if ( (l2_status == _DOWN) || (l2_status == _OFFLINE) )
    {
        _set_l2_status(_DOWN);
        mesc_imp_send_status_msg(255, "Message=ConnectFailed;Cause=Link Down");
        mesc_imp_send_status(255, _DOWN);
    }
    else    /* if ((l2_status == _ONLINE) || (l2_status == _ONLINE_BACKUP)) */
    {
        _set_l2_status(_ONLINE);
        mesc_imp_send_status_msg(255, "Message=ConnectedDiag;Diagnostic=Link Up");
    }

    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
        _dispatch_comm_status(drop, l2_status);

} /* mesc_fsm_process_comm_status() */


/* ----------------------------------------------------------------------- */
/*                      SECTION: LOCAL FUNCTIONS                           */
/* ----------------------------------------------------------------------- */

/* ======================================================================= */
/**
 *  \brief Initializes the MESC finite state machine vector table.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _init_vectors(void)
{
    /* INIT */
    _mesc_fsm_rqst       [_INIT]                        = _init_rqst;
    _mesc_fsm_tmo        [_INIT][_RESP_TMO]             = _timer_do_nothing;
    _mesc_fsm_tmo        [_INIT][_KEEPALIVE_TMO]        = _timer_do_nothing;
    _mesc_fsm_tmo        [_INIT][_DISABLE_TMO]          = _timer_do_nothing;
    _mesc_fsm_app_unso   [_INIT]                        = _app_unso_do_nothing;
    _mesc_fsm_proto_unso [_INIT]                        = _proto_unso_do_nothing;
    _mesc_fsm_app_resp   [_INIT]                        = _resp_do_nothing;
    _mesc_fsm_proto_resp [_INIT][_CFG_RQRD]             = _resp_do_nothing;
    _mesc_fsm_proto_resp [_INIT][_CFG_STEP1_RESP]       = _resp_do_nothing;
    _mesc_fsm_proto_resp [_INIT][_CFG_STEP2_RESP]       = _resp_do_nothing;
    _mesc_fsm_proto_resp [_INIT][_KEEPALIVE_RESP]       = _resp_do_nothing;
    _mesc_fsm_disable    [_INIT]                        = _disable_do_nothing;
    _mesc_fsm_tx_ind     [_INIT]                        = _tx_ind_do_nothing;
    _mesc_fsm_x42_status [_INIT]                        = _x42_drop_status_do_nothing;
    _mesc_fsm_comm_status[_INIT]                        = _init_l2_status_do_nothing;

    /* CS1 */
    _mesc_fsm_rqst       [_CS1]                         = _queue_rqst;
    _mesc_fsm_tmo        [_CS1][_RESP_TMO]              = _cs1_resp_tmo;
    _mesc_fsm_tmo        [_CS1][_KEEPALIVE_TMO]         = _timer_do_nothing;
    _mesc_fsm_tmo        [_CS1][_DISABLE_TMO]           = _timer_do_nothing;
    _mesc_fsm_app_unso   [_CS1]                         = _app_unso;
    _mesc_fsm_proto_unso [_CS1]                         = _proto_unso_do_nothing;
    _mesc_fsm_app_resp   [_CS1]                         = _resp_do_nothing;
    _mesc_fsm_proto_resp [_CS1][_CFG_RQRD]              = _resp_do_nothing;
    _mesc_fsm_proto_resp [_CS1][_CFG_STEP1_RESP]        = _cs1_resp;
    _mesc_fsm_proto_resp [_CS1][_CFG_STEP2_RESP]        = _resp_do_nothing;
    _mesc_fsm_proto_resp [_CS1][_KEEPALIVE_RESP]        = _resp_do_nothing;
    _mesc_fsm_disable    [_CS1]                         = _cs_disable;
    _mesc_fsm_tx_ind     [_CS1]                         = _tx_ind;
    _mesc_fsm_x42_status [_CS1]                         = _cs_x42_drop_status;
    _mesc_fsm_comm_status[_CS1]                         = _cs_l2_status;

    /* CS2 */
    _mesc_fsm_rqst       [_CS2]                         = _queue_rqst;
    _mesc_fsm_tmo        [_CS2][_RESP_TMO]              = _cs2_resp_tmo;
    _mesc_fsm_tmo        [_CS2][_KEEPALIVE_TMO]         = _timer_do_nothing;
    _mesc_fsm_tmo        [_CS2][_DISABLE_TMO]           = _timer_do_nothing;
    _mesc_fsm_app_unso   [_CS2]                         = _app_unso;
    _mesc_fsm_proto_unso [_CS2]                         = _cs2_cfg_rqrd;
    _mesc_fsm_app_resp   [_CS2]                         = _resp_do_nothing;
    _mesc_fsm_proto_resp [_CS2][_CFG_RQRD]              = _cs2_cfg_rqrd;
    _mesc_fsm_proto_resp [_CS2][_CFG_STEP1_RESP]        = _resp_do_nothing;
    _mesc_fsm_proto_resp [_CS2][_CFG_STEP2_RESP]        = _cs2_resp;
    _mesc_fsm_proto_resp [_CS2][_KEEPALIVE_RESP]        = _resp_do_nothing;
    _mesc_fsm_disable    [_CS2]                         = _cs_disable;
    _mesc_fsm_tx_ind     [_CS2]                         = _tx_ind;
    _mesc_fsm_x42_status [_CS2]                         = _cs_x42_drop_status;
    _mesc_fsm_comm_status[_CS2]                         = _cs_l2_status;

    /* IDLE */
    _mesc_fsm_rqst       [_IDLE]                        = _idle_rqst;
    _mesc_fsm_tmo        [_IDLE][_RESP_TMO]             = _timer_do_nothing;
    _mesc_fsm_tmo        [_IDLE][_KEEPALIVE_TMO]        = _idle_keepalive_tmo;
    _mesc_fsm_tmo        [_IDLE][_DISABLE_TMO]          = _timer_do_nothing;
    _mesc_fsm_app_unso   [_IDLE]                        = _app_unso;
    _mesc_fsm_proto_unso [_IDLE]                        = _idle_cfg_rqrd;
    _mesc_fsm_app_resp   [_IDLE]                        = _resp_do_nothing;
    _mesc_fsm_proto_resp [_IDLE][_CFG_RQRD]             = _idle_cfg_rqrd;
    _mesc_fsm_proto_resp [_IDLE][_CFG_STEP1_RESP]       = _resp_do_nothing;
    _mesc_fsm_proto_resp [_IDLE][_CFG_STEP2_RESP]       = _resp_do_nothing;
    _mesc_fsm_proto_resp [_IDLE][_KEEPALIVE_RESP]       = _idle_keepalive_resp;
    _mesc_fsm_disable    [_IDLE]                        = _idle_disable;
    _mesc_fsm_tx_ind     [_IDLE]                        = _tx_ind_do_nothing;
    _mesc_fsm_x42_status [_IDLE]                        = _idle_x42_drop_status;
    _mesc_fsm_comm_status[_IDLE]                        = _idle_l2_status;

    /* PENDING */
    _mesc_fsm_rqst       [_PENDING]                     = _queue_rqst;
    _mesc_fsm_tmo        [_PENDING][_RESP_TMO]          = _pending_resp_tmo;
    _mesc_fsm_tmo        [_PENDING][_KEEPALIVE_TMO]     = _timer_do_nothing;
    _mesc_fsm_tmo        [_PENDING][_DISABLE_TMO]       = _timer_do_nothing;
    _mesc_fsm_app_unso   [_PENDING]                     = _app_unso;
    _mesc_fsm_proto_unso [_PENDING]                     = _pending_cfg_rqrd;
    _mesc_fsm_app_resp   [_PENDING]                     = _pending_resp;
    _mesc_fsm_proto_resp [_PENDING][_CFG_RQRD]          = _pending_cfg_rqrd;
    _mesc_fsm_proto_resp [_PENDING][_CFG_STEP1_RESP]    = _resp_do_nothing;
    _mesc_fsm_proto_resp [_PENDING][_CFG_STEP2_RESP]    = _resp_do_nothing;
    _mesc_fsm_proto_resp [_PENDING][_KEEPALIVE_RESP]    = _resp_do_nothing;
    _mesc_fsm_disable    [_PENDING]                     = _pending_disable;
    _mesc_fsm_tx_ind     [_PENDING]                     = _tx_ind;
    _mesc_fsm_x42_status [_PENDING]                     = _x42_drop_status_do_nothing;
    _mesc_fsm_comm_status[_PENDING]                     = _pending_l2_status;

    /* DISABLED */
    _mesc_fsm_rqst       [_DISABLED]                    = _disabled_rqst;
    _mesc_fsm_tmo        [_DISABLED][_RESP_TMO]         = _timer_do_nothing;
    _mesc_fsm_tmo        [_DISABLED][_KEEPALIVE_TMO]    = _timer_do_nothing;
    _mesc_fsm_tmo        [_DISABLED][_DISABLE_TMO]      = _disabled_disable_tmo;
    _mesc_fsm_app_unso   [_DISABLED]                    = _app_unso;
    _mesc_fsm_proto_unso [_DISABLED]                    = _disabled_cfg_rqrd;
    _mesc_fsm_app_resp   [_DISABLED]                    = _resp_do_nothing;
    _mesc_fsm_proto_resp [_DISABLED][_CFG_RQRD]         = _disabled_cfg_rqrd;
    _mesc_fsm_proto_resp [_DISABLED][_CFG_STEP1_RESP]   = _resp_do_nothing;
    _mesc_fsm_proto_resp [_DISABLED][_CFG_STEP2_RESP]   = _resp_do_nothing;
    _mesc_fsm_proto_resp [_DISABLED][_KEEPALIVE_RESP]   = _resp_do_nothing;
    _mesc_fsm_disable    [_DISABLED]                    = _disabled_disable;
    _mesc_fsm_tx_ind     [_DISABLED]                    = _tx_ind_do_nothing;
    _mesc_fsm_x42_status [_DISABLED]                    = _x42_drop_status_do_nothing;
    _mesc_fsm_comm_status[_DISABLED]                    = _disabled_l2_status;

} /* _init_vectors() */

                        /* ------------------------- */
                        /* SECTION: Common functions */
                        /* ------------------------- */

/* ======================================================================= */
/**
 *  \brief Call timer action for current state and given input.
 *  \param drop The drop address.
 *  \param input Input.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_tmo(u_int8_t drop, int input)
{
    trace("%s(%d): %s ----------v",
        _state2str(_mesc_fsm[drop].state), drop, _tmo2str(input));
    _mesc_fsm_tmo[_mesc_fsm[drop].state][input](drop);
} /* _dispatch_tmo() */


/* ======================================================================= */
/**
 *  \brief Call request action for given state and input.
 *  \param drop The drop address.
 *  \param b Message.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_rqst(u_int8_t drop, buf_t b)
{
    trace("%s(%d): REQUEST ----------v",
        _state2str(_mesc_fsm[drop].state), drop);
    _mesc_fsm_rqst[_mesc_fsm[drop].state](drop, b);
} /* _dispatch_rqst() */


/* ======================================================================= */
/**
 *  \brief Call app unso action for given state.
 *  \param drop The drop address.
 *  \param data Data.
 *  \param length Length
 *  \param dest The destination - applicable for multicast only.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_app_unso(u_int8_t drop, u_int8_t *data, int length, int dest)
{
    if ( drop == 255 )  /* multicast unso */
    {
        trace("MULTICAST APP UNSO ----------v");
        _app_unso(255, data, length, dest);
    }
    else
    {
        trace("%s(%d): APP UNSO ----------v",
            _state2str(_mesc_fsm[drop].state), drop);
        _mesc_fsm_app_unso[_mesc_fsm[drop].state](drop, data, length, 0);
    }

} /* _dispatch_app_unso() */


/* ======================================================================= */
/**
 *  \brief Call proto unso action for given state.
 *  \param drop The drop address.
 *  \param data Data.
 *  \param length Length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_proto_unso(u_int8_t drop, u_int8_t *data, int length)
{
    assert(data);
    trace("%s(%d): PROTO UNSO ----------v",
        _state2str(_mesc_fsm[drop].state), drop);
    if ( (*data != ESCP_CONFIG_REQUIRED) || (length != 1) )
    {
        warn("Invalid protocol unso (len=%d).", length);
        _set_state(drop, _mesc_fsm[drop].state);
        return;
    }
    _mesc_fsm_proto_unso[_mesc_fsm[drop].state](drop, data, length);
} /* _dispatch_proto_unso() */


/* ======================================================================= */
/**
 *  \brief Call app resp action for given state.
 *  \param drop The drop address.
 *  \param data Data.
 *  \param length Length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_app_resp(u_int8_t drop, u_int8_t *data, int length)
{
    _mesc_fsm[drop].cur_tmo_cnt = 0;
    trace("%s(%d): APP RESP ----------v",
        _state2str(_mesc_fsm[drop].state), drop);
    _mesc_fsm_app_resp[_mesc_fsm[drop].state](drop, data, length);
} /* _dispatch_app_resp() */


/* ======================================================================= */
/**
 *  \brief Call proto resp action for given state.
 *  \param drop The drop address.
 *  \param data Data.
 *  \param length Length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_proto_resp(u_int8_t drop, u_int8_t *data, int length)
{
    int input;

    assert(data);
    switch ( *data )
    {
        case ESCP_CONFIG_REQUIRED:  input = _CFG_RQRD; break;
        case ESCP_CONFIG_STEP_1:    input = _CFG_STEP1_RESP; break;
        case ESCP_CONFIG_STEP_2:    input = _CFG_STEP2_RESP; break;
        case ESCP_KEEP_ALIVE:       input = _KEEPALIVE_RESP; break;
        default:
            trace("%s(%d): UNKNOWN PROTO RESP (0x%02X) ----------v",
                _state2str(_mesc_fsm[drop].state), drop, *data);
            _set_state(drop, _mesc_fsm[drop].state);
            return;
    }
    /*
     * NOTE: Here we don't necessarily match the response with the specific
     * request, so it's possible that this response is not what we're looking
     * for; however since we're receiving a response from this SAP regardless,
     * we know we're connected, so no need to force a rotation of hosts,
     * therefore it's reasonable to reset the current timeout count here.
     */
    _mesc_fsm[drop].cur_tmo_cnt = 0;
    trace("%s(%d): %s ----------v",
        _state2str(_mesc_fsm[drop].state), drop, _protoinput2str(input));
    _mesc_fsm_proto_resp[_mesc_fsm[drop].state][input](drop, data, length);
} /* _dispatch_proto_resp() */


/* ======================================================================= */
/**
 *  \brief Call disable action for the given state.
 *  \param drop The drop address.
 *  \param data The new disable timeout (0=enable, 0xFFFF=4eva)
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_disable(u_int8_t drop, u_int16_t data)
{
    _mesc_fsm_disable[_mesc_fsm[drop].state](drop, data);
} /* _dispatch_disable() */


/* ======================================================================= */
/**
 *  \brief Call tx-data-ind action for the given state.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_tx_ind(u_int8_t drop)
{
    _mesc_fsm_tx_ind[_mesc_fsm[drop].state](drop);
} /* _dispatch_tx_ind() */


/* ======================================================================= */
/**
 *  \brief Call x42-drop-status action for the given state.
 *  \param drop The drop address.
 *  \param status The drop status as either "polling" or "slow-polling".
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_x42_drop_status(u_int8_t drop, const char *status)
{
    trace("%s(%d): X.42 POLL STATUS ----------v",
        _state2str(_mesc_fsm[drop].state), drop);
    _mesc_fsm_x42_status[_mesc_fsm[drop].state](drop, status);
    _set_state(drop, _mesc_fsm[drop].state);
} /* _dispatch_x42_drop_status() */


/* ======================================================================= */
/**
 *  \brief Call comm-status action for the given state of each drop.
 *  \param drop The drop address.
 *  \param l2_status The comm L2 status as either _DOWN or _ONLINE.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_comm_status(u_int8_t drop, int l2_status)
{
    _mesc_fsm_comm_status[_mesc_fsm[drop].state](drop, l2_status);
} /* _dispatch_comm_status() */


/* ======================================================================= */
/**
 *  \brief Set current state.
 *  \param drop The drop address.
 *  \param state State.
 */
/* ======================================================================= */
static void _set_state(u_int8_t drop, int state)
{
    _mesc_fsm[drop].state = state;
    trace("v----------> Current State(%d) -> %s\n", drop, _state2str(state));
} /* _set_state() */


/* ======================================================================= */
/**
 *  \brief Set the status and send a status message
 *  \param drop The drop address.
 *  \param status The status - either _ONLINE or _OFFLINE.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _set_status(u_int8_t drop, int status)
{
    _mesc_fsm[drop].status = status;
    mesc_imp_send_status(drop, status);
    info("Status = %s.", _status2str(status));

} /* _set_status() */


/* ======================================================================= */
/**
 *  \brief Convert state to string.
 *  \param state State.
 *  \return State string.
 */
/* ======================================================================= */
static char *_state2str(int state)
{
    switch ( state )
    {
        case _INIT:     return ("INIT");
        case _CS1:      return ("CONFIG STEP 1");
        case _CS2:      return ("CONFIG STEP 2");
        case _IDLE:     return ("IDLE");
        case _PENDING:  return ("PENDING");
        case _DISABLED: return ("DISABLED");
        default:        return ("UNKNOWN STATE");
    }
} /* _state2str() */


/* ======================================================================= */
/**
 *  \brief Convert timeout input to string.
 *  \param input Input.
 *  \return Input string.
 */
/* ======================================================================= */
static char *_tmo2str(int input)
{
    switch ( input )
    {
        case _RESP_TMO:         return ("RESPONSE TMO");
        case _KEEPALIVE_TMO:    return ("KEEPALIVE TMO");
        case _DISABLE_TMO:      return ("DISABLE TMO");
        default:                return ("UNKNOWN TMO");
    }
} /* _tmo2str() */


/* ======================================================================= */
/**
 *  \brief Convert protocol input to string.
 *  \param input protocol iput.
 *  \return Input string.
 */
/* ======================================================================= */
static char *_protoinput2str(int input)
{
    switch ( input )
    {
        case _CFG_RQRD:         return ("CONFIG REQUIRED");
        case _CFG_STEP1_RESP:   return ("CONFIG STEP 1 RESP");
        case _CFG_STEP2_RESP:   return ("CONFIG STEP 2 RESP");
        case _KEEPALIVE_RESP:   return ("KEEP ALIVE RESP");
        default:                return ("UNKNOWN PROTOCOL INPUT");
    }
} /* _protoinput2str() */


/* ======================================================================= */
/**
 *  \brief Convert status input to string.
 *  \param input status input.
 *  \return string.
 */
/* ======================================================================= */
static char *_status2str(int input)
{
    switch ( input )
    {
        case _DOWN:             return ("DOWN");
        case _OFFLINE:          return ("OFFLINE");
        case _ONLINE:           return ("ONLINE");
        case _ONLINE_BACKUP:    return ("ONLINE BACKUP");
        default:                return ("UNKNOWN STATUS");
    }
} /* _status2str() */


/* ======================================================================= */
/**
 *  \brief Clear the queue'd requests for this drop and send response timeouts.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _clear_queue(u_int8_t drop)
{
    buf_t b;

    while ( _mesc_fsm[drop].tx_q )
    {
        b = buf_dequeue(_mesc_fsm[drop].tx_q);
        mesc_imp_send_rsp_tmo(drop, _get_sap(b));
        buf_free(b);
    }
} /* _clear_queue() */


/* ======================================================================= */
/**
 *  \brief Disable this drop.
 *  \param drop The drop to disable
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disable(u_int8_t drop)
{
    _set_status(drop, _OFFLINE);
    _mesc_fsm[drop].dsap = _DSAP_INVALID;
    _mesc_fsm[drop].poll_status = _POLLING;
    mesc_imp_send_status_msg(drop, "Message=MasterDisable");
    _clear_queue(drop);
    if ( _mesc_fsm[drop].disable_tmo != ESCP_DISABLE_FOREVER )
        mesc_tmr_start_disable(drop, _mesc_fsm[drop].disable_tmo);
    _set_state(drop, _DISABLED);

} /* _disable() */


/* ======================================================================= */
/**
 *  \brief Move this drop to the disabled state due to L2 down.
 *  \param drop The drop to disable due to L2 down.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _l2_down(u_int8_t drop)
{
    _mesc_fsm[drop].dsap = _DSAP_INVALID;
    _mesc_fsm[drop].poll_status = _POLLING;
    _clear_queue(drop);
    info("Drop %d disabled due to L2 down.", drop);
    _set_state(drop, _DISABLED);
} /* _l2_down() */


/* ======================================================================= */
/**
 *  \brief Send an app request to the host
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _send_app_rqst(u_int8_t drop)
{
    buf_t b;

    b = buf_dequeue(_mesc_fsm[drop].tx_q);
    _mesc_fsm[drop].dsap = _get_sap(b);
    _send_escp_rqst(drop, b, ESCP_MESSAGE_TYPE_APP);
    if ( !mesc_cfg_is_secure() )
        mesc_tmr_start_resp(drop);
} /* _send_app_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a request message to the host
 *  \param drop The drop address.
 *  \param rqst The request message
 *  \param type The type of request, ESCP_MESSAGE_TYPE_PROTO or
 *      ESCP_MESSAGE_TYPE_APP
 *  \return Nothing.
 */
/* ======================================================================= */
static void _send_escp_rqst(u_int8_t drop, buf_t rqst, u_int8_t type)
{
    buf_t b;
    escp_rqst_hdr_t hdr;
    u_int8_t stat_opts;

    hdr.pid = ESCP_PROTOCOL_ID;
    stat_opts = mesc_stat_get_mask(drop);
    if ( stat_opts )
    {
        hdr.type = ESCP_PACKET_TYPE_REQUEST | ESCP_HEADER_FORMAT_OPTIONAL |
            type;
    }
    else
    {
        hdr.type = ESCP_PACKET_TYPE_REQUEST | ESCP_HEADER_FORMAT_STANDARD |
            type;
    }

    hdr.sap = _mesc_fsm[drop].hsap;
    hdr.seq_no = ++_mesc_fsm[drop].seq_no;
    hdr.prev_resp_tm = htons(mesc_tmr_get_prev_resp_time(drop));
    b = buf_append(0, &hdr, sizeof(hdr));

    /*
     * Update the number of requests before including the statistics.  The
     * statistics will therefore include the current request in the count too.
     */
    mesc_stat_update_escp_rqsts(drop);
    if ( stat_opts )
        b = buf_join(b, mesc_stat_get_escp_stats(drop));

    if ( ++_mesc_fsm[drop].cur_tmo_cnt > _mesc_fsm[drop].tmo_cnt )
    {
        warn("Host Timeout Count Exceeded (%d)", _mesc_fsm[drop].tmo_cnt);
        /* cm: set cur_tmo_cnt to 1, not 0, to account for udp_send() below. */
        _mesc_fsm[drop].cur_tmo_cnt = 1;
        mesc_udp_use_next_addr(drop);
    }

    if ( mesc_cfg_is_secure() )
    {
        info("Sending to MSECP ...");
        mesc_imp_send_msecp_msg(drop, buf_join(b,rqst));
    }
    else
        mesc_udp_send(drop, buf_join(b, rqst));

} /* _send_escp_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a Config Step 1 Request to the host
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _send_escp_cs1_rqst(u_int8_t drop)
{
    _mesc_fsm[drop].seq_no = ESCP_SEQ_NO_UNKNOWN - 1;
    _send_escp_rqst(drop, _make_escp_cs1_rqst(drop), ESCP_MESSAGE_TYPE_PROTO);
    mesc_imp_send_status_msg(drop, "Message=ReqStnCfg");
    if ( !mesc_cfg_is_secure() )
        mesc_tmr_start_resp(drop);
} /* _send_escp_cs1_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a Config Step 2 Request to the host
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _send_escp_cs2_rqst(u_int8_t drop)
{
    _send_escp_rqst(drop, _make_escp_cs2_rqst(drop), ESCP_MESSAGE_TYPE_PROTO);
    mesc_imp_send_status_msg(drop, "Message=RecvStnCfg");
    if ( !mesc_cfg_is_secure() )
        mesc_tmr_start_resp(drop);
} /* _send_escp_cs2_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a Keepalive Request to the host
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _send_escp_keepalive_rqst(u_int8_t drop)
{
    _send_escp_rqst(drop, _make_escp_keepalive_rqst(), ESCP_MESSAGE_TYPE_PROTO);

    /*
    TODO: Requires new vector for tx-ind in idle state, so ka tmr can start
    if ( !mesc_cfg_is_secure() )
    {
        if ( _mesc_fsm[drop].keepalive_tmo != ESCP_DISABLE_KEEPALIVE )
            mesc_tmr_start_keepalive(drop, _mesc_fsm[drop].keepalive_tmo);
    }
     */

    if ( _mesc_fsm[drop].keepalive_tmo != ESCP_DISABLE_KEEPALIVE )
        mesc_tmr_start_keepalive(drop, _mesc_fsm[drop].keepalive_tmo);
} /* _send_escp_keepalive_rqst() */


/* ======================================================================= */
/**
 *  \brief Make a Config Step 1 Request message
 *  \param drop The drop address.
 *  \return A buffer containing the CS1 request
 */
/* ======================================================================= */
static buf_t _make_escp_cs1_rqst(u_int8_t drop)
{
    buf_t b;
    u_int8_t msg_id = ESCP_CONFIG_STEP_1;
    u_int8_t len;
    char *str;
    char hw_id[32]; /* ESPAD needs 21 chars: MAC+drop = "00:11:22:33:44:55:dd"
                       Altura needs 27 chars: Serial ID + drop =
                                              "00:11:22:33:44:55:66:77:dd" */

    b = buf_append(0, &msg_id, 1);

    str = mesc_stat_get_client_name(drop);
    len = strlen(str);
    b = buf_append(b, &len, 1);
    b = buf_append(b, str, len);

    str = (drop == MESC_PAD_DROP) ?
        mesc_cfg_get_pad_rom_id() : mesc_cfg_get_rom_id();
    len = strlen(str);
    b = buf_append(b, &len, 1);
    b = buf_append(b, str, len);

    mesc_cfg_get_hw_id(drop, hw_id);
    len = strlen(hw_id);
    b = buf_append(b, &len, 1);
    b = buf_append(b, hw_id, len);
    return (b);
} /* _make_escp_cs1_rqst() */


/* ======================================================================= */
/**
 *  \brief Make a Config Step 2 Request message
 *  \param drop The drop address.
 *  \return A buffer containing the CS2 request
 */
/* ======================================================================= */
static buf_t _make_escp_cs2_rqst(u_int8_t drop)
{
    buf_t b;
    u_int8_t msg_id = ESCP_CONFIG_STEP_2;
    u_int8_t ulen, mlen;
    char *mstr;
    char unicast[64];       /**< "ESCP/UDP.unicast:www.xxx.yyy.zzz:ppppp"(40)*/

    if ( mesc_cfg_is_secure() )
        snprintf(unicast, sizeof(unicast), "SECP/UDP.unicast:%s:%d",
            _mesc_fsm_nat_addr, _mesc_fsm[drop].nat_port);
    else
        snprintf(unicast, sizeof(unicast), "ESCP/UDP.unicast:%s:%d",
            _mesc_fsm_nat_addr, _mesc_fsm[drop].nat_port);
    unicast[sizeof(unicast) - 1] = '\0';

    b = buf_append(0, &msg_id, 1);

    ulen = strlen(unicast);
    b = buf_append(b, &ulen, 1);
    b = buf_append(b, unicast, ulen);

    mstr = mesc_udp_get_multicast_str();
    if ( (mlen = strlen(mstr)) > 0 )
    {
        b = buf_append(b, &mlen, 1);
        b = buf_append(b, mstr, mlen);
    }
    else
    {
        b = buf_append(b, &ulen, 1);
        b = buf_append(b, unicast, ulen);
    }
    return (b);
} /* _make_escp_cs2_rqst() */


/* ======================================================================= */
/**
 *  \brief Make a Keepalive Request message
 *  \return A buffer containing the keepalive request
 */
/* ======================================================================= */
static buf_t _make_escp_keepalive_rqst(void)
{
    u_int8_t msg_id = ESCP_KEEP_ALIVE;
    return (buf_append(0, &msg_id, 1));
} /* _make_escp_keepalive_rqst() */


/* ======================================================================= */
/**
 *  \brief Compares the target name in the unso header with ours
 *  \param len The maximum number of bytes available for processing
 *  \param hdr Holds the unso header.
 *  \param acn The acn to compare the target name against.
 *      The \a acn arg is expected to be either the client's name in the case
 *      of a unicast unso received, or the application configuration name in
 *      the case of a multicast unso received.
 *  \return 0 if success, -1 if failure.
 */
/* ======================================================================= */
static int _compare_target_name(int len, escp_unso_hdr_t *hdr, _mesc_acn_t *acn)
{
    assert(hdr && acn);
    if ( hdr->tn_len == 0 )
        return (0);     /* always accept if zero length per spec. */

    if ( (len < (sizeof(escp_unso_hdr_t) + acn->len)) || (hdr->tn_len != acn->len) )
        return (-1);    /* not enough bytes or byte count mismatch, so fail */

    if ( memcmp(hdr->data, acn->data, hdr->tn_len) )
        return (-1);    /* comparison failed */

    return (0);         /* match! */

} /* _compare_target_name() */


/* ======================================================================= */
/**
 *  \brief Parse a unicast unsolicited header.
 *  \param drop The drop address.
 *  \param b The unso message, at least 4 bytes in length
 *  \param opt Will be written with the options.
 *  \param name The client's name to compare the unso against.
 *      The application configuration name is also compared as a valid unso
 *      for the client, and this is in case the drop could not join the
 *      multicast group.
 *  \return Length of remaining payload, or _ERROR
 */
/* ======================================================================= */
static int _parse_ucast_unso_header(u_int8_t drop, buf_t b, _opt_t *opt, const char *name)
{
    escp_unso_hdr_t *unso_hdr;
    int len;
    int retval;
    u_int8_t *data;
    _mesc_acn_t my_acn;
    _mesc_acn_t *acn_ptr;

    assert((drop < MESC_CFG_MAX_DROPS) && b && opt && name);
    unso_hdr = (escp_unso_hdr_t *)buf_data(b);
    len = buf_length(b);

    /* Compare against both the drop's acn & the name (client name).
     * Drop any unso that matches neither. */
    if ( drop == MESC_PAD_DROP )
        acn_ptr = &_pad_acn;
    else
        acn_ptr = &_mesc_acn;
    if ( _compare_target_name(len, unso_hdr, acn_ptr) )
    {
        /* ACN match failed -> try client name */
        my_acn.len = strlen(name);
        memcpy(my_acn.data, name, my_acn.len);
        if ( _compare_target_name(len, unso_hdr, &my_acn) )
        {
            warn("Target name mismatch - unso dropped (drop %d).", drop);
            return (_ERROR);
        }
    }
    len -= sizeof(escp_unso_hdr_t) + unso_hdr->tn_len;
    if ( len <= 0 )
    {
        warn("Short message - unso dropped (drop %d).", drop);
        return (_ERROR);
    }
    data = &unso_hdr->data[unso_hdr->tn_len];

    /* Save any options there might be for post-processing of the unso */
    if ( (retval = _parse_options(unso_hdr->type, data, len, opt)) < 0  )
    {
        warn("Bad options - unso dropped (drop %d).", drop);
        return (_ERROR);
    }
    len -= retval;
    if ( len <= 0 )
    {
        warn("Empty/short message - unso dropped (drop %d).", drop);
        return (_ERROR);
    }

    /* According to the ES Connect specification, a disabled client becomes
     * enabled when any of the following occurs: disable timeout expires, an
     * unsolicited message is received, or the client reboots.  But, if an
     * unsolicited message is received with a disable timeout, the behavior
     * is unspecified.  I assume that if a disable is specified in an unso
     * that the disable takes priority; however, if it is absent in an unso,
     * I will treat it as an enable.  If we are already enabled, it doesn't
     * hurt, and if we are disabled, we'll follow the spec.  Note that the
     * behavior is not the same for a response header.  If the disable field
     * is not present in a response header, I don't assume it means enable. */
    if ( (opt->cntrl_opts & ESCP_CNTRL_OPTS_DISABLE_BIT) == 0 )
    {
        opt->cntrl_opts |= ESCP_CNTRL_OPTS_DISABLE_BIT;
        opt->disable = ESCP_ENABLE;
    }

    mesc_stat_update_escp_unsos(drop, 0);
    return (len);

} /* _parse_ucast_unso_header() */


/* ======================================================================= */
/**
 *  \brief Parse an unsolicited header.
 *  \param b The unso message, at least 4 bytes in length
 *  \param opt Will be written with the options.
 *  \param dest The destination - applicable for multicast only.
 *  \return Length of remaining payload, or _ERROR
 */
/* ======================================================================= */
static int _parse_mcast_unso_header(buf_t b, _opt_t *opt, int *dest)
{
    escp_unso_hdr_t *unso_hdr;
    int len;
    int retval;
    u_int8_t *data;

    assert(b && opt && dest);
    *dest = 0;
    unso_hdr = (escp_unso_hdr_t *)buf_data(b);
    len = buf_length(b);

    /* 1st compare against common ACN */
    if ( _compare_target_name(len, unso_hdr, &_mesc_acn) == 0 )
        *dest |= MESC_COMMON_ACN;
    /* 2nd compare against pad-specific ACN */
    if ( _compare_target_name(len, unso_hdr, &_pad_acn) == 0 )
        *dest |= MESC_PAD_ACN;
    if ( *dest == 0 )
    {
        warn("Multicast target name mismatch - unso dropped.");
        return (_ERROR);
    }
    len -= sizeof(escp_unso_hdr_t) + unso_hdr->tn_len;
    if ( len <= 0 )
    {
        warn("Short multicast message - unso dropped.");
        return (_ERROR);
    }
    data = &unso_hdr->data[unso_hdr->tn_len];

    /* Save any options there might be for post-processing of the unso */
    if ( (retval = _parse_options(unso_hdr->type, data, len, opt)) < 0  )
    {
        warn("Bad multicast options - unso dropped.");
        return (_ERROR);
    }
    len -= retval;
    if ( len <= 0 )
    {
        warn("Empty/short multicast message - unso dropped.");
        return (_ERROR);
    }

    /* According to the ES Connect specification, a disabled client becomes
     * enabled when any of the following occurs: disable timeout expires, an
     * unsolicited message is received, or the client reboots.  But, if an
     * unsolicited message is received with a disable timeout, the behavior
     * is unspecified.  I assume that if a disable is specified in an unso
     * that the disable takes priority; however, if it is absent in an unso,
     * I will treat it as an enable.  If we are already enabled, it doesn't
     * hurt, and if we are disabled, we'll follow the spec.  Note that the
     * behavior is not the same for a response header.  If the disable field
     * is not present in a response header, I don't assume it means enable. */
    if ( (opt->cntrl_opts & ESCP_CNTRL_OPTS_DISABLE_BIT) == 0 )
    {
        opt->cntrl_opts |= ESCP_CNTRL_OPTS_DISABLE_BIT;
        opt->disable = ESCP_ENABLE;
    }

    mesc_stat_update_escp_unsos(255, *dest);
    return (len);

} /* _parse_mcast_unso_header() */


/* ======================================================================= */
/**
 *  \brief Parse a response header.
 *  \param drop The drop address.
 *  \param b The response message, at least 4 bytes in length
 *  \param opt Will be written with the options.
 *  \return Length of remaining payload, or _ERROR
 */
/* ======================================================================= */
static int _parse_resp_header(u_int8_t drop, buf_t b, _opt_t *opt)
{
    escp_resp_hdr_t *resp_hdr;
    int len;
    int retval;
    u_int8_t *data;

    resp_hdr = (escp_resp_hdr_t *)buf_data(b);
    len = buf_length(b);

    /* Drop any response whose sequence # does not match */
    if ( resp_hdr->seq_no != _mesc_fsm[drop].seq_no )
    {
        warn("Sequence # mismatch: received %d, expected %d.",
            resp_hdr->seq_no, _mesc_fsm[drop].seq_no);
        return (_ERROR);
    }
    len -= sizeof(escp_resp_hdr_t);
    if ( len <= 0 )
    {
        warn("Short response message dropped (drop %d).", drop);
        return (_ERROR);
    }
    data = (u_int8_t *)resp_hdr->data;

    /* Save any options there might be for post-processing of the response */
    if ( (retval = _parse_options(resp_hdr->type, data, len, opt)) < 0  )
    {
        warn("Response dropped (bad options, drop %d).", drop);
        return (_ERROR);
    }
    len -= retval;
    if ( len <= 0 )
    {
        warn("Empty/short response dropped (drop %d).", drop);
        return (_ERROR);
    }
    mesc_stat_update_escp_resps(drop);
    _mesc_fsm[drop].hsap = resp_hdr->sap;
    return (len);

} /* _parse_resp_header() */


/* ======================================================================= */
/**
 *  \brief Parse the options that may or may not be present in an unso or resp.
 *  \param type An ESCP packet type
 *  \param data The start of the option data
 *  \param len The maximum number of bytes available for processing
 *  \param opt Will hold all the options after parsing.
 *  \return -1 if error; otherwise a byte length
 */
/* ======================================================================= */
static int _parse_options(u_int8_t type, u_int8_t *data, int len, _opt_t *opt)
{
    u_int8_t *ptr = data;
    int cnt = 0;

    assert(ptr && opt);
    memset(opt, 0, sizeof(_opt_t)); /* clear all */

    /* Check if there are any options at all */
    if ( ESCP_HEADER_FORMAT(type) == ESCP_HEADER_FORMAT_STANDARD )
        return (0);

    /* Get the control options field */
    if ( len-- <= 0 )
        return (-1);
    opt->cntrl_opts = ESCP_CNTRL_OPTS_MASK & ptr[cnt++];
    if ( opt->cntrl_opts == 0 )     /* gotta be at least 1 option */
        return (-1);

    /* Check for the statistics mask first */
    if ( opt->cntrl_opts & ESCP_CNTRL_OPTS_STATS_MASK_BIT )
    {
        if ( len-- <= 0 )
            return (-1);
        opt->mask = ptr[cnt++];
    }

    /* Check for the disable w/timeout second */
    if ( opt->cntrl_opts & ESCP_CNTRL_OPTS_DISABLE_BIT )
    {
        if ( len <= 1 )
            return (-1);
        opt->disable = ntohs(*(u_int16_t *)&ptr[cnt]);
        len -= 2;
        cnt += 2;
    }

    /* NOTE: No need to check for "clear statistics" - it doesn't carry data */

    /* Check for keep-alive timeout third */
    if ( opt->cntrl_opts & ESCP_CNTRL_OPTS_KEEP_ALIVE_BIT )
    {
        if ( len <= 1 )
            return (-1);
        opt->keepalive = ntohs(*(u_int16_t *)&ptr[cnt]);
        cnt += 2;
    }
    return (cnt);

} /* _parse_options() */


/* ======================================================================= */
/**
 *  \brief Performs post-processing of any options, including config required
 *  \param drop The drop address.
 *  \param opt Holds all the options
 *  \return Nothing.
 */
/* ======================================================================= */
static void _post_process_options(u_int8_t drop, _opt_t *opt)
{
    u_int16_t prev_ka_tmo;

    assert((drop < MESC_CFG_MAX_DROPS) && opt);

    /* These options DO NOT affect the state of the machine */
    if ( opt->cntrl_opts & ESCP_CNTRL_OPTS_STATS_MASK_BIT )
        mesc_stat_set_mask(drop, opt->mask);
    if ( opt->cntrl_opts & ESCP_CNTRL_OPTS_CLEAR_STATS_BIT )
        mesc_stat_clear_escp_stats(drop);
    if ( opt->cntrl_opts & ESCP_CNTRL_OPTS_KEEP_ALIVE_BIT )
    {
        prev_ka_tmo = _mesc_fsm[drop].keepalive_tmo;
        _mesc_fsm[drop].keepalive_tmo = opt->keepalive;
        if ( (_mesc_fsm[drop].state == _IDLE) &&
            (prev_ka_tmo != _mesc_fsm[drop].keepalive_tmo) )
        {
            mesc_tmr_stop_keepalive(drop);
            if ( _mesc_fsm[drop].keepalive_tmo != ESCP_DISABLE_KEEPALIVE )
                mesc_tmr_start_keepalive(drop, _mesc_fsm[drop].keepalive_tmo);
        }
    }

    /* This option CAN affect the state of the state machine */
    if ( opt->cntrl_opts & ESCP_CNTRL_OPTS_DISABLE_BIT )
        _dispatch_disable(drop, opt->disable);

} /* _post_process_options() */


/* ======================================================================= */
/**
 *  \brief Parse config step 1 communications ID string.
 *  \param drop The drop address.
 *  \param data The communications ID data.  Format = "ESCP/UDP.client@IP:port"
 *  \param len The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _parse_cs1_comm_id(u_int8_t drop, u_int8_t *data, u_int8_t len)
{
    char *b = (char *)alloca(len + 1);
    const char *escp_comm_id_prefix = "ESCP/UDP.client@";
    const char *secp_comm_id_prefix = "SECP/UDP.client@";
    const char *comm_id_prefix;
    char *ptr, *addr;

    /* Don't assume that the data is null-terminated. */
    assert(data);
    memcpy(b, data, len);
    b[len] = '\0';
    ptr = b;

    if ( mesc_cfg_is_secure() )
        comm_id_prefix = secp_comm_id_prefix;
    else
        comm_id_prefix = escp_comm_id_prefix;

    /* Make sure the client prefix is present, then throw it away. */
    if ( (ptr = strstr(b, comm_id_prefix)) == NULL )
        return;
    ptr += strlen(comm_id_prefix);
    if ( ptr == NULL )
        return;

    /* Separate the rest of the string into the IP address & port fields */
    addr = strsep(&ptr, ":");
    if ( (addr == NULL) || (ptr == NULL) )  /* ptr originally null or no ':' */
        return;

    /* Save off the Host's idea of IP address and the client's port */
    if ( strlen(addr) >= sizeof(_mesc_fsm_nat_addr) )
        return;
    strncpy(_mesc_fsm_nat_addr, addr, sizeof(_mesc_fsm_nat_addr));
    _mesc_fsm[drop].nat_port = (u_int16_t)strtoul(ptr, NULL, 0);

} /* _parse_cs1_comm_id() */


/* ======================================================================= */
/**
 *  \brief Parse config step 1 configuration text string.
 *  \param drop The drop to reconfigure.
 *  \param data The configuration data.
 *  \param len The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _parse_cs1_config(u_int8_t drop, u_int8_t *data, u_int8_t len)
{
    cfg_t cfg;
    u_int32_t t0, t1;
    u_int8_t *ptr = (u_int8_t *)alloca(len + 1);

    assert(data);
    /* Don't assume that the data is null-terminated. */
    memcpy(ptr, data, len);
    ptr[len] = '\0';

    /* Process the data, then apply the changes, as reasonable */
    mesc_cfg_process(ptr, &cfg);

    /* Handle adaptive response timer values. */
    if ( (cfg.flags & MESC_CFG_T0) && (cfg.flags & MESC_CFG_T1) )
    {
        if ( cfg.t0 > cfg.t1 )
        {
            mesc_art_set_min_rsp_tmo(drop, cfg.t1);
            mesc_art_set_max_rsp_tmo(drop, cfg.t0);
        }
        else
        {
            mesc_art_set_min_rsp_tmo(drop, cfg.t0);
            mesc_art_set_max_rsp_tmo(drop, cfg.t1);
        }
    }
    else if ( cfg.flags & MESC_CFG_T0 )
    {
        t1 = mesc_art_get_max_rsp_tmo(drop);
        if ( cfg.t0 > t1 )
        {
            mesc_art_set_min_rsp_tmo(drop, t1);
            mesc_art_set_max_rsp_tmo(drop, cfg.t0);
        }
        else
            mesc_art_set_min_rsp_tmo(drop, cfg.t0);
    }
    else if ( cfg.flags & MESC_CFG_T1 )
    {
        t0 = mesc_art_get_min_rsp_tmo(drop);
        if ( t0 > cfg.t1 )
        {
            mesc_art_set_min_rsp_tmo(drop, cfg.t1);
            mesc_art_set_max_rsp_tmo(drop, t0);
        }
        else
            mesc_art_set_max_rsp_tmo(drop, cfg.t1);
    }

    /* Reset the timeout count */
    if ( cfg.flags & MESC_CFG_TC )
    {
        if ( cfg.tc == 0 )  /* Don't allow zero */
            cfg.tc = 1;
        _mesc_fsm[drop].tmo_cnt = cfg.tc;
    }

} /* _parse_cs1_config() */


/* ======================================================================= */
/**
 *  \brief Get stored sap.
 *  \param b Message.
 *  \return SAP.
 */
/* ======================================================================= */
static u_int8_t _get_sap(buf_t b)
{
    return ((u_int8_t)buf_get_tag(b));
} /* _get_sap() */


/* ======================================================================= */
/**
 *  \brief Store the sap in the buffer.
 *  \param b Message.
 *  \param sap The SAP.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _set_sap(buf_t b, u_int8_t sap)
{
    buf_set_tag(b, sap);
} /* _set_sap() */


/* ======================================================================= */
/**
 *  \brief Sends a retry imp message.
 *  \param drop The drop address.
 *  \param id A little identifier as to which message is being re-tried.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _send_imp_retry_msg(u_int8_t drop, const char *id)
{
    char name[256];
    char hw_id[32];
    char msg[256];

    mesc_cfg_get_client_name(drop, name);
    mesc_cfg_get_hw_id(drop, hw_id);
    snprintf(msg, sizeof(msg), "Message=Retrying;Address=%s/%d/%s/%s/%s",
        id, _mesc_fsm[drop].cur_tmo_cnt,
        (drop == MESC_PAD_DROP) ?
        mesc_cfg_get_pad_rom_id() : mesc_cfg_get_rom_id(),
        name, hw_id);
    msg[sizeof(msg) - 1] = '\0';
    mesc_imp_send_status_msg(drop, msg);
} /* _send_imp_retry_msg() */


/* ======================================================================= */
/**
 *  \brief Saves the latest L2 status.
 *  \param l2_status The latest L2 status.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _set_l2_status(int l2_status)
{
    _l2_status = l2_status;
} /* _set_l2_status() */


/* ======================================================================= */
/**
 *  \brief Gets the current L2 status.
 *  \return The current L2 status.
 */
/* ======================================================================= */
static int _get_l2_status(void)
{
    return (_l2_status);
} /* _get_l2_status() */

                    /* ------------------------------------- */
                    /* SECTION: Common state machine vectors */
                    /* ------------------------------------- */

/* ======================================================================= */
/**
 *  \brief Dummy timer vector.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _timer_do_nothing(u_int8_t drop)
{
    _set_state(drop, _mesc_fsm[drop].state);
} /* _timer_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Queue a request for this drop.
 *  \param drop The drop this request is from.
 *  \param b The buffer holding the request.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _queue_rqst(u_int8_t drop, buf_t b)
{
    if ( b )
        buf_queue(_mesc_fsm[drop].tx_q, b);
    _set_state(drop, _mesc_fsm[drop].state);
} /* _queue_rqst() */


/* ======================================================================= */
/**
 *  \brief Dummy application unso vector.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \param dest The destination - applicable for multicast only.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _app_unso_do_nothing(u_int8_t drop, u_int8_t *data, int length, int dest)
{
    if ( drop == 255 )    /* could be a multicast (drop=255) */
        trace("v----------> Nothing to do.\n\n");
    else
        _set_state(drop, _mesc_fsm[drop].state);

} /* _app_unso_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Dummy protocol unso vector.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _proto_unso_do_nothing(u_int8_t drop, u_int8_t *data, int length)
{
    if ( drop == 255 )    /* could be a multicast (drop=255) */
        trace("v----------> Nothing to do.\n\n");
    else
        _set_state(drop, _mesc_fsm[drop].state);

} /* _proto_unso_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Dummy response vector.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _resp_do_nothing(u_int8_t drop, u_int8_t *data, int length)
{
    _set_state(drop, _mesc_fsm[drop].state);
} /* _resp_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Dummy disable vector.
 *  \param drop The drop address.
 *  \param data The data.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disable_do_nothing(u_int8_t drop, u_int16_t data)
{
    /* TODO?: _set_state(drop, _mesc_fsm[drop].state) */
} /* _disable_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Dummy tx ind vector.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _tx_ind_do_nothing(u_int8_t drop)
{
    _set_state(drop, _mesc_fsm[drop].state);

} /* _tx_ind_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Handle a tx-data-ind message.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _tx_ind(u_int8_t drop)
{
    mesc_tmr_start_resp(drop);
    /* TODO?: _set_state(drop, _mesc_fsm[drop].state); */

} /* _tx_ind() */


/* ======================================================================= */
/**
 *  \brief Handles an application unso message.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \param dest The destination - applicable for multicast only.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _app_unso(u_int8_t drop, u_int8_t *data, int length, int dest)
{
    mesc_imp_send_unso(drop, data, length, dest);
    if ( drop == 255 )  /* could be multicast, drop=255 */
        trace("v----------> Multicast Unso Delivered.\n\n");
    else
        _set_state(drop, _mesc_fsm[drop].state);
} /* _app_unso() */


/* ======================================================================= */
/**
 *  \brief Dummy x42-drop-status vector.
 *  \param drop The drop address.
 *  \param status The drop poll status.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _x42_drop_status_do_nothing(u_int8_t drop, const char *status)
{
} /* _x42_drop_status_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Handle a Disable in either CS1 or CS2 states.
 *  \param drop The drop address.
 *  \param data The new disable timeout (0=enable, 0xFFFF=4eva)
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs_disable(u_int8_t drop, u_int16_t data)
{
    trace("%s(%d): %s ----------v",
        _state2str(_mesc_fsm[drop].state), drop, (data==0)?"ENABLE":"DISABLE");
    _mesc_fsm[drop].disable_tmo = data;
    if ( _mesc_fsm[drop].disable_tmo == ESCP_ENABLE )
    {
        _set_state(drop, _mesc_fsm[drop].state);
        return;
    }
    mesc_tmr_stop_resp(drop);
    _disable(drop);
} /* _cs_disable() */


/* ======================================================================= */
/**
 *  \brief Handle x42-drop-status vector in either CS1 or CS2 states.
 *  \param drop The drop address.
 *  \param status The drop poll status.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs_x42_drop_status(u_int8_t drop, const char *status)
{
    /* Mark our flag so we know whether or not to start the keep-alive timer
     * when finishing configuring and entering the idle state. */
    if ( !strcmp(status, "slow-polling") )
    {
        info("Drop %d marked for keep-alive disable.", drop);
        _mesc_fsm[drop].poll_status = _SLOW_POLLING;
    }
    else
    {
        info("Drop %d marked for keep-alive enable.", drop);
        _mesc_fsm[drop].poll_status = _POLLING;
    }
} /* _cs_x42_drop_status() */


/* ======================================================================= */
/**
 *  \brief Handle comm-status vector in either CS1 or CS2 states.
 *  \param drop The drop address.
 *  \param l2_status The L2 comm status, as either _DOWN or _ONLINE.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs_l2_status(u_int8_t drop, int l2_status)
{
    if ( l2_status == _DOWN )
    {
        trace("%s(%d): COMM STATUS: DOWN ----------v",
            _state2str(_mesc_fsm[drop].state), drop);
        mesc_tmr_stop_resp(drop);
        _l2_down(drop);
    }
} /* _cs_l2_status() */

                    /* --------------------------- */
                    /* SECTION: INIT state vectors */
                    /* --------------------------- */

/* ======================================================================= */
/**
 *  \brief Handle a request in the INIT state.
 *  \param drop The drop this request is from.
 *  \param b The buffer holding the request.
 *      If the buffer is NULL, we'll still configure the drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _init_rqst(u_int8_t drop, buf_t b)
{
    if ( b )
        buf_queue(_mesc_fsm[drop].tx_q, b);

    /* First time going to send anything - force DNS lookup */
    mesc_udp_dns_lookup(drop);

    if ( !mesc_cfg_is_secure() )
        mesc_udp_open(drop);

    /* Before we attempt to send the request, check the L2 status. */
    if ( _get_l2_status() == _DOWN )
        _l2_down(drop);
    else
    {
        _send_escp_cs1_rqst(drop);
        _set_state(drop, _CS1);
    }
} /* _init_rqst() */


/* ======================================================================= */
/**
 *  \brief Handle comm-status vector in INIT state.
 *  \param drop The drop address - not used.
 *  \param l2_status The L2 status - not used.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _init_l2_status_do_nothing(u_int8_t drop, int l2_status)
{
} /* _init_l2_status_do_nothing() */

                    /* -------------------------- */
                    /* SECTION: CS1 state vectors */
                    /* -------------------------- */

/* ======================================================================= */
/**
 *  \brief Handle a response timeout in the CS1 state.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs1_resp_tmo(u_int8_t drop)
{
    mesc_art_mark_resp_tmo(drop);
    _send_escp_cs1_rqst(drop);
    _send_imp_retry_msg(drop, "CS1");
    _set_state(drop, _CS1);

} /* _cs1_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Handle a CS1 response in the CS1 state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs1_resp(u_int8_t drop, u_int8_t *data, int length)
{
    escp_cfg_step1_resp_msg_t *msg = (escp_cfg_step1_resp_msg_t *)data;
    u_int8_t *ptr;
    int remaining = length;

    assert(msg);
    mesc_tmr_stop_resp(drop);
    mesc_tmr_mark_end(drop);

    if ( length < (int)sizeof(escp_cfg_step1_resp_msg_t) )
    {
        warn("CS1 Response length mismatch %d vs. %d",
            length, sizeof(escp_cfg_step1_resp_msg_t));
        _send_escp_cs1_rqst(drop);
        _send_imp_retry_msg(drop, "CS1");
        _set_state(drop, _CS1);
        return;
    }
    remaining = length - sizeof(escp_cfg_step1_resp_msg_t);

    /* Save next seq no and target name */
    _mesc_fsm[drop].seq_no = msg->next_seq_no - 1;  /* increment on send */
    if ( drop == MESC_PAD_DROP )
        _pad_acn.len = msg->acn_len;
    else
        _mesc_acn.len = msg->acn_len;
    if ( remaining < msg->acn_len )
    {
        if ( drop == MESC_PAD_DROP )
            _pad_acn.len = 0;
        else
            _mesc_acn.len = 0;
        warn("CS1 Response target name length mismatch %d vs. %d",
            remaining, msg->acn_len);
        _send_escp_cs1_rqst(drop);
        _send_imp_retry_msg(drop, "CS1");
        _set_state(drop, _CS1);
        return;
    }
    if ( drop == MESC_PAD_DROP )
        memcpy(_pad_acn.data, msg->data, msg->acn_len);
    else
        memcpy(_mesc_acn.data, msg->data, msg->acn_len);
    remaining -= msg->acn_len;

    /* Parse the Communications ID:  We need to save the Host's idea of our
     * IP address and port in the event that we go through a NAT box. */
    ptr = msg->data + msg->acn_len;
    if ( remaining < (*ptr + 1) )
    {
        warn("CS1 Response comm ID length mismatch %d vs. %d",
            remaining, *ptr + 1);
        _send_escp_cs1_rqst(drop);
        _send_imp_retry_msg(drop, "CS1");
        _set_state(drop, _CS1);
        return;
    }
    _parse_cs1_comm_id(drop, ptr + 1, *ptr);
    remaining -= (*ptr + 1);
    ptr += *ptr + 1;

    /* Get & Parse Communications Configuration: "tc", "t0", "t1" */
    if ( remaining < (*ptr + 1) )
    {
        warn("CS1 Response comm config length mismatch %d vs. %d",
            remaining, *ptr + 1);
        _send_escp_cs1_rqst(drop);
        _send_imp_retry_msg(drop, "CS1");
        _set_state(drop, _CS1);
        return;
    }
    _parse_cs1_config(drop, ptr + 1, *ptr);
    remaining -= (*ptr + 1);
    ptr += *ptr + 1;

    /* Get multicast address & attempt to join the group */
    if ( remaining != (*ptr + 1) )
    {
        warn("CS1 Response multicast length mismatch %d vs. %d",
            remaining, *ptr + 1);
        _send_escp_cs1_rqst(drop);
        _send_imp_retry_msg(drop, "CS1");
        _set_state(drop, _CS1);
        return;
    }
    mesc_udp_multicast_join(ptr + 1, *ptr);

    _send_escp_cs2_rqst(drop);
    _set_state(drop, _CS2);

} /* _cs1_resp() */

                    /* -------------------------- */
                    /* SECTION: CS2 state vectors */
                    /* -------------------------- */

/* ======================================================================= */
/**
 *  \brief Handle a response timeout in the CS2 state.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs2_resp_tmo(u_int8_t drop)
{
    mesc_art_mark_resp_tmo(drop);
    _send_escp_cs2_rqst(drop);
    _send_imp_retry_msg(drop, "CS2");
    _set_state(drop, _CS2);
} /* _cs2_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Handles a config required unso or resp in the CS2 state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs2_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length)
{
    mesc_tmr_stop_resp(drop);
    _send_escp_cs1_rqst(drop);
    _set_state(drop, _CS1);
} /* _cs2_cfg_rqrd() */


/* ======================================================================= */
/**
 *  \brief Handles a config step 2 response in the CS2 state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _cs2_resp(u_int8_t drop, u_int8_t *data, int length)
{
    mesc_tmr_stop_resp(drop);
    mesc_tmr_mark_end(drop);
    _mesc_fsm[drop].configured = 1;
    mesc_imp_send_status_msg(drop, "Message=ESCIPReady");

    if ( _mesc_fsm[drop].tx_q )
    {
        _send_app_rqst(drop);
        _set_state(drop, _PENDING);
    }
    else
    {
        if ( (_mesc_fsm[drop].poll_status == _POLLING) &&
            (_mesc_fsm[drop].keepalive_tmo != ESCP_DISABLE_KEEPALIVE) )
        {
            mesc_tmr_start_keepalive(drop, _mesc_fsm[drop].keepalive_tmo);
        }
        else
        {
            info("Drop %d keep-alive timer not started: %s.", drop,
                (_mesc_fsm[drop].poll_status == _SLOW_POLLING) ?
                "slow poll" : "timer disabled");
        }
        _set_state(drop, _IDLE);
    }
} /* _cs2_resp() */

                    /* --------------------------- */
                    /* SECTION: IDLE state vectors */
                    /* --------------------------- */

/* ======================================================================= */
/**
 *  \brief Handle a Disable in the IDLE state.
 *  \param drop The drop address.
 *  \param data The new disable timeout (0=enable, 0xFFFF=4eva)
 *  \return Nothing.
 */
/* ======================================================================= */
static void _idle_disable(u_int8_t drop, u_int16_t data)
{
    trace("%s(%d): %s ----------v",
        _state2str(_mesc_fsm[drop].state), drop, (data==0)?"ENABLE":"DISABLE");
    _mesc_fsm[drop].disable_tmo = data;
    if ( _mesc_fsm[drop].disable_tmo == ESCP_ENABLE )
    {
        _set_state(drop, _mesc_fsm[drop].state);
        return;
    }
    mesc_tmr_stop_keepalive(drop);
    _disable(drop);
} /* _idle_disable() */


/* ======================================================================= */
/**
 *  \brief Handle a keepalive timeout in the IDLE state.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _idle_keepalive_tmo(u_int8_t drop)
{
    _send_escp_keepalive_rqst(drop);
    _set_state(drop, _IDLE);
} /* _idle_keepalive_tmo() */


/* ======================================================================= */
/**
 *  \brief Handles a config required unso or resp in the IDLE state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _idle_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length)
{
    _mesc_fsm[drop].configured = 0;
    mesc_tmr_stop_keepalive(drop);
    _send_escp_cs1_rqst(drop);
    _set_state(drop, _CS1);
} /* _idle_cfg_rqrd() */


/* ======================================================================= */
/**
 *  \brief Handles an application request in the IDLE state.
 *  \param drop The drop address.
 *  \param b The buffer
 *  \return Nothing.
 */
/* ======================================================================= */
static void _idle_rqst(u_int8_t drop, buf_t b)
{
    if ( b )
    {
        mesc_tmr_stop_keepalive(drop);
        _mesc_fsm[drop].poll_status = _POLLING;
        buf_queue(_mesc_fsm[drop].tx_q, b);
        _send_app_rqst(drop);
        _set_state(drop, _PENDING);
    }
    else
        _set_state(drop, _IDLE);
} /* _idle_rqst() */


/* ======================================================================= */
/**
 *  \brief Handles a keepalive response in the IDLE state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _idle_keepalive_resp(u_int8_t drop, u_int8_t *data, int length)
{
    _set_state(drop, _mesc_fsm[drop].state);
} /* _idle_keepalive_resp() */


/* ======================================================================= */
/**
 *  \brief Handle x42-drop-status vector in idle state.
 *  \param drop The drop address.
 *  \param status The drop poll status.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _idle_x42_drop_status(u_int8_t drop, const char *status)
{
    if ( !strcmp(status, "slow-polling") )
    {
        /* Ok to stop an already stopped timer */
        mesc_tmr_stop_keepalive(drop);
        _mesc_fsm[drop].poll_status = _SLOW_POLLING;
        info("Keep-alive timer stopped: drop %d in slow poll.", drop);
    }
    else /* "polling" */
    {
        if ( _mesc_fsm[drop].poll_status == _SLOW_POLLING )
        {
            info("Re-enabling keep-alive timer for drop %d ...", drop);
            _send_escp_keepalive_rqst(drop);
            _mesc_fsm[drop].poll_status = _POLLING;
        }
    }
} /* _idle_x42_drop_status() */


/* ======================================================================= */
/**
 *  \brief Handle comm-status vector in the IDLE state.
 *  \param drop The drop address.
 *  \param l2_status The L2 comm status, as either _DOWN or _ONLINE.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _idle_l2_status(u_int8_t drop, int l2_status)
{
    if ( l2_status == _DOWN )
    {
        trace("%s(%d): COMM STATUS: DOWN ----------v",
            _state2str(_mesc_fsm[drop].state), drop);
        mesc_tmr_stop_keepalive(drop);
        _l2_down(drop);
    }
} /* _idle_l2_status() */

                    /* ------------------------------ */
                    /* SECTION: PENDING state vectors */
                    /* ------------------------------ */

/* ======================================================================= */
/**
 *  \brief Handle a Disable in the PENDING state.
 *  \param drop The drop address.
 *  \param data The new disable timeout (0=enable, 0xFFFF=4eva)
 *  \return Nothing.
 */
/* ======================================================================= */
static void _pending_disable(u_int8_t drop, u_int16_t data)
{
    trace("%s(%d): %s ----------v",
        _state2str(_mesc_fsm[drop].state), drop, (data==0)?"ENABLE":"DISABLE");
    _mesc_fsm[drop].disable_tmo = data;
    if ( _mesc_fsm[drop].disable_tmo == ESCP_ENABLE )
    {
        _set_state(drop, _mesc_fsm[drop].state);
        return;
    }
    mesc_tmr_stop_resp(drop);
    mesc_imp_send_rsp_tmo(drop, _mesc_fsm[drop].dsap);
    _disable(drop);
} /* _pending_disable() */


/* ======================================================================= */
/**
 *  \brief Handle a response timeout in the PENDING state.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _pending_resp_tmo(u_int8_t drop)
{
    mesc_art_mark_resp_tmo(drop);
    mesc_imp_send_rsp_tmo(drop, _mesc_fsm[drop].dsap);
    _mesc_fsm[drop].dsap = _DSAP_INVALID;
    if ( _mesc_fsm[drop].tx_q )
    {
        _send_app_rqst(drop);
        _set_state(drop, _PENDING);
    }
    else
    {
        _mesc_fsm[drop].poll_status = _POLLING;
        if ( _mesc_fsm[drop].keepalive_tmo != ESCP_DISABLE_KEEPALIVE )
            mesc_tmr_start_keepalive(drop, _mesc_fsm[drop].keepalive_tmo);
        _set_state(drop, _IDLE);
    }

} /* _pending_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Handles a config required unso or resp in the PENDING state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _pending_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length)
{
    _mesc_fsm[drop].configured = 0;
    mesc_tmr_stop_resp(drop);
    mesc_imp_send_rsp_tmo(drop, _mesc_fsm[drop].dsap);
    _mesc_fsm[drop].dsap = _DSAP_INVALID;
    _send_escp_cs1_rqst(drop);
    _set_state(drop, _CS1);
} /* _pending_cfg_rqrd() */


/* ======================================================================= */
/**
 *  \brief Handles a response in the PENDING state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _pending_resp(u_int8_t drop, u_int8_t *data, int length)
{
    mesc_tmr_stop_resp(drop);
    mesc_tmr_mark_end(drop);

    mesc_imp_send_rsp(drop, data, length, _mesc_fsm[drop].dsap);
    _mesc_fsm[drop].dsap = _DSAP_INVALID;

    if ( _mesc_fsm[drop].tx_q )
    {
        _send_app_rqst(drop);
        _set_state(drop, _PENDING);
    }
    else
    {
        _mesc_fsm[drop].poll_status = _POLLING;
        if ( _mesc_fsm[drop].keepalive_tmo != ESCP_DISABLE_KEEPALIVE )
            mesc_tmr_start_keepalive(drop, _mesc_fsm[drop].keepalive_tmo);
        _set_state(drop, _IDLE);
    }

} /* _pending_resp() */


/* ======================================================================= */
/**
 *  \brief Handle comm-status vector in the PENDING state.
 *  \param drop The drop address.
 *  \param l2_status The L2 comm status, as either _DOWN or _ONLINE.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _pending_l2_status(u_int8_t drop, int l2_status)
{
    if ( l2_status == _DOWN )
    {
        trace("%s(%d): COMM STATUS: DOWN ----------v",
            _state2str(_mesc_fsm[drop].state), drop);
        mesc_tmr_stop_resp(drop);
        mesc_imp_send_rsp_tmo(drop, _mesc_fsm[drop].dsap);
        _l2_down(drop);
    }
} /* _pending_l2_status() */

                    /* ------------------------------- */
                    /* SECTION: DISABLED state vectors */
                    /* ------------------------------- */

/* ======================================================================= */
/**
 *  \brief Handle a Disable in the DISABLED state.
 *  \param drop The drop address.
 *  \param data The new disable timeout (0=enable, 0xFFFF=4eva)
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disabled_disable(u_int8_t drop, u_int16_t data)
{
    trace("%s(%d): %s ----------v",
        _state2str(_mesc_fsm[drop].state), drop, (data==0)?"ENABLE":"DISABLE");
    _mesc_fsm[drop].disable_tmo = data;
    mesc_tmr_stop_disable(drop);

    if ( _mesc_fsm[drop].disable_tmo != ESCP_ENABLE )
    {
        _disable(drop);
        return;
    }

    if ( _get_l2_status() == _DOWN )
    {
        _mesc_fsm[drop].status = _ONLINE;
        _l2_down(drop);
    }
    else
        _disabled_re_enable(drop);

} /* _disabled_disable() */


/* ======================================================================= */
/**
 *  \brief Handles re-enabling from the disabled state.  We either go to the
 *      CS1 or IDLE states.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disabled_re_enable(u_int8_t drop)
{
    /* We're enabled - Check if we have to reconfigure */
    if ( _mesc_fsm[drop].status == _OFFLINE )
        mesc_imp_send_status_msg(drop, "Message=MasterEnable");
    if ( _mesc_fsm[drop].configured == 0 )
    {
        _set_status(drop, _ONLINE);
        _send_escp_cs1_rqst(drop);
        _set_state(drop, _CS1);
    }
    else    /* already configured */
    {
        /* Queue better be empty, but just in case */
        _clear_queue(drop);
        _set_status(drop, _ONLINE);
        _mesc_fsm[drop].poll_status = _POLLING;
        /* TODO: Send a keep-alive reqeust when exiting disabled state? */
        if ( _mesc_fsm[drop].keepalive_tmo != ESCP_DISABLE_KEEPALIVE )
            mesc_tmr_start_keepalive(drop, _mesc_fsm[drop].keepalive_tmo);
        _set_state(drop, _IDLE);
    }
} /* _disabled_re_enable() */


/* ======================================================================= */
/**
 *  \brief Handle a disable timeout in the DISABLED state. (same as enable)
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disabled_disable_tmo(u_int8_t drop)
{
    _disabled_disable(drop, ESCP_ENABLE);
} /* _disabled_disable_tmo() */


/* ======================================================================= */
/**
 *  \brief Handles a config required unso or resp in the DISABLED state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disabled_cfg_rqrd(u_int8_t drop, u_int8_t *data, int length)
{
    _mesc_fsm[drop].configured = 0;
    _set_state(drop, _DISABLED);
} /* _disabled_cfg_rqrd() */


/* ======================================================================= */
/**
 *  \brief Handles a solicited request in the DISABLED state.
 *  \param drop The drop address.
 *  \param b The buffer containing the request.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disabled_rqst(u_int8_t drop, buf_t b)
{
    if ( b )
        buf_queue(_mesc_fsm[drop].tx_q, b);
    _clear_queue(drop);
    _set_state(drop, _DISABLED);
} /* _disabled_rqst() */


/* ======================================================================= */
/**
 *  \brief Handle comm-status vector in the DISABLED state.
 *  \param drop The drop address.
 *  \param l2_status The L2 comm status, as either _DOWN or _ONLINE.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _disabled_l2_status(u_int8_t drop, int l2_status)
{
    if ( l2_status == _DOWN )
    {
        /* We're already here: should be nothing to do, except we could
         * be disabled due to disable from host, so go ahead and report
         * DOWN anyway. */
        trace("%s(%d): COMM STATUS: DOWN ----------v",
            _state2str(_mesc_fsm[drop].state), drop);
        mesc_imp_send_status_msg(255, "Message=ConnectFailed;Cause=Link Down");
        mesc_imp_send_status(255, _DOWN);
        _l2_down(drop);
    }
    else    /* L2 is _ONLINE */
    {
        if ( _mesc_fsm[drop].status == _OFFLINE )
        {
            /* But we're still disabled */
            trace("%s(%d): COMM STATUS: OFFLINE ----------v",
                _state2str(_mesc_fsm[drop].state), drop);
            _disable(drop);
        }
        else
        {
            /* Ok to exit disabled state: Both L2 & drop are _ONLINE */
            trace("%s(%d): COMM STATUS: ONLINE ----------v",
                _state2str(_mesc_fsm[drop].state), drop);
            _disabled_re_enable(drop);
        }
    }
} /* _disabled_l2_status() */

/*
 * End of $Id: mesc_fsm.c,v 1.13 2005/04/22 17:04:03 cmayncvs Exp $
 */


