/* ===[ $RCSfile: mesc_imp.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc_imp.c,v 1.6 2005/03/22 20:23:25 cmayncvs Exp $"
 *
 *  \brief Handle Multi-drop ESCP Inter-Module Protocol (IMP) messages.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/msg.h"
#include "libcom/fd.h"
#include "mesc_imp.h"
#include "mesc_dbg.h"
#include "mesc_cfg.h"
#include "mesc_tmr.h"
#include "mesc_fsm.h"
#include "mesc_stat.h"
#include "mesc.h"
#include "gassert.h"

/** \brief Typedef for Multi-drop ESCP comm imp data. */
typedef struct {
    char *upper_q;      /**< mesc's upper module's queue */
    char *lower_q;      /**< mesc's lower module's queue (ok to be "none") */
    char *pad_q;        /**< pad module's queue (ok to be NULL) */
} _mesc_imp_t;

/* Inbound messages */
static void _mesc_imp_process_comm_msg(buf_t b);
static void _mesc_imp_process_comm_status(buf_t b);
static void _mesc_imp_process_sreq(buf_t b);
static void _mesc_imp_process_timeout(buf_t b);
static void _mesc_imp_set_debug_level(buf_t b);
static void _mesc_imp_stop(buf_t b);
static void _mesc_imp_stats(buf_t b);
static void _mesc_imp_read_queue(int unused, void *unused_also);
static void _mesc_imp_process_rxdata(buf_t b);
static void _mesc_imp_process_txdata_ind(buf_t b);
static void _mesc_imp_process_x42_status(buf_t b);

/** \brief Holds the Multi-drop ESCP IMP information. */
static _mesc_imp_t _mesc_imp;


/* ======================================================================= */
/**
 *  \brief Saves upper queue, creates local queue, registers functions to
 *      handle imp messages, and synchronizes with upper queue.
 *  \param local_q Local queue name.
 *  \param upper_q Upper module's queue name.
 *  \param lower_q Lower module's queue name (Optional for Secure Session).
 *  \param pad_q Optional pad queue for the pad itself (primarily for resets)
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_init(const char *local_q, const char *upper_q,
                   const char *lower_q, const char *pad_q)
{
    assert(local_q && upper_q);

    _mesc_imp.upper_q = (char *)upper_q;
    _mesc_imp.lower_q = (char *)lower_q;
    _mesc_imp.pad_q = (char *)pad_q;
    mq_create((char *)local_q);

    /* Sync with upper/lower modules, where applicable */
    info("imp_init(): Waiting for %s.", _mesc_imp.upper_q);
    mq_sync_with(_mesc_imp.upper_q);
    if ( !_mesc_imp.lower_q || !strncmp("none", _mesc_imp.lower_q, 4) )
        mesc_cfg_set_secure(0);
    else    /* We have a lower queue - it's either security module or ethmon */
    {
        if ( !strncmp("msecp", _mesc_imp.lower_q, 4) )
            mesc_cfg_set_secure(1);
        info("imp_init(): Waiting for %s.", _mesc_imp.lower_q);
        mq_sync_with(_mesc_imp.lower_q);
    }

    if ( _mesc_imp.pad_q )
    {
        info("imp_init(): Waiting for %s.", _mesc_imp.pad_q);
        mq_sync_with(_mesc_imp.pad_q);
    }

    /* Register all inbound messages */
    msg_dispatch_proc("comm-message", _mesc_imp_process_comm_msg);
    msg_dispatch_proc("comm-status", _mesc_imp_process_comm_status);
    msg_dispatch_proc("solicited-request", _mesc_imp_process_sreq);
    msg_dispatch_proc("timeout", _mesc_imp_process_timeout);
    msg_dispatch_proc("set-debug-level", _mesc_imp_set_debug_level);
    msg_dispatch_proc("stop", _mesc_imp_stop);
    msg_dispatch_proc("stats", _mesc_imp_stats);
    msg_dispatch_proc("rx-data", _mesc_imp_process_rxdata);
    msg_dispatch_proc("tx-data-ind", _mesc_imp_process_txdata_ind);
    msg_dispatch_proc("x42-drop-status", _mesc_imp_process_x42_status);

    fd_add (mq_get_socket (), _mesc_imp_read_queue, 0);
    info("imp_init(): Done.");

} /* mesc_imp_init() */


/* ======================================================================= */
/**
 *  \brief Return's the upper module's queue name that mesc talks to.
 *  \return The upper module's queue name.
 */
/* ======================================================================= */
char *mesc_imp_get_upper_q(void)
{
    return (_mesc_imp.upper_q);
} /* mesc_imp_get_upper_q() */


/* ======================================================================= */
/**
 *  \brief Return's the lower module's queue name that mesc talks to.
 *  \return The lower module's queue name.
 */
/* ======================================================================= */
char *mesc_imp_get_lower_q(void)
{
    return (_mesc_imp.lower_q);
} /* mesc_imp_get_lower_q() */


/* ================ */
/* Inbound messages */
/* ================ */
/* ======================================================================= */
/**
 *  \brief Process a comm status message from a lower queue.  If configured
 *      for wireless or dole, we might receive some.  Handle them by just
 *      forwarding them up the chain.
 *      Format is: "comm-message status-message <msg> [drop]"
 *  \param b The IMP message.
 */
/* ======================================================================= */
static void _mesc_imp_process_comm_msg(buf_t b)
{
    char *msg;
    int len;
    u_int8_t *drop;

    /* Verify message format, handling mandatory fields */
    msg = (u_int8_t *)msg_get_field(b, "status-message", &len);
    if ( (msg == NULL) || (len == 0) )
        return;

    /* Send the message up on behalf of the drop, or use 0 if non specified. */
    if ( (drop = (u_int8_t *)msg_get_field(b, "drop", &len)) != NULL )
        mesc_imp_send_status_msg(*drop, msg);
    else
        mesc_imp_send_status_msg(0, msg);

} /* _mesc_imp_process_comm_msg() */


/* ======================================================================= */
/**
 *  \brief Process a comm status message from a lower queue.
 *      Format is: "comm-status status-update <status> [drop]"
 *  \param b The IMP message.
 */
/* ======================================================================= */
static void _mesc_imp_process_comm_status(buf_t b)
{
    char *status;
    int len;

    /* Verify message format, handling mandatory fields */
    status = (u_int8_t *)msg_get_field(b, "status-update", &len);
    if ( (status == NULL) || (len == 0) )
        return;

    /* Inform the state machine of the data link layer status change */
    mesc_fsm_process_comm_status((int)*status);

} /* _mesc_imp_process_comm_status() */


/* ======================================================================= */
/**
 *  \brief Process a solicited request from the upper queue.
 *      Format is: "solicited-request sap [seq-no] [retry] [drop] data"
 *  \param b The IMP message.
 */
/* ======================================================================= */
static void _mesc_imp_process_sreq(buf_t b)
{
    char *data;
    int data_len, len;
    u_int8_t *sap;
    u_int8_t *drop;

    /* Handle mandatory fields */
    sap = (u_int8_t *)msg_get_field(b, "sap", &len);
    assert(sap && (len == 1));
    data = msg_get_field(b, "data", &data_len);

    /* Data field was mandatory, but can now be optional.  This is so that if
     * the PAD resets, but not the terminals, when the PAD reboots and polls
     * again, the empty poll response will cause the terminal to be configured,
     * thus allowing it to receive unso's again. */
    if ( data == NULL )
        data_len = 0;

    /* Check for optional fields we're concerned about */
    /* TODO: Check for retry? */
    if ( (drop = (u_int8_t *)msg_get_field(b, "drop", &len)) != NULL )
    {
        assert((len == 1) && (*drop < MESC_CFG_MAX_DROPS));
        mesc_stat_update_imp_rqsts(*drop);
        mesc_fsm_process_tx_req(*drop, data, data_len, *sap);
    }
    else /* default to drop 0 if not specified */
    {
        mesc_stat_update_imp_rqsts(0);
        mesc_fsm_process_tx_req(0, data, data_len, *sap);
    }

} /* _mesc_imp_process_sreq() */


/* ======================================================================= */
/**
 *  \brief Process a timeout message.  There are 3 timers per drop, response,
 *      keepalive, and disabled timer, each with a unique id.
 *      Format is: "timeout id"
 *  \param b The IMP message.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_imp_process_timeout(buf_t b)
{
    u_int32_t *id;
    u_int8_t drop;
    u_int8_t type;
    int length;

    /* Get and validate the timer ID */
    assert(id = (u_int32_t *)msg_get_field(b, "id", &length));
    drop = mesc_tmr_get_drop(*id);
    type = mesc_tmr_get_type(*id);
    if ( !mesc_tmr_is_mesc_tmr(*id) || (drop >= MESC_CFG_MAX_DROPS) )
    {
        warn("Bad timer ID: 0x%08X.", *id);
        return;
    }

    /* Which timer expired - response, keepalive, or disabled timer? */
    switch ( type )
    {
        case MESC_TMR_RESPONSE:  mesc_fsm_process_resp_tmo(drop); break;
        case MESC_TMR_KEEPALIVE: mesc_fsm_process_keepalive_tmo(drop); break;
        case MESC_TMR_DISABLE:   mesc_fsm_process_disable_tmo(drop); break;
        default:
            warn("Bad timer type: 0x%08X.", *id);
            break;
    }

} /* _mesc_imp_process_timeout() */


/* ======================================================================= */
/**
 *  \brief Allows us to dynamically set a new debug level, while the program
 *      is running.  Format is: "set-debug-level level"
 *  \param b The buffer holding the debug level change message.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_imp_set_debug_level(buf_t b)
{
    char *field;
    unsigned int field_len;

    /* Verify message format */
    field = msg_get_field(b, "level", &field_len);
    assert(field && (field_len == 1));

    /* Message looks good - handle it */
    mesc_dbg_set_level((u_int8_t)*field);

} /* _mesc_imp_set_debug_level() */


/* ======================================================================= */
/**
 *  \brief Allows us to gracefully shut down via the IMP interface.
 *  \param b The buffer holding the stop message.
 *
 *  \return Nothing - Never returns!
 */
/* ======================================================================= */
static void _mesc_imp_stop(buf_t b)
{
    warn("imp_stop(): Shutting down...\n");
    buf_free(b);
    exit(0);

} /* _mesc_imp_stop() */


/* ======================================================================= */
/**
 *  \brief Allows us to send ourselves a SIGUSR2 to generate statistics.
 *  \param b The buffer holding the stats message.
 *
 *  \return Nothing
 */
/* ======================================================================= */
static void _mesc_imp_stats(buf_t b)
{
    info("Statistics written to /tmp/linux-comm-stats.");
    raise(SIGUSR2);
} /* _mesc_imp_stats() */


/* ======================================================================= */
/**
 *  \brief Call back for fd_select.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_imp_read_queue(int unused, void *unused_also)
{
    msg_dispatch(mq_read_without_restart());
} /* _mesc_imp_read_queue() */


/* ======================================================================= */
/**
 *  \brief Process a rx data request from the lower queue.
 *      Format is: "rx-data [drop] data"
 *  \param b The IMP message.
 */
/* ======================================================================= */
static void _mesc_imp_process_rxdata(buf_t b)
{
    int data_len, len;
    u_int8_t *drop_str;
    u_int8_t drop = 0;
    char *data;

    assert(data = msg_get_field(b, "data", &data_len));

    /* Check for optional fields we're concerned about */
    if ( (drop_str = (u_int8_t *)msg_get_field(b, "drop", &len)) != NULL )
    {
        if ( (len == 1) && (*drop_str < MESC_CFG_MAX_DROPS) )
            drop = *drop_str;
    }

    /*
     * Call this since not using mesc_udp_ucast_read() socket reads.
     * mesc_fsm_process_unicast(u_int8_t drop, buf_t b) expects a buf_t buffer
     * with just the ESCP data, so allocate a new buffer via buf_append().
     */
    buf_free(mesc_fsm_process_unicast(drop, buf_append(0, data, data_len)));

} /* _mesc_imp_process_rxdata() */


/* ======================================================================= */
/**
 *  \brief Process a tx data indicator from the lower queue.
 *       Used to start the ESCP Response Timer if the Secure Protocol is
 *       enabled.
 *      Format is: "tx-data-ind [drop]"
 *  \param b The IMP message.
 */
/* ======================================================================= */
static void _mesc_imp_process_txdata_ind(buf_t b)
{
    int len;
    u_int8_t *drop_str;
    u_int8_t drop = 0;

    /* Check for optional fields we're concerned about */
    if ( (drop_str = (u_int8_t *)msg_get_field(b, "drop", &len)) != NULL )
    {
        if ( (len == 1) && (*drop_str < MESC_CFG_MAX_DROPS) )
            drop = *drop_str;
    }

    /* This will start the response timer according to the MESC State Table. */
    mesc_fsm_process_tx_data_ind(drop);

} /* _mesc_imp_process_txdata_ind() */


/* ======================================================================= */
/**
 *  \brief Process an x42 drop status message from x42pp.
 *       Used to stop/start keep-alive timers in converting from x42 poll
 *      status to ESC world of keep-alive status
 *      Format is: "x42-drop-status" [drop] status"
 *  \param b The IMP message.
 */
/* ======================================================================= */
static void _mesc_imp_process_x42_status(buf_t b)
{
    int len;
    u_int8_t *drop_str;
    u_int8_t *status;
    u_int8_t drop = 0;

    /* Check for mandatory status field */
    assert(status = msg_get_field(b, "status", &len));

    /* Check for optional fields we're concerned about */
    if ( (drop_str = (u_int8_t *)msg_get_field(b, "drop", &len)) != NULL )
    {
        if ( (len == 1) && (*drop_str < MESC_CFG_MAX_DROPS) )
            drop = *drop_str;
    }

    /* Inform the state machine of the new poll status for this drop. */
    if ( !strcmp(status, "slow-polling") || !strcmp(status, "polling"))
    {
        mesc_stat_update_imp_x42_drop_status(drop);
        mesc_fsm_process_x42_drop_status(drop, status);
    }
    else
        warn("Ignoring bad X.42 poll status %s for drop %d.\n", status, drop);
} /* _mesc_imp_process_x42_status() */


/* ================= */
/* Outbound messages */
/* ================= */
/* ======================================================================= */
/**
 *  \brief Send solicited response to upper queue.
 *      Format is: solicited-response sap [drop] data
 *  \param drop The drop address: 0-156 (157 reserved).
 *  \param sap The dcp SAP.
 *  \param data Solicited response data.
 *  \param length Solicited response length.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_send_rsp(u_int8_t drop, u_int8_t *data, int length, u_int8_t sap)
{
    buf_t b;
    char *queue;

    assert((drop < MESC_CFG_MAX_DROPS) && data);
    if ( (drop == MESC_CFG_MAX_DROPS - 1) && _mesc_imp.pad_q )
        queue = _mesc_imp.pad_q;
    else
        queue = _mesc_imp.upper_q;

    b = msg_create(queue, 0, "solicited-response");
    b = msg_add_field(b, "sap", &sap, 1);
    b = msg_add_field(b, "drop", &drop, 1);
    b = msg_add_field(b, "data", data, length);
    mq_write(queue, b);
    mesc_stat_update_imp_resps(drop);

} /* mesc_imp_send_rsp() */


/* ======================================================================= */
/**
 *  \brief Send unsolicited data to upper queue.
 *      Format is: unsolicited-data [drop] data
 *  \param drop The drop address: 0-156 (157 reserved), 255 for broadcast.
 *  \param data Unsolicited data.
 *  \param length Unsolicited data length.
 *  \param dest The destination - applicable for multicast only.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_send_unso(u_int8_t drop, u_int8_t *data, int length, int dest)
{
    buf_t b, b2;

    assert(((drop < MESC_CFG_MAX_DROPS) || (drop == 255)) && data);

    if ( (drop == MESC_PAD_DROP) && _mesc_imp.pad_q )
    {
        b2 = msg_create(_mesc_imp.pad_q, 0, "unsolicited-data");
        b2 = msg_add_field(b2, "drop", &drop, 1);
        b2 = msg_add_field(b2, "data", data, length);
        mq_write(_mesc_imp.pad_q, b2);
    }
    else if ( drop == 255 ) /* broadcast */
    {
        /* Use dest to figure out which group the broadcast is for */
        if ( dest & MESC_COMMON_ACN )
        {
            b = msg_create(_mesc_imp.upper_q, 0, "unsolicited-data");
            b = msg_add_field(b, "drop", &drop, 1);
            b = msg_add_field(b, "data", data, length);
            mq_write(_mesc_imp.upper_q, b);
        }
        if ( (dest & MESC_PAD_ACN) && _mesc_imp.pad_q )
        {
            b2 = msg_create(_mesc_imp.pad_q, 0, "unsolicited-data");
            b2 = msg_add_field(b2, "drop", &drop, 1);
            b2 = msg_add_field(b2, "data", data, length);
            mq_write(_mesc_imp.pad_q, b2);
        }
    }
    else    /* unicast unso to a real drop */
    {
        b = msg_create(_mesc_imp.upper_q, 0, "unsolicited-data");
        b = msg_add_field(b, "drop", &drop, 1);
        b = msg_add_field(b, "data", data, length);
        mq_write(_mesc_imp.upper_q, b);
    }
    mesc_stat_update_imp_unsos(drop, dest);

} /* mesc_imp_send_unso() */


/* ======================================================================= */
/**
 *  \brief Send response timeout to upper queue.
 *      Format is: response-timeout sap [drop]
 *  \param drop The drop address: 0-156 (157 reserved).
 *  \param sap The dcp SAP.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_send_rsp_tmo(u_int8_t drop, u_int8_t sap)
{
    buf_t b;
    char *queue;

    assert(drop < MESC_CFG_MAX_DROPS);
    if ( (drop == MESC_PAD_DROP) && _mesc_imp.pad_q )
        queue = _mesc_imp.pad_q;
    else
        queue = _mesc_imp.upper_q;

    b = msg_create(queue, 0, "response-timeout");
    b = msg_add_field(b, "sap", &sap, 1);
    b = msg_add_field(b, "drop", &drop, 1);
    mq_write(queue, b);
    mesc_stat_update_imp_resp_tmos(drop);

} /* mesc_imp_send_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Send status update to upper queue.
 *      Format is: comm-status status-update [drop]
 *  \param drop The drop address: 0-156 (157 reserved), 255 for broadcast.
 *  \param status Status.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_send_status(u_int8_t drop, u_int8_t status)
{
    buf_t b;
    char *queue;

    assert((drop < MESC_CFG_MAX_DROPS) || (drop == 255));
    if ( (drop == MESC_PAD_DROP) && _mesc_imp.pad_q )
        queue = _mesc_imp.pad_q;
    else
        queue = _mesc_imp.upper_q;

    b = msg_create(queue, 0, "comm-status");
    b = msg_add_field(b, "drop", &drop, 1);
    b = msg_add_field(b, "status-update", &status, 1);
    mq_write(queue, b);

} /* mesc_imp_send_status() */


/* ======================================================================= */
/**
 *  \brief Send status message to upper queue.
 *      Format is: comm-message status-message [drop]
 *  \param drop The drop address: 0-156 (157 reserved), 255 for broadcast.
 *  \param msg The status message.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_send_status_msg(u_int8_t drop, char *msg)
{
    buf_t b;
    char *queue;

    assert(((drop < MESC_CFG_MAX_DROPS) || (drop == 255)) && msg);
    if ( (drop == MESC_PAD_DROP) && _mesc_imp.pad_q )
        queue = _mesc_imp.pad_q;
    else
        queue = _mesc_imp.upper_q;

    b = msg_create(queue, 0, "comm-message");
    b = msg_add_field(b, "drop", &drop, 1);
    b = msg_add_field(b, "status-message", msg, 0);
    mq_write(queue, b);

} /* mesc_imp_send_status_msg() */


/* ======================================================================= */
/**
 *  \brief Send Multi-Cast Join message to Lower queue.
 *      Format is:  multicast-join mcast-address mcast-port
 *      Used so that the "MSECP" Security module can join the
 *      Multicast and listen.
 *  \param mcast_addr String format of "156.123.123.123" of the
 *      Multicast IP address.
 *  \param mcast_port Two bytes of the UDP port.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_send_multicast_join_msg(char *mcast_addr, u_int16_t mcast_port)
{
    buf_t b;

    b = msg_create(_mesc_imp.lower_q, 0, "multicast-join");
    b = msg_add_field(b, "mcast-address", mcast_addr, 0);
    b = msg_add_field(b, "mcast-port", (char *)&mcast_port, 2);
    mq_write(_mesc_imp.lower_q, b);

} /* mesc_imp_send_multicast_join_msg() */


/* ======================================================================= */
/**
 *  \brief Send the data to the msecp queue.
 *      Format is: tx-data [drop] data
 *  \param drop The drop address: 0-156 (157 reserved).
 *  \param raw_rqst The raw ESCP Data Request.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_imp_send_msecp_msg(u_int8_t drop, buf_t raw_rqst)
{
    buf_t b;

    assert(drop < MESC_CFG_MAX_DROPS);

    b = msg_create(_mesc_imp.lower_q, 0, "tx-data");
    b = msg_add_field(b, "drop", &drop, 1);
    b = msg_add_field(b, "client_name", mesc_stat_get_client_name(drop), 0);
    b = msg_add_field(b, "data", buf_data(raw_rqst), buf_length(raw_rqst));
    mq_write(_mesc_imp.lower_q, b);
    buf_free(raw_rqst);

} /* mesc_imp_send_msecp_msg() */


/*
 * End of "$Id: mesc_imp.c,v 1.6 2005/03/22 20:23:25 cmayncvs Exp $".
 */


