/* ===[ $RCSfile: mesc_stat.c,v $ ]=======================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc_stat.c,v 1.9 2005/02/17 20:34:46 cmayncvs Exp $"
 *
 *  \brief Multi-drop ES-Connect (MESC) Statistics handling.
 *
 *  This code handles multi-drop ESC statistics functions.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <netinet/in.h>
#include "libcom/buf.h"
#include "libcom/stats.h"
#include "libcom/list.h"
#include "libcom/str.h"
#include "gassert.h"
#include "mesc_cfg.h"
#include "mesc_dbg.h"
#include "mesc_fsm.h"
#include "mesc_udp.h"
#include "mesc.h"

/* ======= */
/* Defines */
/* ======= */
#define _MESC_STAT_CLIENT_NAME_MASK         0x01    /**< client name mask */
#define _MESC_STAT_RQSTS_AND_RESPS_MASK     0x02    /**< rqsts/resps mask */
#define _MESC_STAT_UNSOS_MASK               0x04    /**< unsos mask */

/** Used for setting a new mask; ensures unused/unsupported bits are cleared */
#define _MESC_STAT_SET_MASK(mask)           \
    ((mask) & (_MESC_STAT_UNSOS_MASK |      \
    _MESC_STAT_RQSTS_AND_RESPS_MASK | _MESC_STAT_CLIENT_NAME_MASK))

/** holds all mesc statistics */
typedef struct {
    u_int8_t active;        /**< flag indicating if drop active or not */

    /* ESCP statistics */
    u_int8_t mask;          /**< Tells which of these statistics to send */
    char client_name[32];   /**< The client name for the drop */
    u_int32_t escp_rqsts;   /**< The # of rqsts sent since last cleared */
    u_int32_t escp_resps;   /**< The # of resps received since last cleared */
    u_int32_t escp_unsos;   /**< The # of unsos received since last cleared */

    /* IMP statistics */
    u_int32_t imp_rqsts;    /**< The # of IMP "solicited-requests" received */
    u_int32_t imp_resps;    /**< The # of IMP "solicited-response" sent */
    u_int32_t imp_unsos;    /**< The # of IMP "unsolicited-data" sent */
    u_int32_t imp_resp_tmos;    /**< The # of IMP "response-timeout" sent */
    u_int32_t imp_x42_drop_status;  /**< The # of "x42-drop-status" rcvd */

    /* Timer statistics */
    u_int32_t resp_tmos;        /**< The # of response timeouts */
    u_int32_t keepalive_tmos;   /**< The # of keepalive timeouts */
    u_int32_t disable_tmos;     /**< The # of disable timeouts */

} _mesc_stat_t;

/** Holds the statistics data for all drops */
static _mesc_stat_t _mesc_stat[MESC_CFG_MAX_DROPS];

static char *_mesc_stat_esc_mask2str(u_int8_t drop);
static void _mesc_stat_print(FILE *f);
static void _mesc_stat_print_stats(FILE *f, u_int8_t drop);

/* ======================================================================= */
/**
 *  \brief Initializes all statistics for the requested drop.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_init(void)
{
    u_int8_t drop;

    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
    {
        _mesc_stat[drop].active = 0;

        /* ESCP statistics */
        mesc_cfg_get_client_name(drop, _mesc_stat[drop].client_name);
        _mesc_stat[drop].mask = 0;
        _mesc_stat[drop].escp_rqsts = 0;
        _mesc_stat[drop].escp_resps = 0;
        _mesc_stat[drop].escp_unsos = 0;

        /* IMP statistics */
        _mesc_stat[drop].imp_rqsts = 0;
        _mesc_stat[drop].imp_resps = 0;
        _mesc_stat[drop].imp_unsos = 0;

        /* Timer statistics */
        _mesc_stat[drop].resp_tmos = 0;
        _mesc_stat[drop].keepalive_tmos = 0;
        _mesc_stat[drop].disable_tmos = 0;
    }

    stats_add_proc(_mesc_stat_print);
    info("stat_init(): Done.");
} /* mesc_stat_init() */


/* ======================================================================= */
/**
 *  \brief Gets the current statistics mask of options to include.
 *  \param drop The drop whose mask to get.
 *  \return The mask.
 */
/* ======================================================================= */
u_int8_t mesc_stat_get_mask(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return (_mesc_stat[drop].mask);
} /* mesc_stat_get_mask() */


/* ======================================================================= */
/**
 *  \brief Sets the current statistics mask of options to include, but never
 *      sets the blob bits or unused bits.
 *  \param drop The drop whose mask to set.
 *  \param mask The new statistics mask for this drop.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_set_mask(u_int8_t drop, u_int8_t mask)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].mask = _MESC_STAT_SET_MASK(mask);
} /* mesc_stat_set_mask() */


/* ======================================================================= */
/**
 *  \brief Gets the client name for specified drop.
 *  \param drop The drop whose client name to get.
 *  \return The client name for this drop.
 */
/* ======================================================================= */
char *mesc_stat_get_client_name(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return (_mesc_stat[drop].client_name);
} /* mesc_stat_get_client_name() */


/* ======================================================================= */
/**
 *  \brief Gets the current ESCP requests statistics.
 *  \param drop The drop whose ESCP requests statistics to get.
 *  \return The number of ESCP requests sent since it was last cleared.
 */
/* ======================================================================= */
u_int32_t mesc_stat_get_escp_rqsts(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return (_mesc_stat[drop].escp_rqsts);
} /* mesc_stat_get_escp_rqsts() */


/* ======================================================================= */
/**
 *  \brief Gets the current ESCP responses statistics.
 *  \param drop The drop whose ESCP responses statistics to get.
 *  \return The number of ESCP responses received since it was last cleared.
 */
/* ======================================================================= */
u_int32_t mesc_stat_get_escp_resps(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return (_mesc_stat[drop].escp_resps);
} /* mesc_stat_get_escp_resps() */


/* ======================================================================= */
/**
 *  \brief Gets the current ESCP unsos statistics.
 *  \param drop The drop whose ESCP unsos statistics to get.
 *  \return The number of ESCP unsos received since it was last cleared.
 */
/* ======================================================================= */
u_int32_t mesc_stat_get_escp_unsos(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    return (_mesc_stat[drop].escp_unsos);
} /* mesc_stat_get_escp_unsos() */


/* ======================================================================= */
/**
 *  \brief Writes all the required ESCP statistics to a buffer and returns it.
 *  \warning Returns a buffer that should be joined to a header.
 *  \param drop The drop whose ESCP statistics to get.
 *  \return The buffer containing all required ESCP statistics, based on mask.
 */
/* ======================================================================= */
buf_t mesc_stat_get_escp_stats(u_int8_t drop)
{
    buf_t b;
    u_int8_t len;
    u_int32_t val;

    assert(drop < MESC_CFG_MAX_DROPS);
    b = buf_append(0, &_mesc_stat[drop].mask, 1);
    if ( _mesc_stat[drop].mask & _MESC_STAT_CLIENT_NAME_MASK )
    {
        len = strlen(_mesc_stat[drop].client_name);
        b = buf_append(b, &len, 1);
        b = buf_append(b, _mesc_stat[drop].client_name, len);
    }
    if ( _mesc_stat[drop].mask & _MESC_STAT_RQSTS_AND_RESPS_MASK )
    {
        val = htonl(_mesc_stat[drop].escp_rqsts);
        b = buf_append(b, &val, 4);
        val = htonl(_mesc_stat[drop].escp_resps);
        b = buf_append(b, &val, 4);
    }
    if ( _mesc_stat[drop].mask & _MESC_STAT_UNSOS_MASK )
    {
        val = htonl(_mesc_stat[drop].escp_unsos);
        b = buf_append(b, &val, 4);
    }

    return (b);
} /* mesc_stat_get_escp_stats() */


/* ======================================================================= */
/**
 *  \brief Clears the ESCP statistics counters.  The mask remains unchanged.
 *  \param drop The drop whose ESCP statistics counters to clear.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_clear_escp_stats(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].escp_rqsts = 0;
    _mesc_stat[drop].escp_resps = 0;
    _mesc_stat[drop].escp_unsos = 0;
} /* mesc_stat_clear_escp_stats() */


/* ======================================================================= */
/**
 *  \brief Updates the current ESCP requests statistics.
 *  \param drop The drop whose ESCP requests statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_escp_rqsts(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].escp_rqsts++;
} /* mesc_stat_update_escp_rqsts() */


/* ======================================================================= */
/**
 *  \brief Updates the current ESCP responses statistics.
 *  \param drop The drop whose ESCP responses statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_escp_resps(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].escp_resps++;
} /* mesc_stat_update_escp_resps() */


/* ======================================================================= */
/**
 *  \brief Updates the current ESCP unsos statistics.
 *  \param drop The drop whose ESCP unsos statistics to update (255=all drops)
 *  \param dest For a multicast, specifies which set of drops.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_escp_unsos(u_int8_t drop, int dest)
{
    int d;

    if ( drop == 255 )
    {
        if ( dest & MESC_COMMON_ACN )
            for ( d = 0; d < MESC_PAD_DROP; d++ )
                _mesc_stat[d].escp_unsos++;
        if ( dest & MESC_PAD_ACN )
            _mesc_stat[MESC_PAD_DROP].escp_unsos++;
    }
    else
    {
        assert(drop < MESC_CFG_MAX_DROPS);
        _mesc_stat[drop].escp_unsos++;
    }
} /* mesc_stat_update_escp_unsos() */

        /* -------------------------------------------------- */
        /* IMP statistics - These don't get sent to ES-Admin. */
        /* -------------------------------------------------- */

/* ======================================================================= */
/**
 *  \brief Update the number of "solicited-request" IMP messages received.
 *  \param drop The drop whose "solicited-request" statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_imp_rqsts(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].active = 1;
    _mesc_stat[drop].imp_rqsts++;
} /* mesc_stat_update_imp_rqsts() */


/* ======================================================================= */
/**
 *  \brief Update the number of "solicited-response" IMP messages sent.
 *  \param drop The drop whose "solicited-response" statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_imp_resps(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].imp_resps++;
} /* mesc_stat_update_imp_resps() */


/* ======================================================================= */
/**
 *  \brief Update the number of "unsolicited-data" IMP messages sent.
 *  \param drop The drop whose "unsolicited-data" statistics to update.
 *  \param dest The destination - applicable for multicast only.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_imp_unsos(u_int8_t drop, int dest)
{
    int d;

    if ( drop == 255 )
    {
        if ( dest & MESC_COMMON_ACN )
            for ( d = 0; d < MESC_PAD_DROP; d++ )
                _mesc_stat[d].imp_unsos++;
        if ( dest & MESC_PAD_ACN )
            _mesc_stat[MESC_PAD_DROP].imp_unsos++;
    }
    else
    {
        assert(drop < MESC_CFG_MAX_DROPS);
        _mesc_stat[drop].imp_unsos++;
    }
} /* mesc_stat_update_imp_unsos() */


/* ======================================================================= */
/**
 *  \brief Update the number of "response-timeout" IMP messages received.
 *  \param drop The drop whose "response-timeout" statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_imp_resp_tmos(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].imp_resp_tmos++;
} /* mesc_stat_update_imp_resp_tmos() */


/* ======================================================================= */
/**
 *  \brief Update the number of "x42-drop-status" IMP messages received.
 *  \param drop The drop whose "x42-drop-status" statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_imp_x42_drop_status(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].imp_x42_drop_status++;
} /* mesc_stat_update_imp_x42_drop_status() */

        /* ---------------------------------------------------- */
        /* Timer statistics - These don't get sent to ES-Admin. */
        /* ---------------------------------------------------- */

/* ======================================================================= */
/**
 *  \brief Update the number of response timer expirations.
 *  \param drop The drop whose response timer expirations statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_resp_tmos(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].resp_tmos++;
} /* mesc_stat_update_resp_tmos() */


/* ======================================================================= */
/**
 *  \brief Update the number of keepalive timer expirations.
 *  \param drop The drop whose keepalive timer expirations statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_keepalive_tmos(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].keepalive_tmos++;
} /* mesc_stat_update_keepalive_tmos() */


/* ======================================================================= */
/**
 *  \brief Update the number of disable timer expirations.
 *  \param drop The drop whose disable timer expirations statistics to update.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_stat_update_disable_tmos(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_stat[drop].disable_tmos++;
} /* mesc_stat_update_disable_tmos() */


/* ======================================================================= */
/**
 *  \brief Convert an ESC statistics mask to a human readable string.
 *  \param mask The mask to convert.
 *  \return A string describing the \a mask.
 */
/* ======================================================================= */
static char *_mesc_stat_esc_mask2str(u_int8_t mask)
{
    switch ( mask & 0x06 )
    {
        case 1: return ("name");
        case 2: return ("rqst/resp");
        case 3: return ("rqst/resp,name");
        case 4: return ("unso");
        case 5: return ("unso,name");
        case 6: return ("unso,rqst/resp");
        case 7: return ("unso,rqst/resp,name");
        default: return ("none");
    }
} /* _mesc_stat_esc_mask2str() */


/* ======================================================================= */
/**
 *  \brief Print MESC information to a file
 *  \param f The file to write MESC information to.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_stat_print(FILE *f)
{
    u_int8_t drop;

    assert(f);

    /* Common Parameters: */
    fprintf(f, "---[ ES-PAD MULTI-DROP ES-CONNECT STATISTICS ]----"
        "-------------\n\n");
    fprintf(f, "   Default FW/Comm Debug Level       -> %d/%d (%s/%s)\n",
        mesc_cfg_get_dbg_level(), mesc_dbg_get_level(),
        mesc_cfg_level2str(mesc_cfg_get_dbg_level()),
        mesc_dbg_level2str(mesc_dbg_get_level()));
    fprintf(f, "   Site ID                           -> %d (%s)\n",
        mesc_cfg_get_site_id(), mesc_cfg_siteid2str(mesc_cfg_get_site_id()));
    fprintf(f, "   Hardware Type ID                  -> %d (%s)\n",
        mesc_cfg_get_hw_type_id(), mesc_cfg_hwid2str(mesc_cfg_get_hw_type_id()));
    fprintf(f, "   Unit ID                           -> %d\n",
        mesc_cfg_get_unit_id());
    /* NOTE: ROM ID, host1, host2, are drop-specific. */
    fprintf(f, "   Default Timeout Count (tc)        -> %d\n",
        mesc_cfg_get_tmo_cnt());
    fprintf(f, "   Default Min Response Timeout (t0) -> %s\n",
        sec2str(mesc_cfg_get_min_rsp_tmo() / 1000));
    fprintf(f, "   Default Max Response Timeout (t1) -> %s\n",
        sec2str(mesc_cfg_get_max_rsp_tmo() / 1000));
    fprintf(f, "   Default Keepalive Timeout (ka)    -> %s\n",
        sec2str(mesc_cfg_get_keepalive_tmo()));
    fprintf(f, "   UDP Base Port                     -> %d (0x%04X)\n\n",
        mesc_cfg_get_base_port(), mesc_cfg_get_base_port());

    /* Per-drop statistics & parameters: */
    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
        if ( _mesc_stat[drop].active )
        {
            _mesc_stat_print_stats(f, drop);
            fprintf(f, "\n");
            mesc_fsm_print(f, drop);
            mesc_udp_print(f, drop);
            fprintf(f, "\n");
            fprintf(f, "--------------------------------------------------"
                "-------------\n");
        }

} /* _mesc_stat_print() */


/* ======================================================================= */
/**
 *  \brief Print MESC statistics to a file
 *  \param f The file to write MESC statistics to.
 *  \param drop The drop whose MESC statistics to print.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_stat_print_stats(FILE *f, u_int8_t drop)
{
    assert(f && (drop < MESC_CFG_MAX_DROPS));
    fprintf(f, "Drop %d (%s): %s\n",
        drop, mesc_cfg_drop2ascii(drop), _mesc_stat[drop].client_name);

    /* ESCP statistics */
    fprintf(f, "   ESCP statistics mask              -> %d (%s)\n",
        _mesc_stat[drop].mask, _mesc_stat_esc_mask2str(_mesc_stat[drop].mask));
    fprintf(f, "   ESCP requests                     -> %d\n",
        _mesc_stat[drop].escp_rqsts);
    fprintf(f, "   ESCP responses                    -> %d\n",
        _mesc_stat[drop].escp_resps);
    fprintf(f, "   ESCP unsos                        -> %d\n",
        _mesc_stat[drop].escp_unsos);

    /* IMP statistics */
    fprintf(f, "   IMP  solicited-request            -> %d\n",
        _mesc_stat[drop].imp_rqsts);
    fprintf(f, "   IMP  solicited-response           -> %d\n",
        _mesc_stat[drop].imp_resps);
    fprintf(f, "   IMP  unsolicited-data             -> %d\n",
        _mesc_stat[drop].imp_unsos);
    fprintf(f, "   IMP  response-timeout             -> %d\n",
        _mesc_stat[drop].imp_resp_tmos);
    fprintf(f, "   IMP  x42-drop-status              -> %d\n",
        _mesc_stat[drop].imp_x42_drop_status);

    /* Timer statistics */
    fprintf(f, "   TMR  timeout (response)           -> %d\n",
        _mesc_stat[drop].resp_tmos);
    fprintf(f, "   TMR  timeout (keepalive)          -> %d\n",
        _mesc_stat[drop].keepalive_tmos);
    fprintf(f, "   TMR  timeout (disable)            -> %d\n",
        _mesc_stat[drop].disable_tmos);

} /* _mesc_stat_print_stats() */


/*
 * End of "$Id: mesc_stat.c,v 1.9 2005/02/17 20:34:46 cmayncvs Exp $".
 */


