/* ===[ $RCSfile: mesc_tmr.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc_tmr.c,v 1.3 2005/04/22 17:02:09 cmayncvs Exp $"
 *
 *  \brief Contains the MESC Timer functions.
 */
/* ======================================================================= */
/* ======== */
/* Includes */
/* ======== */
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/time.h>
#include "libcom/buf.h"
#include "libcom/timer.h"
#include "mesc_dbg.h"
#include "mesc_art.h"
#include "mesc_tmr.h"
#include "mesc_cfg.h"
#include "escp.h"
#include "gassert.h"

/* ======= */
/* Defines */
/* ======= */
#define _MESC_TMR_ID_MASK           0xFFFF0000  /**< Timer ID mask */
#define _MESC_TMR_DROP_MASK         0x0000FF00  /**< Timer Drop mask */
#define _MESC_TMR_TYPE_MASK         0x000000FF  /**< Timer type mask */

#define _MESC_TMR_ID_SHIFT          16          /**< Timer ID shift value */
#define _MESC_TMR_DROP_SHIFT        8           /**< Timer Drop shift value */
#define _MESC_TMR_TYPE_SHIFT        0           /**< Timer type shift value */


/** Macro to get the timer ID field */
#define _MESC_TMR_GET_ID(id)        \
    (((id) & _MESC_TMR_ID_MASK) >> _MESC_TMR_ID_SHIFT)
/** Macro to get the timer drop field */
#define _MESC_TMR_GET_DROP(id)      \
    (((id) & _MESC_TMR_DROP_MASK) >> _MESC_TMR_DROP_SHIFT)
/** Macro to get the timer type field */
#define _MESC_TMR_GET_TYPE(id)      \
    (((id) & _MESC_TMR_TYPE_MASK) >> _MESC_TMR_TYPE_SHIFT)

/** Macro to set the timer ID field */
#define _MESC_TMR_SET_ID(id)        ((id) << _MESC_TMR_ID_SHIFT)
/** Macro to get the timer drop field */
#define _MESC_TMR_SET_DROP(drop)    ((drop) << _MESC_TMR_DROP_SHIFT)
/** Macro to get the timer type field */
#define _MESC_TMR_SET_TYPE(type)    ((type) << _MESC_TMR_TYPE_SHIFT)

/** Typedef for timer information */
typedef struct {
    timer resp_tmr;                 /**< Response timer */
    timer keepalive_tmr;            /**< Keep-alive timer */
    timer disable_tmr;              /**< Disable timer */
    struct timeval start_time;      /**< Timer start time */
    u_int16_t response_time;        /**< The last response time */
} _mesc_tmr_t;

/** Holds timer information for all drops */
static _mesc_tmr_t _mesc_tmr[MESC_CFG_MAX_DROPS];

static void _mesc_tmr_mark_start(u_int8_t drop);

/* ======================================================================= */
/**
 *  \brief Initializes the timer component (currently, nothing to do).
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_init(void)
{
    u_int8_t drop;
    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
    {
        _mesc_tmr[drop].resp_tmr = 0;
        _mesc_tmr[drop].keepalive_tmr = 0;
        _mesc_tmr[drop].disable_tmr = 0;
        _mesc_tmr[drop].start_time.tv_sec = 0;
        _mesc_tmr[drop].start_time.tv_usec = 0;
        _mesc_tmr[drop].response_time = ESCP_NO_PREV_RESP_TIME;
    }
    info("tmr_init(): Done.");
} /* mesc_tmr_init() */


/* ======================================================================= */
/**
 *  \brief Starts the response timer.
 *  \param drop The drop address associated with this timer.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_start_resp(u_int8_t drop)
{
    u_int16_t tmo;

    assert(drop < MESC_CFG_MAX_DROPS);

    _mesc_tmr_mark_start(drop);
    tmo = mesc_art_get_tmo(drop);
    _mesc_tmr[drop].resp_tmr = timer_start(
         tmo + mesc_art_get_noise(drop),
        _MESC_TMR_SET_ID(MESC_TMR_ID) |
        _MESC_TMR_SET_DROP(drop) |
        _MESC_TMR_SET_TYPE(MESC_TMR_RESPONSE));
    /* info("Response Timer(%d) started: %d.%d secs.", drop, tmo/1000, tmo%1000); */
} /* mesc_tmr_start_resp() */


/* ======================================================================= */
/**
 *  \brief Starts the keepalive timer.
 *  \param drop The drop address associated with this timer.
 *  \param sec The time in seconds before timer expiration.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_start_keepalive(u_int8_t drop, u_int16_t sec)
{
    assert(drop < MESC_CFG_MAX_DROPS);

    _mesc_tmr[drop].keepalive_tmr = timer_start(1000 * sec,
        _MESC_TMR_SET_ID(MESC_TMR_ID) |
        _MESC_TMR_SET_DROP(drop) |
        _MESC_TMR_SET_TYPE(MESC_TMR_KEEPALIVE));
    info("Keepalive Timer(%d) started: %d seconds.", drop, sec);

} /* mesc_tmr_start_keepalive() */


/* ======================================================================= */
/**
 *  \brief Starts the disable timer.
 *  \param drop The drop address associated with this timer.
 *  \param sec The time in seconds before timer expiration.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_start_disable(u_int8_t drop, u_int16_t sec)
{
    assert(drop < MESC_CFG_MAX_DROPS);

    _mesc_tmr[drop].disable_tmr = timer_start(1000 * sec,
        _MESC_TMR_SET_ID(MESC_TMR_ID) |
        _MESC_TMR_SET_DROP(drop) |
        _MESC_TMR_SET_TYPE(MESC_TMR_DISABLE));
    info("Disable Timer(%d) started: %d seconds.", drop, sec);

} /* mesc_tmr_start_disable() */


/* ======================================================================= */
/**
 *  \brief Stops the response timer.
 *  \param drop The drop address associated with this timer.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_stop_resp(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_tmr[drop].resp_tmr = timer_stop(_mesc_tmr[drop].resp_tmr);

} /* mesc_tmr_stop_resp() */


/* ======================================================================= */
/**
 *  \brief Stops the keepalive timer.
 *  \param drop The drop address associated with this timer.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_stop_keepalive(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_tmr[drop].keepalive_tmr = timer_stop(_mesc_tmr[drop].keepalive_tmr);

} /* mesc_tmr_stop_keepalive() */


/* ======================================================================= */
/**
 *  \brief Stops the disable timer.
 *  \param drop The drop address associated with this timer.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_stop_disable(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_tmr[drop].disable_tmr = timer_stop(_mesc_tmr[drop].disable_tmr);

} /* mesc_tmr_stop_disable() */


/* ======================================================================= */
/**
 *  \brief Gets the ID part of the timer.
 *  \param tmr The timer to get the ID from.
 *  \return The MESC ID.  This should always be \a MESC_TMR_ID.
 */
/* ======================================================================= */
u_int16_t mesc_tmr_get_id(u_int32_t tmr)
{
    return (_MESC_TMR_GET_ID(tmr));
} /* mesc_tmr_get_id() */


/* ======================================================================= */
/**
 *  \brief Gets the drop address part of the timer.
 *  \param tmr The timer to get the drop address from.
 *  \return The drop address.
 */
/* ======================================================================= */
u_int8_t mesc_tmr_get_drop(u_int32_t tmr)
{
    return (_MESC_TMR_GET_DROP(tmr));
} /* mesc_tmr_get_drop() */


/* ======================================================================= */
/**
 *  \brief Gets the type part of the timer.
 *  \param tmr The timer to get the type from.
 *  \return The type as either MESC_TMR[_RESPONSE or _IDLE or _DISABLE].
 */
/* ======================================================================= */
u_int8_t mesc_tmr_get_type(u_int32_t tmr)
{
    return (_MESC_TMR_GET_TYPE(tmr));
} /* mesc_tmr_get_type() */


/* ======================================================================= */
/**
 *  \brief Checks if the timer is a MESC timer or not.
 *  \param tmr The timer to check.
 *  \return 1 if MESC timer; 0 otherwise.
 */
/* ======================================================================= */
int mesc_tmr_is_mesc_tmr(u_int32_t tmr)
{
    return (_MESC_TMR_GET_ID(tmr) == MESC_TMR_ID);
} /* mesc_tmr_is_mesc_tmr() */


/* ======================================================================= */
/**
 *  \brief Get the previous response time for this drop.
 *  \param drop The drop address [range: 0 - (MESC_CFG_MAX_DROPS - 1)]
 *  \return The previous timeout or ESCP_NO_PREV_RESP_TIME if 1st time.
 */
/* ======================================================================= */
u_int16_t mesc_tmr_get_prev_resp_time(u_int8_t drop)
 {
    assert(drop < MESC_CFG_MAX_DROPS);
    return (_mesc_tmr[drop].response_time);
} /* mesc_tmr_get_prev_resp_time() */


/* ======================================================================= */
/**
 *  \brief Resets the previous response time for this drop back to 0xFFFF.
 *  \param drop The drop address [range: 0 - (MESC_CFG_MAX_DROPS - 1)]
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_reset_prev_resp_time(u_int8_t drop)
{
    assert(drop < MESC_CFG_MAX_DROPS);
    _mesc_tmr[drop].response_time = ESCP_NO_PREV_RESP_TIME;
} /* mesc_tmr_reset_prev_resp_time() */


/* ======================================================================= */
/**
 *  \brief Saves the end time
 *  \param drop The drop address
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_tmr_mark_end(u_int8_t drop)
{
    struct timeval end;
    int dif;

    assert(gettimeofday(&end, 0) == 0);

    dif = ((end.tv_sec * 1000) + (end.tv_usec / 1000))
        - ((_mesc_tmr[drop].start_time.tv_sec * 1000) +
        (_mesc_tmr[drop].start_time.tv_usec / 1000));
    if ( dif < 0 )
        dif = 0;
    _mesc_tmr[drop].response_time = (dif < 65535) ? dif : 65534;

    mesc_art_update(drop, dif);
    info ("Reponse time for drop %d was %d", drop, dif);
} /* mesc_tmr_mark_end() */


                        /* --------------- */
                        /* Local functions */
                        /* --------------- */

/* ======================================================================= */
/**
 *  \brief Saves the start time
 *  \param drop The drop address
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_tmr_mark_start(u_int8_t drop)
{
    assert(gettimeofday(&_mesc_tmr[drop].start_time, 0) == 0);
} /* _mesc_tmr_mark_start() */


/*
 * End of "$Id: mesc_tmr.c,v 1.3 2005/04/22 17:02:09 cmayncvs Exp $".
 */

