/* ===[ $RCSfile: mesc_udp.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: mesc_udp.c,v 1.9 2005/04/22 17:01:49 cmayncvs Exp $"
 *
 *  \brief Implements MESC UDP interface.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <unistd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/fd.h"
#include "libcom/list.h"
#include "mesc_dbg.h"
#include "mesc_imp.h"
#include "mesc_cfg.h"
#include "mesc_fsm.h"
#include "mesc_udp.h"
#include "mesc.h"
#include "escp.h"
#include "gassert.h"

/** Number of times to try gethostbyname */
#define _MESC_UDP_GETHOSTBYNAME_RETRIES     3

static list _mesc_udp_dns_lookup(const char *host);
static void _mesc_udp_ucast_read(int s, void *unused);
static void _mesc_udp_mcast_read(int s, void *unused);
static void _multicast_join(const char *addr, const char *port);

/** \brief Typedef for Multi-drop ESCP Unicast UDP information. */
typedef struct {
    struct sockaddr_in to;  /**< UDP address to send to.   */
    int sock;               /**< UDP socket. */
    list host;              /**< List of host(s): Primary, Secondary, ... */
    list cur_host;          /**< Current host. */
    list addr_list;         /**< List of host addresses. */
 } _mesc_udp_ucast_t;

/** \brief Typedef for Multi-drop ESCP Multicast UDP information. */
typedef struct {
    int sock;           /**< UDP socket. */
    char str[257];      /**< "ESCP/UDP.multicast:www.xxx.yyy.zzz:ppppp" (42) */
} _mesc_udp_mcast_t;

/** \brief Typedef for all the Multi-drop ESCP UDP information */
typedef struct {
    _mesc_udp_mcast_t mcast;                        /**< multicast data */
    _mesc_udp_ucast_t ucast[MESC_CFG_MAX_DROPS];    /**< unicast data */
} _mesc_udp_t;

/** \brief Holds all the Multi-drop ESCP UDP information. */
static _mesc_udp_t _mesc_udp;


/* ======================================================================= */
/**
 *  \brief Initializes the udp message handler for all drops.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_udp_init(void)
{
    int len_h1, len_h2;
    int drop;
    char msg[256];

    memset(&_mesc_udp, 0, sizeof(_mesc_udp));

    /* Construct the host list */
    len_h2 = strlen(mesc_cfg_get_host2());
    len_h1 = strlen(mesc_cfg_get_host1());
#if !defined ARCH_ESPAD
    /* Only assert() for Altura since mesc does not run while diags runs, but
     * on ESPAD, 'webs' runs at the same time as mesc, so if there's a
     * parameter error, we'll never be able to correct it! */
    assert(len_h2 || len_h1);
#endif
    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
    {
        if ( len_h2 )
        {
            _mesc_udp.ucast[drop].host =
                cons(mesc_cfg_get_host2(), _mesc_udp.ucast[drop].host);
        }
        if ( len_h1 )
        {
            _mesc_udp.ucast[drop].host =
                cons(mesc_cfg_get_host1(), _mesc_udp.ucast[drop].host);
        }

        _mesc_udp.ucast[drop].cur_host = _mesc_udp.ucast[drop].host;
    }

    if ( _mesc_udp.ucast[MESC_PAD_DROP].cur_host )
        snprintf(msg, sizeof(msg), "Message=ConnectedTo;Address=%s",
            (char *)car(_mesc_udp.ucast[MESC_PAD_DROP].cur_host));
    else
        strncpy(msg, "Message=ConnectedTo;Address=NONE", sizeof(msg));

    msg[sizeof(msg) - 1] = '\0';
    mesc_imp_send_status_msg(MESC_PAD_DROP, msg);
    info("udp_init(): Done.");

} /* mesc_udp_init() */


/* ======================================================================= */
/**
 *  \brief Initializes the udp message handler for the specified drop.
 *      If drop is -1, initializes the multicast udp message handler.
 *  \param drop The drop whose udp message handler to initialize.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_udp_open(int drop)
{
    int s;
    struct sockaddr_in local;
    #if defined ARCH_ESPAD
    int optval = 1;
    #endif /* ARCH_ESPAD */

    assert(drop < MESC_CFG_MAX_DROPS);

    /* Shut it down if it already exists. */
    if ( _mesc_udp.ucast[drop].sock )
    {
        fd_remove(_mesc_udp.ucast[drop].sock);
        assert(close(_mesc_udp.ucast[drop].sock) != -1);
        _mesc_udp.ucast[drop].sock = 0;
    }

    /* Create the socket and bind it to the port designated for that drop */
    assert((s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) != -1);
    #if defined ARCH_ESPAD
    assert(setsockopt(s, SOL_SOCKET, SO_BSDCOMPAT, &optval, sizeof(int)) == 0);
    #endif /* ARCH_ESPAD */
    local.sin_family = AF_INET;
    local.sin_addr.s_addr = htonl(INADDR_ANY);
    local.sin_port = mesc_cfg_get_port(drop);
    info("Local port: %d (0x%0X).", local.sin_port, local.sin_port);
    local.sin_port = htons(mesc_cfg_get_port(drop));
    assert(bind(s, (struct sockaddr *)&local, sizeof(struct sockaddr_in)) != -1);

    /* Add the socket descriptor to our list for select() and to our info. */
    _mesc_udp.ucast[drop].sock = s;
    fd_add(s, _mesc_udp_ucast_read, 0);

} /* mesc_udp_open() */


/* ======================================================================= */
/**
 *  \brief Send a message to the host from this drop.
 *  \param drop The drop sending the message.
 *  \param b Message to send.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_udp_send(u_int8_t drop, buf_t b)
{
    unsigned long addr = (unsigned long)car(_mesc_udp.ucast[drop].addr_list);

    if ( addr == 0 )
    {
        warn("Empty address list - sendto() aborted.");
        buf_free(b);
        return;
    }

    _mesc_udp.ucast[drop].to.sin_addr.s_addr = addr;
    _mesc_udp.ucast[drop].to.sin_family = AF_INET;
    _mesc_udp.ucast[drop].to.sin_port = htons(ESCP_UDP_PORT);
    mesc_dbg_time_stamp();
    mesc_dbg_print_buf(b);

    if ( WITH_RESTART(
        sendto(_mesc_udp.ucast[drop].sock, buf_data(b), buf_length(b),
            MSG_NOSIGNAL, (struct sockaddr *)&_mesc_udp.ucast[drop].to,
            sizeof(struct sockaddr_in))) == -1 )
    {
        warn("Sendto(%s/%s) failed: %s.",
            (char *)car(_mesc_udp.ucast[drop].cur_host),
            inet_ntoa(_mesc_udp.ucast[drop].to.sin_addr), strerror(errno));
        mesc_udp_open(drop);
        buf_free(b);
        /* TODO: Should we do more than simply return? */
        return;
    }

    info("Msg sent to: %s (%s).",
        (char *)car(_mesc_udp.ucast[drop].cur_host),
        inet_ntoa(_mesc_udp.ucast[drop].to.sin_addr));
    buf_free(b);

} /* mesc_udp_send() */


/* ======================================================================= */
/**
 *  \brief The client has given up on the current address.  Use the next one.
 *  \param drop The drop who is about to change host addresses.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_udp_use_next_addr(u_int8_t drop)
{
    char msg[256];
    unsigned long addr;

    /* Get the next address in the list for the current host. */
    _mesc_udp.ucast[drop].addr_list = cdrf(_mesc_udp.ucast[drop].addr_list);
    if ( _mesc_udp.ucast[drop].addr_list == NULL )
    {
        /* Rotate hosts */
        _mesc_udp.ucast[drop].cur_host = cdr(_mesc_udp.ucast[drop].cur_host);
        if ( !_mesc_udp.ucast[drop].cur_host )
            _mesc_udp.ucast[drop].cur_host = _mesc_udp.ucast[drop].host;

        /* Perform DNS lookup on host */
        _mesc_udp.ucast[drop].addr_list =
            _mesc_udp_dns_lookup(car(_mesc_udp.ucast[drop].cur_host));
    }

    addr = (unsigned long)car(_mesc_udp.ucast[drop].addr_list);
    if ( addr )
        snprintf(msg, sizeof(msg), "Message=ConnectedTo;Address=%s",
            inet_ntoa(*(struct in_addr *)&addr));
    else
        strncpy(msg, "Message=ConnectedTo;Address=NONE", sizeof(msg));
    msg[sizeof(msg) - 1] = '\0';
    mesc_imp_send_status_msg(drop, msg);

} /* mesc_udp_use_next_addr() */


/* ======================================================================= */
/**
 *  \brief Attempt to join a multicast group, common to all drops.
 *  \param data A string containing a multicast IP address and port in the
 *      form of "ESCP/UDP.multicast:www.xxx.yyy.zzz:ppppp".
 *  \param len The length of the string, not necessarily null-terminated.
 *
 *  This function changes the multicast address & port for all drops, if
 *  possible, but if any error occurs either in parsing the string or in
 *  attempting to join, the original multicast address & port are unchanged.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_udp_multicast_join(const char *data, u_int8_t len)
{
    const char *escp_mcast_prefix = "ESCP/UDP.multicast:";
    const char *secp_mcast_prefix = "SECP/UDP.multicast:";
    const char *mcast_prefix;

    char *b = (char *)alloca(len + 1);
    char *mcast, *addr, *ptr;

    /* Don't assume that the data is null-terminated. */
    assert(data);
    memcpy(b, data, len);
    b[len] = '\0';

    if ( mesc_cfg_is_secure() )
        mcast_prefix = secp_mcast_prefix;
    else
        mcast_prefix = escp_mcast_prefix;

    /* Find the multicast prefix, then grab everything from here 'til ';' */
    if ( (ptr = strstr(b, mcast_prefix)) == NULL )
        return;
    mcast = strsep(&ptr, ";");

    /* Search the current multicast string for this one.  If it's the same,
     * no need to rejoin the same group, so we're done; otherwise skip past
     * the prefix. */
    if ( !strncmp(_mesc_udp.mcast.str, mcast, sizeof(_mesc_udp.mcast.str)) )
        return;
    ptr = mcast + strlen(mcast_prefix);

    /* Separate the string into the address and port */
    addr = strsep(&ptr, ":");
    if ( ptr == NULL )
        return;

    /* TODO: Somewhere in here I need to check if I've already joined a
     * multicast group.  If so, then we're now attempting to join a NEW group,
     * so I need to remove myself from the current group first! */

    /* Finally, join this multicast group */
    _multicast_join(addr, ptr);

} /* mesc_udp_multicast_join() */


/* ======================================================================= */
/**
 *  \brief Get the multicast address string.
 *  \return The multicast address as a null-terminated string.
 */
/* ======================================================================= */
char *mesc_udp_get_multicast_str(void)
{
    return (_mesc_udp.mcast.str);
} /* mesc_udp_get_multicast_str() */


/* ======================================================================= */
/**
 *  \brief Perform DNS lookup of cur_host for \a drop.  Save addr list.
 *  \param drop The drop address: 0-156
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_udp_dns_lookup(u_int8_t drop)
{
    /* Perform DNS lookup of current host. */
    if ( (_mesc_udp.ucast[drop].addr_list =
        _mesc_udp_dns_lookup(car(_mesc_udp.ucast[drop].cur_host))) == NULL )
    {
        /* Address list is empty - rotate the host list and try again. */
        if ( (_mesc_udp.ucast[drop].cur_host = cdr(_mesc_udp.ucast[drop].cur_host)) == NULL )
            _mesc_udp.ucast[drop].cur_host = _mesc_udp.ucast[drop].host;
        _mesc_udp.ucast[drop].addr_list =
            _mesc_udp_dns_lookup(car(_mesc_udp.ucast[drop].cur_host));
    }

} /* mesc_udp_dns_lookup() */


/* ======================================================================= */
/**
 *  \brief Print MESC UDP infomration for a drop to a file.
 *  \param f The file to write MESC UDP information to.
 *  \param drop The drop whose MESC UDP information to print.
 *  \return Nothing.
 */
/* ======================================================================= */
void mesc_udp_print(FILE *f, u_int8_t drop)
{
    list tmp;
    unsigned long addr;

    fprintf(f, "   Current Host                      -> %s\n",
        strcmp((char *)car(_mesc_udp.ucast[drop].cur_host),
            (char *)car(_mesc_udp.ucast[drop].host)) ? "Secondary" : "Primary");
    for (tmp = _mesc_udp.ucast[drop].addr_list; tmp; tmp = cdr(tmp) )
    {
        addr = (unsigned long)car(tmp);
        fprintf(f, "                                     -> %s\n",
            inet_ntoa(*(struct in_addr *)&addr));
    }
    fprintf(f, "   Multicast Information             -> %s\n",
        mesc_udp_get_multicast_str());
} /* mesc_udp_print() */

                            /* --------------- */
                            /* Local functions */
                            /* --------------- */

/* ======================================================================= */
/**
 *  \brief Perform DNS lookup of host.  Return address list.
 *  \param host The host to lookup.
 *  \return The address list for the specified host.
 */
/* ======================================================================= */
static list _mesc_udp_dns_lookup(const char *host)
{
    int i;
    struct hostent *hip = 0;
    list addr_list = NULL;
    list tmp;
    unsigned long addr;

    info("Looking up host %s.", host);
    for ( i = 0; i < _MESC_UDP_GETHOSTBYNAME_RETRIES; i++ )
    {
        if ( (hip = gethostbyname(host)) != NULL )
            break;
        warn("gethostbyname(%s)!", hstrerror(h_errno));
    }
    if ( i >= _MESC_UDP_GETHOSTBYNAME_RETRIES )
        return (NULL);
    assert(hip);

    /*
     * Construct our list of addresses for this host.
     *
     * NOTE: The list constructor expects a pointer to the object in the list,
     * but I'm casting things here so I'm actually placing the object itself
     * in the list, rather than a pointer to it, as the memory it points to
     * can be overwritten by subsequent calls to gethostbyname().
     */
    for ( i = 0; hip->h_addr_list[i]; i++ )
        addr_list = cons((unsigned long *)
            (*(unsigned long *)hip->h_addr_list[i]), addr_list);

    /* Ensure we can access them by printing them out */
    for ( tmp = addr_list; tmp; tmp = cdr(tmp) )
    {
        addr = (unsigned long)car(tmp);
        info("Address:\t%s", inet_ntoa(*(struct in_addr *)&addr));
    }

    return (addr_list);

} /* _mesc_udp_dns_lookup() */


/* ======================================================================= */
/**
 *  \brief Call back for fd_select.
 *  \param s The socket, which we'll have to figure out from what drop.
 *  \param unused Unused argument required by fd_add().
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_udp_ucast_read(int s, void *unused)
{
    char temp[MAX_MESSAGE_LENGTH];
    struct sockaddr_in from;
    socklen_t from_length = sizeof(struct sockaddr_in);
    int length = 0;
    int drop;

    /* Which drop is it for? */
    for ( drop = 0; drop < MESC_CFG_MAX_DROPS; drop++ )
        if ( _mesc_udp.ucast[drop].sock == s )
            break;
    assert(drop < MESC_CFG_MAX_DROPS);

    if ( WITH_RESTART(length = recvfrom(s, temp, MAX_MESSAGE_LENGTH - 750,
        MSG_NOSIGNAL, (struct sockaddr *)&from, &from_length)) == -1 )
    {
        warn("Recvfrom(%d) failed: %s", drop, strerror(errno));
        mesc_udp_open(drop);
    }
    else
    {
        info("Recvfrom(%d) return %d bytes.", drop, length);
        buf_free(mesc_fsm_process_unicast(drop, buf_append(0, temp, length)));
    }

} /* _mesc_udp_ucast_read() */


/* ======================================================================= */
/**
 *  \brief Call back for fd_select.
 *  \param s The socket, which better match the multicast socket.
 *  \param unused Unused argument required by fd_add().
 *  \return Nothing.
 */
/* ======================================================================= */
static void _mesc_udp_mcast_read(int s, void *unused)
{
    char temp[MAX_MESSAGE_LENGTH];
    int length = 0;
    struct sockaddr_in from;
    socklen_t from_length = sizeof(struct sockaddr_in);

    /* Verify multicast socket */
    assert(_mesc_udp.mcast.sock == s);
    if ( WITH_RESTART(length = recvfrom(s, temp, MAX_MESSAGE_LENGTH - 750,
        MSG_NOSIGNAL, (struct sockaddr *)&from, &from_length)) == -1 )
    {
        warn("Recvfrom failed: %s", strerror(errno));
        return;
    }
    else
    {
        info("Recvfrom return %d bytes.", length);
        buf_free(mesc_fsm_process_multicast(buf_append(0, temp, length)));
    }

} /* _mesc_udp_mcast_read() */


/* ======================================================================= */
/**
 *  \brief Creates a socket, joins the multicast group and binds to the
 *      specified port for accepting multicast datagrams.  If Security is
 *      enabled try to join it, close it and send an IMP message to MSECP.
 *      Only add the Multicast fd to the read list if Security is disabled.
 *  \param addr The IP address/host name to be joined.
 *  \param port The port to bind to for listening for multicast UDP datagrams.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _multicast_join(const char *addr, const char *port)
{
    struct sockaddr_in sa;
    struct ip_mreq mreq;
    struct in_addr mcast_addr;
    struct hostent *hip;
    int sock;
    int optval = 1;
    u_int32_t port_num;
    struct ifreq ifrq;

    assert(addr && port);
    memset(&sa, 0, sizeof(sa));
    memset(&mreq, 0, sizeof(mreq));
    memset(&mcast_addr, 0, sizeof(mcast_addr));
    port_num = strtoul(port, NULL, 0);

    if ( (hip = gethostbyname(addr)) == NULL )
    {
        warn("Gethostbyname Failed: %s.", hstrerror(h_errno));
        return;
    }

    mcast_addr.s_addr = ((struct in_addr *)hip->h_addr_list[0])->s_addr;
    if ( !IN_MULTICAST(ntohl(mcast_addr.s_addr)) )
    {
        warn("Invalid multicast address: %s.", addr);
        return;
    }

    /* Create the socket and bind it to the port specified */
    assert((sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) != -1);
    assert(setsockopt(sock, SOL_SOCKET, SO_BSDCOMPAT, &optval, sizeof(int)) == 0);
    assert(fcntl(sock, F_SETFL, O_NONBLOCK) != -1);
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = htonl(INADDR_ANY);
    sa.sin_port = htons(port_num);
    assert(bind(sock, (struct sockaddr *)&sa, sizeof(struct sockaddr_in)) != -1);

    /* PAD needs this (PowerPC) but PC (Intel) hates it so don't assert(). */
    strcpy(ifrq.ifr_name, "eth0");
    ioctl(sock, SIOCGIFFLAGS, &ifrq);   /* assert(ioctl() != -1) */
    ifrq.ifr_flags |= IFF_ALLMULTI;
    ioctl(sock, SIOCSIFFLAGS, &ifrq);   /* assert(ioctl() != -1) */

    memcpy(&mreq.imr_multiaddr, &mcast_addr, sizeof(mcast_addr));
    mreq.imr_interface.s_addr = htonl(INADDR_ANY);
    if (setsockopt(sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq)) < 0)
    {
        warn("Joining Multicast Group %s Failed\n", addr);
        close(sock);
        return;
    }

    /* Handle security or not */
    if ( mesc_cfg_is_secure() )
    {
        /* All this crap above was just to test if we could join the multicast
         * group or not.  Since it looks like we can, close the socket then
         * send an IMP multicast-join to MSECP, who will join on our behalf. */
        info("Closing Multicast Group so MESC can join on our behalf.\n");
        assert(close(sock) != -1);
        mesc_imp_send_multicast_join_msg((char *)addr, (u_int16_t)port_num);
    }
    else    /* no security, so all this was not in vain. */
    {
        /* Cleanup the old descriptor, if any, then add ours. */
        if ( _mesc_udp.mcast.sock )
        {
            fd_remove(_mesc_udp.mcast.sock);
            assert(close(_mesc_udp.mcast.sock) != -1);
        }
        _mesc_udp.mcast.sock = sock;
        fd_add(_mesc_udp.mcast.sock, _mesc_udp_mcast_read, 0);
        snprintf(_mesc_udp.mcast.str, sizeof(_mesc_udp.mcast.str),
            "ESCP/UDP.multicast:%s:%s", addr, port);
        info("Joining Multicast Group %s successful.", addr);
    }

} /* _multicast_join() */


/*
 * End of $Id: mesc_udp.c,v 1.9 2005/04/22 17:01:49 cmayncvs Exp $
 */


