/*=======================================================================

This item is the property of GTECH Corporation, West Greenwich,
Rhode Island, and contains confidential and trade secret information.
It may not be transferred from the custody or control of GTECH except
as authorized in writing by an officer of GTECH.  Neither this item
nor the information it contains may be used, transferred, reproduced,
published, or disclosed, in whole or in part, and directly or
indirectly, except as expressly authorized by an officer of GTECH,
pursuant to written agreement.

Copyright (c) 2002 GTECH Corporation.  All rights reserved.

======================================================================= */

/**
 * \file crypto.c
 *
 * "$Id: crypto.c,v 1.2 2005/02/09 23:36:36 cmayncvs Exp $"
 *
 * \brief Implements the wrappers for algorithms used in ES Connect Security Protocol.
 */
#include <linux/limits.h>
#include <sys/fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <netdb.h>

//#include "gtypes.h"
//#include "utils.h"
#include "../libcom/buf.h"
#include "../libcom/mq.h"
#include "gassert.h"
//#include "escip.h"
#include "msecp_secure.h"
#include "msecp_dbg.h"

#define MAX_VARTEXT_LENGTH  64
#define G_DBG_LVL 3

//G_DBG_DEFDATA_EXTERN

extern struct secure *pSecure;

#define crypt_close(fd) do { if (fd != -1) close(fd); } while (0)

static int crypt_seed_prng(int bytes)
{
  if (RAND_load_file("/dev/urandom", bytes) < bytes)
    return 0;
  return 1;
}

/*
 * \brief This function generates digest of the files.
 * \param  spath pointer to path to files
 * \param  files pointer to list of file names
 * \param  dgst  pointer to digest data
 * \param  dlen  size of digest data
 * \return \li size of the dgst
 *         \li FAILURE  on error
 */
int crypt_digest(char *buf, int size, unsigned char **dgst, int *dlen)
{
  EVP_MD_CTX ctx;
  int ret = SUCCESS;

  EVP_DigestInit(&ctx, EVP_sha1());

  EVP_DigestUpdate(&ctx, buf, size);

  EVP_DigestFinal(&ctx, *dgst, dlen);

  return ret;
}

/*
 * \brief This function signs data.
 * \param  data pointer to data
 * \param  datalen  pointer to size of data returned
 * \param  pkey pointer to key
 * \param  sig  pointer to Signature
 * \param  siglen  size to Signature
 * \return \li SUCCESS on successful signing
 *         \li FAILURE  on error
 */
int crypt_sign(char *data, unsigned int data_sz, void *pkey, unsigned char **sig, unsigned int *siglen)
{
  int ret = FAILURE;
  int dlen = 0,ecode;
  unsigned char dgst[EVP_MAX_MD_SIZE], *pdgst = dgst;

  if (crypt_digest(data, data_sz, &pdgst, &dlen) < 0)
    return ret;

  if (RSA_sign(NID_sha1, dgst, dlen, *sig, siglen, pkey) != 1)
    {
      ecode = ERR_get_error();
  //    DBGPRT(DBG_ERR, "esec write public %s\n", ERR_error_string(ecode, NULL));
      return ret;
    }

  return SUCCESS;
}

/*
 * \brief This function verifies signature of the data.
 * \param  data pointer to data
 * \param  datalen  size of data
 * \param  pkey pointer to key
 * \param  sig  pointer to Signature
 * \param  siglen  size to Signature
 * \return \li SUCCESS on successful signing
 *         \li FAILURE  on error
 */
int crypt_verify(char *data, unsigned int dlen, void *pkey, unsigned char *sig, int siglen)
{
  EVP_MD_CTX ctx;
  int ret = FAILURE;

  do {
    EVP_VerifyInit(&ctx, EVP_sha1());

    EVP_VerifyUpdate(&ctx, data, dlen);

    if (!EVP_VerifyFinal(&ctx, sig, siglen, pkey))
      break;

    ret = SUCCESS;

  } while (0);

  return ret;
}

/*
 * \brief This function initializes the encryption context.
 * \param  alg pointer to encryption algorithm
 * \param  key pointer to DES key
 * \param  iv  pointer to Initialization Vector
 */
int crypt_sym_encrypt_init(struct secure *psec, int alg, unsigned char *key, unsigned char *iv)
{
  int ret = SUCCESS;
  switch (alg)
    {
    case ALG_DES:
      psec->sym_alg = (EVP_CIPHER *)EVP_des_cbc();
      break;

    case ALG_3DES:
      psec->sym_alg = (EVP_CIPHER *)EVP_des_ede3_cbc();
      break;

    case ALG_AES:
    default:
      ret = FAILURE;
      break;
    }
  psec->sym_key = key;
  psec->sym_iv  = iv;
  return ret;
}

/*
 * \brief This function initializes the decryption context.
 * \param  alg pointer to decryption algorithm
 * \param  key pointer to DES key
 * \param  iv  pointer to Initialization Vector
 */
int crypt_sym_decrypt_init(struct secure *psec, int alg, unsigned char *key, unsigned char *iv)
{
  int ret = SUCCESS;
  switch (alg)
    {
    case ALG_DES:
      psec->sym_alg = (EVP_CIPHER *)EVP_des_cbc();
      break;

    case ALG_3DES:
      psec->sym_alg = (EVP_CIPHER *)EVP_des_ede3_cbc();
      break;

    case ALG_AES:
    default:
      ret = FAILURE;
      break;
    }
  psec->sym_key = key;
  psec->sym_iv  = iv;
  return ret;
}

/*
 * \brief This function encrypts the buf,  returns encrypted buf and its length.
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_sym_encrypt(struct secure *psec, unsigned char *buf, int dlen, unsigned char *ebuf)
{
  EVP_CIPHER_CTX ctx;
  int elen = 0, enc_sz = 0, tlen = 0;

  do {
    EVP_CIPHER_CTX_init(&ctx);

    EVP_EncryptInit(&ctx, psec->sym_alg, psec->sym_key, psec->sym_iv);

    if (!EVP_EncryptUpdate(&ctx, ebuf, &elen, buf, dlen))
      break;

    if (!EVP_EncryptFinal(&ctx, ebuf + elen, &tlen))
      break;

    elen += tlen;

    EVP_CIPHER_CTX_cleanup(&ctx);

    enc_sz = elen;
  } while (0);
  return enc_sz;
}

/*
 * \brief This function decrypts the buf,  returns decrypted buf and its length.
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_sym_decrypt(struct secure *psec, unsigned char *ebuf, int elen, unsigned char *buf)
{
  EVP_CIPHER_CTX ctx;
  int dlen = 0, dec_sz = 0, tlen = 0;

  do {
    EVP_CIPHER_CTX_init(&ctx);

    EVP_DecryptInit(&ctx, psec->sym_alg, psec->sym_key, psec->sym_iv);

    if (!EVP_DecryptUpdate(&ctx, buf, &dlen, ebuf, elen))
      break;

    if (!EVP_DecryptFinal(&ctx, buf + dlen, &tlen))
      break;

    dlen += tlen;

    EVP_CIPHER_CTX_cleanup(&ctx);

    dec_sz = dlen;
  } while (0);

  return dec_sz;
}


static void crypt_sym_cleanup(struct secure *psec)
{
  psec->sym_alg = NULL;
  psec->sym_key = NULL;
  psec->sym_iv = NULL;
}


/*
 * \brief This function encrypts the buf,  returns encrypted buf and its length.
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_sess_encrypt(struct secure *psec, unsigned char *buf, int blen, unsigned char *ebuf)
{
  int elen = 0;

  if (crypt_sym_encrypt_init(psec, ALG_3DES, psec->ses_key, psec->ses_iv) < 0)
    return 0;

  elen = crypt_sym_encrypt(psec, buf, blen, ebuf);

  crypt_sym_cleanup(psec);

  return elen;
}

/*
 * \brief This function encrypts the buf,  returns encrypted buf and its length.
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_sess_decrypt(struct secure *psec, unsigned char *ebuf, int elen, unsigned char *buf)
{
  int len = 0;

  if (crypt_sym_decrypt_init(psec, ALG_3DES, psec->ses_key, psec->ses_iv) < 0)
    return 0;

  len = crypt_sym_decrypt(psec, ebuf, elen, buf);

  crypt_sym_cleanup(psec);

  return len;
}

/*
 * \brief This function encrypts the buf,  returns encrypted buf and its length.
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_unso_encrypt(struct secure *psec, unsigned char *buf, int len, unsigned char *ebuf)
{
  int elen = 0;

  if (crypt_sym_encrypt_init(psec, ALG_3DES, psec->bro_key, psec->bro_iv) < 0)
    return 0;

  elen = crypt_sym_encrypt(psec, buf, len, ebuf);

  crypt_sym_cleanup(psec);

  return elen;
}

/*
 * \brief This function decrypts the buf,  returns decrypted buf and its length.
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_unso_decrypt(struct secure *psec, unsigned char *ebuf, int elen, unsigned char *buf)
{
  int len = 0;

  if (crypt_sym_decrypt_init(psec, ALG_3DES, psec->bro_key, psec->bro_iv) < 0)
    return 0;

  len = crypt_sym_decrypt(psec, ebuf, elen, buf);

  crypt_sym_cleanup(psec);

  return len;
}


static int generate_secret_data(char *sdata, int size)
{
  struct secure *psec = pSecure;
  int cid_sz = strlen(psec->client_id);
  int sz = cid_sz > size ? size : cid_sz;

  memset(sdata, 0, size);

  memcpy(sdata, psec->client_id, sz);
  memcpy(sdata + sz, psec->hwid, 16);

  return sz + 16;
}

static void shred_secret_data(char *sdata, int size)
{
  struct secure *psec = pSecure;
  int cid_sz = strlen(psec->client_id);
  int sz = cid_sz > size ? size : cid_sz;

  memset(sdata, 0, sz + 16);
}

static int  pem_pp_cb(char *buf, int size, int rwflag, void *u)
{
  char sdata[MAX_VARTEXT_LENGTH];
  int ssiz;

  sdata[MAX_VARTEXT_LENGTH - 1] = 0;

  ssiz = generate_secret_data(sdata, sizeof(sdata));

  memcpy(buf, sdata, ssiz);
  return ssiz;
}

/*
 * \brief This function writes the private key to file.
 * \param  pkey pointer to key
 * \param  fd file descriptor
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_write_private(RSA *rsa, int fd)
{
  FILE *fp;
  int ret = FAILURE;
  int ecode, ssiz;
  char sdata[MAX_VARTEXT_LENGTH];

  info("crypt_write_private() 1");

  ssiz = generate_secret_data(sdata, sizeof(sdata));
  sdata[MAX_VARTEXT_LENGTH - 1] = 0;


 // DBGPRT(DBG_INFO4, "crypt_write_private : %s\n", sdata);
  info("crypt_write_private() 2");

  do {
    if ((fp = fdopen(fd, "wb")) == NULL)
      break;

    //    if (!PEM_write_RSAPrivateKey(fp, rsa, EVP_des_ede3_cbc(), sdata, ssiz, NULL,NULL))
    if (!PEM_write_RSAPrivateKey(fp, rsa, EVP_des_ede3_cbc(), NULL, 0, pem_pp_cb, NULL))
      {
        ecode = ERR_get_error();
        //     DBGPRT(DBG_ERR, "esec write private %s\n", ERR_error_string(ecode, NULL));
        fclose(fp);
        break;
      }
    fclose(fp);
    ret = SUCCESS;
  }
  while (0);
  info("crypt_write_private() 3");

  shred_secret_data(sdata, sizeof(sdata));

  system("saveconf \n");   //NEW_TM for ESPAD.

  return ret;
}

/*
 * \brief This function reads the private key from file.
 * \param  pkey pointer to key
 * \param  fd file descriptor
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_read_private(RSA **rsa, int fd)
{
  FILE *fp;
  int ret = FAILURE;
  int ecode;
  char sdata[MAX_VARTEXT_LENGTH];

  generate_secret_data(sdata, sizeof(sdata));
  sdata[MAX_VARTEXT_LENGTH - 1] = 0;

 // DBGPRT(DBG_INFO4, "crypt_read_private : %s\n", sdata);

  do {
    if ((fp = fdopen(fd, "rb")) == NULL)
      break;

    //    if ((*rsa = PEM_read_RSAPrivateKey(fp, NULL, NULL, sdata)) == NULL)
    if ((*rsa = PEM_read_RSAPrivateKey(fp, NULL, pem_pp_cb, NULL)) == NULL)
      {
        ecode = ERR_get_error();
        //     DBGPRT(DBG_ERR, "esec read private %s\n", ERR_error_string(ecode, NULL));
        fclose(fp);
        break;
      }

    fclose(fp);

    ret = SUCCESS;
  } while (0);

  shred_secret_data(sdata, sizeof(sdata));

  return ret;
}

/*
 * \brief This function writes the public key to file.
 * \param  pkey pointer to key
 * \param  fd file descriptor
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_write_public(RSA *rsa, int fd)
{
  FILE *fp;
  int ret = FAILURE;
  int ecode;

  do {
    if ((fp = fdopen(fd, "wb")) == NULL)
      break;

    if (!PEM_write_RSA_PUBKEY(fp, rsa))
      {
        ecode = ERR_get_error();
        //   DBGPRT(DBG_ERR, "esec write public %s\n", ERR_error_string(ecode, NULL));
        fclose(fp);
        break;
      }

    fclose(fp);
    ret = SUCCESS;
  } while (0);

  system("saveconf \n");     // NEW_TM For ESPAD.

  return ret;
}

/*
 * \brief This function reads the public key from file.
 * \param  pkey pointer to key
 * \param  fd file descriptor
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_read_public(RSA **rsa, int fd)
{
  FILE *fp;
  int ret = FAILURE;
  int ecode;

  do {
    if ((fp = fdopen(fd, "rb")) == NULL)
      break;

    if ((*rsa = PEM_read_RSA_PUBKEY(fp, NULL, NULL, NULL)) == NULL)
      {
        ecode = ERR_get_error();
        // DBGPRT(DBG_ERR, "esec read public %s\n", ERR_error_string(ecode, NULL));
        fclose(fp);
        break;
      }

    fclose(fp);
 info("read_public : ");
 print_hex_data((char *)rsa,64);

    ret = SUCCESS;
  }
  while (0);

  return ret;
}

/*
 * \brief This function encrypts the buf,  returns encrypted buf and its length.
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_asym_encrypt(RSA *rsa, unsigned char *buf, int blen, unsigned char *ebuf)
{
  int encSz, ecode;

  if ((encSz = RSA_public_encrypt(blen, buf, ebuf, rsa, RSA_PKCS1_PADDING)) < 0)
    {
      ecode =  ERR_get_error();
   //   DBGPRT(DBG_ERR, "esec write public %s\n", ERR_error_string(ecode, NULL));
      return -1;
    }

  return encSz;
}

/*
 * \brief This function decrypts the buf,  returns decrypted buf and its length.
 * \param  ebuf pointer to encrypted buf
 * \param  elen size of ebuf
 * \param  buf pointer to buf
 * \param  blen size of buf
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_asym_decrypt(RSA *rsa, unsigned char *ebuf, int elen, unsigned char *buf)
{
  int decSz, ecode;

  if ((decSz = RSA_private_decrypt(elen, ebuf, buf, rsa, RSA_PKCS1_PADDING)) < 0)
     {
       ecode = ERR_get_error();
     //  DBGPRT(DBG_ERR, "esec asym decrypt %s\n", ERR_error_string(ecode, NULL));
       return -1;
     }

  return decSz;
}

/*
 * \brief This function DES encodes the RSA public key.
 * \return \li buffer on success
 *         \li NULL  on error
 */
unsigned char *crypt_encode_akey(RSA *rsa, unsigned short *len)
{
  unsigned char *buf, *next;

  *len = i2d_RSA_PUBKEY(rsa, NULL);

  buf = next = (unsigned char *) malloc(*len);
  memset(buf, 0, *len);

  i2d_RSA_PUBKEY(rsa, &next);

  return buf;
}

/* Convert key data to RSA struct */
RSA *decode_pub_key(unsigned char **pubKey, long key_sz)
{
  RSA *rsa = NULL;
  int ecode;
  unsigned char *test;
  test = *pubKey;
 info("decode ");
 print_hex_data(test,64);

  do {
    if ((rsa = d2i_RSA_PUBKEY(NULL, pubKey, key_sz)) == NULL)
      {
       ecode = ERR_get_error();
     //   DBGPRT(DBG_ERR, "error decode public key %s\n", ERR_error_string(ecode, NULL));
      }
  } while (0);

  return rsa;
}

/*
 * \brief This function generates keys and writes them to files.
 * \param bits - number of bits for the key
 * \param spath - path to the private and public key files
 * \param  prvFile pointer to private key filename
 * \param  pubFile pointer to public key filename
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
static int crypt_asym_generate_key(int bits, int prfd, int pufd)
{
  RSA *rsa;
  int ret = FAILURE;

//  DBGPRT(DBG_INFO4, "Generate Keys\n");

  do {

    if (!crypt_seed_prng(128))
      break;

    if ((rsa = RSA_generate_key(bits, 17, NULL, NULL)) == NULL)
      break;

    if (RSA_check_key(rsa) != 1)
      break;

    if (crypt_write_public(rsa, pufd) == FAILURE)
      break;

    if (crypt_write_private(rsa, prfd) == FAILURE)
      break;

    ret = SUCCESS;

  } while (0);

  info("asym_generate_key");
  return ret;
}

/*
 * \brief This function loads the asymmetric keys from persistent storage.
 * \return \li SUCCESS on verification success
 *         \li FAILURE  on error
 */
int crypt_read_keys(struct secure  *esec)
{
  int fd;
  char fname[512];
  int ret = FAILURE;

  fd = -1;

  do {
    /* Read Terminal public key */
    snprintf(fname, sizeof(fname), "%s/cpub.dat", esec->kpath);
    if ((fd = open(fname, O_RDONLY)) < 0)
      break;
    if (crypt_read_public(&esec->t_pubkey, fd) == FAILURE)
      break;
    info("read_keys:");
    print_hex_data((char *) &esec->t_pubkey,64);

    crypt_close(fd);
    esec->valid = ESEC_TERM_PUB_KEY;
    info("read_keys() 1");

    /* Read Terminal private key */
    snprintf(fname, sizeof(fname), "%s/cprv.dat", esec->kpath);
    if ((fd = open(fname, O_RDONLY)) < 0)
      break;

    info("read_keys() 2");
    if (crypt_read_private(&esec->t_prvkey, fd) == FAILURE)
      break;
    crypt_close(fd);
    esec->valid |= ESEC_TERM_PRV_KEY;
    info("read_keys() 3");
    ret = SUCCESS;
  } while (0);

  crypt_close(fd);

  return ret;
}

/*
 * \brief This function generates the asymmetric keys from persistent storage.
 * \return \li SUCCESS on key generation
 *         \li FAILURE  on error
 */
int crypt_generate_keys(struct secure *psec)
{
  int prfd, pufd;
  char fname[512];
  int ret = FAILURE;

  info("crypt_generate_keys() 1");
  do {
    RSA_free(psec->t_pubkey);
    RSA_free(psec->t_prvkey);

    psec->t_pubkey = NULL;
    psec->t_prvkey = NULL;

    crypt_seed_prng(128);

    snprintf(fname, sizeof(fname), "%s/cpub.dat", psec->kpath);
    if ((pufd = open(fname, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU)) < 0)
      break;

    snprintf(fname, sizeof(fname), "%s/cprv.dat", psec->kpath);
    if ((prfd = open(fname, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU)) < 0)
      break;

   info("crypt_generate_keys() 2 ");

    if (crypt_asym_generate_key(2048, prfd, pufd) == FAILURE)
      break;

    crypt_close(pufd);
    crypt_close(prfd);

    pufd = prfd = -1;

    if (crypt_read_keys(psec) == FAILURE) /* assume keys are loaded to memory here */
      break;
    ret = SUCCESS;
  } while(0);
  info("crypt_generate_keys() 3");

  return ret;
}

/*
 * \brief This function saves  the server's public key to persistent storage.
 * \return \li SUCCESS on successful storage
 *         \li FAILURE  on error
 */
int crypt_write_server_pub_key(struct secure *psec)
{
  int pufd;
  char fname[512];
  int ret = FAILURE;

  snprintf(fname, sizeof(fname), "%s/spub.dat", psec->kpath);
  if ((pufd = open(fname, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU)) < 0)
    return ret;

  if (crypt_write_public(psec->s_pubkey, pufd) == FAILURE)
    {
      close (pufd);
      return ret;
    }
  close (pufd);

  return SUCCESS;
}

/*
 * \brief This function reads  the server's public key from persistent storage.
 * \return \li SUCCESS on successful storage
 *         \li FAILURE  on error
 */
int crypt_read_server_pub_key(struct secure *psec)
{
  int fd;
  char fname[512];
  int ret = FAILURE;

  snprintf(fname, sizeof(fname), "%s/spub.dat", psec->kpath);
  if ((fd = open(fname, O_RDONLY)) < 0)
    return ret;

  if (crypt_read_public(&psec->s_pubkey, fd) == FAILURE)
    {
      psec->s_pubkey = NULL;
      close (fd);
      return ret;
    }

  close (fd);
  return SUCCESS;
}
