/* ===[ $RCSfile: msecp.c,v $ ]============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: msecp.c,v 1.2 2004/08/06 17:34:13 tmeiccvs Exp $"
 *
 *  \brief Multi-Drop Security (MSECP) module.
 *
 *  This code handles multi-drop ESC Security functions for an ES-PAD device.
 *  The current device is the Cyclades TS100.  It is based on
 *  ESConnect Security v1.1.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include "libcom/buf.h"
#include "libcom/fd.h"
#include "libcom/mq.h"
#include "gassert.h"

#include "msecp_cfg.h"
#include "msecp_dbg.h"
#include "msecp_secure.h"
#include "msecp_tmr.h"
#include "msecp_art.h"
#include "msecp_udp.h"
#include "msecp_imp.h"
//#include "msecp_stat.h"
#include "msecp_fsm.h"

/* =============== */
/* Local variables */
/* =============== */
/** MSECP version - TODO Update as required.  Based on the document:
                    "ES Connect Secure Protol (SECP) Protocol Definition"  */
static char _msecp_version[] = "SECP Rev 1.0, July 20, 2004";


/* ========================= */
/* Local function prototypes */
/* ========================= */
static void _msecp_init(const char *local_q, const char *upper_q, int level);
static void _msecp_delay(void);
static void _msecp(void);

/* ======================================================================= */
/**
 *  \brief MSECP entry point.
 *  \param argc Should be 3.  More than 3 are ignored.
 *  \param argv - argv[1]=local queue, argv[2]=upper queue,
 *      optional argv[3] holds debug level.
 *
 *  \return Never returns (intentionally at least).
 */
/* ======================================================================= */
int main(int argc, char *argv[])
{
    int level = 1; //-1;

    info("msecp (%d) :  Based on: %s : Started.\n", getpid(), _msecp_version);

    /* Check usage */
    if ( argc < 3 )
    {
        warn("\nUsage: msecp <local_queue> <upper_queue> [-d dbglvl]\n");
        exit(0);
    }

    assert(argv && argv[1] && argv[2]);

    /* Check for command-line override of debug level */
    if ( argc > 3 )
    {
        assert(argv[3]);
        if ( argc == 5 )       /* optional -d dbglvl */
        {
            if ( !strncmp(argv[4], "-d", 2) )
                level = strtol(argv[5], NULL, 0) & 0x07;
            else
            {
                warn("\nUsage: msecp <local_queue> <upper_queue> [-d dbglvl]\n");
                exit(0);
            }
        }
        else                        /* invalid */
        {
            warn("\nUsage: msecp <local_queue> <upper_queue> [-d dbglvl]\n");
            exit(0);
        }
    }

    /* Initialize everything */
    _msecp_init(argv[1], argv[2], level);

    /* Start MSECP - should never return */
    _msecp();
    return (0);

} /* main() */

                        /* =============== */
                        /* Local functions */
                        /* =============== */

/* ======================================================================= */
/**
 *  \brief Initializes everything MSECP needs before it starts running,
 *      including config data, queue's, art, and state machine.
 *  \param local_q msecp queue name.
 *  \param upper_q upper queue name.
 *  \param level An optional forced debug level to set msecp to.  This debug
 *      level is expected to be one from gdebug.h, but it will be mapped to
 *      one of the levels in debug.h.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_init(const char *local_q, const char *upper_q, int level)
{

    msecp_cfg_init();
    msecp_dbg_init(level);
    msecp_secure_init();
    msecp_tmr_init();
    msecp_art_init();
// TODO If statistics required:  msecp_stat_init();
    info("call msecp_imp_init() ");
    msecp_imp_init(local_q, upper_q);
    msecp_udp_init();
    msecp_fsm_init();

    info("msecp_init(): Done.");
} /* _msecp_init() */


/* ======================================================================= */
/**
 *  \brief Waits a random amount of time to avoid terminal sychronization.
 *      (See "man rand" for why it's implemented the way it is.)
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_delay(void)
{
    int delay;
    struct timeval tv;
    struct timezone tz;

    /* Randomly seed the random number generator;
     * otherwise what's the point!? */
    assert(gettimeofday(&tv, &tz) == 0);
    srand((unsigned int)tv.tv_usec);

    /*
     * Wait some random amount of time (between 0-5000 msec's) before starting
     * so that if all es-pads are reset at the same time, they won't be
     * synchronized.
     */
    delay = (int)(5001.0 * rand() / (RAND_MAX + 1.0));
    info("Sleeping for %d.%d seconds ...", delay / 1000, delay % 1000);
    usleep(1000 * delay);
    info("... Ready!\n");

} /* _msecp_delay() */


/* ======================================================================= */
/**
 *  \brief MSECP main function.
 *  \return Nothing (never returns).
 */
/* ======================================================================= */
static void _msecp(void)
{
    _msecp_delay();
    while ( 1 )
        fd_select();

} /* _msecp() */

/*
 * End of "$Id: msecp.c,v 1.2 2004/08/06 17:34:13 tmeiccvs Exp $".
 */


