/* ===[ $RCSfile: msecp_cfg.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: msecp_cfg.c,v 1.1 2004/07/12 18:03:47 tmeiccvs Exp $"
 *
 *  \brief MSECP Run-Time Configuration routines.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include "libcom/buf.h"
#include "libcom/list.h"
#include "libcom/str.h"
#include "libcom/c_config.h"
#include "msecp_dbg.h"
#include "cfg/msecp_cfg.h"
#include "msecp_cfg.h"
//#include "escp.h"
#include "gassert.h"

/* ======= */
/* Defines */
/* ======= */

/* =============== */
/* Local variables */
/* =============== */
/** MSECP configuration parameter data */
static msecp_cfg_t     _msecp_cfg;
static msecp_cfg_misc  _msecp_cfg_misc;


static void _msecp_cfg_set_ip_addr_and_hw_id_prefix(void);

/* ======================================================================= */
/**
 *  \brief Initializes the MSECP run-time configuration parameters.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_cfg_init(void)
{
    /* Read the msecp parameters */
    c_config_read(msecp_cfg_record, &_msecp_cfg, &msecp_cfg_defaults,
        sizeof(msecp_cfg_t));

    /* Register our config processing functions */
   // _mesc_cfg_misc.proc_list = 0;
  //  _mesc_cfg_proc("T0", _t0);
   // _mesc_cfg_proc("T1", _t1);
   // _mesc_cfg_proc("TC", _tc);
      _msecp_cfg_set_ip_addr_and_hw_id_prefix();

    info("msecp cfg_init(): Done.");

} /* msecp_cfg_init() */


/* ======================================================================= */
/**
 *  \brief Get the debug level.
 *  \return The debug level.
 */
/* ======================================================================= */
u_int8_t msecp_cfg_get_dbg_level(void)
{
    return (_msecp_cfg.dbg_level);
} /* msecp_cfg_get_dbg_level() */


/* ======================================================================= */
/**
 *  \brief Get the primary host address.
 *  \return The primary host address.
 */
/* ======================================================================= */
char *msecp_cfg_get_host1(void)
{
    return (_msecp_cfg.host1);
} /* msecp_cfg_get_host1() */


/* ======================================================================= */
/**
 *  \brief Get the secondary host address.
 *  \return The secondary host address.
 */
/* ======================================================================= */
char *msecp_cfg_get_host2(void)
{
    return (_msecp_cfg.host2);
} /* msecp_cfg_get_host2() */


/* ======================================================================= */
/**
 *  \brief Get the UDP Base Port.
 *  \return The UDP Base Port.
 */
/* ======================================================================= */
u_int16_t msecp_cfg_get_udpbase_port(void)
{
    return (_msecp_cfg.udp_base_port);
} /* msecp_cfg_get_udpbase_port() */



/* ======================================================================= */
/**
 *  \brief Get the Session Test Flag.
 *  \return The Session Test Flag.
 */
/* ======================================================================= */
u_int8_t msecp_cfg_get_session_test(void)
{
    return (_msecp_cfg.session_test);
} /* msecp_cfg_get_session_test() */


/* ======================================================================= */
/**
 *  \brief Get the timeout count to rotate hosts.
 *  \return The timeout count to rotate hosts.
 */
/* ======================================================================= */
u_int32_t msecp_cfg_get_tmo_cnt(void)
{
    return (_msecp_cfg.tmo_cnt);
} /* msecp_cfg_get_tmo_cnt() */


/* ======================================================================= */
/**
 *  \brief Get the minimum response timeout value.
 *  \return The minimum response timeout value, in milliseconds.
 */
/* ======================================================================= */
u_int32_t msecp_cfg_get_min_rsp_tmo(void)
{
    return (1000 * _msecp_cfg.min_rsp_tmo);
} /* msecp_cfg_get_min_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Get the maximum response timeout value.
 *  \return The maximum response timeout value, in milliseconds.
 */
/* ======================================================================= */
u_int32_t msecp_cfg_get_max_rsp_tmo(void)
{
    return (1000 * _msecp_cfg.max_rsp_tmo);
} /* secp_cfg_get_max_rsp_tmo() */


/* ======================================================================= */
/**
 *  \brief Get the port to use.
 *  \param drop The drop address requesting its port to use.
 *  \return The port.
 */
/* ======================================================================= */
u_int16_t msecp_cfg_get_port(u_int8_t drop)
{
    assert(drop < MSECP_CFG_MAX_DROPS);

    return ( (msecp_cfg_get_udpbase_port()) - drop);
} /* msecp_cfg_get_port() */


/* ======================================================================= */
/**
 *  \brief Sets the IP address and hardware ID prefix as the ethernet address
 *      of the pad.  Later, the drop address will be appended as the virtual
 *      hardware id of the specific drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_cfg_set_ip_addr_and_hw_id_prefix(void)
{
    int s;
    struct ifreq ifr;
    struct sockaddr_in *addr;
    u_int8_t *hwaddr;

    /* Create a dummy socket. */
    assert((s = socket(AF_INET, SOCK_DGRAM, 0)) != -1);
    strcpy(ifr.ifr_name, "eth0");
    ifr.ifr_addr.sa_family = AF_INET;

    /* Get the IP address */
    while ( ioctl(s, SIOCGIFADDR, &ifr) == -1 )
        sleep(1);
    addr = (struct sockaddr_in *)&ifr.ifr_addr;
    snprintf(_msecp_cfg_misc.ip_addr, sizeof(_msecp_cfg_misc.ip_addr),
        "%s", inet_ntoa(addr->sin_addr));
    _msecp_cfg_misc.ip_addr[sizeof(_msecp_cfg_misc.ip_addr) - 1] = '\0';/*2bsure*/
    info("IP address is %s", _msecp_cfg_misc.ip_addr);

    /* Get the ethernet address for the interface & save as our hw_id prefix */
    assert(ioctl(s, SIOCGIFHWADDR, &ifr) != -1);
    hwaddr = (u_int8_t *)&ifr.ifr_addr;
    snprintf(_msecp_cfg_misc.hw_id_prefix, 19, "%02X:%02X:%02X:%02X:%02X:%02X:",
        *(hwaddr + 2), *(hwaddr + 3), *(hwaddr + 4),
        *(hwaddr + 5), *(hwaddr + 6), *(hwaddr + 7));
    _msecp_cfg_misc.hw_id_prefix[18] = '\0';     /* just to be sure */
    assert(close(s) != -1);
    info("Hardware ID prefix set to %s", _msecp_cfg_misc.hw_id_prefix);

} /* _msecp_cfg_set_ip_addr_and_hw_id_prefix() */


/* ======================================================================= */
/**
 *  \brief Get the hw id.
 *  \return The HW ID.
 */
/* ======================================================================= */
char *msecp_cfg_get_hwid(void)
{
  return (_msecp_cfg_misc.hw_id_prefix);
} /* msecp_cfg_get_hwid() */


/* ======================================================================= */
/**
 *  \brief Get the client prefix.
 *  \return The Client prefix.
 */
/* ======================================================================= */
char *msecp_cfg_get_client_id(void)
{
  return (_msecp_cfg.client_id);
} /* msecp_cfg_get_client_id() */


/*
 * End of "$Id: msecp_cfg.c,v 1.1 2004/07/12 18:03:47 tmeiccvs Exp $".
 */

