/* ===[ $RCSfile: msecp_dbg.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: msecp_dbg.c,v 1.1 2004/07/12 18:03:47 tmeiccvs Exp $"
 *
 *  \brief Implements the debug functions.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include "libcom/buf.h"
#include "libcom/c_config.h"
#include "gdebug.h"
#include "gfault.h"
#include "msecp_dbg.h"
#include "msecp_cfg.h"

/* =============== */
/* Local variables */
/* =============== */
/** Debug level. */
u_int8_t msecp_dbg_level = PRINT_LEVEL;

static u_int8_t _msecp_dbg_fw2comm(u_int8_t level);
static char *_msecp_dbg_level2str(u_int8_t level);

/* ======================================================================= */
/**
 *  \brief Initialize debug.
 *  \param level If >=0, the forced debug level.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_dbg_init(int level)
{
    /* Set the debug level, either a forced level, or our configured one. */
    if ( level >= 0 )
        msecp_dbg_set_level((u_int8_t)level);
    else
        msecp_dbg_set_level(msecp_cfg_get_dbg_level());
    info("dbg_init(): Done.");

} /* msecp_dbg_init() */


/* ======================================================================= */
/**
 *  \brief Set debug level in this module and the underlying shared libraries.
 *  \param level Debug level.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_dbg_set_level(u_int8_t level)
{
    msecp_dbg_level = _msecp_dbg_fw2comm(level);
    warn("Debug level changed to %s (%X).",
        _msecp_dbg_level2str(msecp_dbg_level), msecp_dbg_level);

    f_fault_set_debug_level(level);
    c_config_set_debug_level(level);

} /* msecp_dbg_set_level() */


/* ======================================================================= */
/**
 *  \brief Get MSECP debug level.
 *  \return The debug level (see debug.h for comm values).
 */
/* ======================================================================= */
u_int8_t msecp_dbg_get_level(void)
{
    return (msecp_dbg_level);
} /* msecp_dbg_get_level() */


/* ======================================================================= */
/**
 *  \brief Get current time.
 *  \return Time in milliseconds.
 */
/* ======================================================================= */
u_int32_t msecp_dbg_time(void)
{
    struct timeval t;

    gettimeofday (&t, 0);
    return ((u_int32_t)((t.tv_sec * 1000) + (t.tv_usec / 1000)));
} /* msecp_dbg_time() */


/* ======================================================================= */
/**
 *  \brief Print current time.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_dbg_time_stamp(void)
{
    info("Time is %u.", msecp_dbg_time());
} /* msecp_dbg_time_stamp() */


/* ======================================================================= */
/**
 *  \brief Print a buffer.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_dbg_print_buf(buf_t b)
{
    if ( msecp_dbg_level >= PRINT_LEVEL )
        buf_print(b);

} /* msecp_dbg_print_buf() */

                        /* --------------- */
                        /* Local functions */
                        /* --------------- */

/* ======================================================================= */
/**
 *  \brief Convert firmware debug level to comm debug level.
 *  \param level The firmware debug level (see debug.h for firmware values).
 *  \return The comm debug level (see debug.h for comm values).
 */
/* ======================================================================= */
static u_int8_t _msecp_dbg_fw2comm(u_int8_t level)
{
    static u_int8_t _fw2comm[8] = {
        [DBG_NONE] NONE_LEVEL,
        [DBG_ALL] PRINT_LEVEL,
        [DBG_INFO4] PRINT_LEVEL,
        [DBG_INFO3] TRACE_LEVEL,
        [DBG_INFO2] TRACE_LEVEL,
        [DBG_INFO1] INFO_LEVEL,
        [DBG_WARN] WARN_LEVEL,
        [DBG_ERR] WARN_LEVEL
    };
    return (_fw2comm[level & 0x07]);

} /* _msecp_dbg_fw2comm() */


/* ======================================================================= */
/**
 *  \brief Returns a human-readable text string of the debug level.
 *  \return The string.
 */
/* ======================================================================= */
static char *_msecp_dbg_level2str(u_int8_t level)
{
    switch ( level )
    {
        case NONE_LEVEL:    return ("NONE_LEVEL");
        case WARN_LEVEL:    return ("WARN_LEVEL");
        case INFO_LEVEL:    return ("INFO_LEVEL");
        case TRACE_LEVEL:   return ("TRACE_LEVEL");
        case PRINT_LEVEL:   return ("PRINT_LEVEL");
        default:            return ("UNKNOWN LEVEL");
    }
} /* _msecp_dbg_level2str() */


/*
 * End of $Id: msecp_dbg.c,v 1.1 2004/07/12 18:03:47 tmeiccvs Exp $
 */


