/* ===[ $RCSfile: msecp_fsm.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: msecp_fsm.c,v 1.4 2005/02/09 23:36:37 cmayncvs Exp $"
 *
 *  \brief Multi-drop Secure ES Connect Finite State Machine (MSECP FSM).
 *      See the msecp_summary.txt file for details of the state machine.
 *
 *    TODO:  Handle the Insecure Channel Command.
 *           Signature Block appending/verifying.
 *           Multicast handling.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <netinet/in.h>
#include "libcom/buf.h"
#include "gassert.h"
#include "msecp_dbg.h"
#include "msecp_cfg.h"

//#include "mesc_stat.h"  TODO
#include "msecp_art.h"
#include "msecp_imp.h"
#include "msecp_tmr.h"
#include "msecp_fsm.h"
#include "msecp_udp.h"
//#include "msecp.h"
#include "msecp_cfg.h"
#include "msecp_secure.h"
#include "crypto.h"

#include <malloc.h>
#include <sys/mman.h>

  /** \brief Typedef for Multi-drop ESCP finite state machine per drop data. */
typedef struct {
    int state;              /**< INIT, CONFIG, READY.. */
    int cfg_rqst_state;     /**< Tracks the Security Config Request state */
    int session_valid;      /**< Boolean flag session valid */
    buf_q tx_q;             /**< Queue of requests */
    u_int8_t  sec_details;  /**< The Security Packet Details */
    u_int16_t nat_port;     /**< Host's idea of local port (NAT port) */
    u_int8_t seq_no;        /**< Host/client sequence number */
    u_int32_t tmo_cnt;      /**< The timeout counter value ("tc" of 1.7.4) */
    u_int32_t cur_tmo_cnt;  /**< The current timeout counter value */
    char client_name[32];   /**< The client name for each drop. */
} _msecp_fsm_t;

/* ======= */
/* Defines */
/* ======= */
/* MSECP defines */
struct secure *psec;

/** Holds all the MSECP finite state machine data. */
static _msecp_fsm_t _msecp_fsm[MSECP_CFG_MAX_DROPS];


/* =========== SECP Request functions =================== */
void msecp_fsm_process_app_rqst(u_int8_t drop, char *client_id, char *data, int len);

static void _send_secp_app_rqst(u_int8_t drop);
static void _send_secp_rqst(u_int8_t drop, buf_t rqst, u_int8_t msg_type);

static void _send_secp_config_rqst(u_int8_t drop);
static void _send_secp_sigtest_rqst(u_int8_t drop);
static void _send_secp_sess_rqst(u_int8_t drop);
static void _send_secp_encryptest_rqst(u_int8_t drop);
static void _send_secp_secure_failure_rqst(u_int8_t drop, u_int8_t error_code);

static buf_t _make_secp_config_rqst(u_int8_t drop);
static buf_t _make_secp_sigtest_rqst(u_int8_t drop);
static buf_t _make_secp_sess_rqst(u_int8_t drop);
static buf_t _make_secp_encryptest_rqst(u_int8_t drop);
static buf_t _session_encrypt_msg(u_int8_t drop, buf_t b);


/* =========== SECP Response functions =================== */
static int  _process_msg_body(u_int8_t drop, struct secure *psec, unsigned char *msg, int msg_sz);
static int  _parse_msg_header(u_int8_t drop, buf_t b);

static void _dispatch_app_rqst(u_int8_t drop, buf_t b);
static void _dispatch_app_resp(u_int8_t drop, u_int8_t *data, int length);
static void _dispatch_app_unso(u_int8_t drop, u_int8_t *data, int length);
static void _dispatch_proto_msg(u_int8_t drop, u_int8_t *data, int length);
static void _dispatch_tmo(u_int8_t drop, int input);

/* states: _MSECP_FSM_STATE */
#define _INIT                           0       /**< INIT state */
#define _CONFIG                         1       /**< Secure Config state */
#define _SESSION                        2       /**< Session state */
#define _READY                          3       /**< Ready state */
#define _ENCTEST                        4       /**< Encryption state */
#define _SIGTEST                        5       /**< SIG Test state */
#define _NUM_STATES                     6       /**< The number of states */

/* Inputs */
#define _RESP_TMO                       0       /**< response timeout */
#define _NUM_TMOS                       1       /**< The # of timers */

/* SECP PROTO RESP Inputs  */
#define _CFG_RESP                       0       /**< Secure Config */
#define _SESS_RESP                      1       /**< Session */
#define _TEST_OK_RESP                   2       /**< Sign/Encry Test */
#define _GOTO_CFG_RQST                  3       /**< Secure Config */
#define _GOTO_SESS_RQST                 4       /**< Session */
#define _NUM_PROTO_INPUTS               5       /**< The # of proto inputs */

/* SECP UNSO PROTO Inputs  */
#define _CFG_RQST                       0       /**< Secure Config */
#define _SESS_RQST                      1       /**< Session */
#define _ENCTEST_RQST                   2       /**< Encrypt Test */
#define _SIGTEST_RQST                   3       /**< Signature Test */
#define _INSECURE_CMD                   4       /**< Insecure Command */
#define _NUM_UNSO_PROTO_INPUTS          5       /**< The # of proto inputs */

/* SECP Config Request states  */
#define  CFG_INVALID                    0
#define  FIRST_CFG_RQST                 1
#define  QUEUED_CFG_RQST                2
#define  CFG_VALID                      3

/* SECP Message Types          */
#define  SECP_APP_MSG                   0
#define  SECP_PROTO_MSG                 1


/* These directly appear in the vector tables. */
/* Shared vectors */
static void _timer_do_nothing(u_int8_t drop);
static void _app_unso_do_nothing(u_int8_t drop, u_int8_t *data, int length);
static void _proto_unso_do_nothing(u_int8_t drop);
static void _resp_do_nothing(u_int8_t drop, u_int8_t *data, int length);

static void _queue_rqst(u_int8_t drop, buf_t b);
static void _app_unso(u_int8_t drop, u_int8_t *data, int length);
static void _config_rqst(u_int8_t drop);
static void _sess_rqst(u_int8_t drop);
static void _enctest_rqst(u_int8_t drop);
static void _sigtest_rqst(u_int8_t drop);
static void _insecure_cmd(u_int8_t drop);

/* INIT */
static void _init_rqst(u_int8_t drop, buf_t b);
/*CONFIG */
static void _config_resp(u_int8_t drop, u_int8_t *data, int length);
static void _config_resp_tmo(u_int8_t drop);
/*SESSION */
static void _sess_resp(u_int8_t drop, u_int8_t *data, int length);
static void _sess_resp_tmo(u_int8_t drop);
/*READY*/
static void _ready_rqst(u_int8_t drop, buf_t b);
static void _app_resp(u_int8_t drop, u_int8_t *data, int length);
/*ENCTEST*/
static void _enctest_resp(u_int8_t drop, u_int8_t *data, int length);
static void _enctest_resp_tmo(u_int8_t drop);
/*SIGTEST*/
static void _sigtest_resp(u_int8_t drop, u_int8_t *data, int length);
static void _sigtest_resp_tmo(u_int8_t drop);

static void _set_state(u_int8_t drop, int state);
static char *_state2str(int state);
static void _set_cfg_rqst_state(u_int8_t drop);

    /** Array of request vectors. */
static void (*_msecp_fsm_app_rqst[_NUM_STATES])(u_int8_t, buf_t);
/** Array of timeout vectors. */
static void (*_msecp_fsm_proto_tmo[_NUM_STATES][_NUM_TMOS])(u_int8_t);
/** Array of application unso vectors */
static void (*_msecp_fsm_app_unso[_NUM_STATES])(u_int8_t, u_int8_t *, int);
/** Array of protocol Unso vectors */
static void (*_msecp_fsm_proto_unso[_NUM_STATES][_NUM_UNSO_PROTO_INPUTS])(u_int8_t);
/** Array of application response vectors */
static void (*_msecp_fsm_app_resp[_NUM_STATES])(u_int8_t, u_int8_t *, int);
/** Array of protocol response vectors */
static void (*_msecp_fsm_proto_resp[_NUM_STATES][_NUM_PROTO_INPUTS])(u_int8_t, u_int8_t *, int);



/* ======================================================================= */
/**
 *  \brief Initializes the MSECP finite state machine vector table.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _init_vectors(void)
{
  /************ INIT ***************/
    _msecp_fsm_app_rqst  [_INIT]                      = _init_rqst;
    _msecp_fsm_proto_tmo [_INIT][_RESP_TMO]           = _timer_do_nothing;
    _msecp_fsm_app_unso  [_INIT]                      = _app_unso_do_nothing;
    _msecp_fsm_proto_unso[_INIT][_CFG_RQST]           = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_INIT][_SESS_RQST]          = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_INIT][_ENCTEST_RQST]       = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_INIT][_SIGTEST_RQST]       = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_INIT][_INSECURE_CMD]       = _proto_unso_do_nothing;

    _msecp_fsm_app_resp  [_INIT]                      = _resp_do_nothing;
    _msecp_fsm_proto_resp[_INIT][_CFG_RESP]           = _resp_do_nothing;
    _msecp_fsm_proto_resp[_INIT][_SESS_RESP]          = _resp_do_nothing;
    _msecp_fsm_proto_resp[_INIT][_TEST_OK_RESP]       = _resp_do_nothing;

    /************ CONFIG *************/
    _msecp_fsm_app_rqst  [_CONFIG]                    = _queue_rqst;
    _msecp_fsm_proto_tmo [_CONFIG][_RESP_TMO]         = _config_resp_tmo;
    _msecp_fsm_app_unso  [_CONFIG]                    = _app_unso_do_nothing;
    _msecp_fsm_proto_unso[_CONFIG][_CFG_RQST]         = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_CONFIG][_SESS_RQST]        = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_CONFIG][_ENCTEST_RQST]     = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_CONFIG][_SIGTEST_RQST]     = _proto_unso_do_nothing;
    _msecp_fsm_proto_unso[_CONFIG][_INSECURE_CMD]     = _proto_unso_do_nothing;

    _msecp_fsm_app_resp  [_CONFIG]                    = _resp_do_nothing;
    _msecp_fsm_proto_resp[_CONFIG][_CFG_RESP]         = _config_resp;
    _msecp_fsm_proto_resp[_CONFIG][_SESS_RESP]        = _resp_do_nothing;
    _msecp_fsm_proto_resp[_CONFIG][_TEST_OK_RESP]     = _resp_do_nothing;
    _msecp_fsm_proto_resp[_CONFIG][_GOTO_CFG_RQST]    = _config_rqst;

    /************ SESSION ****************/
    _msecp_fsm_app_rqst  [_SESSION]                   = _queue_rqst;
    _msecp_fsm_proto_tmo [_SESSION][_RESP_TMO]        = _sess_resp_tmo;
    _msecp_fsm_app_unso  [_SESSION]                   = _app_unso_do_nothing;
    _msecp_fsm_proto_unso[_SESSION][_CFG_RQST]        = _config_rqst;
    _msecp_fsm_proto_unso[_SESSION][_SESS_RQST]       = _sess_rqst;
    _msecp_fsm_proto_unso[_SESSION][_ENCTEST_RQST]    = _enctest_rqst;
    _msecp_fsm_proto_unso[_SESSION][_SIGTEST_RQST]    = _sigtest_rqst;
    _msecp_fsm_proto_unso[_SESSION][_INSECURE_CMD]    = _insecure_cmd;

    _msecp_fsm_app_resp  [_SESSION]                   = _resp_do_nothing;
    _msecp_fsm_proto_resp[_SESSION][_CFG_RESP]        = _resp_do_nothing;
    _msecp_fsm_proto_resp[_SESSION][_SESS_RESP]       = _sess_resp;
    _msecp_fsm_proto_resp[_SESSION][_TEST_OK_RESP]    = _resp_do_nothing;
    _msecp_fsm_proto_resp[_SESSION][_GOTO_SESS_RQST]  = _sess_rqst;
    _msecp_fsm_proto_resp[_SESSION][_GOTO_CFG_RQST]   = _config_rqst;


    /************** READY *******************/
    _msecp_fsm_app_rqst  [_READY]                     = _ready_rqst;
    _msecp_fsm_proto_tmo [_READY][_RESP_TMO]          = _timer_do_nothing;
    _msecp_fsm_app_unso  [_READY]                     = _app_unso;
    _msecp_fsm_proto_unso[_READY][_CFG_RQST]          = _config_rqst;
    _msecp_fsm_proto_unso[_READY][_SESS_RQST]         = _sess_rqst;
    _msecp_fsm_proto_unso[_READY][_ENCTEST_RQST]      = _enctest_rqst;
    _msecp_fsm_proto_unso[_READY][_SIGTEST_RQST]      = _sigtest_rqst;
    _msecp_fsm_proto_unso[_READY][_INSECURE_CMD]      = _insecure_cmd;

    _msecp_fsm_app_resp  [_READY]                     = _app_resp;
    _msecp_fsm_proto_resp[_READY][_CFG_RESP]          = _resp_do_nothing;
    _msecp_fsm_proto_resp[_READY][_SESS_RESP]         = _resp_do_nothing;
    _msecp_fsm_proto_resp[_READY][_TEST_OK_RESP]      = _resp_do_nothing;

    /**************** ENCTEST ***************/
    _msecp_fsm_app_rqst  [_ENCTEST]                   = _queue_rqst;
    _msecp_fsm_proto_tmo [_ENCTEST][_RESP_TMO]        = _enctest_resp_tmo;
    _msecp_fsm_app_unso  [_ENCTEST]                   = _app_unso;
    _msecp_fsm_proto_unso[_ENCTEST][_CFG_RQST]        = _config_rqst;
    _msecp_fsm_proto_unso[_ENCTEST][_SESS_RQST]       = _sess_rqst;
    _msecp_fsm_proto_unso[_ENCTEST][_ENCTEST_RQST]    = _enctest_rqst;
    _msecp_fsm_proto_unso[_ENCTEST][_SIGTEST_RQST]    = _sigtest_rqst;
    _msecp_fsm_proto_unso[_ENCTEST][_INSECURE_CMD]    = _insecure_cmd;

    _msecp_fsm_app_resp  [_ENCTEST]                   = _resp_do_nothing;
    _msecp_fsm_proto_resp[_ENCTEST][_CFG_RESP]        = _resp_do_nothing;
    _msecp_fsm_proto_resp[_ENCTEST][_SESS_RESP]       = _resp_do_nothing;
    _msecp_fsm_proto_resp[_ENCTEST][_TEST_OK_RESP]    = _enctest_resp;
    _msecp_fsm_proto_resp[_ENCTEST][_SESS_RQST]       = _sess_rqst;

     /************ SIGTEST ******************/
    _msecp_fsm_app_rqst  [_SIGTEST]                   = _queue_rqst;
    _msecp_fsm_proto_tmo [_SIGTEST][_RESP_TMO]        = _sigtest_resp_tmo;
    _msecp_fsm_app_unso  [_SIGTEST]                   = _app_unso;
    _msecp_fsm_proto_unso[_SIGTEST][_CFG_RQST]        = _config_rqst;
    _msecp_fsm_proto_unso[_SIGTEST][_SESS_RQST]       = _sess_rqst;
    _msecp_fsm_proto_unso[_SIGTEST][_ENCTEST_RQST]    = _enctest_rqst;
    _msecp_fsm_proto_unso[_SIGTEST][_SIGTEST_RQST]    = _sigtest_rqst;
    _msecp_fsm_proto_unso[_SIGTEST][_INSECURE_CMD]    = _insecure_cmd;

    _msecp_fsm_app_resp  [_SIGTEST]                   = _resp_do_nothing;
    _msecp_fsm_proto_resp[_SIGTEST][_CFG_RESP]        = _resp_do_nothing;
    _msecp_fsm_proto_resp[_SIGTEST][_SESS_RESP]       = _resp_do_nothing;
    _msecp_fsm_proto_resp[_SIGTEST][_TEST_OK_RESP]    = _sigtest_resp;
    _msecp_fsm_proto_resp[_SIGTEST][_GOTO_CFG_RQST]   = _config_rqst;
    _msecp_fsm_proto_resp[_SIGTEST][_GOTO_SESS_RQST]  = _sess_rqst;

} /* _init_vectors() */

/* ======================================================================= */
/**
 *  \brief Initializes the MSECP finite state machine.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_fsm_init(void)
{
    u_int8_t drop;

    info("msecp_fsm_init()");

    /*Security keys shared by all drops */
    psec= get_secure_struct();
    info("psec client %s ", psec->client_id);

    /* FSM vectors */
    _init_vectors();

    /* Drop-specific state information */
    for ( drop = 0; drop < MSECP_CFG_MAX_DROPS; drop++ )
    {
        /* FSM variables */
        _msecp_fsm[drop].state = _INIT;
        _msecp_fsm[drop].cfg_rqst_state = CFG_INVALID;
        _msecp_fsm[drop].session_valid = 0;
        _msecp_fsm[drop].tx_q = 0;

        /* Request header:
         *   ART has previous response time,
         *   STAT has statistics option mask, client name, and
         *      # of requests, responses, and unsolicited's  */
        _msecp_fsm[drop].nat_port = msecp_cfg_get_port(drop);
        _msecp_fsm[drop].seq_no = SECP_SEQ_NO_INITIAL;

        _msecp_fsm[drop].tmo_cnt = msecp_cfg_get_tmo_cnt();   /* tc, sec 1.7.4 */
        _msecp_fsm[drop].cur_tmo_cnt = 0;
    }

    info("msecp_fsm_init(): Done.");

} /* msecp_fsm_init() */


/* ======================================================================= */
/**
 *  \brief Send an app request to the host
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void _send_secp_app_rqst(u_int8_t drop)
{
    buf_t b, b_app, enc_rqst, b_raw;
    u_int8_t len,  msg_id = ESEC_APP_REQ;
    u_int16_t msg_len;
    char client_str[32];
    unsigned short  enc_sz, raw_sz, enc_sess_sz;


    info("send_secp_app_rqst(drop= %d) ", drop);

    strcpy(client_str, _msecp_fsm[drop].client_name);
    len = strlen(client_str);
    b = buf_append(0, &len, 1);
    b = buf_append(b, client_str, len);

    b_raw = buf_append(0, &msg_id, 1);
    b_app = buf_dequeue(_msecp_fsm[drop].tx_q);

    info("session_encrypt_msg");

   /*Allocate the Encrypted size and memory */
    raw_sz = buf_length(b_app);

   if( (psec->sec_cfg_word | (CONFIG_WORD_SYM_BIT | CONFIG_WORD_ASYM_BIT)) )
     enc_sz = SYM_BLOCK_SZ(raw_sz+3);
   else
     enc_sz = raw_sz+3;

   msg_len = htons(raw_sz);
   info("msg_len=%d   enc_sz =%d", msg_len, enc_sz);
   b_raw = buf_append(b_raw, &msg_len, 2);      /*Insert APP Length */
   b_raw = buf_join(b_raw, b_app);              /*Insert Raw APP Rqst */

   /* Encrypt the data if System Security Config Word requires before CRC.*/
   if( (psec->sec_cfg_word & (CONFIG_WORD_SYM_BIT | CONFIG_WORD_ASYM_BIT)) )
   {

     assert(enc_rqst = _session_encrypt_msg(drop,  b_raw));
     buf_free(b_raw);             /*Free the raw data buffer */
   }
   b = buf_join(b, enc_rqst);               /* Add Client name and APP Rqst */

   info("SECP data buf:");
   print_hex_data(buf_data(b),64);
   _send_secp_rqst(drop, b, SECP_APP_MSG);   /* Let App level handle Response Timer */

   info("send_secp_app_rqst(drop= %d) DONE.", drop);

} /* _send_secp_app_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a request message to the host
 *  \param drop The drop address.
 *  \param rqst The request message
 *  \return Nothing.
 */
/* ======================================================================= */
void _send_secp_rqst(u_int8_t drop, buf_t rqst, u_int8_t msg_type)
{
    buf_t b, b_rqst, enc_rqst;
    secp_rqst_hdr_t hdr;
    u_int32_t crc;

    info("   send_secp_rqst(drop= %d) sec_cfg_word: %x ", drop,psec->sec_cfg_word);

    hdr.pid = SECP_PROTO_ID;

    hdr.crc_sum = 0;
    if( msg_type == SECP_APP_MSG)
      hdr.seq_no = 0;
    else
      hdr.seq_no = _msecp_fsm[drop].seq_no;
    hdr.packet_details =0;

    /* Encrypt the data if System Security Config Word requires before CRC.*/
    if( (msg_type == SECP_APP_MSG) && (psec->sec_cfg_word & (CONFIG_WORD_SYM_BIT | CONFIG_WORD_ASYM_BIT)) )
    {
      if(psec->sec_cfg_word & CONFIG_WORD_SYM_BIT)
        hdr.packet_details = PKT_DETAIL_SYM_BIT;
      else
        hdr.packet_details = PKT_DETAIL_ASYM_BIT;
    }

    if(psec->sec_cfg_word & CONFIG_WORD_SIGNATURE_BIT)
      hdr.packet_details = (hdr.packet_details | PKT_DETAIL_SIGNATURE_BIT);

    hdr.packet_size = htons( buf_length(rqst));

    b = buf_append(0, &hdr, sizeof(hdr));
    buf_join(b, rqst);                       /* Add MSG to SECP header */

    /* Calculate CRC, free the calculated buffer and create
       the final buffer with the CRC added. */
    crc = crc32( -1, buf_data(b), 0, buf_length(b));
    hdr.crc_sum  = htonl(crc);
    buf_free(b);

    /* Add final header with CRC. */
    b = buf_append(0, &hdr, sizeof(hdr));
    buf_join(b, rqst);
    info("with crc");
    print_hex_data( buf_data(b), buf_length(b) >32 ? 64: buf_length(b));

    hdr.packet_size = htons(buf_length(rqst));
    b_rqst = buf_dup(rqst);

    b = buf_append(0, &hdr, sizeof(hdr));
    buf_join(b, b_rqst);


    /*TODO: Add Signature block functions if bit set.
            Block size is not included in SECP Header packet size. */
    if(psec->sec_cfg_word & CONFIG_WORD_SIGNATURE_BIT)
    {
       info("Add Signature Block here to tx buffer");
    }

   /* Send it up to the ESC Server, will free b also. */
   if(msg_type == SECP_PROTO_MSG)
   {
     if( ++_msecp_fsm[drop].cur_tmo_cnt > _msecp_fsm[drop].tmo_cnt )
    {
        warn("Host Timeout Count Exceeded (%d)", _msecp_fsm[drop].tmo_cnt);
        _msecp_fsm[drop].cur_tmo_cnt = 0;
        msecp_udp_use_next_host(drop);
    }
   }
   msecp_udp_send(drop, b);
   info("send_secp_rqst(drop= %d) DONE.", drop);
} /* _send_secp_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a Security Config Public Key Request to the host.
 *         Stop the timer just in case a pending Proto Request is pending.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void _send_secp_config_rqst(u_int8_t drop)
{
    info("send_secp_config_rqst(drop= %d) ", drop);
    msecp_tmr_stop_resp(drop);
    _send_secp_rqst(drop, _make_secp_config_rqst(drop), SECP_PROTO_MSG);

    info("send_secp_config_rqst(drop= %d) DONE.", drop);

    msecp_tmr_start_resp(drop);

} /* _send_secp_config_rqst() */



/* ======================================================================= */
/**
 *  \brief Send a Signature Test Request to the host
 *         Stop the timer just in case a pending Proto Request is pending.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void _send_secp_sigtest_rqst(u_int8_t drop)
{
    info("send_secp_sigtest_rqst(drop= %d) ", drop);

    msecp_tmr_stop_resp(drop);
    _send_secp_rqst(drop, _make_secp_sigtest_rqst(drop), SECP_PROTO_MSG );

    info("send_sec_sigtest_rqst(drop= %d) DONE", drop);

    msecp_tmr_start_resp(drop);
} /* _send_secp_sigtest_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a Secure Session Request to the host
 *         Stop the timer just in case a pending Proto Request is pending.
  *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void _send_secp_sess_rqst(u_int8_t drop)
{
    info("send_secp_sess_rqst(drop= %d) ", drop);

    msecp_tmr_stop_resp(drop);
    _send_secp_rqst(drop, _make_secp_sess_rqst(drop), SECP_PROTO_MSG );

    info("send_sec_sess_rqst(drop= %d) DONE. ", drop);

    msecp_tmr_start_resp(drop);
} /* _send_secp_sess_rqst() */



/* ======================================================================= */
/**
 *  \brief Send a Encrption Test Request to the host
 *         Stop the timer just in case a pending Proto Request is pending.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
 void _send_secp_encryptest_rqst(u_int8_t drop)
{
    info("send_secp_encryptest_rqst(drop= %d) ", drop);

    msecp_tmr_stop_resp(drop);

    _send_secp_rqst(drop, _make_secp_encryptest_rqst(drop), SECP_PROTO_MSG );

    info("send_secp_encryptest_rqst(drop= %d) DONE.", drop);

    msecp_tmr_start_resp(drop);
} /* _send_secp_encryptest_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a Security Failure request to the host.  Currently not used.
 *         Stop the timer just in case a pending Proto Request is pending.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
 void _send_secp_secure_failure_rqst(u_int8_t drop, u_int8_t error_code)
{
    buf_t b;
    u_int8_t msg_id = ESEC_SEC_FAIL_REQ;

    b = buf_append(0, &msg_id, 1);
    b = buf_append(b, &error_code, 1);

    msecp_tmr_stop_resp(drop);

    _send_secp_rqst(drop, b, SECP_PROTO_MSG);

    msecp_tmr_start_resp(drop);

} /* _send_secp_secure_failure_rqst() */



/* ======================================================================= */
/**
 *  \brief Make a Security Config Request message
 *  \param drop The drop address.
 *  \return A buffer containing the SECP Config Request.
 */
/* ======================================================================= */
static buf_t _make_secp_config_rqst(u_int8_t drop)
{
    buf_t b;
    u_int8_t msg_id = ESEC_CFG_REQ;
    u_int8_t len;
    unsigned char *pkey;
    u_int16_t net_sz, pkey_sz = 0;
    char client_str[32];

    info("make secp_config ");
    strcpy(client_str, _msecp_fsm[drop].client_name);
    len = strlen(client_str);

    info("make secp_config %s ", client_str);
   b = buf_append(0, &len, 1);
   b = buf_append(b, client_str, len);
info("buf_append ");
    b = buf_append(b, &msg_id, 1);
info("buf app--- t_pubkey:");
print_hex_data((char *)psec->t_pubkey ,64);

    assert((pkey = crypt_encode_akey(psec->t_pubkey, &pkey_sz)) != NULL);
    net_sz = htons(pkey_sz);
info("htons pkey_sz=%d net=%d", pkey_sz, net_sz);

    b = buf_append(b, &net_sz, sizeof(net_sz)); /* Size of Encoded Client Public Key*/
    b = buf_append(b, pkey, pkey_sz);           /* Encoded Client Public Key */
    free(pkey);                                 /* Free from malloc in crypt_encode_akey().*/

    return (b);
} /* _make_secp_config_rqst() */


/* ======================================================================= */
/**
 *  \brief Make a Signature Test Request message
 *  \param drop The drop address.
 *  \return A buffer containing the SECP Signature Test Request.
 */
/* ======================================================================= */
static buf_t _make_secp_sigtest_rqst(u_int8_t drop)
{
    buf_t b;
    u_int8_t msg_id = ESEC_STST_REQ, cli_len;
    u_int16_t  len, net_sz;
    unsigned char *signed_data;
    char client_str[32];
    unsigned char  sigBuf[512], *sigp = sigBuf;
    u_int32_t sig_sz=0;

    info("make secp_sigtest ");
    strcpy(client_str, _msecp_fsm[drop].client_name);
    cli_len = strlen(client_str);

    b = buf_append(0, &cli_len, 1);
    b = buf_append(b, client_str, cli_len);
    b = buf_append(b, &msg_id, 1);

    info("call crypt_sign: tdata_sz : %x", psec->tdata_sz);
    if(crypt_sign(psec->tdata, psec->tdata_sz, psec->t_prvkey, &sigp, &sig_sz) < 0)
      return -1;
    net_sz = htons((u_int16_t)sig_sz);
    b = buf_append(b, &net_sz,2);
    b = buf_append(b, sigBuf, sig_sz);

    return (b);
} /* _make_secp_sigtest_rqst() */


/* ======================================================================= */
/**
 *  \brief Make a Secure Session Request message
 *  \param drop The drop address.
 *  \return A buffer containing the Session request.
 */
/* ======================================================================= */
static buf_t _make_secp_sess_rqst(u_int8_t drop)
{
    buf_t b;
    u_int8_t msg_id = ESEC_SESS_REQ;
    u_int8_t cli_len;
    char client_str[32];

    info("make secp_sess_rqst ");
    strcpy(client_str, _msecp_fsm[drop].client_name);

    cli_len = strlen(client_str);
    b = buf_append(0, &cli_len, 1);
    b = buf_append(b, client_str, cli_len);
    b = buf_append(b, &msg_id,1);

    return (b);
} /* _make_secp_sess_rqst() */


/* ======================================================================= */
/**
 *  \brief Make a Encryption Test Request message
 *  \param drop The drop address.
 *  \return A buffer containing the SECP Encryption Test Request.
 */
/* ======================================================================= */
static buf_t _make_secp_encryptest_rqst(u_int8_t drop)
{
    buf_t b;
    u_int8_t  cli_len;
    u_int8_t msg_id = ESEC_ETST_REQ;
    u_int16_t  len, net_sz, edata_sz;
    unsigned char *session_data, *broad_data, *enc_ptr;
    char client_str[32];

    strcpy(client_str, _msecp_fsm[drop].client_name);
    cli_len = strlen(client_str);
    b = buf_append(0, &cli_len, 1);
    b = buf_append(b, client_str, cli_len );
    b = buf_append(b, &msg_id, 1);

    /*** Session Key Encrypted Test Data ***/
    edata_sz = SYM_BLOCK_SZ(psec->tdata_sz); /* Total Encrypted Data Size */
    enc_ptr = (unsigned char *) malloc(edata_sz);
    memset(enc_ptr, 0, edata_sz);

    net_sz = htons(edata_sz);
    b = buf_append(b, &net_sz, 2);        /* Insert length */

    if (edata_sz != crypt_sess_encrypt(psec, psec->tdata, psec->tdata_sz, enc_ptr))
    {
      info("build enc test req: Sess Encryption Failed %d : %d\n", psec->tdata_sz, edata_sz);
      return -1;
    }
    b = buf_append(b, enc_ptr, edata_sz); /* Insert Data   */

    net_sz = htons(edata_sz);
    b = buf_append(b, &net_sz, 2);        /* Insert length */

    memset(enc_ptr, 0, edata_sz);
    info("encyrpt bro");
    if (edata_sz != crypt_unso_encrypt(psec, psec->tdata, psec->tdata_sz, enc_ptr))
    {
      info("build enc test req: Bro Encryption Failed\n");
      return -1;
    }
    b = buf_append(b, enc_ptr, edata_sz); /* Insert Data   */
    free(enc_ptr);

    return (b);
} /* _make_secp_encryptest_rqst() */




/* ======================================================================= */
/**
 *  \brief Process a unicast message from the host.
 *  \param drop The drop address.
 *  \param b The buffer.
 *  \return The buffer passed in.
 */
/* ======================================================================= */
buf_t msecp_fsm_process_unicast(u_int8_t drop, buf_t b)
{
    u_int8_t   *secp_msg;
    u_int16_t  temp;
    int  len, msg_sz;
    secp_resp_hdr_t  *secp_hdr;

    secp_hdr = (secp_resp_hdr_t *)buf_data(b);
    temp = ntohs(secp_hdr->packet_size);
    secp_hdr->packet_size = temp;

    info("unicast rcvd ------------------");
    print_hex_data( buf_data(b), buf_length(b) >32 ? 64: buf_length(b));

    if ( secp_hdr->pid != SECP_PROTO_ID )
    {
        warn("Invalid protocol ID (%d)", secp_hdr->pid);
        return (b);
    }

    /* Basic header checking */
    if ( (len = buf_length(b)) < (sizeof(secp_resp_hdr_t)) )
    {
        warn("Received Header too small (%d)", len);
        return (b);
    }

    /*Check the CRC, Seq #, Packet Details, Client Name */
    if ( (msg_sz = _parse_msg_header(drop, b)) < 0 )
    {
        warn("esec msg_sz %d < 0 \n", msg_sz);
        return (b);
    }

    /* Point to the SECP Message type, skip over header and client name */
    secp_msg = ((u_int8_t *)secp_hdr +
                (sizeof(secp_resp_hdr_t) + secp_hdr->client_len));

    info("secp_msg %x %x sz=%x",*secp_msg, secp_hdr->client_len, sizeof(secp_resp_hdr_t));
    /*Unecrypt the Message */
    if ((msg_sz = _process_msg_body(drop,psec, (u_int8_t *) (secp_msg), msg_sz))<0)
    {
      warn("esec body bad\n");
      return (b);
    }

    /* Pass the buffer down to the next layer */
    if( *secp_msg == ESEC_APP_RSP )    /* 0xDB Process SECP Application UNSO/Response */
    {
       msg_sz-=3;    /* Remove Msg ID and Len before passing it to MESC. */
       secp_msg+=3;

       if ( PKT_DETAIL_UNSO_BIT & (secp_hdr->packet_details) )
       {
          info("SECP UNSO msg rcvd.");
          _dispatch_app_unso(drop, secp_msg, msg_sz);
       }else{
          info("SECP Solicited App Response rcvd.");
         _dispatch_app_resp(drop, secp_msg, msg_sz);
       }
    }
    else{                              /*Process the SECP Msg Protocol */
       if ( PKT_DETAIL_UNSO_BIT & (secp_hdr->packet_details) )
          info("SECP UNSO msg rcvd.");
       else
          info("SECP Solicited msg rcvd.");

       _dispatch_proto_msg(drop, secp_msg, msg_sz);
    }

    return (b);
} /* msecp_fsm_process_unicast() */


/* ======================================================================= */
/**
 *  \brief Parse a received SECP msg header.
 *  \param drop The drop address.
 *  \param b The response message.
 *  \return Length of remaining payload, or -1 ERROR
 */
/* ======================================================================= */
static int _parse_msg_header(u_int8_t drop, buf_t b)
{
    secp_resp_hdr_t *secp_hdr;
    int len;
    u_int16_t temp;
    u_int32_t crc, calc_crc;

    secp_hdr = (secp_resp_hdr_t *)buf_data(b);
    temp = ntohs(secp_hdr->packet_size);
    secp_hdr->packet_size = temp;
    len = buf_length(b);

    /*Get the CRC32 sum and compare.*/
    crc = secp_hdr->crc_sum;
    crc = ntohl(crc);
    secp_hdr->crc_sum=0;        /*Set it to zero to calculate.*/
    if (crc != (calc_crc = crc32( -1, buf_data(b), 0, len)) )
    {
      warn("CRC32 Sum mismatch: received %x, expected %x.",
            crc, calc_crc);
      return (-1);
    }

    len = secp_hdr->client_len;

   /* Check Seq# for SECP proto responses only. Can't check the Message ID
      for APP Responses since this may be encrypted. SECP Proto Responses
      will not be encrypted. */
    info("sec pkt %x", secp_hdr->packet_details);
    if( !( secp_hdr->packet_details & PKT_DETAIL_UNSO_BIT) &&
         ( ( !(secp_hdr->packet_details & PKT_DETAIL_SYM_BIT) &&
             !(secp_hdr->packet_details & PKT_DETAIL_ASYM_BIT) ) &&
              ( ((*secp_hdr->data) + len) != ESEC_APP_RSP)) )
    {
      if( secp_hdr->seq_no != _msecp_fsm[drop].seq_no )
      {
        warn("Sequence # mismatch: received %d, expected %d.",
             secp_hdr->seq_no, _msecp_fsm[drop].seq_no);
        return (-1);
      }
    }

    if( !( secp_hdr->packet_details & PKT_DETAIL_UNSO_BIT) )
    {
      if( memcmp(secp_hdr->data, _msecp_fsm[drop].client_name, secp_hdr->client_len) )
      {
        warn("Client Name mismatch unso dropped (drop %d). ", drop);
        return (-1);
      }
    }

     len = secp_hdr->packet_size - secp_hdr->client_len - 1;
     info("len = %x, pak sz = %x  cli len =%x", len, secp_hdr->packet_size, secp_hdr->client_len);
    /*Save the Packet details, replaces psec->pkt. */
    _msecp_fsm[drop].sec_details = secp_hdr->packet_details;

    return (len);
} /* _parse_msg_header() */



/* ======================================================================= */
/*
 * \brief This function processes the security messages based on the packet
          information. It modifies the incoming buffer.

          NOTE: TODO Signature Applied processing not implemented on the Server.

 * \return \li size of the message
 *         \li FAILURE  on error
*/
/* ======================================================================= */
static int _process_msg_body(u_int8_t drop, struct secure *psec, unsigned char *msg, int msg_sz)
{
   int sz;
   int enc_sz;
   unsigned char *buf;
   int buf_sz = SYM_BLOCK_SZ(msg_sz);
   sz = msg_sz;

   /*If encrypted, decrypt */
   if((_msecp_fsm[drop].sec_details & PKT_DETAIL_ASYM_BIT ) ||
      (_msecp_fsm[drop].sec_details & PKT_DETAIL_SYM_BIT))
   {
      buf = malloc(buf_sz);
      memset(buf, 0, buf_sz);

      if (_msecp_fsm[drop].sec_details & PKT_DETAIL_UNSO_BIT)
      {
        if ((enc_sz = crypt_unso_decrypt(psec, msg, msg_sz, buf)) == 0)
        {
          free(buf);
          return -1;
        }
      }else{
        if ((enc_sz = crypt_sess_decrypt(psec, msg, msg_sz, buf)) == 0)
        {
          free(buf);
          return -1;
         }
      }

      info("Decrypted RXData: ----------");
      print_hex_data(buf, enc_sz);

      sz = enc_sz;
      memcpy(msg, buf, enc_sz);
      free(buf);
    }
    else    /* Check invalid encryption (bits 1,5 set) */
      if( _msecp_fsm[drop].sec_details & (PKT_DETAIL_ASYM_BIT | PKT_DETAIL_SYM_BIT))
        return(-1);

   if(_msecp_fsm[drop].sec_details & PKT_DETAIL_SIGNATURE_BIT)
   {
      /*TODO Process/compare Signature block ?*/
   }

   return (sz);
}

/* ======================================================================= */
/**
 *  \brief Process a SECP protocol response timeout.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_fsm_process_proto_tmo(u_int8_t drop)
{
    _dispatch_tmo(drop, _RESP_TMO);
} /* msecp_fsm_process_proto_tmo() */


/* ======================================================================= */
/*
 * \brief This function encryptes the request message.
 *  \param drop The drop address.
 *  \param b The raw data message.
 *  \return Encrypted buffer or -1 ERROR
 */
/* ======================================================================= */
static buf_t _session_encrypt_msg(u_int8_t drop, buf_t b)
{
  unsigned char  *enc_ptr;
  unsigned char *raw_ptr;
  unsigned short  enc_sz, raw_sz, enc_sess_sz, i;
  buf_t  enc_buf;


  info("-----session_encrypt_msg() -----");

  /*Allocate the Encrypted size and memory */
  enc_sz = SYM_BLOCK_SZ( buf_length(b));
  enc_ptr = (unsigned char *) malloc(enc_sz+8);
  memset(enc_ptr, 0, enc_sz);

  /*Get the unecrypted data */
  raw_ptr =  buf_data(b);

  enc_sess_sz = crypt_sess_encrypt(psec, raw_ptr, enc_sz, enc_ptr);
  info("enc_sess_sz = %x  enc_sz=%x", enc_sess_sz, enc_sz);
  if(enc_sess_sz != enc_sz && (enc_sess_sz % BLOCK_SZ))
  {
      info("free enc_ptr");
      buf_free(b);
      free(enc_ptr);
      return ((buf_t) -1);
  }

  buf_free(b);

  enc_buf = buf_append(0, enc_ptr, enc_sess_sz);

  free(enc_ptr);

  info("----ret enc_buf----");
  return (enc_buf);
}


/* ======================================================================= */
/**
 *  \brief Call app resp action for given state.
 *  \param drop The drop.
 *  \param b    The data buffer.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_app_rqst(u_int8_t drop, buf_t b)
{
     trace("%s(%d): APP RQST ----------v",
        _state2str(_msecp_fsm[drop].state), drop);

    _msecp_fsm_app_rqst[_msecp_fsm[drop].state](drop, b);

} /* _dispatch_app_rqst() */

/* ======================================================================= */
/**
 *  \brief Call app resp action for given state.
 *  \param drop The drop.
 *  \param data Data.
 *  \param length Length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_app_resp(u_int8_t drop, u_int8_t *data, int length)
{
     trace("%s(%d): APP RESP ----------v",
        _state2str(_msecp_fsm[drop].state), drop);

    //Calls msecp_imp_send_rxdata().
    _msecp_fsm_app_resp[_msecp_fsm[drop].state](drop, data, length);

} /* _dispatch_app_resp() */


/* ======================================================================= */
/**
 *  \brief Call proto resp/unso action for given state.
 *  \param drop The drop.
 *  \param data Data.
 *  \param length Length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_proto_msg(u_int8_t drop, u_int8_t *data, int length)
{
    int input;

    assert(data);
    switch ( *data )    /* MSG code */
    {
        case ESEC_CFG_RSP:       //0xD1
             input = _CFG_RESP;
             break;
        case ESEC_SESS_RSP:      //0xD5
             input = _SESS_RESP;
             break;
        case ESEC_ACK_CMD_RSP:   //0xD9
            data++;   /* Status Code */

            if(*data == ESEC_COMMAND_RESP )  //0xFF
            {
              data++;
              switch(*data){       /*Command Code */
                case   ESEC_CFG_REQ:
                            input = _CFG_RQST; break;
                case   ESEC_STST_REQ:
                            input = _SIGTEST_RQST; break;
                case   ESEC_SESS_REQ:
                            input = _SESS_RQST; break;
                case   ESEC_ETST_REQ:
                            input = _ENCTEST_RQST; break;
                case   ESEC_UNSECURE_CMD:
                            input = _INSECURE_CMD; break;
                default:
                    trace("%s(%d): UNKNOWN Sec ACK/Command Code (0x%02X) ----------v",
                    _state2str(_msecp_fsm[drop].state), drop, *data);
                    _set_state(drop, _msecp_fsm[drop].state);
                    return;
              }
            }else
            {
             switch(*data){       /*Status Codes */
                case   ESEC_STATUS_OK:
                            input = _TEST_OK_RESP; break;
                case   ESEC_INVALID_PKEY:
                case   ESEC_GENERAL_ERR:
                case   ESEC_STST_FAILED:
                           info("Failed Sigtest/General/pkey invalid");
                            _msecp_fsm[drop].seq_no++;
                            input = _GOTO_CFG_RQST; break;
                case   ESEC_SESSION_FAIL:
                case   ESEC_SESSION_ERR:
                            _msecp_fsm[drop].seq_no++;
                            input = _GOTO_SESS_RQST; break;
                case   ESEC_PKEY_EXP:     /*Generate Keys and send CFG_REQUEST*/
/* Stop all other drops from generating keys until the current keys are
   generated, since on the ESPAD it may take a few minutes. */
                            if(psec->gen_key_flag != KEYS_GENERATING)
                            {
                               psec->gen_key_flag = KEYS_GENERATING;
                               assert( crypt_generate_keys(psec));
                               psec->gen_key_flag = KEYS_GENERATED;
                            }
                            _msecp_fsm[drop].seq_no++;
                            input = _GOTO_CFG_RQST; break;

                default:
                    trace("%s(%d): UNKNOWN SESP Status Code (0x%02X) ----------v",
                    _state2str(_msecp_fsm[drop].state), drop, *data);
                    _set_state(drop, _msecp_fsm[drop].state);
                    return;
              }
            }
             break;

        default:
              trace("%s(%d): UNKNOWN Proto MSG ID (0x%02X) ----------v",
              _state2str(_msecp_fsm[drop].state), drop, *data);
              _set_state(drop, _msecp_fsm[drop].state);
        return;
    }

    _msecp_fsm[drop].cur_tmo_cnt = 0;

    /*Check if this is an UNSO Msg. */
    if( _msecp_fsm[drop].sec_details & PKT_DETAIL_UNSO_BIT )
      _msecp_fsm_proto_unso[_msecp_fsm[drop].state][input](drop);
    else
      _msecp_fsm_proto_resp[_msecp_fsm[drop].state][input](drop, data, length);

} /* _dispatch_proto_msg() */



/* ======================================================================= */
/**
 *  \brief Call timer action for current state and given input.
 *  \param drop The drop.
 *  \param input Input.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_tmo(u_int8_t drop, int input)
{
    _msecp_fsm_proto_tmo[_msecp_fsm[drop].state][input](drop);
} /* _dispatch_tmo() */


/* ======================================================================= */
/**
 *  \brief Queue a request for this drop.
 *  \param drop The drop this request is from.
 *  \param b The buffer holding the request.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _queue_rqst(u_int8_t drop, buf_t b)
{
    if ( b )
        buf_queue(_msecp_fsm[drop].tx_q, b);
    _set_state(drop, _msecp_fsm[drop].state);
} /* _queue_rqst() */



/* ======================================================================= */
/**
 *  \brief Handle a request in the INIT state.
 *  \param drop The drop this request is from.
 *  \param b The buffer holding the request.
 *      If the buffer is NULL, we'll still configure the drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _init_rqst(u_int8_t drop, buf_t b)
{
   if ( b )
     buf_queue(_msecp_fsm[drop].tx_q, b);

   msecp_udp_open(drop);


   if( psec->spubkey_valid == VALID )
   {
        info("======== spubkey VALID ==================== ");
       /*Check if Sig/Encrypt Test required. EncryTest will be sent
         after a Session is established. */
        if( msecp_cfg_get_session_test())
        {
            _sigtest_rqst(drop);
        }else{
            _send_secp_sess_rqst(drop);
            _set_state(drop, (int) _SESSION);
        }

   }else{
        _config_rqst(drop);
        _set_state(drop, (int) _CONFIG);
   }

} /* _init_rqst() */


/* ======================================================================= */
/**
 *  \brief Handles a Secure Session response in the SESSION state.
 *         Re-request a Session upon an error.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _sess_resp(u_int8_t drop, u_int8_t *data, int length)
{
    u_int8_t *dataptr;
    u_int16_t net_sz, w_ksz, ksz;

    msecp_tmr_stop_resp(drop);
    if(length <= 0)
    {
       _send_secp_sess_rqst(drop);
       _set_state(drop, (int) _SESSION);
       return;
    }

    /*TODO Save the Session keys on per drop basis. */
    dataptr=++data;                        /*skip Msg type */

    if (psec->ses_key) free(psec->ses_key);
    if (psec->bro_key) free(psec->bro_key);
    memset(psec->ses_iv, 0, 8);
    memset(psec->bro_iv, 0, 8);

    /*Wrapped Init Session Vector*/
    memcpy(&net_sz, dataptr, sizeof(net_sz));
    w_ksz = ntohs(net_sz);
    dataptr += sizeof(w_ksz);
    if(w_ksz != 8)
    {
       _send_secp_sess_rqst(drop);
       _set_state(drop, (int) _SESSION);
       return;
    }
    memcpy(psec->ses_iv, dataptr, 8);
    dataptr += w_ksz;

    /*Wrapped Session Key*/
    memcpy(&net_sz, dataptr, sizeof(net_sz));
    w_ksz = ntohs(net_sz);
    dataptr += sizeof(w_ksz);
    if (w_ksz > length)
    {
       _send_secp_sess_rqst(drop);
       _set_state(drop, (int) _SESSION);
       return;
    }

    ksz = 0;
    if ((psec->ses_key = get_sym_key(psec, dataptr, w_ksz, &ksz)) == NULL)
    {
      warn("Error getting session key\n");
      _send_secp_sess_rqst(drop);
      _set_state(drop, (int) _SESSION);
      return;
    }
    psec->ses_ksz = ksz;
    dataptr += w_ksz;

    /*Wrapped Init Bro Vector*/
    memcpy(&net_sz, dataptr, sizeof(net_sz));
    w_ksz = ntohs(net_sz);
    dataptr += sizeof(w_ksz);
    if (w_ksz != 8)
    {
       _send_secp_sess_rqst(drop);
       _set_state(drop, (int) _SESSION);
       return;
    }
    memcpy(psec->bro_iv, dataptr, 8);
    dataptr += w_ksz;

    /*Wrapped Bro  Session Key */
    memcpy(&net_sz, dataptr, sizeof(net_sz));
    w_ksz = ntohs(net_sz);
    dataptr += sizeof(w_ksz);
    if (w_ksz > length)
    {
       _send_secp_sess_rqst(drop);
       _set_state(drop, (int) _SESSION);
       return;
    }

    ksz = 0;
    if ((psec->bro_key = get_sym_key(psec, dataptr, w_ksz, &ksz)) == NULL)
    {
      warn("Error getting session key\n");
      _send_secp_sess_rqst(drop);
      _set_state(drop, (int) _SESSION);
      return;
    }

    psec->bro_ksz = ksz;
    dataptr += w_ksz;

    _msecp_fsm[drop].session_valid = 1;
    _msecp_fsm[drop].seq_no++;

    if( msecp_cfg_get_session_test())   /*Check if we need to send Encryp Test*/
    {
       _enctest_rqst(drop);
    }else{
        _set_state(drop, (int) _READY);
        if ( _msecp_fsm[drop].tx_q )
        {
            _send_secp_app_rqst(drop);
            msecp_imp_send_txdata_ind(drop);
        }
    }

} /* _sess_resp() */


/* ======================================================================= */
/**
 *  \brief Handle a response timeout in the SESSION state.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _sess_resp_tmo(u_int8_t drop)
{
    msecp_art_mark_resp_tmo(drop);
    _send_secp_sess_rqst(drop);
    _set_state(drop, (int) _SESSION);

} /* _sess_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Handles a Secure Config response in the CONFIG state.
 *         Allows only one outstanding Config Response to be processes at
 *         time, since the ESPAD crypto functions and NVRAM saves are
 *         time consuming.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _config_resp(u_int8_t drop, u_int8_t *data, int length)
{
    u_int16_t enc_spub_sz, net_sz, enc_test_sz, orig_sec_cfg_word;
    unsigned char *dataptr;
    RSA *decoded_spub = NULL;
    unsigned char  tdata[256];

    msecp_tmr_stop_resp(drop);

    info("Secure Config resp:");
    print_hex_data( data, 32);

    /* Re-Send again until the first Request from other drop is received/validated.*/
    if((_msecp_fsm[drop].cfg_rqst_state == QUEUED_CFG_RQST) &&
       (psec->gen_key_flag == KEYS_GENERATING) )
    {
       warn("Config Rqst Pending. \n");
       _config_rqst(drop);
       _set_state(drop, (int) _CONFIG);
       return;
    }

    /* TODO:Update the MSECP/PSEC structure with Server Public key, and Test Data info
      RSA Signature Required, Encryption type (Asym/Sym) . */
    data++;
    dataptr=data;                        /*skip Msg type */
    memcpy(&net_sz, dataptr, 2);
    enc_spub_sz = ntohs(net_sz);

    dataptr++;
    dataptr++;

    print_hex_data( dataptr, 32 );
    if((decoded_spub = decode_pub_key(&dataptr, enc_spub_sz)) == NULL)
    {
      warn("Decode SPUB Key failed. \n");
      _config_rqst(drop);
      _set_state(drop, (int) _CONFIG);
      return;
    }

    info("comparing keys");
    /*TODO Compare the the server public key with the stored one, if different
           update with current one. Eliminates excessive writes to NVRAM */
//    if( !(memcmp(decoded_spub, psec->s_pubkey, 4 ))) // enc_spub_sz)) )
//    {
//      if(psec->s_pubkey)
//        RSA_free(psec->s_pubkey);
//       psec->s_pubkey = NULL;

       info("write_server");
       psec->s_pubkey = decoded_spub;
       assert( crypt_write_server_pub_key(psec));
//    }

    data++;
    data++;
    data+= enc_spub_sz;         /* point to Security Config Word */
    memcpy(&net_sz, data, sizeof(net_sz));
    /* Check for invalid Sym and Asym bits set */
// TODO  if( (ntohs(net_sz)) & (CONFIG_WORD_SYM_BIT | CONFIG_WORD_ASYM_BIT))
//    {
//     warn("Security Word Invalid");
//     _config_rqst(drop);
//     _set_state(drop,(int) _CONFIG);
//     return;
//    }

    orig_sec_cfg_word = psec->sec_cfg_word;
    psec->sec_cfg_word = ntohs(net_sz);    /*Update new Security Word*/
    info("Security Config Word: %x enc_spub_sz=%x dataptr:%x \n", psec->sec_cfg_word,enc_spub_sz,*data);

    data += sizeof(net_sz);               /*point to Test Data size */
    memcpy(&net_sz, data, sizeof(net_sz));
    enc_test_sz = ntohs(net_sz);
    /* point to Test Data */
    data++;
    data++;

    /*Copy original test data for comparision.*/
    memset(tdata, 0, sizeof(tdata));
    memcpy(tdata, psec->tdata, psec->tdata_sz);

    if(get_test_data(psec, data, enc_test_sz, _msecp_fsm[drop].sec_details) < 0)
    {
      psec->tdata_sz = 0;
      warn("Error getting test data \n");
      _config_rqst(drop);
      _set_state(drop, (int) _CONFIG);
      return;
    }

    /* Compare Security Word,Test Data with "/etc/gtech/sec.cfg"
       save it to NVRAM if required. */
    if( (memcmp(tdata, psec->tdata, psec->tdata_sz) != 0) ||
       (orig_sec_cfg_word != psec->sec_cfg_word))
       assert( esec_write_config(psec) );

     psec->spubkey_valid = VALID;
    _msecp_fsm[drop].cfg_rqst_state = CFG_VALID;
    _msecp_fsm[drop].seq_no++;

    /*Check if Signature and Encryption tests are required. Perform Sig test
      first, then perform Encryption Test once a Session has been establised.*/
    if( msecp_cfg_get_session_test())
    {
      _sigtest_rqst(drop);
    }
    else
      _sess_rqst(drop);

} /* _config_resp() */



/* ======================================================================= */
/**
 *  \brief Handle a response timeout in the CONFIG state.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _config_resp_tmo(u_int8_t drop)
{
    msecp_art_mark_resp_tmo(drop);
    _config_rqst(drop);
    _set_state(drop, (int) _CONFIG);

} /* _config_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Send a Secure Config Request.
 *        If another drop is in the process of generating keys
 *        (could take a few minutes), don't send the Request,
 *        rely on Resp Timer to retry.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _config_rqst(u_int8_t drop)
{

  _set_cfg_rqst_state(drop);       /*Process only One CFG response at time */
  _send_secp_config_rqst(drop);

  _set_state(drop, (int) _CONFIG);
} /* _config_rqst() */


/* ======================================================================= */
/**
 *  \brief Set the correct SECP Config Request state.
 *        Only one SECP Config Response is handled at a time.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _set_cfg_rqst_state(u_int8_t drop)
{

  if( _msecp_fsm[drop].cfg_rqst_state != FIRST_CFG_RQST)
  {
     if(psec->spubkey_valid != CFG_PENDING)
     {
       psec->spubkey_valid = CFG_PENDING;
       _msecp_fsm[drop].cfg_rqst_state = FIRST_CFG_RQST;
     }else
       _msecp_fsm[drop].cfg_rqst_state = QUEUED_CFG_RQST;
  }
}


/* ======================================================================= */
/**
 *  \brief Send a Secure Session Request.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _sess_rqst(u_int8_t drop)
{
    _send_secp_sess_rqst(drop);
    _set_state(drop, (int) _SESSION);
} /* _sess_rqst() */

/* ======================================================================= */
/**
 *  \brief Send a Encyrption Test Request.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _enctest_rqst(u_int8_t drop)
{
    _send_secp_encryptest_rqst(drop);
    _set_state(drop, (int) _ENCTEST);
} /* _enctest_rqst() */


/* ======================================================================= */
/**
 *  \brief Send a Signature Test Request.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _sigtest_rqst(u_int8_t drop)
{
    _send_secp_sigtest_rqst(drop);
    _set_state(drop, (int) _SIGTEST);
} /* _sigtest_rqst() */


/* ======================================================================= */
/**
 *  \brief Process a Encyrption Test Response.
 *  \param drop The drop.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _enctest_resp(u_int8_t drop, u_int8_t *data, int length)
{
   msecp_tmr_stop_resp(drop);

   if( _msecp_fsm[drop].tx_q)
   {
     _send_secp_app_rqst(drop);
     msecp_imp_send_txdata_ind(drop);
   }
   _set_state(drop, (int) _READY);
   _msecp_fsm[drop].seq_no++;

} /* _enctest_resp() */


/* ======================================================================= */
/**
 *  \brief Process a Encyrption Test Response TMO.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _enctest_resp_tmo(u_int8_t drop)
{
    msecp_art_mark_resp_tmo(drop);

   _enctest_rqst(drop);
}
/* _enctest_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Handles an application request in the READY state.
 *  \param drop The drop address.
 *  \param b The buffer
 *  \return Nothing.
 */
/* ======================================================================= */
static void _ready_rqst(u_int8_t drop, buf_t b)
{
    if ( b )
    {
        buf_queue(_msecp_fsm[drop].tx_q, b);
        _send_secp_app_rqst(drop);
        msecp_imp_send_txdata_ind(drop);
    }

    _set_state(drop, (int) _READY);

} /* _ready_rqst() */



/* ======================================================================= */
/**
 *  \brief Process a Signature Test Response.
 *         If there is no current Secure session, request one.
 *  \param drop The drop.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _sigtest_resp(u_int8_t drop, u_int8_t *data, int length)
{
    msecp_tmr_stop_resp(drop);

    _msecp_fsm[drop].seq_no++;

    if( _msecp_fsm[drop].session_valid)
       _set_state(drop, _READY);
    else
       _sess_rqst(drop);

} /* _sigtest_resp() */


/* ======================================================================= */
/**
 *  \brief Handle a response timeout in the CONFIG state.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _sigtest_resp_tmo(u_int8_t drop)
{
    msecp_art_mark_resp_tmo(drop);
    _sigtest_rqst(drop);

} /* _sigtest_resp_tmo() */


/* ======================================================================= */
/**
 *  \brief Handle the Insecure Channel Command.
 *   TODO:  Could add a new state (INSECURE) to the msecp state machine.
 *          Where the UDP data would no be decrypted.
 *          This command is currently not implemented.
 *  \param drop The drop.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _insecure_cmd(u_int8_t drop)
{
    trace("Insecure Cmd ignored.\n\n");
    _set_state(drop, _msecp_fsm[drop].state);
} /* _insecure_cmd() */


/* ======================================================================= */
/**
 *  \brief Process a transmit request from a drop
 *  \param drop The drop address.
 *  \param data The data to send
 *  \param len The number of data bytes
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_fsm_process_app_rqst(u_int8_t drop, char *client_id, char *data, int len)
{
    buf_t b = NULL;

    if ( data && len )
    {
        b = buf_append(0, data, len);
    }

    /*Save client name for SECP Requests */
    strcpy(_msecp_fsm[drop].client_name, client_id);

    _dispatch_app_rqst(drop, b);

} /* msecp_fsm_process_app_rqst() */



/* ======================================================================= */
/**
 *  \brief Call app unso action for given state.
 *  \param drop The drop.
 *  \param data Data.
 *  \param length Length
 *  \return Nothing.
 */
/* ======================================================================= */
static void _dispatch_app_unso(u_int8_t drop, u_int8_t *data, int length)
{
    if ( drop == 255 )  /* multicast unso, send it immediately to all drops.*/
    {
        trace("MULTICAST APP UNSO ----------v");
        _app_unso(255, data, length);
    }
    else
    {
        trace("%s(%d): APP UNSO ----------v",
            _state2str(_msecp_fsm[drop].state), drop);
        _msecp_fsm_app_unso[_msecp_fsm[drop].state](drop, data, length);
    }

} /* _dispatch_app_unso() */



/* ======================================================================= */
/**
 *  \brief Handles an application unso message.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _app_unso(u_int8_t drop, u_int8_t *data, int length)
{
    msecp_imp_send_rxdata(drop, data, length);

    if ( drop == 255 )  /* could be multicast, drop=255 */
        trace("v----------> Multicast Unso Delivered.\n\n");
    else
        _set_state(drop, _msecp_fsm[drop].state);
// The state should stay the same, for the case where an Ecrypt/Session
// is in progress and an App Unso is received.

} /* _app_unso() */


/* ======================================================================= */
/**
 *  \brief Dummy application unso vector.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _app_unso_do_nothing(u_int8_t drop, u_int8_t *data, int length)
{
    if ( drop == 255 )    /* could be a multicast (drop=255) */
        trace("v----------> Nothing to do.\n\n");
    else
        _set_state(drop, _msecp_fsm[drop].state);

} /* _app_unso_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Dummy protocol unso vector.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
// TODO Multicast: static void _proto_unso_do_nothing(u_int8_t drop, u_int8_t *data, int length)
static void _proto_unso_do_nothing(u_int8_t drop)
{
    if ( drop == 255 )    /* could be a multicast (drop=255) */
        trace("v----------> Nothing to do.\n\n");
    else
        _set_state(drop, _msecp_fsm[drop].state);

} /* _proto_unso_do_nothing() */

/* ======================================================================= */
/**
 *  \brief Handles a response in the READY state.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _app_resp(u_int8_t drop, u_int8_t *data, int length)
{
    msecp_imp_send_rxdata(drop, data, length);

    if ( _msecp_fsm[drop].tx_q )
    {
        _send_secp_app_rqst(drop);
    }

    _set_state(drop, _READY);

} /* _app_resp() */


/* ======================================================================= */
/**
 *  \brief Dummy response vector.
 *  \param drop The drop address.
 *  \param data The data.
 *  \param length The length.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _resp_do_nothing(u_int8_t drop, u_int8_t *data, int length)
{
    _set_state(drop, _msecp_fsm[drop].state);
} /* _resp_do_nothing() */

/* ======================================================================= */
/**
 *  \brief Dummy timer vector.
 *  \param drop The drop address.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _timer_do_nothing(u_int8_t drop)
{
    _set_state(drop, _msecp_fsm[drop].state);
} /* _timer_do_nothing() */


/* ======================================================================= */
/**
 *  \brief Set current state.
 *  \param drop The drop.
 *  \param state State.
 */
/* ======================================================================= */
static void _set_state(u_int8_t drop, int state)
{
    _msecp_fsm[drop].state = state;
    trace("v----------> Current State(%d) -> %s\n\n", drop, _state2str(state));

} /* _set_state() */


/* ======================================================================= */
/**
 *  \brief Convert state to string.
 *  \param state State.
 *  \return State string.
 */
/* ======================================================================= */
static char *_state2str(int state)
{
    switch ( state )
    {
        case _INIT:     return ("INIT");
        case _CONFIG:   return ("CONFIG RQST");
        case _SESSION:  return ("SESSION RQST");
        case _READY:    return ("READY");
        case _ENCTEST:  return ("ENC TEST");
        case _SIGTEST:  return ("SIG TEST");
        default:        return ("UNKNOWN STATE");
    }
} /* _state2str() */

 /* End of $Id: msecp_fsm.c,v 1.4 2005/02/09 23:36:37 cmayncvs Exp $
 */
