/* ===[ $RCSfile: msecp_imp.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: msecp_imp.c,v 1.2 2004/08/06 17:35:16 tmeiccvs Exp $"
 *
 *  \brief Handle Multi-drop SECP Inter-Module Protocol (IMP) messages.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/msg.h"
#include "libcom/fd.h"

#include "msecp_imp.h"
#include "msecp_dbg.h"
#include "msecp_cfg.h"
#include "msecp_udp.h"
#include "msecp_tmr.h"
#include "msecp_fsm.h"
//#include "msecp_stat.h"
#include "gassert.h"

/** \brief Typedef for Multi-drop ESCP comm imp data. */
typedef struct {
    char *upper_q;      /**< msecp's upper module's queue */
    char *pad_q;        /**< pad module's queue (ok to be NULL) */
} _msecp_imp_t;

/* Inbound messages */
static void _msecp_imp_process_txdata(buf_t b);
static void _msecp_imp_process_multicast(buf_t b);
static void _msecp_imp_process_timeout(buf_t b);
static void _msecp_imp_set_debug_level(buf_t b);
static void _msecp_imp_read_queue(int unused, void *unused_also);

/* Outbound messages */
void msecp_imp_send_txdata_ind(u_int8_t drop);
void msecp_imp_send_rxdata(u_int8_t drop, u_int8_t *data, int length);

/** \brief Holds the Multi-drop SECP IMP information. */
static _msecp_imp_t _msecp_imp;


/* ======================================================================= */
/**
 *  \brief Saves upper queue, creates local queue, registers functions to
 *      handle imp messages, and synchronizes with upper queue.
 *  \param local_q Local queue name.
 *  \param upper_q Upper module's queue name.
 *  \param pad_q Optional pad queue for the pad itself (primarily for resets)
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_imp_init(const char *local_q, const char *upper_q)
{
    info("msecp_imp_init local %s upper %s ", local_q,upper_q);

    assert(local_q && upper_q);

    _msecp_imp.upper_q = (char *)upper_q;

    mq_create((char *)local_q);

    /* Synch with upper module */
    info("msecp_imp_init(): Waiting for %s.", _msecp_imp.upper_q);
    mq_sync_with(_msecp_imp.upper_q);

    /* Register all inbound messages */
    msg_dispatch_proc("tx-data", _msecp_imp_process_txdata);
    msg_dispatch_proc("multicast-join", _msecp_imp_process_multicast);
    msg_dispatch_proc("timeout", _msecp_imp_process_timeout);
    msg_dispatch_proc("set-debug-level", _msecp_imp_set_debug_level);

    fd_add (mq_get_socket (), _msecp_imp_read_queue, 0);
    info("msecp_imp_init(): Done.");

} /* msecp_imp_init() */


/* ======================================================================= */
/**
 *  \brief Return's the upper module's queue name that msecp talks to.
 *  \return The upper module's queue name.
 */
/* ======================================================================= */
char *msecp_imp_get_upper_q(void)
{
    return (_msecp_imp.upper_q);
} /* msecp_imp_get_upper_q() */


/* ================ */
/* Inbound messages */
/* ================ */
/* ======================================================================= */
/**
 *  \brief Process a transmit data request from the upper queue.
 *      Format is: "tx-data drop client-name data"
 *  \param b The IMP message.
 */
/* ======================================================================= */
static void _msecp_imp_process_txdata(buf_t b)
{
    char *data, *client_id;
    int data_len, len;
    u_int8_t *drop;

    info("msecp process_txdata: ");

    /* Handle mandatory fields */
    data = msg_get_field(b, "data", &data_len);

    /* Check for optional fields we're concerned about */
    if ( (drop = (u_int8_t *)msg_get_field(b, "drop", &len)) != NULL )
      assert((len == 1) && (*drop < MSECP_CFG_MAX_DROPS));
    else /* default to drop 0 if not specified */
      *drop=0;

    assert( client_id = (u_int8_t *)msg_get_field(b, "client_name", &len));
    info("client =%s len %d", client_id,len);

    print_hex_data( buf_data(b), buf_length(b) >32 ? 64: buf_length(b));

    msecp_fsm_process_app_rqst(*drop, client_id, data, data_len);

} /* _msecp_imp_process_txdata() */


/* ======================================================================= */
/**
 *  \brief Process a Multicast join message.
 *      Format is: "multicast-join mcast-address mcast-port"
 *
 *  \param b The IMP message. Address is non-NULL terminated.

 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_imp_process_multicast(buf_t b)
{
    u_int16_t  *mcast_port;
    char *mcast_add = (char *)alloca(16);
    char *mcast_address;
    int len;

    info("msecp _msecp_imp_process_multicast: ");
    memset(mcast_add,0,16);
    mcast_address =  msg_get_field(b, "mcast-address", &len);

    memcpy(mcast_add,mcast_address,len);

    info("addr: %s  len :%d", mcast_add, len);
    mcast_port = (u_int16_t *) msg_get_field(b, "mcast-port", &len);
    info("len :%d",len);
    assert(mcast_address && mcast_port);

    print_hex_data(buf_data(b), 74);

    info("msecp process_multicast: mcast_address:%s   ;mcast_port:%d",
                                            mcast_add, *mcast_port);
  //  Process the multicast join messages, open the socket for listening.
  //  If its a new  Multicast address close the existing one and open
  //  the new one.
    info("call join");
    msecp_udp_multicast_join( mcast_add, mcast_port);

} /* _msecp_imp_process_multicast() */


/* ======================================================================= */
/**
 *  \brief Process a timeout message.  There is 1 timer per drop, response,
 *         with a unique id.
 *      Format is: "timeout id"
 *  \param b The IMP message.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_imp_process_timeout(buf_t b)
{
    u_int32_t *id;
    u_int8_t drop;
    u_int8_t type;
    int length;

    /* Get and validate the timer ID */
    assert(id = (u_int32_t *)msg_get_field(b, "id", &length));
    drop = msecp_tmr_get_drop(*id);
    type = msecp_tmr_get_type(*id);
    if ( !msecp_tmr_is_msecp_tmr(*id) || (drop >= MSECP_CFG_MAX_DROPS) )
    {
        warn("Bad timer ID: 0x%08X.", *id);
        return;
    }

    /* Which timer expired - response, keepalive, or disabled timer? */
    switch ( type )
    {
        case MSECP_TMR_RESPONSE:  msecp_fsm_process_proto_tmo(drop); break;
        default:
            warn("Bad timer type: 0x%08X.", *id);
            break;
    }

} /* _msecp_imp_process_timeout() */

/* ======================================================================= */
/**
 *  \brief Allows us to dynamically set a new debug level, while the program
 *      is running.  Format is: "set-debug-level level"
 *  \param b The buffer holding the debug level change message.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_imp_set_debug_level(buf_t b)
{
    char *field;
    unsigned int field_len;

    /* Verify message format */
    field = msg_get_field(b, "level", &field_len);
    assert(field && (field_len == 1));

    /* Message looks good - handle it */
    msecp_dbg_set_level((u_int8_t)*field);

} /* _msecp_imp_set_debug_level() */



/* ======================================================================= */
/**
 *  \brief Call back for fd_select.
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_imp_read_queue(int unused, void *unused_also)
{
    msg_dispatch(mq_read_without_restart());
} /* _msecp_imp_read_queue() */


/* ================= */
/* Outbound messages */
/* ================= */
/* ======================================================================= */
/**
 *  \brief Send transmit data indicator to upper queue.
 *      Format is: tx-data-ind [drop]
 *
 *  \param drop The drop address: 0-156 (157 reserved).
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_imp_send_txdata_ind(u_int8_t drop)
{
    buf_t b;
    char *queue;

    info("send_txdata_ind: ");

    assert((drop < MSECP_CFG_MAX_DROPS) );

    queue = _msecp_imp.upper_q;
    b = msg_create(queue, 0, "tx-data-ind");
    b = msg_add_field(b, "drop", &drop, 1);

    info("send_txdata_ind: drop:%x ",drop);
    mq_write(queue, b);

} /* msecp_imp_send_txdata_ind() */


/* ======================================================================= */
/**
 *  \brief Send requested and unsolicited data to upper queue.
 *      Format is: rx-data [drop] data
 *  \param drop The drop address: 0-156 (157 reserved), 255 for broadcast.
 *  \param data Received data.
 *  \param length Received data length.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_imp_send_rxdata(u_int8_t drop, u_int8_t *data, int length)
{
    buf_t b;

    info("send_rxdata: ");

    assert(((drop < MSECP_CFG_MAX_DROPS) || (drop == 255)) && data);

    b = msg_create(_msecp_imp.upper_q, 0, "rx-data");
    b = msg_add_field(b, "drop", &drop, 1);
    b = msg_add_field(b, "data", data, length);

    info("send_rxdata: drop:%d", drop);
    mq_write(_msecp_imp.upper_q, b);

} /* msecp_imp_send_rxdata() */

/*
 * End of "$Id: msecp_imp.c,v 1.2 2004/08/06 17:35:16 tmeiccvs Exp $".
 */
