/* ===[ $RCSfile: msecp_secure.c,v $ ]============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: msecp_secure.c,v 1.2 2004/08/06 17:41:00 tmeiccvs Exp $"
 *
 *  \brief SESC SSL module.
 *
 *  This code handles SSL ESC functions for an ES-PAD device.  The
 *  current device is the Cyclades TS100.  It is based on ESConnect v1.3.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <sys/fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include "libcom/buf.h"
#include "libcom/fd.h"
#include "libcom/mq.h"
#include "gassert.h"
#include "gdebug.h"
#include "msecp_dbg.h"
#include "msecp_secure.h"
#include "crypto.h"
#include "msecp_cfg.h"

struct secure *pSecure;
const char *kpath ="/etc/gtech";

/* =============== */
/* Local variables */
/* =============== */

/* ========================= */
/* Local function prototypes */
/* ========================= */

/* ======================================================================= */
/**
 *  \brief Initializes the MSECP security structures.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_secure_init(void)
{
   struct secure *psec = NULL;

   info("msecp_secure_init() ");

   pSecure = malloc(sizeof(struct secure));
   memset(pSecure,0,sizeof(struct secure));
   psec= pSecure;

   strcpy(psec->kpath, kpath);

   /* Reads from "etc/gtech/sec.cfg", if error returned,
      delete Server key file to force a SECP Config Request */
   if( esec_read_config(psec) == FAILURE )
       system("rm -f /etc/gtech/spub.dat");

   strcpy(psec->client_id, (char *)msecp_cfg_get_client_id());
   strcpy((char *) psec->hwid, (char *) msecp_cfg_get_hwid());

   info("1 %s   : %s ", psec->client_id, (char *)psec->hwid);

   info("open ssl ");
   OpenSSL_add_all_algorithms();
   info(" read keys");

   psec->gen_key_flag = KEYS_GENERATING;
   if( crypt_read_keys(psec) == -1)
   {
     psec->gen_key_flag = KEYS_GENERATING;
     assert(crypt_generate_keys(psec));
   }
   psec->gen_key_flag = KEYS_GENERATED;

   info("gen key");
   psec->spubkey_valid = INVALID;

   if( crypt_read_server_pub_key(psec) != -1)
     psec->spubkey_valid = VALID;       /* Key exists */

   info("spubkey = %d", psec->spubkey_valid);
}

/* ======================================================================= */
/**
 *  \brief Returns the pSecure struct.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
struct secure * get_secure_struct(void)
{
 return(pSecure);
}



/* ======================================================================= */
/**
 * \brief Decrypts the test data.
 *
 *  \param *psec    Secure struct.
 *  \param *ebuf    Encrypted buffer.
 *  \param enc_sz   Encrypted size.
 *  \return Test data size.
 */
/* ======================================================================= */
int get_test_data(struct secure *psec, unsigned char *ebuf,
                        unsigned short enc_sz, u_int8_t packet_details)
{
  short dec_sz=-1;

  memset(psec->tdata, 0, sizeof(psec->tdata));
  psec->tdata_sz = 0;

  info("get_test_data:");
  print_hex_data((char *)psec->t_prvkey,32);

#ifdef CHECK_PACK_DETAILS
  /*TODO check whether to use symmetric or asymetric decrypt via Packet Details.*/
   if(packet_details & PKT_DETAIL_ASYM_BIT )
   {   if ((dec_sz = crypt_asym_decrypt( psec->t_pubkey, ebuf, enc_sz, psec->tdata)) < 0)
       {
         return -1;
       }
   }else{
     if(packet_details & PKT_DETAIL_SYM_BIT)
     {
      if ((dec_sz = crypt_sym_decrypt(psec, ebuf, enc_sz, psec->tdata)) < 0)
      {
        return -1;
      }
    }
   }
#endif

  info("call crypt_asym_decrypt ");
  if((dec_sz = crypt_asym_decrypt(psec->t_prvkey,ebuf,enc_sz,psec->tdata)) <0)
  {
   info("error");
    return -1;
  }
  info("crypt_asym_decrypt ok");

  psec->tdata_sz = dec_sz;
  return dec_sz;
}


/* ======================================================================= */
/**
 *  \brief This function reads the esec configuration.
 *
 *  \param *psec    Secure struct.
 #
 *  \return SUCCESS or FAILURE.
 */
/* ======================================================================= */
int esec_read_config(struct secure *psec)
{
  int fd;
  char fname[512];
  int ret = FAILURE;
  struct secure_config *cfg;

  cfg = (struct secure_config *) psec;

  snprintf(fname, sizeof(fname), "%s/sec.cfg", psec->kpath);
  if ((fd = open(fname, O_RDONLY)) < 0)
    return ret;

  if (read(fd, cfg, sizeof(struct secure_config)) < sizeof(struct secure_config))
  {
     close (fd);
     return ret;
  }

  close (fd);
  return SUCCESS;
}


/* ======================================================================= */
/**
 *  \brief This function writes the esec configuration.
 *
 *  \param *psec    Secure struct.
 #
 *  \return SUCCESS or FAILURE.
 */
/* ======================================================================= */
int esec_write_config(struct secure *psec)
{
  int fd;
  char fname[512];
  int ret = FAILURE;
  struct secure_config *cfg;

  cfg = (struct secure_config *) psec;

  snprintf(fname, sizeof(fname), "%s/sec.cfg", psec->kpath);
  if ((fd = open(fname, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU)) < 0)
    return ret;

  if (write(fd, cfg, sizeof(struct secure_config)) < sizeof(struct secure_config))
  {
    close (fd);
    return ret;
  }
  close (fd);

  system("saveconf \n");

  return SUCCESS;
}


/* This function unwraps the symmetric key wrapped at the server. */
unsigned char *get_sym_key(struct secure * psec, unsigned char *wkey, unsigned short wkey_sz, unsigned short *ksz)
{
  unsigned char *key;
  int psz;

  key = malloc(RSA_size(psec->t_prvkey));

  if ((psz = crypt_asym_decrypt(psec->t_prvkey, wkey, wkey_sz, key)) < 0)
      return NULL;

  *ksz = psz;

  return key;
}



/*
 * End of "$Id: msecp_secure.c,v 1.2 2004/08/06 17:41:00 tmeiccvs Exp $".
 */

