#ifndef __MSECP_SECURE_H__
#define __MSECP_SECURE_H__

#include <openssl/evp.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/md5.h>
#include <openssl/rand.h>
#include <openssl/rsa.h>

/* Definitions based on SECP Protocol document Rev 1.0 */
#define SECP_PROTO_ID      0x1C
#define ESEC_CFG_REQ       0xD0
#define ESEC_CFG_RSP       0xD1
#define ESEC_STST_REQ      0xD2
#define ESEC_ACK_CMD_RSP   0xD9
#define ESEC_SESS_REQ      0xD4
#define ESEC_SESS_RSP      0xD5
#define ESEC_ETST_REQ      0xD6
#define ESEC_SEC_FAIL_REQ  0xD8
#define ESEC_APP_REQ       0xDA
#define ESEC_APP_RSP       0xDB

/* Error/Status Codes */
#define ESEC_STATUS_OK     0x00
#define ESEC_INVALID_PKEY  0x01
#define ESEC_STST_FAILED   0x02
#define ESEC_SESSION_FAIL  0x05
#define ESEC_PKEY_EXP      0x0C
#define ESEC_GENERAL_ERR   0x0E
#define ESEC_SESSION_ERR   0x0F
#define ESEC_COMMAND_RESP  0xFF

#define ESEC_UNSECURE_CMD  0x00

/* Config Word */
#define CONFIG_WORD_SYM_BIT         0x20
#define CONFIG_WORD_ASYM_BIT        0x02
#define CONFIG_WORD_SIGNATURE_BIT   0x04

/* Packet Details */
#define PKT_DETAIL_ASYM_BIT         0x02
#define PKT_DETAIL_SIGNATURE_BIT    0x04
#define PKT_DETAIL_SYM_BIT          0x20
#define PKT_DETAIL_UNSO_BIT         0x40


#define SECP_SEQ_NO_INITIAL         0      /**< Initial sequence # */

#define KEYS_GENERATING             0
#define KEYS_GENERATED              1

#define INVALID                     0      /**< psec->spubkey_valid  */
#define CFG_PENDING                 1
#define VALID                       2


/** Typedef for an SECP request header */
struct  secp_rqst_hdr{
    u_int8_t pid;             /**< Protocol ID: Always SECP_PROTOCOL_ID */
    u_int8_t packet_details;  /**< Packet type: type + sign + encryption type */
    u_int32_t crc_sum;        /**< CRC: CRC 32 checksum per ANSI X3.66, not icluding signature block */
    u_int8_t seq_no;          /**< Sequence number */
    u_int16_t packet_size;    /**< Packet Size */
   /**< Variable length client name include in Start of data */
}  __attribute__ ((packed));

typedef struct secp_rqst_hdr  secp_rqst_hdr_t;

/** Typedef for an SECP Response header */
struct  secp_resp_hdr{
    u_int8_t pid;             /**< Protocol ID: Always SECP_PROTOCOL_ID */
    u_int8_t packet_details;  /**< Packet type: type + sign + encryption type */
    u_int32_t crc_sum;        /**< CRC: CRC 32 checksum per ANSI X3.66, not icluding signature block */
    u_int8_t seq_no;          /**< Sequence number */
    u_int16_t packet_size;    /**< Packet Size */
    u_int8_t client_len;      /**< Target name length */
    /* variable length client name and variable length data follow */
    u_int8_t data[0];         /**< Start of data */
} __attribute__ ((packed));

typedef struct secp_resp_hdr  secp_resp_hdr_t;



#define ALG_DES     (0x01)
#define ALG_3DES    (0x02)
#define ALG_AES     (0x03)
#define ALG_RSA     (0x04)
#define ALG_DSA     (0x05)

#define ESEC_ENCRYPTION_REQUIRED  (5)

/* Valid flags for the contents of the secure structure. */
#define ESEC_TERM_PRV_KEY   (0x01)
#define ESEC_TERM_PUB_KEY   (0x02)
#define ESEC_SERV_PUB_KEY   (0x04)
#define ESEC_SECRET_KEY     (0x08)
#define ESEC_BROADCAST_KEY  (0x10)

#define BLOCK_SZ (8)
#define SYM_BLOCK_SZ(siz) (((siz) % (BLOCK_SZ)) ? (((siz) + (BLOCK_SZ)) / (BLOCK_SZ)) * (BLOCK_SZ) : (siz) )


/* ESPAD test */
#define FAILURE -1
#define SUCCESS 1

/* Security configuration information stored in a file. */
struct secure_config
{
  unsigned short sec_cfg;
  unsigned char  tdata_sz;
  unsigned char  tdata[128];
} __attribute__ ((packed));


struct secure
{
  unsigned short sec_cfg_word;
  unsigned char  tdata_sz;
  unsigned char  tdata[128];

  unsigned int   hwid[19];
  char           client_id[16];
  unsigned char  cid_sz;
  unsigned int   valid;          /* set in crypt_read_keys( ) */
  unsigned char  gen_key_flag;   /* Flag to allow crypt_generate_keys(psec) to be called once.*/
  unsigned char  spubkey_valid;
  unsigned char  kpath[256];

  RSA *t_pubkey;            /* Terminal Public Key */
  RSA *t_prvkey;            /* Terminal Private Key */
  RSA *s_pubkey;            /* Server Symmetric Key */

  unsigned char  bk_typ;    /* Broadcast Key Type like DES, 3DES, AES */
  unsigned char *bro_key;   /* Server Broadcast Key */
  unsigned char  bro_ksz;
  unsigned char  bro_iv[8];

  unsigned char  sk_typ;    /* Secret Key Type like DES, 3DES, AES */
  unsigned char *ses_key;   /* Secret Key */
  unsigned char  ses_ksz;
  unsigned char  ses_iv[8];

  EVP_CIPHER *sym_alg;
  unsigned char *sym_key;
  unsigned char *sym_iv;

} __attribute__ ((packed));

void msecp_secure_init(void);
struct secure *get_secure_struct(void);
int get_test_data(struct secure *psec, unsigned char *ebuf,
                        unsigned short enc_sz, u_int8_t packet_details);
int esec_read_config(struct secure *psec);
int esec_write_config(struct secure *psec);
unsigned char *get_sym_key(struct secure * psec,
                 unsigned char *wkey, unsigned short wkey_sz, unsigned short *ksz);

int crc32( int accum, u_int8_t *data_ptr, int offset, int length );

#endif
