/* ===[ $RCSfile: msecp_udp.c,v $ ]========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2004 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: msecp_udp.c,v 1.2 2004/08/06 17:36:05 tmeiccvs Exp $"
 *
 *  \brief Implements MSECP UDP interface.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <unistd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/fd.h"
#include "libcom/list.h"
#include "msecp_dbg.h"
#include "msecp_imp.h"
#include "msecp_cfg.h"
#include "msecp_fsm.h"
#include "msecp_udp.h"
#include "gassert.h"

/** Number of times to retry gethostbyname. */
#define _MSECP_UDP_GETHOSTBYNAME_RETRIES     1



static void _msecp_udp_ucast_read(int s, void *unused);
static void _msecp_udp_mcast_read(int s, void *unused);

/** \brief Typedef for Multi-drop SECP Unicast UDP information. */
typedef struct {
    struct sockaddr_in to;  /**< UDP address to send to.   */
    int sock;               /**< UDP socket. */
    list host_addr;         /**< List of host IP addresses */
    list cur_host_addr;     /**< Current host IP address */
 } _msecp_udp_unicast_t;

/** \brief Typedef for Multi-drop SECP Multicast UDP information. */
typedef struct {
    int sock;               /**< UDP socket. */
    char str[257];      /**< "SECP/UDP.multicast:www.xxx.yyy.zzz:ppppp" (42) */
} _msecp_udp_multicast_t;

/** \brief Typedef for all the Multi-drop SECP UDP information */
typedef struct {
    _msecp_udp_multicast_t multicast;                     /**< multicast data */
    _msecp_udp_unicast_t unicast[MSECP_CFG_MAX_DROPS];    /**< unicast data */
} _msecp_udp_t;

/** \brief Holds all the Multi-drop SECP UDP information. */
static _msecp_udp_t _msecp_udp;

/* ======================================================================= */
/**
 *  \brief Initializes the udp message handler for all drops.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_udp_init(void)
{
    int len_h1, len_h2;
    int drop;
    char t[256];

    memset(&_msecp_udp, 0, sizeof(_msecp_udp));

    /* Construct the host list */
    len_h2 = strlen(msecp_cfg_get_host2());
    len_h1 = strlen(msecp_cfg_get_host1());
    assert(len_h2 || len_h1);

    for ( drop = 0; drop < MSECP_CFG_MAX_DROPS; drop++ )
    {
        if ( len_h2 )
        {
            _msecp_udp.unicast[drop].host_addr =
                cons(msecp_cfg_get_host2(), _msecp_udp.unicast[drop].host_addr);
        }
        if ( len_h1 )
        {
            _msecp_udp.unicast[drop].host_addr =
                cons(msecp_cfg_get_host1(), _msecp_udp.unicast[drop].host_addr);
        }

        _msecp_udp.unicast[drop].cur_host_addr =
            _msecp_udp.unicast[drop].host_addr;
    }

    info("msecp_udp_init(): Done.");
} /* msecp_udp_init() */


/* ======================================================================= */
/**
 *  \brief Initializes the udp message handler for the specified drop.
 *      If drop is -1, initializes the multicast udp message handler.
 *  \param drop The drop whose udp message handler to initialize.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_udp_open(int drop)
{
    int s, optval = 1;
    struct sockaddr_in local;

    assert(drop < MSECP_CFG_MAX_DROPS);

    /* Shut it down if it already exists. */
    if ( _msecp_udp.unicast[drop].sock )
    {
        fd_remove(_msecp_udp.unicast[drop].sock);
        assert(close(_msecp_udp.unicast[drop].sock) != -1);
        _msecp_udp.unicast[drop].sock = 0;
    }

    /* Create the socket and bind it to the port designated for that drop */
    assert((s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) != -1);
    assert(setsockopt(s, SOL_SOCKET, SO_BSDCOMPAT, &optval, sizeof(int)) == 0);
    local.sin_family = AF_INET;
    local.sin_addr.s_addr = htonl(INADDR_ANY);
    local.sin_port = htons(msecp_cfg_get_port(drop));
    info("Local port: %d (0x%0X).", local.sin_port, local.sin_port);
    assert(bind(s, (struct sockaddr *)&local, sizeof(struct sockaddr_in)) != -1);

    /* Add the socket descriptor to our list for select() and to our info. */
    _msecp_udp.unicast[drop].sock = s;
    fd_add(s, _msecp_udp_ucast_read, 0);

} /* msecp_udp_open() */


/* ======================================================================= */
/**
 *  \brief Send a message to the host from this drop.
 *  \param drop The drop sending the message.
 *  \param b Message to send.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_udp_send(u_int8_t drop, buf_t b)
{
    int i;
    struct hostent *hip = 0;
    char t[256];

    assert((drop < MSECP_CFG_MAX_DROPS) && b);
    info("msecp_udpsend () Looking up host %s.",
                         (char *)car(_msecp_udp.unicast[drop].cur_host_addr));
    for ( i = 0; i < (_MSECP_UDP_GETHOSTBYNAME_RETRIES + 1); i++ )
    {
        if ( ((hip =
            gethostbyname((char *)car(_msecp_udp.unicast[drop].cur_host_addr))) == NULL)
            && (h_errno != TRY_AGAIN) )
        {
            warn("Gethostbyname Failed: %s.", hstrerror(h_errno));
        }
        else if ( h_errno != TRY_AGAIN )
            break;
    }
    if ( i == (_MSECP_UDP_GETHOSTBYNAME_RETRIES + 1) )
    {
        warn("Gethostbyname Retries Exceeded!");
        buf_free(b);
        return;
    }
    assert(hip);
    _msecp_udp.unicast[drop].to.sin_addr.s_addr =
        *((unsigned long *)hip->h_addr_list[0]);
    _msecp_udp.unicast[drop].to.sin_family = AF_INET;
    _msecp_udp.unicast[drop].to.sin_port = htons(MSECP_UDP_PORT);

    if ( WITH_RESTART(
        sendto(_msecp_udp.unicast[drop].sock, buf_data(b), buf_length(b),
            MSG_NOSIGNAL, (struct sockaddr *)&_msecp_udp.unicast[drop].to,
            sizeof(struct sockaddr_in))) == -1 )
    {
        warn("Sendto failed: %s.", strerror(errno));
        msecp_udp_open(drop);
    }

//    snprintf(t, sizeof(t), "Message=SendingTo;Address=%s",
//        inet_ntoa(_msecp_udp.unicast[drop].to.sin_addr));
//    t[sizeof(t)-1] = '\0';
//  TODO: If status required.  mesc_imp_send_status_msg(drop, t);
    info("UDP Message Sent To: %s.",
          inet_ntoa(_msecp_udp.unicast[drop].to.sin_addr));
    buf_free(b);

} /* msecp_udp_send() */


/* ======================================================================= */
/**
 *  \brief Send a message to the host from this drop.
 *  \param drop The drop who is about to change hosts.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_udp_use_next_host(u_int8_t drop)
{
    char t[256];

    /* Set the current host to the next host in the list */
    _msecp_udp.unicast[drop].cur_host_addr =
        cdr(_msecp_udp.unicast[drop].cur_host_addr);

    /* If we're at the end, set the current host to the 1st one in the list */
    if ( !_msecp_udp.unicast[drop].cur_host_addr )
    {
        _msecp_udp.unicast[drop].cur_host_addr =
            _msecp_udp.unicast[drop].host_addr;
    }
//   snprintf(t, sizeof(t), "Message=ConnectedTo;Address=%s",
//        (char *)car(_msecp_udp.unicast[drop].cur_host_addr));
//    t[sizeof(t)-1] = '\0';
//  TODO: If status required.  msecp_imp_send_status_msg(drop, t);
} /* msecp_udp_use_next_host() */


/* ======================================================================= */
/**
 *  \brief Get the multicast address string.
 *  \return The multicast address as a null-terminated string.
 */
/* ======================================================================= */
char *msecp_udp_get_multicast_str(void)
{
    return (_msecp_udp.multicast.str);
} /* msecp_udp_get_multicast_str() */

                            /* --------------- */
                            /* Local functions */
                            /* --------------- */
/* ======================================================================= */
/**
 *  \brief Call back for fd_select.
 *  \param s The socket, which we'll have to figure out from what drop.
 *  \param unused Unused argument required by fd_add().
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_udp_ucast_read(int s, void *unused)
{
    char temp[MAX_MESSAGE_LENGTH];
    struct sockaddr_in from;
    socklen_t from_length = sizeof(struct sockaddr_in);
    int length = 0;
    int drop;

    /* Which drop is it for? */
    for ( drop = 0; drop < MSECP_CFG_MAX_DROPS; drop++ )
        if ( _msecp_udp.unicast[drop].sock == s )
            break;
    assert(drop < MSECP_CFG_MAX_DROPS);

    if ( WITH_RESTART(length = recvfrom(s, temp, MAX_MESSAGE_LENGTH - 750,
        MSG_NOSIGNAL, (struct sockaddr *)&from, &from_length)) == -1 )
    {
        warn("Recvfrom(%d) failed: %s", drop, strerror(errno));
        msecp_udp_open(drop);
    }
    else
    {
        info("Recvfrom(%d) return %d bytes.", drop, length);
        buf_free(msecp_fsm_process_unicast(drop, buf_append(0, temp, length)));
    }

} /* _msecp_udp_ucast_read() */



/* ======================================================================= */
/**
 *  \brief Call back for fd_select.
 *  \param s The socket, which better match the multicast socket.
 *  \param unused Unused argument required by fd_add().
 *  \return Nothing.
 */
/* ======================================================================= */
static void _msecp_udp_mcast_read(int s, void *unused)
{
    char temp[MAX_MESSAGE_LENGTH];
    int length = 0;
    struct sockaddr_in from;
    socklen_t from_length = sizeof(struct sockaddr_in);

    /* Verify multicast socket */
    assert(_msecp_udp.multicast.sock == s);
    if ( WITH_RESTART(length = recvfrom(s, temp, MAX_MESSAGE_LENGTH - 750,
        MSG_NOSIGNAL, (struct sockaddr *)&from, &from_length)) == -1 )
    {
        warn("Recvfrom failed: %s", strerror(errno));
        return;
    }
    else
    {
        info("Recvfrom return %d bytes.", length);
// TODO_TM       buf_free(msecp_fsm_process_multicast(buf_append(0, temp, length)));
    }

} /* _msecp_udp_mcast_read() */



/* ======================================================================= */
/**
 *  \brief Creates a socket, joins the multicast group and binds to the
 *      specified port for accepting multicast datagrams.
 *  \param addr The IP address/host name to be joined.
 *  \param port The port to bind to for listening for multicast UDP datagrams.
 *  \return Nothing.
 */
/* ======================================================================= */
void msecp_udp_multicast_join(char *addr, u_int16_t *port)
{
    struct sockaddr_in sa;
    struct ip_mreq mreq;
    struct in_addr mcast_addr;
    struct hostent *hip;
    int sock;
    int optval = 1;
    u_int16_t port_num;
    struct ifreq ifrq;

    assert(addr && port);

    memset(&sa, 0, sizeof(sa));
    memset(&mreq, 0, sizeof(mreq));
    memset(&mcast_addr, 0, sizeof(mcast_addr));
    port_num = *port;

    if ( (hip = gethostbyname(addr)) == NULL )
    {
        warn("Gethostbyname Failed: %s.", hstrerror(h_errno));
        return;
    }

    mcast_addr.s_addr = ((struct in_addr *)hip->h_addr_list[0])->s_addr;
    if ( !IN_MULTICAST(ntohl(mcast_addr.s_addr)) )
    {
        warn("Invalid multicast address: %s.", addr);
        return;
    }

    /* Create the socket and bind it to the port specified */
    assert((sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) != -1);
    assert(setsockopt(sock, SOL_SOCKET, SO_BSDCOMPAT, &optval, sizeof(int)) == 0);
    assert(fcntl(sock, F_SETFL, O_NONBLOCK) != -1);
    sa.sin_family = AF_INET;
    sa.sin_addr.s_addr = htonl(INADDR_ANY);
    sa.sin_port = htons(port_num);
    assert(bind(sock, (struct sockaddr *)&sa, sizeof(struct sockaddr_in)) != -1);

    /* PAD needs this (PowerPC) but PC (Intel) hates it so don't assert(). */
    strcpy(ifrq.ifr_name, "eth0");
    ioctl(sock, SIOCGIFFLAGS, &ifrq);   /* assert(ioctl() != -1) */
    ifrq.ifr_flags |= IFF_ALLMULTI;
    ioctl(sock, SIOCSIFFLAGS, &ifrq);   /* assert(ioctl() != -1) */

    memcpy(&mreq.imr_multiaddr, &mcast_addr, sizeof(mcast_addr));
    mreq.imr_interface.s_addr = htonl(INADDR_ANY);
    if (setsockopt(sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq)) < 0)
    {
        warn("Joining Multicast Group %s  %d Failed\n", addr,port_num);
        close(sock);
        return;
    }

    /* Cleanup the old descriptor, if any, then add ours. */
    if ( _msecp_udp.multicast.sock )
    {
        fd_remove(_msecp_udp.multicast.sock);
        assert(close(_msecp_udp.multicast.sock) != -1);
    }
    _msecp_udp.multicast.sock = sock;
    fd_add(_msecp_udp.multicast.sock, _msecp_udp_mcast_read, 0);
    snprintf(_msecp_udp.multicast.str, sizeof(_msecp_udp.multicast.str),
              "SECP/UDP.multicast:%s:%d", addr, port_num);

    info("Joining Multicast Group %s successful\n", addr);
} /* msecp_udp_multicast_join() */

/*
 * End of $Id: msecp_udp.c,v 1.2 2004/08/06 17:36:05 tmeiccvs Exp $
 */
