/* ===[ $RCSfile: pad_imp.c,v $ ]=========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: pad_imp.c,v 1.2 2005/02/08 23:58:23 cmayncvs Exp $"
 *
 *  \brief Handle PAD Inter-Module Protocol (IMP) messages.
 */
/* ======================================================================= */

/* ============= */
/* Include Files */
/* ============= */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/reboot.h>
#include <linux/reboot.h>
#include <string.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/msg.h"
#include "libcom/stats.h"
#include "mesc/mesc_cfg.h"
#include "pad.h"
#include "pad_imp.h"
#include "gassert.h"

/* =========================== */
/* Local functions & variables */
/* =========================== */
static char *pad_imp_lower_q;   /**< The PAD's lower_q */

static void _pad_imp_default_proc(buf_t b, const char *file, int line);


/* ======================================================================= */
/**
 *  \brief Saves lower queue, creates pad's queue, registers functions
 *      to handle imp messages, and synchronizes with lower queue.
 *  \param local_q pad queue name.
 *  \param lower_q The lower queue name.
 *
 *  \return Nothing.
 */
/* ======================================================================= */
void pad_imp_init(const char *local_q, const char *lower_q)
{
    assert(local_q && lower_q);

    /* Take care of the queue's */
    pad_imp_lower_q = (char *)lower_q;
    mq_create((char *)local_q);

    /* Synch with lower q */
    info("imp_init(): Waiting for %s.", lower_q);
    mq_sync_with((char *)lower_q);

    /* Register messages */
    msg_dispatch_proc("unsolicited-data", pad_imp_rcv_unso);
    msg_default_dispatch_proc(_pad_imp_default_proc);
    info("imp_init(): Done.");

} /* pad_imp_init() */


/* ======================================================================= */
/**
 *  \brief Sends a Solicited Request IMP message to the PAD's lower q.
 *  \param data The buffer of data to send.
 *  \param size The amount of data to send.
 *  \param sap_id The sap ID.
 *  \param drop The PAD's "drop address".
 *
 *  \return Nothing.
 */
/* ======================================================================= */
void pad_imp_send_rqst(const char *data, size_t size, u_int8_t sap_id, u_int8_t drop)
{
    buf_t imp_buf;

    /* Build message */
    imp_buf = msg_create(pad_imp_lower_q, NULL, "solicited-request");
    imp_buf = msg_add_field(imp_buf, "sap", (char *)&sap_id, 1);
    imp_buf = msg_add_field(imp_buf, "drop", (char *)&drop, 1);
    if ( data && size )
        imp_buf = msg_add_field(imp_buf, "data", (char *)data, size);

    /* Send it */
    mq_write(pad_imp_lower_q, imp_buf);
    info("pad: Sent Request.");

} /* pad_imp_send_rqst() */


/* ======================================================================= */
/**
 *  \brief Receives an Unsolicited Data IMP message from the lower q.
 *  \param b The buffer containing the unso message.
 *      If the unso turns out to be a soft, hard, or ultimate reset, the PAD
 *      will reset after a short delay to allow the x42pp module time to send
 *      the unso to all of its drops, in case it's a broadcast reset.
 *  \return Nothing.
 */
/* ======================================================================= */
void pad_imp_rcv_unso(buf_t b)
{
    u_int8_t *field;
    unsigned int field_len;
    u_int8_t soft_reset[] = {0x30, 0x77};
    u_int8_t hard_reset[] = {0x30, 0x79};
    u_int8_t ultimate_reset[] = {0x30, 0x75};

    /* Verify message format - check drop, and data */
    field = (u_int8_t *)msg_get_field(b, "drop", &field_len);
    assert(field && (field_len == 1));
    if ( (*field != PAD_DROP_ADDRESS) && (*field != 255) )
    {
        warn("Invalid drop address %d.", *field);
        return;
    }
    field = (u_int8_t *)msg_get_field(b, "data", &field_len);
    assert(field);
    if ( field_len < 2 )
    {
        warn("Invalid data length %d.", field_len);
        return;
    }

    /* Reset the pad if the unso is a soft, hard, or ultimate reset. */
    if ( !memcmp(soft_reset, field, 2) || !memcmp(hard_reset, field, 2) ||
        !memcmp(ultimate_reset, field, 2) )
    {
        info("About to reboot ...");
        sleep(5);
        system("/sbin/reboot");
        exit(0);
    }

} /* pad_imp_rcv_unso() */

                        /* =============== */
                        /* LOCAL FUNCTIONS */
                        /* =============== */

/* ======================================================================= */
/**
 *  \brief Default message dispatch procedure.
 *  \param b The IMP message.
 *  \param file The file this function was called from.
 *  \param line The line number this function was called from.
 *
 *  This replaces the standard default procedure and prevents the unimplemented
 *  messages from being written to the debug log.
 */
/* ======================================================================= */
static void _pad_imp_default_proc(buf_t b, const char *file, int line)
{
} /* _pad_imp_default_proc() */


/*
 * End of "$Id: pad_imp.c,v 1.2 2005/02/08 23:58:23 cmayncvs Exp $".
 */


