/* ===[ $RCSfile: imp.c,v $ ]=============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: imp.c,v 1.2 2004/01/26 18:21:57 gspatcvs Exp $"
 *
 * \brief STTY IMP interface.
 *
 * STTY uses the serial messages defined for the Altura serial interface. A
 * new TX confirm message has been added.
 *
 */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "gassert.h"
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/msg.h"
#include "libcom/fd.h"
#include "line-monitor.h"
#include "serial.h"
#include "imp.h"
#include "stty_cfg.h"
#include "stty_debug.h"

/** Module debug level */
unsigned char stty_debug_level = WARN_LEVEL | INFO_LEVEL | TRACE_LEVEL | PRINT_LEVEL;

/** Upper queue name */
static char *upper_q;

/**
 * \brief Send TX Confirm.
 * \param port Port ID.
 */
void
imp_send_tx_cfm (char *port)
{
    buf_t b;

    b = msg_create (upper_q, 0, "serial-TX-confirm");
    b = msg_add_field (b, "portid", port, 0);
    mq_write (upper_q, b);
}

/**
 * \brief Send RX Data.
 * \param port Port ID.
 * \param data Pointer to data.
 * \param length Data length.
 */
void
imp_send_data (char *port, char *data, int length)
{
    buf_t b;

    b = msg_create (upper_q, 0, "serial-RX-data");
    b = msg_add_field (b, "portid", port, 0);
    b = msg_add_field (b, "data", data, length);
    mq_write (upper_q, b);
}

/**
 * \brief Process a serial control.
 * \param b Message buffer.
 */
static void
process_serial_control (buf_t b)
{
    char *portid;
    char *baud;
    int length;

    assert (portid = msg_get_field (b, "portid", &length));

    /* Make the default buad rate 19200 */
    if ((baud = msg_get_field (b, "baud-rate", &length)) == 0)
        baud = "19200";

    open_port (portid, baud);
}

/**
 * \brief Process tx request.
 * \param b Message buffer.
 */
static void
process_tx_request (buf_t b)
{
    char *data, *portid;
    int length;

    assert (portid = msg_get_field (b, "portid", &length));
    assert (data = msg_get_field (b, "data", &length));

    write_data (portid, data, length);
}


/**
 * \brief Call back for fd_select.
 */
static void
read_queue (int unused, void *unused_also)
{
    msg_dispatch (mq_read_without_restart ());
}

/**
 * \brief Initialize the IMP message handlers.
 * \param self Own queue name.
 * \param upper Upper queue name.
 */
static void
imp_init (char *self, char *upper)
{
    upper_q = upper;

    mq_create (self);
    //mq_sync_with (upper);

    msg_dispatch_proc ("serial-control", process_serial_control);
    msg_dispatch_proc ("serial-TX-data", process_tx_request);

    fd_add (mq_get_socket (), read_queue, 0);

    for (;;)
        fd_select ();
}

/**
 * \brief Main function.
 * If there are no command line arguements, I default to usefull queue names
 * for testing.
 */
int
main (int argc, char **argv)
{
    char *self, *upper;

    if (argc < 3)
    {
        self = "stty";
        upper = "x42pp";
    }
    else
    {
        self = argv[1];
        upper = argv[2];
    }

    stty_cfg_init ();
    lm_init ();
    imp_init (self, upper);

    return 0;
}

/*
 * End of "$Id: imp.c,v 1.2 2004/01/26 18:21:57 gspatcvs Exp $".
 */

