/* ===[ $RCSfile: serial.c,v $ ]=============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: serial.c,v 1.7 2005/07/11 21:04:25 cmayncvs Exp $"
 *
 * \brief TTY interface functions.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include "gassert.h"
#include "libcom/list.h"
#include "libcom/buf.h"
#include "libcom/fd.h"
#include "libcom/mq.h"
#include "stty_debug.h"
#include "line-monitor.h"
#include "imp.h"
#include "serial.h"
#include "stty_cfg.h"

#define RS232   0   /**< Serial port configuration for RS-232 */
#define RS485   11  /**< Serial port configuration for RS-485 */

/** Hash: key = port, value = file descriptor */
static list fd_list;
/** Hash: key = file descriptor, value = port */
static list port_list;

/**
 * \brief Lookup port from file descriptor.
 * \param fd File descriptor to lookup.
 * \return Port.
 */
static char *
get_port (int fd)
{
    list t;
    int eq (void *a, void *b) {return a == b;}

    if ((t = assoc ((void *)fd, port_list, eq)))
        return nth (1, t);
    return 0;
}

/**
 * \brief Lookup file descriptor from port.
 * \param port Port to lookup.
 * \return File descriptor.
 */
static int
get_fd (char *port)
{
    list t;

    if ((t = assoc (port, fd_list, 0)))
        return (int)nth (1, t);
    return 0;
}

/**
 * \brief Callback from fd_select for when socket becomes readable.
 * \param fd File descriptor.
 * \param unused Not used.
 */
static void
read_data (int fd, void *unused)
{
    char b[1024];
    int len;

    assert (WITH_RESTART (len = read (fd, b, sizeof (b))) != -1);
    imp_send_data (get_port (fd), b, len);
    lm_rx_data (b, len);
}

/**
 * \brief Convert baud rate string to termios define.
 * \param baud Baud rate string.
 * \return Termios define.
 */
static speed_t
baud_str_to_num (char *baud)
{
    if (strcmp (baud, "300") == 0) return B300;
    if (strcmp (baud, "1200") == 0) return B1200;
    if (strcmp (baud, "1800") == 0) return B1800;
    if (strcmp (baud, "2400") == 0) return B2400;
    if (strcmp (baud, "4800") == 0) return B4800;
    if (strcmp (baud, "9600") == 0) return B9600;
    if (strcmp (baud, "19200") == 0) return B19200;
    if (strcmp (baud, "38400") == 0) return B38400;
    if (strcmp (baud, "576000") == 0) return B57600;
    if (strcmp (baud, "115200") == 0) return B115200;
    return B19200;
}

/**
 * \brief Set serial port options.
 * \param fd File descriptor for the serial port.
 * \param baud Baud rate string.
 */
static void
set_options (int fd, char *baud)
{
    struct termios options;
    int mt;
    int MLines = 0x20000;

    ioctl(fd, TIOCSERCYC, &MLines);
    mt = stty_cfg_get_media ();
    if (mt == RS485)
    {
        MLines |= RS485;
        //ioctl(fd, TIOCSERCONFIG, &mt);
        info ("Media Type set to RS485");
    }
    else
    {
        MLines |= RS232;
        info ("Media Type set to RS232");
    }
    ioctl(fd, TIOCSERCYC, &MLines);

    /* Get current options */
    tcgetattr(fd, &options);
    /* Set baud rate */
    cfsetispeed(&options, baud_str_to_num (baud));
    cfsetospeed(&options, baud_str_to_num (baud));
    /* Set port into raw mode */
    options.c_iflag &= 0;
    //options.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);
    options.c_lflag = 0;
    //options.c_oflag &= ~OPOST;
    options.c_oflag = 0;
    options.c_cflag |= CLOCAL;
    options.c_cflag &= ~CRTSCTS;
    /* Write options */
    tcsetattr(fd, TCSANOW, &options);
}

/**
 * \brief Convert port id to file name.
 * \param port Port ID.
 * \return File name.
 */
static char *
get_port_name (char *port)
{
    if (strcmp (port, "port1") == 0) return "/dev/ttyS0";
    return "/dev/ttyS0";
}

/**
 * \brief Open serial port.
 * \param port Port ID.
 * \param baud Baud rate string.
 */
void
open_port (char *port, char *baud)
{
    int fd;

    assert (fd = open(get_port_name (port), O_RDWR | O_NOCTTY | O_NDELAY));
    set_options (fd, baud);
    fd_add (fd, read_data, 0);

    /* Save fd and port in hashes. */
    fd_list = acons (strdup (port), (void *)fd, fd_list);
    port_list = acons ((void *)fd, strdup (port), port_list);
}

/**
 * \brief Write serial data.
 * \param port Port ID.
 * \param data Pointer to data.
 * \param length Data length.
 *
 * Data must be completely sent form serial port before a
 * TX confirm is sent.
 */
void
write_data (char *port, char *data, int length)
{
    int rt, drt;
    int i = 0;
    int slen = length;

    while (length)
    {
        WITH_RESTART (rt = write (get_fd (port), &data[i], length));
        assert (rt != -1 || errno == EIO);
        if (errno == EIO)
        {
            /* This error occurs when the cable is disconnected, sometimes. */
            warn ("EIO error while writing");
            break;
        }
        /* Wait for data to be written from the serial port */
        WITH_RESTART (drt = tcdrain (get_fd (port)));
        assert (drt != -1 || errno == EIO);
        if (errno == EIO)
        {
            /* This error occurs when the cable is disconnected, sometimes. */
            warn ("EIO error while waiting for write to complete");
            break;
        }
        length -= rt;
        i += rt;
    }

    imp_send_tx_cfm (port);
    lm_tx_data (data, slen);
}


/********************************TEST******************************************/
#if 0

void
imp_send_data (char *port, char *data, int length)
{
    int i;

    printf ("Received on: %s\n", port);
    for (i = 0; i < length; i++)
        printf ("%c", data[i]);
    printf ("\n");
}

void
imp_send_tx_cfm (char *port)
{
    printf ("Tx Confirmed for: %s\n", port);
}

int
main (void)
{
    char *p = "port1";
    int fd;
    FILE *f;
    int i;

    //assert ((fd = open ("/dev/lm", O_RDWR | O_NONBLOCK)) != -1);
    //assert (f = fdopen (fd, "w"));
    //for (i = 0; i < 1030; i++){
    //if (fprintf (f, "File test %d\n", i) < 11)
      //  printf ("Write error: %d\n", i);

    //if (write (fd, "test", 4) < 4)
      //  printf ("Write error\n");

    //fflush(f);
    //}

    lm_init ();
    open_port (p, "9600");
    write_data (p, "test", 4);
    for (;;)
        fd_select ();
}

#endif

/*
 * End of "$Id: serial.c,v 1.7 2005/07/11 21:04:25 cmayncvs Exp $".
 */

