/* ===[ $RCSfile: x42pp_imp.c,v $ ]==========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: x42pp_imp.c,v 1.4 2005/02/09 17:35:27 cmayncvs Exp $"
 *
 * \brief X42pp IMP interface.
 *
 * X42pp takes the place of Devcom in the standard modules. X42pp sends and
 * receives data as if it was an application. An optional IMP field, "drop",
 * has been added to support multiple terminals.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "libcom/buf.h"
#include "libcom/mq.h"
#include "libcom/msg.h"
#include "libcom/fd.h"
#include "x42pp_types.h"
#include "x42pp_rx.h"
#include "x42pp_tx.h"

/** Saved upper queue name */
static char *upper_q;
/** Saved lower queue name */
static char *lower_q;
/** Saved port ID */
static char *portid;

/**
 * \brief Send serial data.
 * \param frame Buffer to send to serial port.
 */
void
x42pp_imp_send_frame (buf_t frame)
{
    buf_t b;

    b = msg_create (lower_q, 0, "serial-TX-data");
    b = msg_add_field (b, "portid", portid, 0);
    b = msg_add_field (b, "data", buf_data (frame), buf_length (frame));
    mq_write (lower_q, b);
}

/**
 * \brief Send solicited request to MESC.
 * \param drop Drop request is from.
 * \param data Pointer to request.
 * \param length Length of request.
 */
void
x42pp_imp_sreq (ubyte_1 drop, char *data, int length)
{
    buf_t b;
    char sap = 0;

    b = msg_create (upper_q, 0, "solicited-request");
    b = msg_add_field (b, "sap", &sap, 1);
    b = msg_add_field (b, "drop", &drop, 1);
    if (length)
        b = msg_add_field (b, "data", data, length);
    mq_write (upper_q, b);
}

/**
 * \brief Send empty solicited request.
 * \param drop Drop request from.
 */
void
x42pp_imp_send_empty (ubyte_1 drop)
{
    x42pp_imp_sreq (drop, 0, 0);
}

/**
 * \brief Send drop status to MESC (Polling or Slow Polling).
 * \param drop Drop status is for.
 * \param status "polling" or "slow-polling".
 */
void
x42pp_imp_send_drop_status (ubyte_1 drop, char *status)
{
    buf_t b;

    b = msg_create (upper_q, 0, "x42-drop-status");
    b = msg_add_field (b, "drop", &drop, 1);
    b = msg_add_field (b, "status", status, 0);
     mq_write (upper_q, b);
}

/**
 * \brief Process timeout.
 * \param b The IMP message.
 */
static void
process_timeout (buf_t b)
{
    unsigned int *id;
    int length;

    assert (id = (unsigned int *)msg_get_field (b, "id", &length));
    switch ((*id) & 0xFF00)
    {
        case X42PP_RX:
            x42pp_rx_poll_tmo ((ubyte_1)(*id & 0xFF));
            break;
        case X42PP_TX:
            x42pp_tx_idle_tmo ();
            break;
    }
}

/**
 * \brief Process Solicited Response.
 * \param b The IMP message.
 */
static void
process_srsp (buf_t b)
{
    char *data;
    int length;
    unsigned char *drop;
    buf_t msg;

    assert (drop = msg_get_field (b, "drop", &length));
    assert (data = msg_get_field (b, "data", &length));

    msg = buf_append (0, data, length);
    x42pp_tx_req (msg, *drop, SAD);
    buf_free (msg);
}

/**
 * \brief Process Unsolicited data.
 * \param b The IMP message.
 *
 * Drop 255 is a broadcast unso.
 */
static void
process_unso (buf_t b)
{
    char *data;
    int length;
    unsigned char *drop;
    buf_t msg;

    assert (drop = msg_get_field (b, "drop", &length));
    assert (data = msg_get_field (b, "data", &length));

    msg = buf_append (0, data, length);
    x42pp_tx_req (msg, *drop, *drop == 255 ? BROC : UAD);
    buf_free (msg);
}

/**
 * \brief Process Response Timeout
 * \param b The IMP message.
 */
static void
process_rsp_tmo (buf_t b)
{
    int length;
    unsigned char *drop;

    assert (drop = msg_get_field (b, "drop", &length));
    x42pp_tx_rsp_tmo (*drop);
}

/**
 * \brief Process Comm Status message.
 * \param b The IMP message.
 *
 * Drop 255 disables or enables all drops.
 */
static void
process_comm_status (buf_t b)
{
    int length;
    unsigned char *status;
    unsigned char *drop;

    assert (status = msg_get_field (b, "status-update", &length));
    assert (drop = msg_get_field (b, "drop", &length));

    if (*status == 0 || *status == 1)
        x42pp_tx_disable (*drop);
    else
        x42pp_tx_enable (*drop);
}

/**
 * \brief Process received serial data.
 * \param b The IMP message.
 */
static void
process_rx_data (buf_t b)
{
    char *data;
    int length;

    assert (data = msg_get_field (b, "data", &length));
    x42pp_rx_ind (data, length);
}

/**
 * \brief Process serial transmit confirm.
 * \param b The IMP message.
 */
static void
process_tx_cfm (buf_t b)
{
    x42pp_tx_cfm ();
}

/**
 * \brief Open serial port.
 * \param baud_rate Baud rate.
 */
static void
open_port (char *baud_rate)
{
    buf_t b;

    b = msg_create (lower_q, 0, "serial-control");
    b = msg_add_field (b, "portid", portid, 0);
    b = msg_add_field (b, "baud-rate", baud_rate, 0);
    mq_write (lower_q, b);
}

/**
 * \brief Default message dispatch procedure.
 * \param b The IMP message.
 * \param file The file this function was called from.
 * \param line The line number this function was called from.
 *
 * This replaces the standard default procedure and prevents the unimplemented
 * messages from being written to the debug log.
 */
static void
default_proc (buf_t b, const char *file, int line)
{
}

/**
 * \brief Initalize IMP message handlers.
 * \param self Own queue name.
 * \param upper Upper queue name.
 * \param lower Lower queue name.
 * \param port The serial port.
 * \param baud_rate The baud rate.
 */
void
x42pp_imp_init (char *self, char *upper, char *lower, char *port, char *baud_rate)
{
    upper_q = upper;
    lower_q = lower;
    portid = port;

    mq_create (self);
    mq_sync_with (lower);
    mq_sync_with (upper);

    msg_dispatch_proc ("solicited-response", process_srsp);
    msg_dispatch_proc ("unsolicited-data", process_unso);
    msg_dispatch_proc ("response-timeout", process_rsp_tmo);
    msg_dispatch_proc ("comm-status", process_comm_status);
    msg_dispatch_proc ("serial-RX-data", process_rx_data);
    msg_dispatch_proc ("serial-TX-confirm", process_tx_cfm);
    msg_dispatch_proc ("timeout", process_timeout);
    //msg_dispatch_proc ("set-debug-level", process_set_debug_level);
    msg_default_dispatch_proc (default_proc);

    open_port (baud_rate);
    x42pp_tx_poll_tmo ();

    for (;;)
        msg_dispatch (mq_read ());
}

/*
 * End of "$Id: x42pp_imp.c,v 1.4 2005/02/09 17:35:27 cmayncvs Exp $".
 */

