/* ===[ $RCSfile: x42pp_rx.c,v $ ]===========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: x42pp_rx.c,v 1.5 2005/02/09 23:27:36 cmayncvs Exp $"
 *
 * \brief X42pp Recieve State Machine.
 *
 * The RX state machine is started by the TX state machine after all data is
 * finished being transmitted by the serial port. The poll timer is restarted
 * after every valid character received.
 *
 */
#include "libcom/buf.h"
#include "libcom/timer.h"
#include "x42pp_types.h"
#include "x42pp_debug.h"
#include "x42pp_utils.h"
#include "x42pp_tx.h"
#include "x42pp_imp.h"
#include "x42pp_rx.h"

/** Maximum size of a received X42 frame. */
#define MAX_RX_DATA     2048

/* States */
#define READY       0   /**< Ready state */
#define ADDR_BYTE   1   /**< Address byte handling state */
#define ETX_BYTE    2   /**< ETX byte handling state */
#define LRC_BYTE    3   /**< Longitudinal Redundancy Check (LRC) handling state */
#define DRC_BYTE    4   /**< Diagonal Redundancy Check (DRC) handling state */
#define NUM_STATES  5   /**< The number of states */

/** State machine */
static void (*matrix[NUM_STATES]) (ubyte_1);

/* Timer macros */
/** \brief Starts a timer */
#define start_timer(t,m,d) do{ if (m) { timer_stop (t); \
                               t = timer_start (m, X42PP_RX | ((d) & 0xFF)); }}while(0)
/** \brief Stops a timer */
#define stop_timer(t) do{ t = timer_stop (t); }while(0)

/* Action prototypes */
static void
no_action (ubyte_1 not_used);
static void
check_addr (ubyte_1 data);
static void
rx_data (ubyte_1 data);
static void
save_lrc (ubyte_1 data);
static void
check_fcs (ubyte_1 data);

/* Helper function prototypes */
static void
dispatch (ubyte_1 data);
static void
set_state (ubyte_1 new_state);

/** Current State Machine state. */
static struct
{
    int         state;                  ///< Current state.

    ubyte_1     address;                ///< Current Address.
    ubyte_1     type;                   ///< Current type: poll, solicited, unso.

    ubyte_1     data[MAX_RX_DATA];      ///< Received data buffer.
    ubyte_2     index;                  ///< Buffer length.

    int         compliment_next;        ///< Flag: compliment next byte.

    ubyte_1     clrc;                   ///< Calculated LRC.
    ubyte_1     cdrc;                   ///< Calculated DRC.
    ubyte_1     rlrc;                   ///< Received LRC.

    ubyte_4     poll_tmo;               ///< Poll timeout value.
    timer       poll_timer;             ///< Poll timer.
} state;

/* External Functions */

/**
 * \brief Initialized state machine.
 * \param poll_tmo Poll timeout in milliseconds.
 */
void
x42pp_rx_init (ubyte_4 poll_tmo)
{
    matrix [READY]      = no_action;
    matrix [ADDR_BYTE]  = check_addr;
    matrix [ETX_BYTE]   = rx_data;
    matrix [LRC_BYTE]   = save_lrc;
    matrix [DRC_BYTE]   = check_fcs;

    set_state (READY);
    state.index = 0;
    state.poll_tmo = poll_tmo;
}

/**
 * \brief Reset state machine and begin searching for first byte.
 * \param address Current address.
 * \param type Current type: poll, solicited, unos.
 */
void
x42pp_rx_reset (ubyte_1 address, ubyte_1 type)
{
    set_state (ADDR_BYTE);
    state.address = address;
    state.type = type;
    state.index = 0;
    state.compliment_next = 0;
    state.clrc = 0;
    state.cdrc = 0;
    start_timer (state.poll_timer, state.poll_tmo, state.address);
}

/**
 * \brief Drive state machine with received data.
 * \param data Pointer to data buffer.
 * \param length Length of buffer.
 */
void
x42pp_rx_ind (ubyte_1 *data, ubyte_2 length)
{
    ubyte_2 i;

    for (i = 0; i < length; i++)
        dispatch (data[i]);
}

/**
 * \brief Handle poll timeout.
 * \param drop Drop timeout is associated with.
 */
void
x42pp_rx_poll_tmo (ubyte_1 drop)
{
    if (drop == state.address)
    {
        set_state (READY);
        x42pp_tx_poll_tmo ();
    }
}

/* Actions */

/**
 * \brief Do nothing.
 * \param not_used Not used.
 */
static void
no_action (ubyte_1 not_used)
{
}

/**
 * \brief Look for address byte.
 * \param data Received character.
 */
static void
check_addr (ubyte_1 data)
{
    ubyte_1 acb, cacb, mdata;

    if (state.address < 30)
    {
        acb = make_acb (state.address, state.type);
        cacb = ((ubyte_1)(~acb)) & 0x7F;
    }
    else
    {
        acb = make_eacb (state.address);
        cacb = ((ubyte_1)(~acb)) | 0x40;
    }

    /* Mask off or set old GSC encryption bit. */
    if (state.address < 30)
        mdata = data & 0x7F;
    else
        mdata = data | 0x40;

    if (acb == mdata)
    {
        stop_timer (state.poll_timer);
        update_fcs (data, &state.clrc, &state.cdrc);

        if (state.type == PAD)
        {
            /* Terminal is sending a message. */
            start_timer (state.poll_timer, state.poll_tmo, state.address);
            set_state (ETX_BYTE);
        }
        else
        {
            /* Terminal acknowledged a message sent to it. */
            set_state (READY);
            x42pp_tx_ack_ind ();
        }
    }
    if (mdata == cacb)
    {
        /* Terminal has nothing to send or did not correctly receive a
         * message sent to it. */
        stop_timer (state.poll_timer);
        set_state (READY);
        x42pp_tx_nack_ind ();
    }
}

/**
 * \brief Look for ETX.
 * \param data Received character.
 */
static void
rx_data (ubyte_1 data)
{
    stop_timer (state.poll_timer);
    start_timer (state.poll_timer, state.poll_tmo, state.address);

    if (state.compliment_next)
    {
        state.compliment_next = 0;
        state.data[state.index++] = ~data;
        update_fcs (~data, &state.clrc, &state.cdrc);
    }
    else
    {
        if (data == CNB)
        {
            /* LRC/DRC is caluclated before escaping the frame, so we don't
             * update the LRC/DRC when a CNB is received. */
            state.compliment_next = 1;
        }
        else if (data == ETX)
        {
            set_state (LRC_BYTE);
            update_fcs (data, &state.clrc, &state.cdrc);
        }
        else
        {
            state.data[state.index++] = data;
            update_fcs (data, &state.clrc, &state.cdrc);
        }
    }
}

/**
 * \brief Save off LRC.
 * \param data Received character.
 *
 * X42 is half-duplex protocol. We don't check the LRC until we have received
 * the DRC also. We don't want to send any before the terminal has finished
 * talking.
 */
static void
save_lrc (ubyte_1 data)
{
    stop_timer (state.poll_timer);
    start_timer (state.poll_timer, state.poll_tmo, state.address);

    if (state.compliment_next)
    {
        state.rlrc = ~data;
        state.compliment_next = 0;
        set_state (DRC_BYTE);
    }
    else
    {
        if (data == CNB)
        {
            state.compliment_next = 1;
        }
        else
        {
            state.rlrc = data;
            set_state (DRC_BYTE);
        }
    }
}

/**
 * \brief Get the DRC and check the LRC/DRC.
 * \param data Received character.
 */
static void
check_fcs (ubyte_1 data)
{
    ubyte_1 rdrc;

    stop_timer (state.poll_timer);

    if (state.compliment_next)
    {
        rdrc = ~data;
    }
    else
    {
        if (data == CNB)
        {
            state.compliment_next = 1;
            start_timer (state.poll_timer, state.poll_tmo, state.address);
            return;
        }
        else
        {
            rdrc = data;
        }
    }
    if (state.clrc == state.rlrc && state.cdrc == rdrc)
    {
        info ("Drop %d: Rx Frame: Length = %d", state.address, state.index);
        print_hex_data (state.data, state.index > 32 ? 32 : state.index);
        x42pp_imp_sreq (state.address, state.data, state.index);
        x42pp_tx_set_pending ();
        x42pp_tx_ack_ind ();
    }
    else
    {
        warn ("Drop %d: Bad FCS: CLRC -> %u RLRC -> %u CDRC-> %u RDRC -> %u", state.address, state.clrc,
            state.rlrc, state.cdrc, rdrc);
        x42pp_tx_fcs_error ();
    }
}

/* Helper functions */

/**
 * \brief Call state machine with RX data.
 * \param data Received character.
 */
static void
dispatch (ubyte_1 data)
{
    //trace ("Dispatch: %d <- %2.2X", state.state, data);
    matrix [state.state] (data);
}

/**
 * \brief Set new state.
 * \param new_state The new state.
 */
static void
set_state (ubyte_1 new_state)
{
    state.state = new_state;
}

/*
 * End of "$Id: x42pp_rx.c,v 1.5 2005/02/09 23:27:36 cmayncvs Exp $".
 */

