/* ===[ $RCSfile: x42pp_tx.c,v $ ]===========================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003-2005 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: x42pp_tx.c,v 1.3 2005/07/18 19:01:48 cmayncvs Exp $"
 *
 * \brief X42pp Transmit State Machine.
 *
 */
#include "libcom/buf.h"
#include "libcom/timer.h"
#include "x42pp_types.h"
#include "x42pp_debug.h"
#include "x42pp_utils.h"
#include "x42pp_imp.h"
#include "x42pp_status.h"
#include "x42pp_rx.h"
#include "x42pp_tx.h"

/* Tx States */
#define IDLE        0   /**< Idle state */
#define POLLING     1   /**< Polling state */
#define DATA        2   /**< Data state */
#define NUM_STATES  3   /**< Number of states */

/* Terminal States */
#define DISABLED    0   /**< Drop is in disabled state */
#define POLLING     1   /**< Drop is responding to polls */
#define SLOW_POLL   2   /**< Drop is in slow polll state */
#define PENDING     3   /**< Drop has a pending transaction */
#define NOT_CFG     4   /**< Drop is not configured */

/** TX state machine */
static void (*matrix[NUM_STATES]) (ubyte_1);

/* Timer macros */
/** \brief Starts a timer. */
#define start_timer(t,m) do{ if (m) { timer_stop (t); \
                               t = timer_start (m, X42PP_TX); }}while(0)
/** \brief Stops a timer. */
#define stop_timer(t) do{ t = timer_stop (t); }while(0)

/* Actions */
static int
all_pending (void);
static int
next_drop_to_poll (void);
static void
poll (ubyte_1 link);
static void
retry_data (ubyte_1 link);
static void
send_idle_link (ubyte_1 link);
static void
send_poll (ubyte_1 link);
static void
send_frame (buf_t b, ubyte_1 link);

/* Helpers */
static void
dispatch (ubyte_1 link);

/** Current TX state machine state. */
static struct _x42pp_tx_state
{
    int     state;              ///< Current state.

    buf_q   solicited;          ///< Solicited message queue.
    buf_q   unsolicited;        ///< Unsolicited message queue.
    buf_q   broadcast;          ///< Broadcast message queue.

    ubyte_1 curr_drop;          ///< Current drop.
    ubyte_1 curr_type;          ///< Current type: poll, solicited, unso.
    int     tries;              ///< Number of tried to same drop.
    buf_t   curr_frame;         ///< Last frame sent.

    ubyte_4 poll_tmo;           ///< Poll and Idle timeout value.
    timer   idle_timer;         ///< Idle timer.

    /** Drop state. */
    struct
    {
        int     state;          ///< Current state.
        int     no_rsp_cnt;     ///< Number of times drop hasn't responded.
        int     skip_cnt;       ///< Number of times the drop has been skipped.
        int     send_empty;     ///< Send empty solicited request.
    } drops[MAX_DROPS];
} state;

/* External Functions */

/**
 * \brief Initialize TX state machine.
 * \param drop_list Drops to poll.
 * \param length The length of the poll list.
 * \param poll_tmo Poll timeout.
 */
void
x42pp_tx_init (char *drop_list, int length, ubyte_4 poll_tmo)
{
    int i;

    matrix [IDLE]    = poll;
    matrix [POLLING] = poll;
    matrix [DATA]    = retry_data;

    for (i = 0; i < MAX_DROPS; i++)
    {
        state.drops[i].send_empty = 1;
        if (i < length && drop_list[i])
        {
            x42pp_status_polling (i);
            state.drops[i].state = POLLING;
        }
        else
            state.drops[i].state = NOT_CFG;
    }

    state.poll_tmo = poll_tmo;
}

/**
 * \brief Queue message from MESC.
 * \param msg Message from MESC.
 * \param drop Drop message is for.
 * \param type Message type.
 */
void
x42pp_tx_req (buf_t msg, ubyte_1 drop, ubyte_1 type)
{
    buf_t frame;

    if (drop < MAX_DROPS && state.drops [drop].state == NOT_CFG)
    {
        warn ("Tx Request for unconfigured Drop: %d", drop);
        return;
    }

    frame = make_frame (msg, drop, type);
    /* Save the drop address with the frame */
    buf_set_tag (frame, drop);

    if (drop != 255)
    {
        info ("Tx Request: %s for Drop %u",
            type == SAD ? "Solicited" : "Unsolicited", drop);
    }
    else
    {
        info ("Tx Request: Broadcast");
    }
    print_hex_data (buf_data (msg), buf_length (msg) > 32 ? 32 : buf_length (msg));

    if (type == SAD)
    {
        if (state.drops [drop].state == PENDING)
        {
            x42pp_status_polling (drop);
            state.drops [drop].state = POLLING;
        }
        buf_queue (state.solicited, frame);
    }
    if (type == UAD)
        buf_queue (state.unsolicited, frame);
    if (type == BROC)
        buf_queue (state.broadcast, frame);
}

/**
 * \brief Handle a poll timeout.
 */
void
x42pp_tx_poll_tmo (void)
{
    if (state.curr_drop < MAX_DROPS)
    {
        if (++state.drops [state.curr_drop].no_rsp_cnt == 3)
        {
             state.drops [state.curr_drop].no_rsp_cnt = 0;

            if (state.drops [state.curr_drop].state != DISABLED &&
                state.drops [state.curr_drop].state != PENDING)
            {
                if (state.drops [state.curr_drop].state != SLOW_POLL)
                    warn ("Drop %u put in Slow Poll", state.curr_drop);
                state.drops [state.curr_drop].state = SLOW_POLL;
                x42pp_status_slow_poll (state.curr_drop);
            }
        }
    }
    /* Poll next drop */
    dispatch (LINK);
}

/**
 * \brief Positive acknowledgement from a terminal.
 */
void
x42pp_tx_ack_ind (void)
{
    state.drops [state.curr_drop].no_rsp_cnt = 0;
    if (state.drops [state.curr_drop].state == SLOW_POLL)
    {
        x42pp_status_polling (state.curr_drop);
        info ("Drop %u taken out of Slow Poll", state.curr_drop);
        state.drops [state.curr_drop].state = POLLING;
    }
    if (state.drops [state.curr_drop].send_empty)
    {
        state.drops [state.curr_drop].send_empty = 0;
        if (state.drops [state.curr_drop].state != PENDING)
            x42pp_imp_send_empty (state.curr_drop);
    }
    if (state.state == DATA)
        state.state = POLLING;
    dispatch (ACKN);
}

/**
 * \brief Negative acknowledgement from a terminal
 */
void
x42pp_tx_nack_ind (void)
{
    state.drops [state.curr_drop].no_rsp_cnt = 0;
    if (state.drops [state.curr_drop].state == SLOW_POLL)
    {
        x42pp_status_polling (state.curr_drop);
        info ("Drop %u taken out of Slow Poll", state.curr_drop);
        state.drops [state.curr_drop].state = POLLING;
    }
    if (state.drops [state.curr_drop].send_empty)
    {
        state.drops [state.curr_drop].send_empty = 0;
        x42pp_imp_send_empty (state.curr_drop);
    }
    dispatch (ACKN);
}

/**
 * \brief LRC/DRC received from was wrong.
 */
void
x42pp_tx_fcs_error (void)
{
    state.drops [state.curr_drop].no_rsp_cnt = 0;
    if (state.drops [state.curr_drop].state != DISABLED)
    {
        x42pp_status_polling (state.curr_drop);
        if (state.drops [state.curr_drop].state == SLOW_POLL)
            info ("Drop %u taken out of Slow Poll", state.curr_drop);
        state.drops [state.curr_drop].state = POLLING;
    }
    dispatch (LINK);
}

/**
 * \brief All data sent from the serial port.
 */
void
x42pp_tx_cfm (void)
{
    if (state.state == IDLE)
        start_timer (state.idle_timer, state.poll_tmo);
    else if (state.state == DATA && state.curr_drop == 255)
    {
        state.state = POLLING;
        dispatch (LINK);
    }
    else
        x42pp_rx_reset (state.curr_drop, state.curr_type);
}

/**
 * \brief Idle timeout.
 */
void
x42pp_tx_idle_tmo (void)
{
    if (state.state == IDLE)
        dispatch (LINK);
}

/**
 * \brief Set drop pending.
 */
void
x42pp_tx_set_pending (void)
{
    x42pp_status_pending (state.curr_drop);
    state.drops [state.curr_drop].state = PENDING;
}

/**
 * \brief Response timeout from MESC.
 * \param drop Drop that timed out.
 */
void
x42pp_tx_rsp_tmo (ubyte_1 drop)
{
    warn ("Response timeout for Drop %u", drop);
    if (state.drops[drop].state == PENDING)
    {
        x42pp_status_polling (drop);
        state.drops[drop].state = POLLING;
    }
}

/**
 * \brief Disable a drop.
 * \param drop Drop to disable.
 */
void
x42pp_tx_disable (ubyte_1 drop)
{
    int i;

    if (drop < MAX_DROPS && state.drops [drop].state != NOT_CFG)
    {
        info ("Disabling Drop %u", drop);
        state.drops [drop].state = DISABLED;
        x42pp_status_disabled (drop);
    }
    if (drop == 255)
    {
        info ("Disabling all drops");
        for (i = 0; i < MAX_DROPS; i++)
        {
            if (state.drops[i].state != NOT_CFG)
            {
                x42pp_status_disabled (i);
                state.drops [i].state = DISABLED;
            }
        }
    }
}

/**
 * \brief Enable a drop.
 * \param drop Drop to enable.
 */
void
x42pp_tx_enable (ubyte_1 drop)
{
    int i;

    if (drop < MAX_DROPS && state.drops [drop].state == DISABLED)
    {
        info ("Enabling Drop %u", drop);
        state.drops [drop].state = POLLING;
        x42pp_status_polling (drop);
    }
    if (drop == 255)
    {
        info ("Enabling all drops");
        for (i = 0; i < MAX_DROPS; i++)
        {
            if (state.drops [i].state == DISABLED)
            {
                x42pp_status_polling (i);
                state.drops [i].state = POLLING;
            }
        }
    }
}

/* Actions */

/**
 * \brief Check if all drops have pending transactions.
 * \return True if all drops have a pending transaction.
 */
static int
all_pending (void)
{
    int i;

    for (i = 0; i < MAX_DROPS; i ++)
    {
        if (state.drops[i].state == POLLING ||
            state.drops[i].state == SLOW_POLL)
            return 0;
    }
    return 1;
}

/**
 * \brief Find next drop to poll.
 * \return Drop to poll.
 */
static int
next_drop_to_poll (void)
{
    int i;

    if (all_pending ())
        return MAX_DROPS;

    for (i = (state.curr_drop + 1) % MAX_DROPS; ; i = (i + 1) % MAX_DROPS)
    {
        if (state.drops[i].state == SLOW_POLL && ++state.drops[i].skip_cnt == 3)
        {
            state.drops[i].skip_cnt = 0;
            return i;
        }
        if (state.drops[i].state == POLLING)
            return i;
    }
}

/**
 * \brief Poll next drop.
 * \param link First characted of next frame.
 */
static void
poll (ubyte_1 link)
{
    buf_t frame = 0;

    state.curr_frame = buf_free (state.curr_frame);
    state.tries = 0;

    /* Check if there is a queued message first.
    Broadcasts are given priority to so resets
    get to the terminals before the pad resets. */
    if ((frame = buf_dequeue (state.broadcast)) ||
        (frame = buf_dequeue (state.unsolicited)) ||
        (frame = buf_dequeue (state.solicited)))
    {
        state.curr_drop = (ubyte_1)buf_get_tag (frame);
        state.curr_type = ((ubyte_1 *)buf_data (frame))[1] & 0x60;
        if (state.curr_drop != 255)
        {
            info ("Sending %s frame to Drop %u",
                state.curr_type == SAD ? "Solicited" : "Unsolicited", state.curr_drop);
        }
        else
        {
            info ("Sending Broadcast frame");
        }
        state.state = DATA;
        send_frame (frame, link);
    }
    /* If not, find next drop to poll or go idle. */
    else
    {
        state.curr_type = PAD;
        state.curr_drop = next_drop_to_poll ();
        if (state.curr_drop == MAX_DROPS)
        {
            if (state.state != IDLE)
                info ("Link is Idle");
            state.state = IDLE;
            send_idle_link (link);
        }
        else
        {
            if (state.state != POLLING)
                info ("Polling");
            state.state = POLLING;
            send_poll (link);
        }
    }
}

/**
 * \brief Resend last message to terminal.
 * \param link First characted of next frame.
 */
static void
retry_data (ubyte_1 link)
{
    info ("Resending last frame");
    if (++state.tries == 3)
        poll (link);
    else
        send_frame (state.curr_frame, link);
}

/**
 * \brief Send idle byte.
 * \param link First characted of next frame.
 */
static void
send_idle_link (ubyte_1 link)
{
    send_frame (buf_append (0, &link, 1), link);
}

/**
 * \brief Send poll.
 * \param link First characted of next frame.
 */
static void
send_poll (ubyte_1 link)
{
    buf_t b;
    ubyte_1 acb, eacb;

    b = buf_append (0, &link, 1);
    acb = make_acb (state.curr_drop, PAD);
    b = buf_append (b, &acb, 1);
    if (state.curr_drop >= 30)
    {
        eacb = make_eacb (state.curr_drop);
        b = buf_append (b, &eacb, 1);
    }
    send_frame (b, link);
}

/**
 * \brief Send a frame.
 * \param b Frame to send.
 * \param link First characted of next frame.
 */
static void
send_frame (buf_t b, ubyte_1 link)
{
    ((ubyte_1 *)buf_data (b))[0] = link;
    state.curr_frame = b;
    x42pp_imp_send_frame (b);
}

/**
 * \brief Call state machine.
 * \param link First characted of next frame.
 */
static void
dispatch (ubyte_1 link)
{
    matrix [state.state] (link);
}

/*
 * End of "$Id: x42pp_tx.c,v 1.3 2005/07/18 19:01:48 cmayncvs Exp $".
 */

