/* ===[ $RCSfile: x42pp_utils.c,v $ ]=============================================

    This item is the property of GTECH Corporation, West Greenwich,
    Rhode Island, and contains confidential and trade secret information.
    It may not be transferred from the custody or control of GTECH except
    as authorized in writing by an officer of GTECH.  Neither this item
    nor the information it contains may be used, transferred, reproduced,
    published, or disclosed, in whole or in part, and directly or
    indirectly, except as expressly authorized by an officer of GTECH,
    pursuant to written agreement.

    Copyright (c) 2003 GTECH Corporation.  All rights reserved.

   ======================================================================= */

/** \file
 *
 *  "$Id: x42pp_utils.c,v 1.1.1.1 2003/12/02 15:53:30 sbobcvs Exp $"
 *
 * \brief A few shared utility functions.
 *
 */
#include "libcom/buf.h"
#include "x42pp_types.h"
#include "x42pp_utils.h"

/**
 * \brief Calculate the address byte.
 * \param address Drop address.
 * \param type Message type.
 */
ubyte_1
make_acb (ubyte_1 address, ubyte_1 type)
{
    if (type == BROC)
    {
        return 0x7F;
    }
    else
    {
        if (address < 30)
            return type + (address & 0x1F);
        else
            return type + 0x1E;
    }
}

/**
 * \brief Calculate the extended address byte.
 * \param address Drop address.
 */
ubyte_1
make_eacb (ubyte_1 address)
{
    return (((address - 30) & 0x40) << 1) + ((address - 30) & 0x3F) + 0x40;
}

/**
 * \brief Update LRC and DRC.
 * \param val Next byte.
 * \param lrc LRC to update.
 * \param drc DRC to update.
 */
void
update_fcs (ubyte_1 val, ubyte_1 *lrc, ubyte_1 *drc)
{
    *lrc ^= val;
     if (*drc & 1)
        *drc = ((*drc >> 1) + 0x80) ^ val;
     else
        *drc = (*drc >> 1) ^ val;
}

/**
 * \brief Construct a X42 frame.
 * \param msg Message to put in frame.
 * \param address Drop address frame is for.
 * \param type Frame type.
 * \return X42 frame.
 */
buf_t
make_frame (buf_t msg, ubyte_1 address, ubyte_1 type)
{
    ubyte_1 frame [(buf_length (msg) * 2) + 8];
    ubyte_2 index = 0;
    ubyte_1 lrc = 0, drc = 0;
    ubyte_2 i;

    /* Return true is byte is a resevered X42 character. */
    char is_reserved (ubyte_1 val)
    {
        if (val == LINK || val == ACKN || val == ETX || val == CNB
            || val == ENQ || val == ETB_NEW || val == ETB_OLD)
            return 1;
        return 0;
    }

    /* Add data to frame and escape if needed. */
    void add_data (ubyte_1 data)
    {
        if (is_reserved (data))
        {
            frame[index++] = CNB;
            frame[index++] = ~data;
        }
        else
        {
            frame[index++] = data;
        }
    }

    frame[index++] = LINK;
    frame[index] = make_acb (address, type);
    update_fcs (frame[index++], &lrc, &drc);
    if (address >= 30 && type != BROC)
    {
        frame[index] = make_eacb (address);
        update_fcs (frame[index++], &lrc, &drc);
    }

    for (i = 0; i < buf_length (msg); i++)
    {
        ubyte_1 data = ((ubyte_1 *)buf_data (msg))[i];

        update_fcs (data, &lrc, &drc);
        add_data (data);
    }

    frame[index] = ETX;
    update_fcs (frame[index++], &lrc, &drc);
    add_data (lrc);
    add_data (drc);

    return buf_append (0, frame, index);
}

/*
 * End of "$Id: x42pp_utils.c,v 1.1.1.1 2003/12/02 15:53:30 sbobcvs Exp $".
 */

