/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <pthread.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <netdb.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/time.h>
#include <execinfo.h>
#include <stdarg.h>
#include <ctype.h>

#include "annex_common.h"

extern "C" {

char *basename_safe(const char *path) {
  const char *cp = strrchr(path, '/');
  if(cp) {
    cp++;
    return (char *) cp;
  } else {
    return (char *) path;
  }
}

int str_ends_with(const char *str, const char *sub) {
  size_t str_len = strlen(str);
  size_t sub_len = strlen(sub);

  if (sub_len > str_len) {
    return 0;
  }

  if (!strcmp(str + (str_len - sub_len), sub)) {
    return 1;
  } else {
    return 0;
  }
}

void sleep_msec(size_t msec) {
  int tmp;
  struct timespec req;
  struct timespec rem;

  req.tv_sec = msec / 1000;
  req.tv_nsec = (msec % 1000) * 1000 * 1000;

  while (1) {
    tmp = nanosleep(&req, &rem);
    if (tmp < 0) {
      if (errno == EINTR) {
        req = rem;
      } else {
        log_err("nanosleep failed: %m");
        return;
      }
    } else {
      return;
    }
  }
}

#define BACKTRACE_LEN 128

void log_backtrace() {
  int i;
  int n_addrs;
  void *addrs[BACKTRACE_LEN];
  char **symbols;

  n_addrs = backtrace(addrs, BACKTRACE_LEN);

  symbols = backtrace_symbols(addrs, n_addrs);
  if (!symbols) {
    log_err("backtrace_symbols failed: %m");
    return;
  }

  log_err("backtrace:");
  for (i = 1; i < n_addrs; i++) {
    log_err("> %s", symbols[i]);
  }

  free(symbols);
}

int mutex_lock(pthread_mutex_t *mutex) {
  int tmp = pthread_mutex_lock(mutex);
  if (tmp) {
    log_emerg("pthread_mutex_lock failed: %d", tmp);
    log_backtrace();
    exit(1);
  }

  return tmp;
}

int mutex_unlock(pthread_mutex_t *mutex) {
  int tmp = pthread_mutex_unlock(mutex);
  if (tmp) {
    log_emerg("pthread_mutex_unlock failed: %d", tmp);
    log_backtrace();
    exit(1);
  }

  return tmp;
}

int cond_wait(pthread_cond_t *cond, pthread_mutex_t *mutex) {
  int tmp = pthread_cond_wait(cond, mutex);
  if (tmp) {
    log_emerg("pthread_cond_wait failed: %d", tmp);
    log_backtrace();
    exit(1);
  }

  return tmp;
}

int cond_timedwait(
    pthread_cond_t *cond,
    pthread_mutex_t *mutex,
    const struct timespec *abstime) {
  int tmp = pthread_cond_timedwait(cond, mutex, abstime);
  if (tmp && tmp != ETIMEDOUT) {
    log_emerg("pthread_cond_wait failed: %d", tmp);
    log_backtrace();
    exit(1);
  }

  return tmp;
}

int cond_signal(pthread_cond_t *cond) {
  int tmp = pthread_cond_signal(cond);
  if (tmp) {
    log_emerg("pthread_cond_signal failed: %d", tmp);
    log_backtrace();
    exit(1);
  }

  return tmp;
}

int nonblocking(int fd, int nonblocking)
{
  int tmp;

  tmp = fcntl(fd, F_GETFL, 0);
  if (tmp < 0) {
    log_err("fcntl F_GETFL failed: %m");
    return tmp;
  }

  if (nonblocking) {
    tmp |= O_NONBLOCK;
  } else {
    tmp &= ~O_NONBLOCK;
  }

  tmp = fcntl(fd, F_SETFL, tmp);
  if (tmp < 0) {
    log_err("fcntl F_SETFL failed: %m");
    return tmp;
  }

  return tmp;
}

int abstimeout_tv_msec(struct timeval *tv, int msec) {
  struct timeval now;
  struct timeval add;
  int tmp;

  tmp = gettimeofday(&now, NULL);
  if (tmp < 0) {
    log_err("gettimeofday failed: %m");
    return tmp;
  }

  add.tv_sec = msec / 1000;
  add.tv_usec = (msec % 1000) * 1000;
  timeradd(&now, &add, tv);

  return 0;
}

int abstimeout_ts_msec(struct timespec *ts, int msec) {
  struct timeval now;
  struct timeval add;
  struct timeval res;
  int tmp;

  tmp = gettimeofday(&now, NULL);
  if (tmp < 0) {
    log_err("gettimeofday failed: %m");
    return tmp;
  }

  add.tv_sec = msec / 1000;
  add.tv_usec = (msec % 1000) * 1000;
  timeradd(&now, &add, &res);

  ts->tv_sec = res.tv_sec;
  ts->tv_nsec = res.tv_usec * 1000;

  return 0;
}

#if SSL_SUPPORT
void ssl_log_err(const char *prefix, unsigned long err) {
  char err_str[256];

  log_err("%s: %s", prefix, ERR_error_string(err, err_str));
}

void ssl_dump_err_queue(const char *prefix) {
  unsigned long err;

  log_err("%s", prefix);

  while ((err = ERR_get_error())) {
    ssl_log_err(prefix, err);
  }
}

void ssl_log_ssl_error(const char *prefix, SSL *ssl, int ret) {
  char buf[256];
  int errnum = errno;
  unsigned long err;

  switch (SSL_get_error(ssl, ret)) {
  case SSL_ERROR_NONE:
    log_err("%s: SSL_ERROR_NONE", prefix);
    break;
  case SSL_ERROR_SSL:
    log_err("%s: SSL_ERROR_SSL", prefix);
    break;
  case SSL_ERROR_WANT_READ:
    log_err("%s: SSL_ERROR_WANT_READ", prefix);
    break;
  case SSL_ERROR_WANT_WRITE:
    log_err("%s: SSL_ERROR_WANT_WRITE", prefix);
    break;
  case SSL_ERROR_WANT_X509_LOOKUP:
    log_err("%s: SSL_ERROR_WANT_X509_LOOKUP", prefix);
    break;
  case SSL_ERROR_SYSCALL:
    err = ERR_get_error();

    if (err) {
        log_err("%s: SSL_ERROR_SYSCALL %s", prefix, ERR_error_string(err, buf));
    } else {
      if (ret == -1) {
        strerror_r(errnum, buf, sizeof(buf));
        log_err("%s: SSL_ERROR_SYSCALL [%d] %s", prefix, errnum, buf);
      } else if (ret == 0) {
        log_err("%s: SSL_ERROR_SYSCALL EOF", prefix);
      } else {
        log_err("%s: SSL_ERROR_SYSCALL ?", prefix);
      }
    }
    break;
  case SSL_ERROR_ZERO_RETURN:
    log_err("%s: SSL_ERROR_ZERO_RETURN", prefix);
    break;
  case SSL_ERROR_WANT_CONNECT:
    log_err("%s: SSL_ERROR_WANT_CONNECT", prefix);
    break;
  case SSL_ERROR_WANT_ACCEPT:
    log_err("%s: SSL_ERROR_WANT_ACCEPT", prefix);
    break;
  default:
    log_err("%s: Unknown Error", prefix);
    break;
  }
}

int sasl_log_callback(
    void *context __attribute__ ((unused)),
    int priority,
    const char *message) {
  log_debug("begin");

  if (!message) {
    log_err("bad param");
    return SASL_BADPARAM;
  }

  switch (priority) {
  case SASL_LOG_ERR:
    log_err("SASL: %s", message);
    break;
  case SASL_LOG_NOTE:
    log_notice("SASL: %s", message);
    break;
  default:
    log_info("SASL: %s", message);
  }

  log_debug("end");

  return SASL_OK;
}

int sasl_realm_callback(
    void *context,
    int id,
    const char **availrealms __attribute__ ((unused)),
    const char **result)
{
  char *value = (char *) context;

  log_debug("begin");

  if (!value || id != SASL_CB_GETREALM) {
    log_err("bad param");
    return SASL_FAIL;
  }

  *result = value;

  log_debug("end");

  return SASL_OK;
}

int sasl_simple_callback(
    void *context,
    int id,
    const char **result,
    unsigned *len) {
  char *value = (char *) context;

  log_debug("begin");

  if (!result) {
    log_err("bad param");
    return SASL_BADPARAM;
  }

  switch (id) {
  case SASL_CB_USER:
    if (!value) {
      log_err("user required");
      return SASL_BADPARAM;
    }
    *result = value;
    if (len) {
      *len = (unsigned) strlen(value);
    }
    break;
  case SASL_CB_AUTHNAME:
    if (!value) {
      log_err("authname required");
      return SASL_BADPARAM;
    }
    *result = value;
    if (len) {
      *len = (unsigned) strlen(value);
    }
    break;
  case SASL_CB_LANGUAGE:
    *result = NULL;
    if (len) {
      *len = 0;
    }
    break;
  default:
    return SASL_BADPARAM;
  }

  log_debug("end");

  return SASL_OK;
}

int sasl_secret_callback(
    sasl_conn_t *conn,
    void *context,
    int id,
    sasl_secret_t **psecret) {
  char *value = (char *) context;
  unsigned len;

  log_debug("begin");

  if (!value || !conn || !psecret || id != SASL_CB_PASS) {
    log_err("bad param");
    return SASL_BADPARAM;
  }

  len = (unsigned) strlen(value);

  *psecret = (sasl_secret_t *) malloc(sizeof(sasl_secret_t) + len);
  if (!*psecret) {
    log_err("malloc sasl_secret_t failed");
    return SASL_NOMEM;
  }

  (*psecret)->len = len;
  strcpy((char *) (*psecret)->data, value);

  log_debug("end");

  return SASL_OK;
}

int sasl_challenge_callback(
    void *context,
    int id,
    const char *challenge __attribute__ ((unused)),
    const char *prompt,
    const char *defresult __attribute__ ((unused)),
    const char **result,
    unsigned *len) {
  char *value = (char *) context;

  log_debug("begin");

  if (!value || !prompt || !result || !len ||
      (id != SASL_CB_ECHOPROMPT && id != SASL_CB_NOECHOPROMPT)) {
    log_err("bad param");
    return SASL_BADPARAM;
  }

  *result = strdup(value);
  if (!*result) {
    log_err("strdup failed");
    return SASL_NOMEM;
  }
  *len = (unsigned) strlen(value);

  log_debug("end");

  return SASL_OK;
}

int sasl_auth_callbacks_init(
    sasl_callback_t *callbacks,
    size_t n_callbacks,
    const char *username,
    const char *passwd,
    const char *realm) {
  int i = 0;

  log_debug("begin");

  if (n_callbacks < 8) {
    return SASL_BADPARAM;
  }

  callbacks[i].id = SASL_CB_LOG;
  callbacks[i].proc = (int (*)()) sasl_log_callback;
  callbacks[i].context = NULL;
  i++;

  callbacks[i].id = SASL_CB_USER;
  callbacks[i].proc = (int (*)()) sasl_simple_callback;
  callbacks[i].context = (void *) username;
  i++;

  callbacks[i].id = SASL_CB_AUTHNAME;
  callbacks[i].proc = (int (*)()) sasl_simple_callback;
  callbacks[i].context = (void *) username;
  i++;

  callbacks[i].id = SASL_CB_GETREALM;
  callbacks[i].proc = (int (*)()) sasl_realm_callback;
  callbacks[i].context = (void *) realm;
  i++;

  callbacks[i].id = SASL_CB_PASS;
  callbacks[i].proc = (int (*)()) sasl_secret_callback;
  callbacks[i].context = (void *) passwd;
  i++;

  callbacks[i].id = SASL_CB_ECHOPROMPT;
  callbacks[i].proc = (int (*)()) sasl_challenge_callback;
  callbacks[i].context = (void *) passwd;
  i++;

  callbacks[i].id = SASL_CB_NOECHOPROMPT;
  callbacks[i].proc = (int (*)()) sasl_challenge_callback;
  callbacks[i].context = (void *) passwd;
  i++;

  callbacks[i].id = SASL_CB_LIST_END;
  callbacks[i].proc = NULL;
  callbacks[i].context = NULL;
  i++;

  log_debug("end");

  return SASL_OK;
}

int sasl_auth_start(
    sasl_conn_t **conn,
    sasl_callback_t *callbacks,
    char *mech_list,
    const char **mech,
    char *token_out,
    size_t token_out_size) {
  int tmp;
  int ret;

  const char *data;
  unsigned data_len;
  unsigned token_out_len;

  log_debug("begin");

  ret = sasl_client_new("annex", "test.example.com", NULL, NULL, callbacks, 0, conn);
  if (ret != SASL_OK) {
    log_err("sasl_client_new failed: %d", ret);
    return ret;
  }

  ret = sasl_client_start(*conn, mech_list, NULL, &data, &data_len, mech);
  if (ret != SASL_OK && ret != SASL_CONTINUE) {
    log_err("sasl_client_start failed: %s", sasl_errdetail(*conn));
    sasl_dispose(conn);
    return ret;
  }

  if (data && data_len) {
    tmp = sasl_encode64(data, data_len, token_out, token_out_size, &token_out_len);
    if (tmp != SASL_OK) {
      log_err("sasl_encode64 failed");
      sasl_dispose(conn);
      return tmp;
    }
  } else {
    *token_out = '\0';
  }

  log_debug("token_out: %s", token_out);

  log_debug("end");

  return ret;
}

int sasl_auth_reply(
    sasl_conn_t *conn,
    const char *token_in,
    char *token_out,
    size_t token_out_size) {
  int tmp;
  int ret;
  const char *data;
  unsigned data_len;
  unsigned token_out_len;
  char *reply_data = NULL;
  unsigned reply_data_len = 0;
  unsigned token_in_len;

  log_debug("begin");

  token_in_len = strlen(token_in);
  reply_data = (char *) malloc(token_in_len);
  if (!reply_data) {
    log_err("malloc reply_data failed");
    ret = SASL_NOMEM;
    goto done;
  }

  ret = sasl_decode64(token_in, token_in_len, reply_data, token_in_len, &reply_data_len);
  if (ret != SASL_OK) {
    log_err("sasl_decode64 failed");
    goto done;
  }

  ret = sasl_client_step(conn, reply_data, reply_data_len, NULL, &data, &data_len);
  if (ret != SASL_OK && ret != SASL_CONTINUE) {
    log_err("sasl_client_step failed: %s", sasl_errdetail(conn));
    goto done;
  }

  if (data && data_len) {
    tmp = sasl_encode64(data, data_len, token_out, token_out_size, &token_out_len);
    if (tmp != SASL_OK) {
      log_err("sasl_encode64 failed");
      ret = tmp;
      goto done;
    }
  } else {
    *token_out = '\0';
  }

  log_debug("token_out: %s", token_out);

done:
  free(reply_data);

  log_debug("end");

  return ret;
}

void sasl_auth_end(sasl_conn_t **conn) {
  sasl_dispose(conn);
}
#endif

int file_exists(const char *path) {
  struct stat st;

  return stat(path, &st) < 0 ? 0 : 1;
}

int file_is_dir(const char *path) {
  struct stat st;

  if (stat(path, &st) < 0) {
    return 0;
  }

  return S_ISDIR(st.st_mode);
}

int file_is_reg(const char *path) {
  struct stat st;

  if (stat(path, &st) < 0) {
    return 0;
  }

  return S_ISREG(st.st_mode);
}

int path_scanf(const char *path, const char *format, ...) {
  va_list ap;
  int tmp;
  FILE *stream;

  stream = fopen(path, "r");
  if (!stream) {
    log_debug("unable to open file %s: %m", path);
    return EOF;
  }

  va_start(ap, format);
  tmp = vfscanf(stream, format, ap);
  va_end(ap);

  fclose(stream);

  return tmp;
}

int path_printf(const char *path, const char *format, ...) {
  va_list ap;
  int tmp;
  FILE *stream;

  stream = fopen(path, "w");
  if (!stream) {
    log_debug("unable to open file %s: %m", path);
    return -1;
  }

  va_start(ap, format);
  tmp = fprintf(stream, format, ap);
  va_end(ap);

  fclose(stream);

  return tmp;
}

int path_read_address(const char *path, char *addr) {
  int tmp=0;
  int count=0;
  FILE *stream;
  char buf[128];
  unsigned int hex_val;

  stream = fopen(path, "r");
  if (!stream) {
    log_debug("unable to open file %s: %m", path);
    return EOF;
  }

  tmp = fread(buf, sizeof(char), 17, stream); 
  if (tmp != 17) {
    log_debug("error reading the stream - buf = %s, tmp = %d\n", buf, tmp);
    return -1;
  }

  buf[tmp] = '\0';
  tmp = 0;
  while (buf[tmp]) {
    if (buf[tmp] == ':') {
      buf[tmp] = '\0';
      count++;
    }
    tmp++;
  }

#if 0
  sscanf(&buf[0], "%x", &hex_val); *addr = hex_val;
  sscanf(&buf[3], "%x", &hex_val); *(addr+1) = hex_val;
  sscanf(&buf[6], "%x", &hex_val); *(addr+2) = hex_val;
  sscanf(&buf[9], "%x", &hex_val); *(addr+3) = hex_val;
  if (count == 5) {
    sscanf(&buf[12], "%x", &hex_val); *(addr+4) = hex_val;
    sscanf(&buf[15], "%x", &hex_val); *(addr+5) = hex_val;
  }
#else
  *addr = (char)hex_atoi(&buf[0]);
  *(addr+1) = (char)hex_atoi(&buf[3]);
  *(addr+2) = (char)hex_atoi(&buf[6]);
  *(addr+3) = (char)hex_atoi(&buf[9]);
  if (count == 5) {
    *(addr+4) = (char)hex_atoi(&buf[12]);
    *(addr+5) = (char)hex_atoi(&buf[15]);
  }
#endif

  log_debug ("Addr Read = %x:%x:%x:%x:%x:%x\n", *addr, *(addr+1), *(addr+2), *(addr+3), *(addr+4), *(addr+5));

  fclose(stream);

  return (count+1);
}

int hex_atoi(char *buf)
{
  int val = 0;

  while (*buf) {
    val <<= 4;

    if ((*buf >= '0') && (*buf <= '9'))
      val += (*buf - '0');
    else if ((*buf >= 'a') && (*buf <= 'z'))
      val += ((*buf - 'a') + 10);
    else if ((*buf >= 'A') && (*buf <= 'Z'))
      val += ((*buf - 'A') + 10);

    buf++;
  }
  return val;
}

char *strrstrip(char *str) {
  char *end;
  char *prev;

  if (!str || !*str) {
    return NULL;
  }
  prev = end = str + strlen(str);

  while (end > str && isspace(*(end - 1))) {
    end--;
  }
  *end = '\0';

  return prev == end ? NULL : str;
}

char *atcmd_value_tok(char **str) {
  char *next;
  char *begin;
  int count = 0;

  begin = *str;
  if (!begin || !*begin) {
    return NULL;
  }

  begin += strspn(begin, " \t");

  switch (*begin) {
  case '\"':
    next = ++begin;

    next = strchr(next, '\"');
    if (!next) {
      return NULL;
    }
    *next++ = '\0';

    next = strchr(next, ',');
    if (next) {
      *next++ = '\0';
    }

    break;
  case '(':
    next = ++begin;

    while (1) {
      switch (*next) {
      case ')':
        if (!count) {
            goto found;
        }
        count--;

        break;
      case '(':
        count++;

        break;
      case '\0':
        return NULL;
      }

      next++;
    }

    found:

    *next++ = '\0';

    next = strchr(next, ',');
    if (next) {
      *next++ = '\0';
    }

    break;
  default:
    next = begin;

    next = strchr(next, ',');
    if (next) {
      *next++ = '\0';
    }

    strrstrip(begin);
  }

  *str = next;

  return begin;
}

char *atcmd_response_brk(char **str) {
  char *next;
  char *begin;

  begin = *str;
  if (!begin || !*begin) {
    return NULL;
  }

  begin += strspn(begin, " \t");

  next = begin;

  next = strstr(next, ": ");
  if (!next) {
    return NULL;
  }
  *next++ = '\0';
  next += strspn(next, " \t");

  *str = next;

  return begin;
}

int is_ppp_link_up()
{
    FILE *fp_ppp = NULL;
    char ppp_stat_buf[32] = "";

    fp_ppp = fopen("/var/run/pppstatus.pid", "r");
    if(fp_ppp != NULL) {
        fgets(ppp_stat_buf, sizeof(ppp_stat_buf), fp_ppp);
        fclose(fp_ppp);
        if(strstr(ppp_stat_buf, "Link is up")) {
            return (1);
        }
    }
    return (0);
}

int bring_ppp_link_up()
{
    char ppp_stat_buf[32] = "";
    FILE *fp_ppp = NULL;
    int i, ppp_up = 0;

    log_debug ("Bringing up PPP");
    printf ("Bringing up PPP\n");
#if 1
    system ("ping -c 1 -w 20 www.google.com 1>/dev/null 2>/dev/null");
#else
    i = execl ("/bin/ping", "/bin/ping", "-c", "1", "-w", "20", "www.google.com", NULL);
    printf ("ret value from execl = %d\n", i);
#endif

    printf ("Checking for link status\n");
    for (i=0; i<20; i++) {
        fp_ppp = fopen ("/var/run/pppstatus.pid", "r");
        if (fp_ppp != NULL) {
            fgets (ppp_stat_buf, sizeof(ppp_stat_buf), fp_ppp);
            fclose(fp_ppp);
            if (strstr(ppp_stat_buf,"Link is up")) {
                ppp_up = 1;
                printf ("Link is up\n");
                log_debug ("PPP Link is up");
                break;
            }
            else {
                printf ("Sleep for 2 Secs\n");
                sleep_msec(2000);
                continue;
            }
        }
        else {
            printf ("Sleep for 2 Secs\n");
            sleep_msec(2000);
            continue;
        }
    }
    return (ppp_up);
}

}
