/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#ifndef ANNEX_TRANSFER_H_
#define ANNEX_TRANSFER_H_

#include <pthread.h>
#include <list>
#include <sys/time.h>

#include <annex_common.h>
#include <annex.pb.h>
#include <annex_transfer_agent.h>

namespace mts {

class AnnexTransfer {
 public:
  AnnexTransfer() {
    timerclear(&sendq_abstimeout_);
    events_ = 0;
    sending_ = false;
    tracked_ = false;
    owner_ = NULL;
  }

  ~AnnexTransfer() {
    for (std::list<AnnexTransfer *>::iterator iter = replyq_.begin();
         iter != replyq_.end();
         ++iter) {
      delete *iter;
    }
  }

  bool SetSendqTimeoutMsec(int to_msec) {
    int tmp;

    if (to_msec < 0) {
      timerclear(&sendq_abstimeout_);
    } else {
      tmp = abstimeout_tv_msec(&sendq_abstimeout_, to_msec);
      if (tmp < 0) {
        log_err("gettimeofday failed: %m");
        return false;
      }
    }

    return true;
  }

  bool SendqTimedout(struct timeval *now) {
    if (timerisset(&sendq_abstimeout_) && timercmp(&sendq_abstimeout_, now, <=)) {
      log_debug("transfer timeout");
      return true;
    }

    return false;
  }

  void set_sending(bool value) {
    sending_ = value;
  }

  bool sending() {
    return sending_;
  }

  bool tracked() {
    return tracked_;
  }

  void set_tracked(bool tracked) {
    tracked_ = tracked;
  }

  AnnexTransferAgent *owner() {
    return owner_;
  }

  void set_owner(AnnexTransferAgent *owner) {
    owner_ = owner;
  }

  void AddTransferEvent(AnnexTransferAgent::TransferEvent event) {
    events_ |= event;
  }

  int events() {
    return events_;
  }

  std::list<AnnexTransfer *> *mutable_replyq() {
    return &replyq_;
  }

  annex::Package *mutable_package() {
    return &package_;
  }

  const annex::Package &package() const {
    return package_;
  }

  void AddReply(AnnexTransfer *transfer) {
    replyq_.push_back(transfer);
  }

 private:
  annex::Package package_;
  struct timeval sendq_abstimeout_;
  bool sending_;
  bool tracked_;
  int events_;
  std::list<AnnexTransfer *> replyq_;
  AnnexTransferAgent *owner_;

  DISALLOW_COPY_AND_ASSIGN(AnnexTransfer);
};

}
#endif  // ANNEX_TRANSFER_H_
