/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>

#include "annex_common.h"
#include "annex_transfer_agent.h"
#include "courier.h"

namespace mts {

AnnexTransferAgent::AnnexTransferAgent() {
  courier_ = NULL;
  reply_status_code_ = annex::Package::STATUS_CODE_NONE;

  pthread_mutex_init(&event_mutex_, NULL);
  pthread_cond_init(&event_cond_, NULL);
}

AnnexTransferAgent::~AnnexTransferAgent() {
  pthread_mutex_destroy(&event_mutex_);
  pthread_cond_destroy(&event_cond_);
}

void AnnexTransferAgent::StartListening() {
  if (courier_) {
    courier_->AddTransferListener(this);
  }
}

void AnnexTransferAgent::StopListening() {
  if (courier_) {
    courier_->RemoveTransferListener(this);
  }
}

void AnnexTransferAgent::ReplyStatusReset() {
  reply_status_code_ = annex::Package::STATUS_CODE_NONE;
  reply_status_message_ = "";
}

bool AnnexTransferAgent::TransferEventRecv(AnnexTransfer *transfer) {
  return false;
}

bool AnnexTransferAgent::TransferEventFired(AnnexTransfer *transfer, TransferEvent event) {
  int rc = false;

  switch (event) {
  case TRANSFER_EVENT_SENDQ_TIMEOUT:
  case TRANSFER_EVENT_SENT:
    rc = TransferEventComplete(transfer);
    break;
  case TRANSFER_EVENT_REPLY:
    rc = TransferEventReply(transfer);
    break;
  case TRANSFER_EVENT_RECV:
    rc = TransferEventRecv(transfer);
    break;
  }

  if (rc) {
    EventSignal();
  }

  return rc;
}

void AnnexTransferAgent::AddTransfer(AnnexTransfer *transfer) {
  mutex_lock(&event_mutex_);
  transferq_.push_back(transfer);
  mutex_unlock(&event_mutex_);
}

void AnnexTransferAgent::RemoveTransfer(AnnexTransfer *transfer) {
  mutex_lock(&event_mutex_);
  transferq_.remove(transfer);
  mutex_unlock(&event_mutex_);
}

void AnnexTransferAgent::DeleteTransfer(AnnexTransfer *transfer) {
  RemoveTransfer(transfer);
  delete transfer;
}

AnnexTransfer *AnnexTransferAgent::FindTransferWithEvent(TransferEvent event) {
  AnnexTransfer *transfer = NULL;

  for (std::list<AnnexTransfer *>::iterator iter = transferq_.begin();
       iter != transferq_.end();
       ++iter) {
    if ((*iter)->events() & event) {
      transfer = *iter;
      break;
    }
  }

  return transfer;
}

void AnnexTransferAgent::EventSignal() {
  mutex_lock(&event_mutex_);
  cond_signal(&event_cond_);
  mutex_unlock(&event_mutex_);
}

AnnexTransfer *AnnexTransferAgent::TransferWaitRecv(int timeout_msec) {
  int tmp;
  struct timespec abstimeout;
  AnnexTransfer *transfer = NULL;

  abstimeout_ts_msec(&abstimeout, timeout_msec);

  mutex_lock(&event_mutex_);

  while (!(transfer = FindTransferWithEvent(TRANSFER_EVENT_RECV))) {
    if (timeout_msec >= 0) {
      tmp = cond_timedwait(&event_cond_, &event_mutex_, &abstimeout);
      if (tmp == ETIMEDOUT) {
        break;
      }
    } else {
      cond_wait(&event_cond_, &event_mutex_);
    }
  }

  if (transfer) {
    transferq_.remove(transfer);
  }

  mutex_unlock(&event_mutex_);

  return transfer;
}

void AnnexTransferAgent::TransferStart(
    AnnexTransfer *transfer,
    int sendq_timeout_msec,
    bool tracked) {
  transfer->set_tracked(tracked);
  transfer->set_owner(this);
  transfer->SetSendqTimeoutMsec(sendq_timeout_msec);

  courier()->QueueTransfer(transfer);
}

bool AnnexTransferAgent::TransferEventComplete(AnnexTransfer *transfer) {
  if (transfer->owner() == this) {
    if (transfer->tracked()) {
      AddTransfer(transfer);
    } else {
      delete transfer;
    }
    return true;
  } else {
    return false;
  }
}

bool AnnexTransferAgent::TransferIsComplete(AnnexTransfer *transfer) {
  for (std::list<AnnexTransfer *>::iterator iter = transferq_.begin();
       iter != transferq_.end();
       ++iter) {
    if (*iter == transfer) {
      return true;
    }
  }

  return false;
}

bool AnnexTransferAgent::TransferWaitComplete(
    AnnexTransfer *transfer,
    int timeout_msec) {
  int tmp;
  int rc;
  struct timespec abstimeout;

  if (timeout_msec >= 0) {
    abstimeout_ts_msec(&abstimeout, timeout_msec);
  }

  mutex_lock(&event_mutex_);

  while (!TransferIsComplete(transfer)) {
    if (timeout_msec >= 0) {
      tmp = cond_timedwait(&event_cond_, &event_mutex_, &abstimeout);
      if (tmp == ETIMEDOUT) {
        break;
      }
    } else {
      cond_wait(&event_cond_, &event_mutex_);
    }
  }

  rc = TransferIsComplete(transfer);

  mutex_unlock(&event_mutex_);

  return rc;
}

AnnexTransfer *AnnexTransferAgent::TransferWaitReply(
    AnnexTransfer *transfer,
    int timeout_msec) {
  int tmp;
  struct timespec abstimeout;
  AnnexTransfer *reply = NULL;
  std::list<AnnexTransfer *> *replyq = transfer->mutable_replyq();

  abstimeout_ts_msec(&abstimeout, timeout_msec);

  mutex_lock(&event_mutex_);

  while (replyq->empty()) {
    if (timeout_msec >= 0) {
      tmp = cond_timedwait(&event_cond_, &event_mutex_, &abstimeout);
      if (tmp == ETIMEDOUT) {
        break;
      }
    } else {
      cond_wait(&event_cond_, &event_mutex_);
    }
  }

  if (!replyq->empty()) {
    reply = replyq->front();
    replyq->pop_front();
  }

  mutex_unlock(&event_mutex_);

  if (reply) {
    reply_status_code_ = reply->package().status_code();
    reply_status_message_ = reply->package().status_message();
  }

  return reply;
}

bool AnnexTransferAgent::StatusReplyRaw(
    int sendq_timeout_msec,
    uint32_t correlation_id,
    annex::Package_StatusCode status_code,
    const char *status_msg) {
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return false;
  }
  annex::Package *package = transfer->mutable_package();

  package->set_correlation_id(correlation_id);
  package->set_status_code(status_code);
  if (status_msg) {
    package->set_status_message(status_msg);
  }

  TransferStart(transfer, sendq_timeout_msec, false);

  return true;
}

bool AnnexTransferAgent::StatusReply(
    int sendq_timeout_msec,
    uint32_t correlation_id,
    annex::Package_StatusCode status_code,
    const char *status_fmt, ...) {
  va_list ap;
  int tmp;
  int rc;
  char *status_msg = NULL;

  if (status_fmt) {
    va_start(ap, status_fmt);
    tmp = vasprintf(&status_msg, status_fmt, ap);
    va_end(ap);
    if (tmp < 0) {
      log_err("status_msg formatting failed");
      return false;
    }
  }

  rc = StatusReplyRaw(sendq_timeout_msec, correlation_id, status_code, status_msg);

  free(status_msg);

  return rc;
}

AnnexTransfer *AnnexTransferAgent::NotificationMessageStartRaw(
    int sendq_timeout_msec,
    annex::Notification_Level level,
    const char *msg) {
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return NULL;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_NOTIFICATION);
  annex::Notification *notification = content->mutable_notification();

  notification->set_level(level);
  notification->set_message(msg);

  TransferStart(transfer, sendq_timeout_msec, true);

  return transfer;
}

AnnexTransfer *AnnexTransferAgent::NotificationMessageStart(
    int sendq_timeout_msec,
    annex::Notification_Level level,
    const char *msg_fmt, ...) {
  va_list ap;
  int tmp;
  char *msg = NULL;

  va_start(ap, msg_fmt);
  tmp = vasprintf(&msg, msg_fmt, ap);
  va_end(ap);
  if (tmp < 0) {
    log_err("msg formatting failed");
    return NULL;
  }

  AnnexTransfer *transfer = NotificationMessageStartRaw(sendq_timeout_msec, level, msg);

  free(msg);

  return transfer;
}

bool AnnexTransferAgent::AnnexFileCloseInternal(const annex::File &file) {
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return NULL;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ACTION);
  annex::Action *action = content->mutable_action();
  action->set_type(annex::Action::TYPE_DO);
  annex::Function *func = action->mutable_function();
  func->set_type(annex::Function::TYPE_FILE_CLOSE);
  annex::FileClose *fc = func->mutable_file_close();
  *fc->mutable_file() = file;

  TransferStart(transfer, 10000, true);
  TransferWaitComplete(transfer, -1);
  if (transfer->events() & TRANSFER_EVENT_SENDQ_TIMEOUT) {
    log_debug("sendq timeout");
    DeleteTransfer(transfer);
    return false;
  }

  AnnexTransfer *reply = TransferWaitReply(transfer, 5 * 60 * 1000);
  if (!reply) {
    log_err("TransferWaitReply failed");
    DeleteTransfer(transfer);
    return false;
  }

  if (reply_status_code_ != annex::Package::STATUS_CODE_OK) {
    log_err("FileClose failed: %d (%s)",
        reply_status_code_, reply_status_message_.c_str());
    return false;
  }

  return true;
}

bool AnnexTransferAgent::AnnexFileClose(annex::File *file) {
  ReplyStatusReset();

  int rc = AnnexFileCloseInternal(*file);

  delete file;

  return rc;
}

annex::File *AnnexTransferAgent::AnnexFileOpen(const char *path, const char *mode) {
  ReplyStatusReset();

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return NULL;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ACTION);
  annex::Action *action = content->mutable_action();
  action->set_type(annex::Action::TYPE_DO);
  annex::Function *func = action->mutable_function();
  func->set_type(annex::Function::TYPE_FILE_OPEN);
  annex::FileOpen *fo = func->mutable_file_open();
  fo->set_path(path);
  fo->set_mode(mode);

  TransferStart(transfer, 10000, true);
  TransferWaitComplete(transfer, -1);
  if (transfer->events() & TRANSFER_EVENT_SENDQ_TIMEOUT) {
    log_debug("sendq timeout");
    DeleteTransfer(transfer);
    return false;
  }

  AnnexTransfer *reply = TransferWaitReply(transfer, 5 * 60 * 1000);
  if (!reply) {
    log_err("TransferWaitReply failed");
    DeleteTransfer(transfer);
    return NULL;
  }

  const annex::Package &rpackage = reply->package();

  if (reply_status_code_ != annex::Package::STATUS_CODE_OK) {
    log_err("FileOpen failed: %d (%s)",
        reply_status_code_, reply_status_message_.c_str());
    return false;
  }

  if (!(rpackage.has_content() &&
      rpackage.content().type() == annex::Content::TYPE_ATTRIBUTE &&
      rpackage.content().attribute().type() == annex::Attribute::TYPE_FILE &&
      rpackage.content().attribute().file().has_fileno())) {
    log_err("invalid response");
    return NULL;
  }

  annex::File *file = new(std::nothrow) annex::File(rpackage.content().attribute().file());
  if (!file) {
    AnnexFileCloseInternal(rpackage.content().attribute().file());
    log_err("malloc annex::File failed");
    return NULL;
  }

  return file;
}

bool AnnexTransferAgent::AnnexFileRead(
    annex::File *file,
    annex::FileBuffer *buffer,
    int len) {
  ReplyStatusReset();

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return false;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ACTION);
  annex::Action *action = content->mutable_action();
  action->set_type(annex::Action::TYPE_DO);
  annex::Function *func = action->mutable_function();
  func->set_type(annex::Function::TYPE_FILE_READ);
  annex::FileRead *fr = func->mutable_file_read();
  *fr->mutable_file() = *file;

  if (len >= 0) {
    fr->set_length(len);
  }

  TransferStart(transfer, 10000, true);
  TransferWaitComplete(transfer, -1);
  if (transfer->events() & TRANSFER_EVENT_SENDQ_TIMEOUT) {
    log_debug("sendq timeout");
    DeleteTransfer(transfer);
    return false;
  }

  AnnexTransfer *reply = TransferWaitReply(transfer, 5 * 60 * 1000);
  if (!reply) {
    log_err("TransferWaitReply failed");
    DeleteTransfer(transfer);
    return false;
  }

  const annex::Package &rpackage = reply->package();

  if (reply_status_code_ != annex::Package::STATUS_CODE_OK) {
    log_err("FileRead failed: %d (%s)",
        reply_status_code_, reply_status_message_.c_str());
    return false;
  }

  if (!(rpackage.has_content() &&
      rpackage.content().type() == annex::Content::TYPE_ATTRIBUTE &&
      rpackage.content().attribute().type() == annex::Attribute::TYPE_FILE_BUFFER)) {
    log_err("invalid response");
    return false;
  }

  *buffer = rpackage.content().attribute().file_buffer();

  return true;
}

bool AnnexTransferAgent::AnnexFileSeek(
    annex::File *file,
    int len) {
  ReplyStatusReset();

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return false;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ACTION);
  annex::Action *action = content->mutable_action();
  action->set_type(annex::Action::TYPE_DO);
  annex::Function *func = action->mutable_function();
  func->set_type(annex::Function::TYPE_FILE_SET_POS);
  annex::FileSetPos *fsp = func->mutable_file_set_pos();
  *fsp->mutable_file() = *file;
  annex::FilePos *fp = fsp->mutable_pos();
  fp->set_pos(len);

  TransferStart(transfer, 10000, true);
  TransferWaitComplete(transfer, -1);
  if (transfer->events() & TRANSFER_EVENT_SENDQ_TIMEOUT) {
    log_debug("sendq timeout");
    DeleteTransfer(transfer);
    return false;
  }

  AnnexTransfer *reply = TransferWaitReply(transfer, 5 * 60 * 1000);
  if (!reply) {
    log_err("TransferWaitReply failed");
    DeleteTransfer(transfer);
    return false;
  }

  if (reply_status_code_ != annex::Package::STATUS_CODE_OK) {
    log_err("FileSetPos failed: %d (%s)",
        reply_status_code_, reply_status_message_.c_str());
    return false;
  }

  return true;
}

bool AnnexTransferAgent::TransferEventReply(AnnexTransfer *transfer) {
  uint32_t correlation_id = transfer->package().correlation_id();
  int rc = false;

  mutex_lock(&event_mutex_);
  for (std::list<AnnexTransfer *>::iterator iter = transferq_.begin();
       iter != transferq_.end();
       ++iter) {
    if ((*iter)->package().message_id() == correlation_id) {
      (*iter)->AddReply(transfer);
      rc = true;
      break;
    }
  }
  mutex_unlock(&event_mutex_);

  return rc;
}

bool AnnexTransferAgent::set_courier(Courier *courier) {
  courier_ = courier;

  return true;
}

Courier *AnnexTransferAgent::courier() const {
  return courier_;
}

}
