/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#ifndef ANNEX_TRANSFER_AGENT_H_
#define ANNEX_TRANSFER_AGENT_H_

#include <annex_common.h>
#include <annex.pb.h>
#include <stdarg.h>
#include <list>

namespace mts {

class AnnexTransfer;
class Courier;

class AnnexTransferAgent {
 public:
  enum TransferEvent {
    TRANSFER_EVENT_SENDQ_TIMEOUT       = 0x0002,
    TRANSFER_EVENT_SENT                = 0x0004,
    TRANSFER_EVENT_REPLY               = 0x0008,
    TRANSFER_EVENT_RECV                = 0x0010,
  };

  AnnexTransferAgent();
  virtual ~AnnexTransferAgent();

  bool TransferEventFired(AnnexTransfer *transfer, TransferEvent event);

  bool set_courier(Courier *courier);
  Courier *courier() const;

 protected:
  void StartListening();
  void StopListening();

  void AddTransfer(AnnexTransfer *transfer);
  void RemoveTransfer(AnnexTransfer *transfer);
  void DeleteTransfer(AnnexTransfer *transfer);

  void EventSignal();

  virtual bool TransferEventComplete(AnnexTransfer *transfer);
  virtual bool TransferEventReply(AnnexTransfer *transfer);
  virtual bool TransferEventRecv(AnnexTransfer *transfer);

  AnnexTransfer *TransferWaitReply(
      AnnexTransfer *transfer,
      int timeout_msec);

  AnnexTransfer *FindTransferWithEvent(TransferEvent event);
  AnnexTransfer *TransferWaitRecv(int timeout_msec);

  void TransferStart(
      AnnexTransfer *transfer,
      int sendq_timeout_msec,
      bool tracked);
  bool TransferIsComplete(AnnexTransfer *transfer);
  bool TransferWaitComplete(
      AnnexTransfer *transfer,
      int timeout_msec);

  bool StatusReplyRaw(
      int sendq_timeout_msec,
      uint32_t correlation_id,
      annex::Package_StatusCode status_code,
      const char *status_message);
  bool StatusReply(
      int sendq_timeout_msec,
      uint32_t correlation_id,
      annex::Package_StatusCode status_code,
      const char *status_fmt, ...);

  AnnexTransfer *NotificationMessageStartRaw(
      int sendq_timeout_msec,
      annex::Notification_Level level,
      const char *msg);
  AnnexTransfer *NotificationMessageStart(
      int sendq_timeout_msec,
      annex::Notification_Level level,
      const char *msg_fmt, ...);

  void ReplyStatusReset();
  bool AnnexFileCloseInternal(const annex::File &file);
  bool AnnexFileClose(annex::File *file);
  annex::File *AnnexFileOpen(const char *path, const char *mode);
  bool AnnexFileRead(
      annex::File *file,
      annex::FileBuffer *buffer,
      int len);
  bool AnnexFileSeek(annex::File *file, int len);

  std::list<AnnexTransfer *> transferq_;
  pthread_mutex_t event_mutex_;
  pthread_cond_t event_cond_;
  Courier *courier_;
  annex::Package_StatusCode reply_status_code_;
  std::string reply_status_message_;
};

}
#endif  // ANNEX_TRANSFER_AGENT_H_
