/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <pthread.h>
#include <stdio.h>
#include <sys/types.h>
#include <netdb.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <courier.h>
#include <container_buffer.h>

#define SSL_SUPPORT 1
#define PPP_SUPPORT 0

namespace mts {

const char *Courier::kTimestampFormat = "%Y-%m-%dT%H:%M:%SZ";

Courier::Courier() {
  port_ = kAnnexPort;

  recv_timeout_msec_ = kRecvTimeoutMsecDefault;
  send_timeout_msec_ = kSendTimeoutMsecDefault;

  sd_ = -1;
  ssl_ = NULL;
#if SSL_SUPPORT
  ssl_ctx_ = NULL;
  sasl_conn_ = NULL;
#endif

  inspect_containers_ = false;

  synchronous_ = false;
  restart_courier_ = false;

  message_id_ = 0;

  rpd_interval_ = kRpdInterval;

  gps_interval_ = 0;
  stats_interval_ = 0;
  cell_interval_ = 0;
  trigger_interval_ = kTriggerInterval;

  when_ppp_up_ = false;
  bring_ppp_up_ = true;

  firmware_upgrade_ = false;
  config_upgrade_ = false;

  pthread_mutex_init(&sendq_mutex_, NULL);
  pthread_mutex_init(&transfer_listeners_mutex_, NULL);
  pthread_mutex_init(&message_id_mutex_, NULL);

  validator_.set_courier(this);
  AddTransferListener(&validator_);
}

Courier::~Courier() {
  Stop();

  RemoveTransferListener(&validator_);

  pthread_mutex_destroy(&message_id_mutex_);
  pthread_mutex_destroy(&transfer_listeners_mutex_);
  pthread_mutex_destroy(&sendq_mutex_);
}

bool Courier::TestMe() {
  return true;
}

void Courier::InspectContainer(const char *prefix, const annex::Container &container) {
  if (inspect_containers_) {
    log_info("%s: %s", prefix, container.DebugString().c_str());
  }
}

ssize_t Courier::RecvBuffer(void *buf, size_t len) {
  ssize_t cc;
  ssize_t total = 0;

  while (len > 0) {
    if (ssl_) {
#if SSL_SUPPORT
      cc = SSL_read(ssl_, buf, len);
      if (cc <= 0) {
        log_err("SSL_read failed: %d %d", static_cast<int>(cc), SSL_get_error(ssl_, cc));
        return cc;
      }
#endif
    } else {
      cc = read(sd_, buf, len);
      if (cc < 0) {
        log_err("read failed: %m");
        return cc;
      } else if(cc == 0) {
        log_err("read to eof");
        return -1;
      }
    }

    total += cc;
    buf = static_cast<uint8_t *>(buf) + cc;
    len -= cc;
  }

  return total;
}

ssize_t Courier::SendBuffer(const void *buf, size_t len) {
  ssize_t cc;
  ssize_t total = 0;

  while (len > 0) {
    if (ssl_) {
#if SSL_SUPPORT
      cc = SSL_write(ssl_, buf, len);
      if (cc <= 0) {
        log_err("SSL_write failed: %d %d", static_cast<int>(cc), SSL_get_error(ssl_, cc));
        return cc;
      }
#endif
    } else {
      cc = write(sd_, buf, len);
      if (cc < 0) {
        log_err("write failed: %m");
        return cc;
      }
    }

    total += cc;
    buf = static_cast<const uint8_t *>(buf) + cc;
    len -= cc;
  }

  return total;
}

bool Courier::AnnexTimestamp(time_t time_at, std::string *str) {
  struct tm tm_at;
  struct tm *tm_check;

  tm_check = gmtime_r(&time_at, &tm_at);
  if (!tm_check) {
    log_err("breaking down time failed");
    return false;
  }

  int tmp;
  char buf[kTimestampSize];

  tmp = strftime(buf, sizeof(buf), kTimestampFormat, &tm_at);
  if (!tmp) {
    log_err("strftime failed: %d", tmp);
    return false;
  }

  *str = buf;

  return true;
}

bool Courier::PackageExpired(const annex::Package &package) {
  struct tm tm;
  char *cp;

  if (!package.has_time_expires()) {
    return false;
  }

  cp = (char *) strptime(package.time_expires().c_str(), kTimestampFormat, &tm);
  if (!cp || *cp) {
    return false;
  }

  time_t now = time(NULL);

  if (timegm(&tm) <= now) {
    log_info("Package expired: %s", package.DebugString().c_str());
    return true;
  }

  return false;
}

void Courier::SetTimePackaged(annex::Package *package) {
  int rc;
  std::string timestamp;

  rc = AnnexTimestamp(time(NULL), &timestamp);
  if (rc) {
    package->set_time_packaged(timestamp);
  }
}

uint32_t Courier::NextMessageId() {
  uint32_t id;

  mutex_lock(&message_id_mutex_);
  id = message_id_++;
  mutex_unlock(&message_id_mutex_);

  return id;
}

void Courier::SetMessageId(annex::Package *package) {
  package->set_message_id(NextMessageId());
}

bool Courier::SetTimeSent(annex::Container *container) {
  bool rc;
  std::string timestamp;

  rc = AnnexTimestamp(time(NULL), &timestamp);
  if (rc) {
    for (int i = 0; i < container->packages_size(); ++i) {
      container->mutable_packages(i)->set_time_sent(timestamp);
    }
  }

  return true;
}

bool Courier::SendContainer(annex::Container *container) {
#if SI_WITH_BLOCKING_METHODS
  bool rc;
  ssize_t count;

  SetTimeSent(container);

  InspectContainer("send container", *container);

  std::string data;

  rc = container->SerializeToString(&data);
  if (!rc) {
    log_err("SerializeToString failed");
    return false;
  }

  uint32_t len = htonl(data.length());
  count = SendBuffer(&len, sizeof(len));
  if (count < 0) {
    log_err("SendBuffer failed");
    return false;
  }

  count = SendBuffer(data.c_str(), data.length());
  if (count < 0) {
    log_err("SendBuffer failed");
    return false;
  }

  return true;
#else
  ContainerBuffer *send_cbuf;
  ContainerBuffer::BufferEvent send_event = ContainerBuffer::BUFFER_EVENT_START;

  send_cbuf = new class ContainerBuffer();
  if (send_cbuf == NULL) {
     log_err ("SendBuffer failed");
     return false;
  }

  SetTimeSent(container);

  InspectContainer("send container", *container);

  while (true) {
    switch (send_event) {
    case ContainerBuffer::BUFFER_EVENT_START:
      send_event = send_cbuf->SendStart(sd_, ssl_, send_timeout_msec_, container, true);
      break;
    case ContainerBuffer::BUFFER_EVENT_STEP:
      send_event = send_cbuf->SendStep();
      break;
    case ContainerBuffer::BUFFER_EVENT_COMPLETE:
      return true;
    default:
      log_err("SendContainer failed with BufferEvent %d", send_event);
      return false;
    }
  }

  if (send_cbuf)
    delete send_cbuf;
#endif
}

bool Courier::RecvContainer(annex::Container *container) {
#if SI_WITH_BLOCKING_METHODS
  bool rc;
  ssize_t count;
  uint32_t len;
  uint8_t *buf;

  count = RecvBuffer(&len, sizeof(len));
  if (count < 0) {
    log_err("RecvBuffer failed");
    return false;
  }
  len = ntohl(len);

  buf = new(std::nothrow) uint8_t [len];
  if (!buf) {
    log_err("malloc buf failed");
    return false;
  }

  count = RecvBuffer(buf, len);
  if (count < 0) {
    log_err("RecvBuffer failed");
    delete [] buf;
    return false;
  }

  container->Clear();
  rc = container->ParseFromArray(buf, len);
  if (!rc) {
    log_err("ParseFromArray failed");
    delete [] buf;
    return false;
  }

  delete [] buf;

  InspectContainer("recv container", *container);

  return true;
#else
  ContainerBuffer *recv_cbuf = NULL;
  ContainerBuffer::BufferEvent recv_event = ContainerBuffer::BUFFER_EVENT_START;

  recv_cbuf = new class ContainerBuffer;
  if (recv_cbuf == NULL) {
     log_err("RecvBuffer failed");
     return false;
  }

  while (true) {
    switch (recv_event) {
    case ContainerBuffer::BUFFER_EVENT_START:
      recv_event = recv_cbuf->RecvStart(sd_, ssl_, recv_timeout_msec_, true);
      break;
    case ContainerBuffer::BUFFER_EVENT_STEP:
      recv_event = recv_cbuf->RecvStep();
      break;
    case ContainerBuffer::BUFFER_EVENT_COMPLETE:
      if(!recv_cbuf->ParseContainer(container)) {
        log_err("ParseContainer failed: ignoring");
        return false;
      }
      InspectContainer("recv container", *container);
      return true;
    default:
      log_err("RecvContainer failed with BufferEvent %d", recv_event);
      return false;
    }
  }

  if (recv_cbuf)
      delete recv_cbuf;
#endif
}

annex::Package *Courier::AddPackage(annex::Container *container) {
  annex::Package *package = container->add_packages();

  SetTimePackaged(package);
  SetMessageId(package);

  return package;
}

AnnexTransfer *Courier::NewTransfer() {
  AnnexTransfer *transfer = new(std::nothrow) AnnexTransfer;
  if (!transfer) {
    log_err("malloc AnnexTransfer failed");
    return false;
  }
  SetTimePackaged(transfer->mutable_package());

  return transfer;
}

void Courier::HandleSetClientConfig(
    const annex::Package &package_in,
    const annex::ClientConfig &client_config) {

  if (client_config.has_rpd_interval()) {
    if (client_config.rpd_interval() <= 0) {
      return;
    }
    set_rpd_interval(client_config.rpd_interval());
  }
  if (client_config.has_gps_interval()) {
    set_gps_interval(client_config.gps_interval());
  }
  if (client_config.has_stats_interval()) {
    set_stats_interval(client_config.stats_interval());
  }
  if (client_config.has_cell_interval()) {
    set_cell_interval(client_config.cell_interval());
  }
  if (client_config.has_trigger_interval()) {
    set_trigger_interval(client_config.trigger_interval());
  }

  // FIXME: need to add code for supporting SSL options

  if (client_config.has_dm_host_name()) {
    if (host().compare(client_config.dm_host_name())) {
      set_host(client_config.dm_host_name());
      restart_courier_ = true;
    }
  }
  if (client_config.has_dm_host_port()) {
    if (port() != client_config.dm_host_port()) {
      set_port(client_config.dm_host_port());
      restart_courier_ = true;
    }
  }
}

#if SSL_SUPPORT
bool Courier::SaslAuthMore(
    const annex::Container &recv_container,
    annex::Package_StatusCode *status_code) {
  bool rc;
  char token_out[kSaslTokenSize];
  char token_in[kSaslTokenSize];

#if 0
  if (recv_container.packages_size() != 1) {
    log_err("expected 1 package but got %d", recv_container.packages_size());
    return false;
  }
#endif

  if (!recv_container.packages(0).has_status_code()) {
    log_err("expected status_code");
    return false;
  }

  *status_code = recv_container.packages(0).status_code();
  if (recv_container.packages(0).status_code() !=
      annex::Package::STATUS_CODE_WAITING_FOR_RESPONSE) {
    return true;
  }

  if (!recv_container.packages(0).has_content() ||
      !recv_container.packages(0).content().has_attribute() ||
      !recv_container.packages(0).content().attribute().has_type() ||
      recv_container.packages(0).content().attribute().type() !=
        annex::Attribute::TYPE_SASL_TOKEN ||
      !recv_container.packages(0).content().attribute().has_sasl_token()) {
    log_err("expected sasl_token");
    return false;
  }

  token_in[sizeof(token_in) - 1] = '\0';

  do {
    strncpy(
        token_in,
        recv_container.packages(0).content().attribute().sasl_token().base64_data().c_str(),
        sizeof(token_in) - 1
    );
    int sasl_code = sasl_auth_reply(sasl_conn_, token_in, token_out, sizeof(token_out));
    if (sasl_code != SASL_OK && sasl_code != SASL_CONTINUE) {
      log_err("sasl_client_step failed: %s", sasl_errdetail(sasl_conn_));
      return false;
    }

    annex::Container container;
    annex::Package *package;
    annex::Content *content;

    package = AddPackage(&container);
    if (!package) {
      log_err("AddPackage failed");
      return false;
    }
    content = package->mutable_content();
    content->set_type(annex::Content::TYPE_SESSION_INIT);
    annex::Attribute *attr = content->mutable_attribute();

    attr->set_type(annex::Attribute::TYPE_SASL_TOKEN);
    annex::SaslToken *sasl_token = attr->mutable_sasl_token();
    sasl_token->set_base64_data(token_out);

    rc = SendContainer(&container);
    if (!rc) {
      log_err("SendContainer failed");
      return false;
    }

    rc = RecvContainer(&container);
    if (!rc) {
      log_err("RecvContainer failed");
      return false;
    }

    return SaslAuthMore(container, status_code);
  } while (*status_code == annex::Package::STATUS_CODE_WAITING_FOR_RESPONSE);
}

bool Courier::SaslServerMechList(char *list, size_t list_size) {
  bool rc;
  annex::Container container;
  annex::Package *package;
  annex::Content *content;
  annex::Attribute *attr;
  annex::Action *action;

  package = AddPackage(&container);
  if (!package) {
    log_err("AddPackage failed");
    return false;
  }
  content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ACTION);
  action = content->mutable_action();
  action->set_type(annex::Action::TYPE_GET);
  attr = action->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_SASL_SERVER_MECH_LIST);
  annex::SaslServerMechList *mech_list = attr->mutable_sasl_server_mech_list();
  (void) mech_list;

  rc = SendContainer(&container);
  if (!rc) {
    log_err("SendContainer failed");
    return false;
  }

  rc = RecvContainer(&container);
  if (!rc) {
    log_err("RecvContainer failed");
    return false;
  }

  if (container.packages_size() != 1) {
    log_err("expected 1 package but got %d", container.packages_size());
    return false;
  }

  if (!container.packages(0).has_status_code()) {
    log_err("expected status_code");
    return false;
  }

  if (container.packages(0).status_code() !=
      annex::Package::STATUS_CODE_OK) {
    return false;
  }

  if (!container.packages(0).has_content() ||
      !container.packages(0).content().has_attribute() ||
      !container.packages(0).content().attribute().has_type() ||
      container.packages(0).content().attribute().type() !=
        annex::Attribute::TYPE_SASL_SERVER_MECH_LIST ||
      !container.packages(0).content().attribute().has_sasl_server_mech_list()) {
    log_err("expected sasl_server_mech_list");
    return false;
  }

  const annex::SaslServerMechList ml =
      container.packages(0).content().attribute().sasl_server_mech_list();

  if (ml.mechanism_size() == 0) {
    log_err("expected at least 1 mechanism");
    return false;
  }

  size_t count = 0;
  for (int i = 0; i < ml.mechanism_size(); ++i) {
    int mech_len = ml.mechanism(i).length();
    if (count + mech_len + 1 < list_size) {
      if (count) {
        strcpy(list + count, " ");
        count++;
      }
      strcpy(list + count, ml.mechanism(i).c_str());
      count += mech_len;
    }
  }

  log_debug("mech_list: %s", list);

  return true;
}
#endif

bool Courier::SessionInit(annex::Package_StatusCode *status_code) {
  bool rc;
  int tmp;
  int sasl_code;
  const char *mech;
  char token_out[kSaslTokenSize];
  char hostname[256];

  tmp = gethostname(hostname, sizeof(hostname));
  if (tmp < 0) {
    log_err("gethostname failed: %m");
    *hostname = '\0';
  }

  annex::Container container;
  annex::Package *package;
  annex::Content *content;

  package = AddPackage(&container);
  if (!package) {
    log_err("AddPackage failed");
    return false;
  }
  content = package->mutable_content();
  content->set_type(annex::Content::TYPE_SESSION_INIT);
  annex::SessionInit *si = content->mutable_session_init();
  si->set_hostname(hostname);
  si->set_vendor_id(vendor_id_);
  si->set_product_id(product_id_);
  si->set_device_id(device_id_);
  si->set_synchronized(synchronous_);
  si->set_container_buffer_max(ContainerBuffer::kSerializedMax);

#if SSL_SUPPORT
  if (sasl_force_mech_.length()) {
    if (sasl_force_mech_.length() >= sizeof(sasl_mech_list_)) {
      log_err("sasl_force_mech_ is longer than buffer");
      return false;
    }
    strncpy(sasl_mech_list_, sasl_force_mech_.c_str(), sizeof(sasl_mech_list_));
  } else {
    SaslServerMechList(sasl_mech_list_, sizeof(sasl_mech_list_));
  }

  if (sasl_username_.empty() || sasl_passwd_.empty()) {
    log_err("sasl username and passwd are required");
    return false;
  }

  sasl_code = sasl_auth_callbacks_init(
      sasl_callbacks_,
      sizeof(sasl_callbacks_) / sizeof(sasl_callbacks_[0]),
      sasl_username_.c_str(),
      sasl_passwd_.c_str(),
      sasl_realm_.c_str()
  );
  if (sasl_code != SASL_OK) {
    log_err("sasl_auth_callbacks_init failed: %d", sasl_code);
    return false;
  }

  sasl_code = sasl_auth_start(
      &sasl_conn_,
      sasl_callbacks_,
      sasl_mech_list_,
      &mech, token_out, sizeof(token_out)
  );
  if (sasl_code != SASL_OK && sasl_code != SASL_CONTINUE) {
    log_err("sasl_auth_start failed: %d", sasl_code);
    return false;
  }

  annex::SaslAuth *sasl_auth = si->mutable_sasl_auth();
  sasl_auth->set_mechanism(mech);
  annex::SaslToken *sasl_token = sasl_auth->mutable_sasl_token();
  sasl_token->set_base64_data(token_out);
#endif

  rc = SendContainer(&container);
  if (!rc) {
    log_err("SendContainer failed");
    return false;
  }

  rc = RecvContainer(&container);
  if (!rc) {
    log_err("RecvContainer failed");
    return false;
  }

#if SSL_SUPPORT
  rc = SaslAuthMore(container, status_code);
  if (!rc) {
    log_err("SaslAuthMore failed");
    return false;
  }
#else
  *status_code = container.packages(0).status_code();
#endif

  log_info("session initialized");

  // If client config is there, handle it.
  if (container.packages_size() == 2) {
    if (container.packages(1).has_content()) {
      if (container.packages(1).content().has_action() &&
        (container.packages(1).content().action().type() == annex::Action::TYPE_SET) &&
        (container.packages(1).content().action().has_attribute()) &&
        (container.packages(1).content().action().attribute().type() == annex::Attribute::TYPE_CLIENT_CONFIG)) {
          HandleSetClientConfig(container.packages(1), 
            container.packages(1).content().action().attribute().client_config());
      }
    }
  }

  return true;
}

void Courier::AddTransferListener(AnnexTransferAgent *listener) {
  mutex_lock(&transfer_listeners_mutex_);

  transfer_listeners_.push_back(listener);

  mutex_unlock(&transfer_listeners_mutex_);
}

void Courier::RemoveTransferListener(AnnexTransferAgent *listener) {
  mutex_lock(&transfer_listeners_mutex_);

  for (std::list<AnnexTransferAgent *>::iterator iter = transfer_listeners_.begin();
       iter != transfer_listeners_.end(); ) {
    if (*iter == listener) {
      iter = transfer_listeners_.erase(iter);
    } else {
      ++iter;
    }
  }

  mutex_unlock(&transfer_listeners_mutex_);
}

int Courier::FireTransferEvent(AnnexTransfer *transfer, AnnexTransferAgent::TransferEvent event) {
  int rc = false;

  transfer->AddTransferEvent(event);

  mutex_lock(&transfer_listeners_mutex_);

  for (std::list<AnnexTransferAgent *>::iterator iter = transfer_listeners_.begin();
       iter != transfer_listeners_.end();
       ++iter) {
    rc = (*iter)->TransferEventFired(transfer, event);
    if (rc) {
      break;
    }
  }

  mutex_unlock(&transfer_listeners_mutex_);

  return rc;
}

bool Courier::QueueTransfer(AnnexTransfer *transfer) {
  std::list<AnnexTransfer *>::iterator iter;

  uint32_t message_id = NextMessageId();

  transfer->mutable_package()->set_message_id(message_id);

  mutex_lock(&sendq_mutex_);

  for (iter = sendq_.begin(); iter != sendq_.end(); ++iter) {
    if (transfer->package().priority() <= (*iter)->package().priority()) {
      break;
    }
  }

  sendq_.insert(iter, transfer);

  mutex_unlock(&sendq_mutex_);

  return true;
}

bool Courier::SwitchNonblocking() {
  int tmp;
  struct timeval tv;

  tv.tv_sec = 0;
  tv.tv_usec = 0;

  tmp = setsockopt(sd_, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
  if (tmp < 0) {
    log_err("setsockopt SO_RCVTIMEO: %m");
    return false;
  }

  tmp = setsockopt(sd_, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
  if (tmp < 0) {
    log_err("setsockopt SO_SNDTIMEO: %m");
    return false;
  }

  tmp = nonblocking(sd_, 1);
  if (tmp < 0) {
    log_err("set_nonblocking failed");
    return false;
  }

  return true;
}

void Courier::HandleSendqTimeouts() {
  int tmp;
  struct timeval now;

  tmp = gettimeofday(&now, NULL);
  if (tmp < 0) {
    log_err("gettimeofday failed: %m");
    return;
  }

  mutex_lock(&sendq_mutex_);

  for (std::list<AnnexTransfer *>::iterator iter = sendq_.begin();
       iter != sendq_.end(); ) {
    if (!(*iter)->sending() && (*iter)->SendqTimedout(&now)) {
      AnnexTransfer *transfer = *iter;
      iter = sendq_.erase(iter);
      int handled = FireTransferEvent(transfer, AnnexTransferAgent::TRANSFER_EVENT_SENDQ_TIMEOUT);
      if (!handled) {
        delete transfer;
      }
    } else {
      ++iter;
    }
  }

  mutex_unlock(&sendq_mutex_);
}

bool Courier::SendqStartSending(annex::Container *container) {
  int size_estimate = 0;

  container->Clear();

  mutex_lock(&sendq_mutex_);

  for (std::list<AnnexTransfer *>::reverse_iterator iter = sendq_.rbegin();
       iter != sendq_.rend();
       ++iter) {
    size_estimate += (*iter)->package().ByteSize() + 128;
    if (size_estimate >= ContainerBuffer::kSerializedMax) {
      break;
    }

    *container->add_packages() = (*iter)->package();
    (*iter)->set_sending(true);
  }

  mutex_unlock(&sendq_mutex_);

  if (container->packages_size() == 0) {
    return false;
  }

  SetTimeSent(container);

  InspectContainer("send container", *container);

  return true;
}

bool Courier::SendqFinishSending(bool success) {
  std::list<AnnexTransfer *>::iterator iter = sendq_.begin();

  mutex_lock(&sendq_mutex_);

  while (iter != sendq_.end()) {
    if (success) {
      if ((*iter)->sending()) {
        AnnexTransfer *transfer = *iter;
        iter = sendq_.erase(iter);
        int handled = FireTransferEvent(transfer, AnnexTransferAgent::TRANSFER_EVENT_SENT);
        if (!handled) {
          delete transfer;
        }
      } else {
        ++iter;
      }
    } else {
      (*iter)->set_sending(false);
      ++iter;
    }
  }

  mutex_unlock(&sendq_mutex_);

  return true;
}

bool Courier::NotifyPackagesRecv(const annex::Container &container) {
  int handled;

  for (int i = 0; i < container.packages_size(); ++i) {
    AnnexTransfer *transfer = new(std::nothrow) AnnexTransfer;
    if (!transfer) {
      log_err("malloc AnnexTransfer failed");
      continue;
    }
    *transfer->mutable_package() = container.packages(i);

#if 0
    if (log_level() == LOG_DEBUG) {
      container.packages(i).PrintDebugString();
    }
#endif

    if (container.packages(i).has_correlation_id()) {
      handled = FireTransferEvent(transfer, AnnexTransferAgent::TRANSFER_EVENT_REPLY);
    } else {
      handled = FireTransferEvent(transfer, AnnexTransferAgent::TRANSFER_EVENT_RECV);
    }
    if (!handled) {
      delete transfer;
    }
  }

  return true;
}

void Courier::RunnableCore() {
  bool rc;
  annex::Package_StatusCode status_code;
  annex::Container container;

  ContainerBuffer *send_cbuf = NULL;
  ContainerBuffer::BufferEvent send_event;

  ContainerBuffer *recv_cbuf = NULL;
  ContainerBuffer::BufferEvent recv_event;

reopen:
  if (runnable_stop_) {
    return;
  }

  Close();

  HandleSendqTimeouts();

#if PPP_SUPPORT // specific to EN2/EN3 models
  if (when_ppp_up()) {
    if (! is_ppp_link_up()) {
      printf ("PPP link is not up. Not sending RM data\n");
      return;
    }
  }
#if 0
  else {
    if (bring_ppp_up()) {
      if (! is_ppp_link_up()) {
        printf ("Bringing the PPP link up\n");
        if (! bring_ppp_link_up()) {
          log_err("Unable to bring the PPP link up");
          return;
        }
        printf ("PPP link is up now\n");
      }
    }
  }
#endif
#endif

  rc = Open();
  if (!rc) {
    log_err("Open failed");
    sleep_msec(1000);
    goto reopen;
  }

  rc = SessionInit(&status_code);
  if (!rc) {
    log_err("SessionInit failed");
    sleep_msec(1000);
    goto reopen;
  }

  if (status_code != annex::Package::STATUS_CODE_OK) {
    log_err("SessionInit returned bad status code %u", status_code);
    sleep_msec(1000);
    goto reopen;
  }

#if SI_WITH_BLOCKING_METHODS
  rc = SwitchNonblocking();
  if (!rc) {
    sleep_msec(1000);
    goto reopen;
  }
#endif

  if (send_cbuf == NULL)
    send_cbuf = new class ContainerBuffer;
  if (recv_cbuf == NULL)
    recv_cbuf = new class ContainerBuffer;
  if ((send_cbuf == NULL) || (recv_cbuf == NULL)) {
    log_err("No memory for send OR receive buffers. Courier failed");
    Close();
  }

  send_event = ContainerBuffer::BUFFER_EVENT_START;
  recv_event = ContainerBuffer::BUFFER_EVENT_START;

  while (!runnable_stop_) {
    HandleSendqTimeouts();

    if (restart_courier_) {
      log_info("Restarting courier due to change in either hostname or host port");
      restart_courier_ = false;
      goto reopen;
    }

    switch (send_event) {
    case ContainerBuffer::BUFFER_EVENT_START:
      if (SendqStartSending(&container)) {
        send_event = send_cbuf->SendStart(sd_, ssl_, send_timeout_msec_, &container, false);
      }
      break;
    case ContainerBuffer::BUFFER_EVENT_STEP:
      send_event = send_cbuf->SendStep();
      break;
    case ContainerBuffer::BUFFER_EVENT_COMPLETE:
      SendqFinishSending(true);
      send_event = ContainerBuffer::BUFFER_EVENT_START;
      break;
    default:
      SendqFinishSending(false);
      send_event = ContainerBuffer::BUFFER_EVENT_START;
      goto reopen;
    }

    switch (recv_event) {
    case ContainerBuffer::BUFFER_EVENT_START:
      recv_event = recv_cbuf->RecvStart(sd_, ssl_, recv_timeout_msec_, false);
      break;
    case ContainerBuffer::BUFFER_EVENT_STEP:
      recv_event = recv_cbuf->RecvStep();
      break;
    case ContainerBuffer::BUFFER_EVENT_COMPLETE:
      if(!recv_cbuf->ParseContainer(&container)) {
        log_err("ParseContainer failed: ignoring");
      } else {
        InspectContainer("recv container", container);
        NotifyPackagesRecv(container);
      }
      recv_event = ContainerBuffer::BUFFER_EVENT_START;
      break;
    default:
      recv_event = ContainerBuffer::BUFFER_EVENT_START;
      goto reopen;
    }

    sleep_msec(200);
  }

  if (recv_cbuf)
      delete recv_cbuf;
  if (send_cbuf)
      delete send_cbuf;

  Close();
}

bool Courier::Close() {
  int tmp;

#if SSL_SUPPORT
  if (sasl_conn_) {
    sasl_auth_end(&sasl_conn_);
    sasl_conn_ = NULL;
  }

  if (ssl_) {
    SSL_shutdown(ssl_);
    SSL_free(ssl_);
    ssl_ = NULL;
  }
  if (ssl_ctx_) {
    SSL_CTX_free(ssl_ctx_);
    ssl_ctx_ = NULL;
  }
#endif

  if (sd_ >= 0) {
    tmp = close(sd_);
    if (tmp < 0) {
      log_err("close: %m");
    }
    log_info("closed connection with %s:%u", host_.c_str(), port_);
    sd_ = -1;
  }

  return true;
}

bool Courier::Open() {
  struct sockaddr_in serv_addr;

  struct hostent h;
  struct hostent *hp;
  char buf[1024];
  int herrno;

  int tmp;

  if (sd_ >= 0) {
    log_err("connection already open");
    return false;
  }

  log_info("opening connection with %s:%u", host_.c_str(), port_);

  tmp = gethostbyname_r(host_.c_str(), &h, buf, sizeof(buf), &hp, &herrno);
  if (!hp) {
    log_err("gethostbyname_r: %d", herrno);
    return false;
  }

  int sd = socket(PF_INET, SOCK_STREAM, 0);
  if (sd < 0) {
    log_err("socket: %m");
    return false;
  }

  memset(&serv_addr, 0, sizeof(serv_addr));
  serv_addr.sin_family = AF_INET;
  serv_addr.sin_port = htons(port_);
  memcpy(&serv_addr.sin_addr.s_addr, h.h_addr_list[0], 4);

#if SI_WITH_BLOCKING_METHODS
  struct timeval tv;

  tv.tv_sec = recv_timeout_msec_ / 1000;
  tv.tv_usec = (recv_timeout_msec_ % 1000) * 1000;
  tmp = setsockopt(sd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
  if (tmp < 0) {
    log_err("setsockopt SO_RCVTIMEO: %m");
    close(sd);
    return false;
  }

  tv.tv_sec = send_timeout_msec_ / 1000;
  tv.tv_usec = (send_timeout_msec_ % 1000) * 1000;
  tmp = setsockopt(sd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
  if (tmp < 0) {
    log_err("setsockopt SO_SNDTIMEO: %m");
    close(sd);
    return false;
  }
#endif

  tmp = connect(sd, (struct sockaddr *) &serv_addr, sizeof(serv_addr));
  if (tmp < 0) {
    log_err("connect: %m");
    close(sd);
    return false;
  }

#if SSL_SUPPORT
  if (ssl_method_.length() && ssl_method_.compare("none")) {
    if (ssl_method_.compare("ssl")) {
      ssl_ctx_ = SSL_CTX_new(SSLv23_client_method());
    } else if (ssl_method_.compare("tls")) {
      ssl_ctx_ = SSL_CTX_new(TLSv1_client_method());
    } else {
      log_err("bad ssl-method: %s", ssl_method_.c_str());
      close(sd);
      return false;
    }
    if (!ssl_ctx_) {
      ssl_dump_err_queue("SSL_CTX_new failed");
      close(sd);
      return false;
    }

    ssl_ = SSL_new(ssl_ctx_);
    if (!ssl_) {
      ssl_dump_err_queue("SSL_new failed");
      SSL_CTX_free(ssl_ctx_);
      ssl_ctx_ = NULL;
      close(sd);
      return false;
    }

    tmp = SSL_set_fd(ssl_, sd);
    if (!tmp) {
      ssl_dump_err_queue("SSL_set_fd failed");
      SSL_shutdown(ssl_);
      SSL_free(ssl_);
      ssl_ = NULL;
      SSL_CTX_free(ssl_ctx_);
      ssl_ctx_ = NULL;
      close(sd);
      return false;
    }

    tmp = SSL_connect(ssl_);
    if (tmp != 1) {
      ssl_log_ssl_error("SSL_connect failed", ssl_, tmp);
      SSL_shutdown(ssl_);
      SSL_free(ssl_);
      ssl_ = NULL;
      SSL_CTX_free(ssl_ctx_);
      ssl_ctx_ = NULL;
      close(sd);
      return false;
    }
  }
#endif

#if !SI_WITH_BLOCKING_METHODS
  tmp = nonblocking(sd, 1);
  if (tmp < 0) {
    log_err("set_nonblocking failed");
    close(sd);
    return false;
  }
#endif

  server_addr_ = serv_addr.sin_addr;

  sd_ = sd;

  log_info("opened connection with %s:%u", host_.c_str(), port_);

  return true;
}

bool Courier::set_recv_timeout_msec(int value) {
  if (value <= 0) {
    return false;
  }
  recv_timeout_msec_ = value;

  return true;
}

int Courier::recv_timeout_msec() const {
  return recv_timeout_msec_;
}

bool Courier::set_send_timeout_msec(int value) {
  if (value <= 0) {
    return false;
  }
  send_timeout_msec_ = value;

  return true;
}

int Courier::send_timeout_msec() const {
  return send_timeout_msec_;
}

bool Courier::set_host(const char *value) {
  host_ = value;

  return true;
}

bool Courier::set_host(const std::string &value) {
  host_ = value;

  return true;
}

const std::string &Courier::host() const {
  return host_;
}

bool Courier::set_port(uint16_t value) {
  port_ = value;

  return true;
}

uint16_t Courier::port() const {
  return port_;
}

bool Courier::set_vendor_id(const char *value) {
  vendor_id_ = value;

  return true;
}

bool Courier::set_vendor_id(const std::string &value) {
  vendor_id_ = value;

  return true;
}

const std::string &Courier::vendor_id() const {
  return vendor_id_;
}

bool Courier::set_product_id(const char *value) {
  product_id_ = value;

  return true;
}

bool Courier::set_product_id(const std::string &value) {
  product_id_ = value;

  return true;
}

const std::string &Courier::product_id() const {
  return product_id_;
}

bool Courier::set_device_id(const char *value) {
  device_id_ = value;

  return true;
}

bool Courier::set_device_id(const std::string &value) {
  device_id_ = value;

  return true;
}

const std::string &Courier::device_id() const {
  return device_id_;
}

bool Courier::set_ssl_method(const char *value) {
  ssl_method_ = value;

  return true;
}

bool Courier::set_ssl_method(const std::string &value) {
  ssl_method_ = value;

  return true;
}

const std::string &Courier::set_ssl_method() const {
  return ssl_method_;
}

bool Courier::set_sasl_username(const char *value) {
  sasl_username_ = value;

  return true;
}

bool Courier::set_sasl_username(const std::string &value) {
  sasl_username_ = value;

  return true;
}

const std::string &Courier::sasl_username() const {
  return sasl_username_;
}

bool Courier::set_sasl_passwd(const char *value) {
  sasl_passwd_ = value;

  return true;
}

bool Courier::set_sasl_passwd(const std::string &value) {
  sasl_passwd_ = value;

  return true;
}

const std::string &Courier::sasl_passwd() const {
  return sasl_passwd_;
}

bool Courier::set_sasl_realm(const char *value) {
  sasl_realm_ = value;

  return true;
}

bool Courier::set_sasl_realm(const std::string &value) {
  sasl_realm_ = value;

  return true;
}

const std::string &Courier::sasl_realm() const {
  return sasl_realm_;
}

bool Courier::set_sasl_force_mech(const char *value) {
  sasl_force_mech_ = value;

  return true;
}

bool Courier::set_sasl_force_mech(const std::string &value) {
  sasl_force_mech_ = value;

  return true;
}

const std::string &Courier::sasl_force_mech() const {
  return sasl_force_mech_;
}

void Courier::set_synchronous(bool value) {
  synchronous_ = value;
}

bool Courier::synchronous() const {
  return synchronous_;
}

bool Courier::set_rpd_interval(int value) {
  if (rpd_interval_ < 0)
    return false;
  rpd_interval_ = value;
  return true;
}

int Courier::rpd_interval() const {
  return rpd_interval_;
}

bool Courier::set_log_level(int value) {
  log_level_ = value;

  inspect_containers_ = (log_level_ == LOG_DEBUG) ? true : false;

  return true;
}

int Courier::log_level() const {
  return log_level_;
}

bool Courier::set_gps_interval(int value) {
  if (gps_interval_ < 0)
    return false;
  gps_interval_ = value;
  return true;
}

int Courier::gps_interval() const {
  return gps_interval_;
}

bool Courier::set_stats_interval(int value) {
  if (stats_interval_ < 0)
    return false;
  stats_interval_ = value;
  return true;
}

int Courier::stats_interval() const {
  return stats_interval_;
}

bool Courier::set_cell_interval(int value) {
  if (cell_interval_ < 0)
    return false;
  cell_interval_ = value;
  return true;
}

int Courier::cell_interval() const {
  return cell_interval_;
}

bool Courier::set_trigger_interval(int value) {
  if (trigger_interval_ < 0)
    return false;
  trigger_interval_ = value;
  return true;
}

bool Courier::when_ppp_up() const {
    return when_ppp_up_;
}

bool Courier::set_when_ppp_up(bool value) {
    when_ppp_up_ = value;
    return (when_ppp_up_);
}

bool Courier::bring_ppp_up() const {
  return bring_ppp_up_;
}

bool Courier::set_bring_ppp_up(bool value) {
    bring_ppp_up_ = value;
    return (bring_ppp_up_);
}

bool Courier::firmware_upgrade() const {
  return firmware_upgrade_;
}

void Courier::set_firmware_upgrade(int value) {
    firmware_upgrade_ = value ? true : false;
}

bool Courier::config_upgrade() const {
  return config_upgrade_;
}

void Courier::set_config_upgrade(int value) {
    config_upgrade_ = value ? true : false;
}

int Courier::trigger_interval() const {
  return trigger_interval_;
}

void Courier::restart_courier() {
  restart_courier_ = true;
}

}
