/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#ifndef COURIER_H_
#define COURIER_H_

#include <pthread.h>
#include <string>
#include <stdint.h>
#include <list>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/time.h>

#include <annex.pb.h>

#include <annex_common.h>
#include <runnable.h>
#include <gps_receiver.h>
#include <annex_transfer_agent.h>
#include <annex_transfer.h>
#include <validator.h>

#define SSL_SUPPORT 1

namespace mts {

class Courier : public Runnable {
 public:
  static const uint16_t kAnnexPort = 5798;
  static const size_t kTimestampSize = 26;
  static const int kRecvTimeoutMsecDefault = 30000;
  static const int kSendTimeoutMsecDefault = 30000;
  static const char *kTimestampFormat;
  static const int kRpdInterval = 1000;
  static const int kTriggerInterval = 1000;

  Courier();
  ~Courier();

  bool set_recv_timeout_msec(int value);
  int recv_timeout_msec() const;

  bool set_send_timeout_msec(int value);
  int send_timeout_msec() const;

  bool set_host(const char *value);
  bool set_host(const std::string &value);
  const std::string &host() const;

  bool set_port(uint16_t value);
  uint16_t port() const;

  bool set_vendor_id(const char *value);
  bool set_vendor_id(const std::string &value);
  const std::string &vendor_id() const;

  bool set_product_id(const char *value);
  bool set_product_id(const std::string &value);
  const std::string &product_id() const;

  bool set_device_id(const char *value);
  bool set_device_id(const std::string &value);
  const std::string &device_id() const;

  bool set_ssl_method(const char *value);
  bool set_ssl_method(const std::string &value);
  const std::string &set_ssl_method() const;

  bool set_sasl_username(const char *value);
  bool set_sasl_username(const std::string &value);
  const std::string &sasl_username() const;

  bool set_sasl_passwd(const char *value);
  bool set_sasl_passwd(const std::string &value);
  const std::string &sasl_passwd() const;

  bool set_sasl_realm(const char *value);
  bool set_sasl_realm(const std::string &value);
  const std::string &sasl_realm() const;

  bool set_sasl_force_mech(const char *value);
  bool set_sasl_force_mech(const std::string &value);
  const std::string &sasl_force_mech() const;

  bool set_rpd_interval(int value);
  int rpd_interval() const;

  bool set_log_level(int value);
  int log_level() const;

  bool set_gps_interval(int value);
  int gps_interval() const;

  bool set_stats_interval(int value);
  int stats_interval() const;

  bool set_cell_interval(int value);
  int cell_interval() const;

  bool set_when_ppp_up(bool value);
  bool when_ppp_up() const;

  bool set_bring_ppp_up(bool value);
  bool bring_ppp_up() const;

  bool firmware_upgrade() const;
  void set_firmware_upgrade(int value);

  bool config_upgrade() const;
  void set_config_upgrade(int value);

  bool set_trigger_interval(int value);
  int trigger_interval() const;

  void set_synchronous(bool value);
  bool synchronous() const;

  void restart_courier();

  uint32_t NextMessageId();
  static bool AnnexTimestamp(time_t time_at, std::string *str);
  static bool PackageExpired(const annex::Package &package);
  static void SetTimePackaged(annex::Package *package);
  static AnnexTransfer *NewTransfer();
  bool QueueTransfer(AnnexTransfer *transfer);
  void AddTransferListener(AnnexTransferAgent *listener);
  void RemoveTransferListener(AnnexTransferAgent *listener);
  int FireTransferEvent(AnnexTransfer *transfer, AnnexTransferAgent::TransferEvent event);
 private:
  static const int kSaslTokenSize = (1 << 11);

  void RunnableCore();

  bool TestMe();

  void InspectContainer(const char *prefix, const annex::Container &container);
  void SetMessageId(annex::Package *package);
  bool SetTimeSent(annex::Container *container);

  ssize_t RecvBuffer(void *buf, size_t len);
  ssize_t SendBuffer(const void *buf, size_t len);
  bool SendContainer(annex::Container *container);
  bool RecvContainer(annex::Container *container);

  void HandleSetClientConfig( const annex::Package &package_in,
                              const annex::ClientConfig &client_config);
  bool SaslServerMechList(char *list, size_t list_size);
  bool SaslAuthMore(const annex::Container &recv_container,
                    annex::Package_StatusCode *status_code);
  annex::Package *AddPackage(annex::Container *container);
  bool SessionInit(annex::Package_StatusCode *status_code);

  bool Close();
  bool Open();

  bool SwitchNonblocking();
  void HandleSendqTimeouts();
  bool SendqStartSending(annex::Container *container);
  bool SendqFinishSending(bool success);
  bool NotifyPackagesRecv(const annex::Container &container);

  std::string host_;
  uint16_t port_;

  std::string hostname_;
  std::string vendor_id_;
  std::string product_id_;
  std::string device_id_;

  bool synchronous_;

  int recv_timeout_msec_;
  int send_timeout_msec_;

  int rpd_interval_;
  int log_level_;

  int gps_interval_;
  int stats_interval_;
  int cell_interval_;
  int trigger_interval_;

  bool when_ppp_up_;
  bool bring_ppp_up_;

  bool firmware_upgrade_;
  bool config_upgrade_;

  bool restart_courier_;

  uint32_t message_id_;
  pthread_mutex_t message_id_mutex_;

  int sd_;
#if SSL_SUPPORT
  SSL *ssl_;
  SSL_CTX *ssl_ctx_;
#else
  char *ssl_;
#endif

  std::string ssl_method_;

#if SSL_SUPPORT
  sasl_conn_t *sasl_conn_;
  sasl_callback_t sasl_callbacks_[8];
#endif
  char sasl_mech_list_[256];
  std::string sasl_server_mech_list_;
  std::string sasl_username_;
  std::string sasl_passwd_;
  std::string sasl_realm_;
  std::string sasl_force_mech_;

  std::list<AnnexTransfer *> sendq_;
  pthread_mutex_t sendq_mutex_;

  std::list<AnnexTransferAgent *> transfer_listeners_;
  pthread_mutex_t transfer_listeners_mutex_;

  struct in_addr server_addr_;

  bool inspect_containers_;
  Validator validator_;

  DISALLOW_COPY_AND_ASSIGN(Courier);
};

}
#endif  // COURIER_H_
