/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <pthread.h>
#include <delivery_agent.h>
#include <courier.h>
#include <container_buffer.h>

#include <linux/if.h>
#include <unistd.h>
#include <sys/reboot.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

namespace mts {

const char *DeliveryAgent::kSysfsNetDir = "/sys/class/net";

DeliveryAgent::DeliveryAgent() {
  synchronous_ = false;
  restart_ = false;

  su_uri_ = "";
  su_md5sum_ = "";
  su_prev_uri_ = "";
  su_prev_md5sum_ = "";
  su_resume_ = false;
  su_name_ = "";
  su_rcell_ = 0;
  su_rcell_config_ = 0;
  su_cdp_ = 0;
  su_download_file_path_ = "";
}

DeliveryAgent::~DeliveryAgent() {
  Stop();
}

bool DeliveryAgent::TransferEventRecv(AnnexTransfer *transfer) {
  AddTransfer(transfer);

  return true;
}

void DeliveryAgent::AddIpAddrs(annex::InetSettings *is, const char *dev) {
  FILE *ip_stream;
  char buf[1024];
  char *line;
  char *cp, *cp1;
  int tmp;

  ip_stream = popen(std::string("ip address show dev " + std::string(dev)).c_str(), "r");
  if (!ip_stream) {
    return;
  }

  while ((line = fgets(buf, sizeof(buf), ip_stream))) {
    if (strncmp(line, "    inet ", strlen("    inet "))) {
      continue;
    }
    line += strlen("    inet ");

    cp = strchr(line, ' ');
    if (cp) {
      *cp++ = '\0';
      cp1 = cp;
    }
    else {
      log_debug("invalid inet line: '%s'", buf);
      continue;
    }

    cp = strchr(line, '/');
    if (!cp) {
      cp = strchr(cp1, '/');
      if (! cp) {
        log_debug("invalid inet line: '%s'", buf);
        continue;
      }
      *cp++ = '\0';
    }
    else
      *cp++ = '\0';

    struct in_addr addr;
    tmp = inet_pton(AF_INET, line, &addr);
    if (!tmp) {
      log_err("inet_pton failed for %s", line);
      continue;
    }

    annex::IpAddrNet *ip_addr = is->add_ip_addrs();
    ip_addr->set_addr(addr.s_addr);
    ip_addr->set_network_prefix_length(atoi(cp));
  }

  pclose(ip_stream);
}

void DeliveryAgent::HandleGetNetworkInterface(
    const annex::Package &package_in,
    const annex::NetworkInterface &ni_in) {
  int tmp;

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }
  annex::Package *package = transfer->mutable_package();
  package->set_correlation_id(package_in.message_id());
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_NETWORK_INTERFACE);
  annex::NetworkInterface *ni = attr->mutable_network_interface();

  if (!ni_in.has_name()) {
    delete transfer;
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "network_interface.name expected");
    return;
  }

  if (ni_in.name().empty() || strchr(ni_in.name().c_str(), '/')) {
    delete transfer;
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "network_interface.name invalid");
    return;
  }

  std::string sysfs_path;
  int int_value;
  unsigned uint_value;
  unsigned carrier_value;
  uint8_t hwaddr_bytes[6];
  std::string net_dir = std::string(kSysfsNetDir) + "/" + ni_in.name();

  if (!file_exists(net_dir.c_str())) {
    delete transfer;
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_FOUND,
        "network_interface.name (%s) not found", ni_in.name().c_str());
    return;
  }

  ni->set_name(ni_in.name());

  sysfs_path = net_dir + "/type";
  tmp = path_scanf(sysfs_path.c_str(), "%d", &int_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    if (annex::NetworkInterface_LinkType_IsValid(int_value)) {
      ni->set_link_type(static_cast<annex::NetworkInterface_LinkType>(int_value));
    }
  }

  sysfs_path = net_dir + "/address";
#if 0
  tmp = path_scanf(sysfs_path.c_str(), "%02X:%02X:%02X:%02X:%02X:%02X",
      &hwaddr_bytes[0],
      &hwaddr_bytes[1],
      &hwaddr_bytes[2],
      &hwaddr_bytes[3],
      &hwaddr_bytes[4],
      &hwaddr_bytes[5]
  );
#else
  // had to write this special function because the above function
  // call results in 'alignment trap' on arm based cdp hardware.

  tmp = path_read_address(sysfs_path.c_str(), (char *)hwaddr_bytes);
#endif
  if (tmp != 6) {
    log_err("error reading %s, val = %d", sysfs_path.c_str(), tmp);
  } else {
    ni->set_addr(hwaddr_bytes, 6);
  }

  sysfs_path = net_dir + "/broadcast";
#if 0
  tmp = path_scanf(sysfs_path.c_str(), "%02X:%02X:%02X:%02X:%02X:%02X",
      &hwaddr_bytes[0],
      &hwaddr_bytes[1],
      &hwaddr_bytes[2],
      &hwaddr_bytes[3],
      &hwaddr_bytes[4],
      &hwaddr_bytes[5]
  );
#else
  tmp = path_read_address(sysfs_path.c_str(), (char *)hwaddr_bytes);
#endif
  if (tmp != 6) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ni->set_broadcast_addr(hwaddr_bytes, 6);
  }

  sysfs_path = net_dir + "/mtu";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ni->set_mtu(uint_value);
  }

  sysfs_path = net_dir + "/carrier";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &carrier_value);
  if (tmp != 1) {
    carrier_value = 0;
  }

  sysfs_path = net_dir + "/flags";
  tmp = path_scanf(sysfs_path.c_str(), "0x%X", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    annex::NetworkInterface::Flags *flags = ni->mutable_flags();

    flags->set_up(uint_value & IFF_UP);
    flags->set_lower_up(carrier_value);
    flags->set_loopback(uint_value & IFF_LOOPBACK);
    flags->set_broadcast(uint_value & IFF_BROADCAST);
    flags->set_pointtopoint(uint_value & IFF_POINTOPOINT);
    flags->set_multicast(uint_value & IFF_MULTICAST);
    flags->set_dynamic(uint_value & IFF_DYNAMIC);
    flags->set_noarp(uint_value & IFF_NOARP);
    flags->set_allmulti(uint_value & IFF_ALLMULTI);
    flags->set_promisc(uint_value & IFF_PROMISC);
  }

  annex::InetSettings *is = ni->mutable_inet_settings();
  is->set_method(annex::InetSettings::METHOD_MANUAL);
  AddIpAddrs(is, ni_in.name().c_str());

#if 0
  is->set_default_route(0);
  is->add_name_servers(0);
  is->add_name_servers(0);
#endif

  annex::LinkStatistics *ls = ni->mutable_link_statistics();
  sysfs_path = net_dir + "/statistics/rx_bytes";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_bytes(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_packets";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_packets(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_errors(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_dropped";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_dropped(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_over_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_overrun(uint_value);
  }

  sysfs_path = net_dir + "/statistics/multicast";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_multicast(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_bytes";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_bytes(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_packets";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_packets(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_errors(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_dropped";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_dropped(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_carrier_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_carrier(uint_value);
  }

  sysfs_path = net_dir + "/statistics/collisions";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_collisions(uint_value);
  }

  TransferStart(transfer, 1000, false);
}

void DeliveryAgent::SendNetworkInterfaceStats(const char *if_name)
{
  int tmp;

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_NETWORK_INTERFACE);
  annex::NetworkInterface *ni = attr->mutable_network_interface();

  std::string sysfs_path;
  int int_value;
  unsigned uint_value;
  unsigned carrier_value;
  uint8_t hwaddr_bytes[6];
  std::string net_dir = std::string(kSysfsNetDir) + "/" + if_name;

  if (!file_exists(net_dir.c_str())) {
    delete transfer;
    return;
  }

  ni->set_name(if_name);

  sysfs_path = net_dir + "/type";
  tmp = path_scanf(sysfs_path.c_str(), "%d", &int_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    if (annex::NetworkInterface_LinkType_IsValid(int_value)) {
      ni->set_link_type(static_cast<annex::NetworkInterface_LinkType>(int_value));
    }
  }

  sysfs_path = net_dir + "/address";
  tmp = path_read_address(sysfs_path.c_str(), (char *)hwaddr_bytes);
  if (tmp != 6) {
    log_err("error reading %s, val = %d", sysfs_path.c_str(), tmp);
  } else {
    ni->set_addr(hwaddr_bytes, 6);
  }

  sysfs_path = net_dir + "/broadcast";
  tmp = path_read_address(sysfs_path.c_str(), (char *)hwaddr_bytes);
  if (tmp != 6) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ni->set_broadcast_addr(hwaddr_bytes, 6);
  }

  sysfs_path = net_dir + "/mtu";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ni->set_mtu(uint_value);
  }

  sysfs_path = net_dir + "/carrier";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &carrier_value);
  if (tmp != 1) {
    carrier_value = 0;
  }

  sysfs_path = net_dir + "/flags";
  tmp = path_scanf(sysfs_path.c_str(), "0x%X", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    annex::NetworkInterface::Flags *flags = ni->mutable_flags();

    flags->set_up(uint_value & IFF_UP);
    flags->set_lower_up(carrier_value);
    flags->set_loopback(uint_value & IFF_LOOPBACK);
    flags->set_broadcast(uint_value & IFF_BROADCAST);
    flags->set_pointtopoint(uint_value & IFF_POINTOPOINT);
    flags->set_multicast(uint_value & IFF_MULTICAST);
    flags->set_dynamic(uint_value & IFF_DYNAMIC);
    flags->set_noarp(uint_value & IFF_NOARP);
    flags->set_allmulti(uint_value & IFF_ALLMULTI);
    flags->set_promisc(uint_value & IFF_PROMISC);
  }

  annex::InetSettings *is = ni->mutable_inet_settings();
  is->set_method(annex::InetSettings::METHOD_MANUAL);
  AddIpAddrs(is, if_name);

  annex::LinkStatistics *ls = ni->mutable_link_statistics();
  sysfs_path = net_dir + "/statistics/rx_bytes";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_bytes(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_packets";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_packets(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_errors(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_dropped";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_dropped(uint_value);
  }

  sysfs_path = net_dir + "/statistics/rx_over_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_overrun(uint_value);
  }

  sysfs_path = net_dir + "/statistics/multicast";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_rx_multicast(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_bytes";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_bytes(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_packets";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_packets(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_errors(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_dropped";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_dropped(uint_value);
  }

  sysfs_path = net_dir + "/statistics/tx_carrier_errors";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_carrier(uint_value);
  }

  sysfs_path = net_dir + "/statistics/collisions";
  tmp = path_scanf(sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sysfs_path.c_str());
  } else {
    ls->set_tx_collisions(uint_value);
  }

  TransferStart(transfer, 1000, false);
}

void DeliveryAgent::SendNetworkStats()
{
  //send the network stats for - lo, eth0, ppp0
  SendNetworkInterfaceStats ("lo");
  SendNetworkInterfaceStats ("eth0");
  SendNetworkInterfaceStats ("ppp0");
}

void DeliveryAgent::CellularRadioFillFromLog(annex::CellularRadio *cr) {
  struct stat st;
  FILE *stream;
  std::string log_file;
  char buf[1024];
  char *line;
  char *match;
  char *save;
  char *token;

  annex::Sim *sim = NULL;

  if (stat("/var/log/ppp_trace", &st) == 0) {
    log_file = "/var/log/ppp_trace";

    stream = fopen(log_file.c_str(), "r");
    if (!stream) {
      log_err("fopen %s failed: %m", log_file.c_str());
      return;
    }

    while ((line = fgets(buf, sizeof(buf), stream))) {
      if ((match = strcasestr(line, "+CPIN: "))) {
        save = match;
        token = atcmd_response_brk(&save);
        if (token) {
          token = atcmd_value_tok(&save);
          if (token) {
            if (!sim) {
              sim = cr->mutable_sim();
            }
            match = strstr(token, "^M");
            if (match) {
              *match = '\0';
            }
            sim->set_cpin(token);
          }
        }
      } else if ((match = strcasestr(line, "AT+CSQ"))) {
        size_t prefix_len = match - line;
        line = fgets(buf, sizeof(buf), stream);
        line = fgets(buf, sizeof(buf), stream);
        line = fgets(buf, sizeof(buf), stream);
        if (line && strlen(line) > prefix_len) {
          match = line + prefix_len;

          if (strcasestr(line, "CSQ: ")) {
            save = match;
            token = atcmd_response_brk(&save);
            if (token) {
              token = atcmd_value_tok(&save);
              if (token) {
                cr->set_rssi(atoi(token));
                token = atcmd_value_tok(&save);
                if (token) {
                  cr->set_fer(atoi(token));
                }
              }
            }
          } else if (strcasestr(line, "ERROR")) {
            save = match;
            token = atcmd_value_tok(&save);
            if (token) {
              cr->set_rssi(atoi(token));
              token = atcmd_value_tok(&save);
              if (token) {
                cr->set_fer(atoi(token));
              }
            }
          }
        }
      } else if ((match = strcasestr(line, "+CNUM: "))) {
        save = match;
        token = atcmd_response_brk(&save);
        if (token) {
          token = atcmd_value_tok(&save);
          if (token) {
            token = atcmd_value_tok(&save);
            if (token) {
              match = strstr(token, "^M");
              if (match) {
                *match = '\0';
              }
              cr->set_phone_number(token);
            }
          }
        }
      } else if ((match = strcasestr(line, "AT+CGSN"))) {
        size_t prefix_len = match - line -1;
        line = fgets(buf, sizeof(buf), stream);
        line = fgets(buf, sizeof(buf), stream);
        line = fgets(buf, sizeof(buf), stream);
        if (line && strlen(line) > prefix_len) {
          token = line + prefix_len;
          match = strstr(token, "^M");
          if (match) {
            *match = '\0';
          }
          match = strpbrk(token, "\r\n\t ");
          if (match) {
            *match = '\0';
          }
          cr->set_imei(token);
        }
      } else if ((match = strcasestr(line, "AT+CIMI"))) {
        size_t prefix_len = match - line - 1;
        line = fgets(buf, sizeof(buf), stream);
        line = fgets(buf, sizeof(buf), stream);
        line = fgets(buf, sizeof(buf), stream);
        if (line && strlen(line) > prefix_len) {
          token = line + prefix_len;
          match = strstr(token, "^M");
          if (match) {
            *match = '\0';
          }
          match = strpbrk(token, "\r\n\t ");
          if (match) {
            *match = '\0';
          }
          if (!sim) {
            sim = cr->mutable_sim();
          }
          sim->set_imsi(token);
        }
      }
    }

    fclose(stream);
  } else if (stat("/var/log/messages", &st) == 0) {
    // XXX: Check rotated logs also messages.{0,n}
    log_file = "/var/log/messages";

    stream = fopen(log_file.c_str(), "r");
    if (!stream) {
      log_err("fopen %s failed: %m", log_file.c_str());
      return;
    }

    while ((line = fgets(buf, sizeof(buf), stream))) {
      if ((match = strcasestr(line, "+CPIN: "))) {
        save = match;
        token = atcmd_response_brk(&save);
        if (token) {
          token = atcmd_value_tok(&save);
          if (token) {
            if (!sim) {
              sim = cr->mutable_sim();
            }
            match = strstr(token, "^M");
            if (match) {
              *match = '\0';
            }
            sim->set_cpin(token);
          }
        }
      } else if ((match = strcasestr(line, ": AT+CSQ"))) {
        size_t prefix_len = match - line + 2;
        line = fgets(buf, sizeof(buf), stream);
        if (line && strlen(line) > prefix_len) {
          match = line + prefix_len;

          if (strcasestr(line, "CSQ:")) {
            save = match;
            token = atcmd_response_brk(&save);
            if (token) {
              token = atcmd_value_tok(&save);
              if (token) {
                cr->set_rssi(atoi(token));
                token = atcmd_value_tok(&save);
                if (token) {
                  cr->set_fer(atoi(token));
                }
              }
            }
          } else if (strcasestr(line, "ERROR")) {
            save = match;
            token = atcmd_value_tok(&save);
            if (token) {
              cr->set_rssi(atoi(token));
              token = atcmd_value_tok(&save);
              if (token) {
                cr->set_fer(atoi(token));
              }
            }
          }
        }
      } else if ((match = strcasestr(line, "+CNUM: "))) {
        save = match;
        token = atcmd_response_brk(&save);
        if (token) {
          token = atcmd_value_tok(&save);
          if (token) {
            token = atcmd_value_tok(&save);
            if (token) {
              match = strstr(token, "^M");
              if (match) {
                *match = '\0';
              }
              cr->set_phone_number(token);
            }
          }
        }
      } else if ((match = strcasestr(line, ": AT+CGSN"))) {
        size_t prefix_len = match - line + 2;
        line = fgets(buf, sizeof(buf), stream);
        if (line && strlen(line) > prefix_len) {
          token = line + prefix_len;
          match = strstr(token, "^M");
          if (match) {
            *match = '\0';
          }
          match = strpbrk(token, "\r\n\t ");
          if (match) {
            *match = '\0';
          }
          cr->set_imei(token);
        }
      } else if ((match = strcasestr(line, ": AT+CIMI"))) {
        size_t prefix_len = match - line + 2;
        line = fgets(buf, sizeof(buf), stream);
        if (line && strlen(line) > prefix_len) {
          token = line + prefix_len;
          match = strstr(token, "^M");
          if (match) {
            *match = '\0';
          }
          match = strpbrk(token, "\r\n\t ");
          if (match) {
            *match = '\0';
          }
          if (!sim) {
            sim = cr->mutable_sim();
          }
          sim->set_imsi(token);
        }
      }
    }

    fclose(stream);
  }
}

void DeliveryAgent::HandleGetCellularRadio(
    const annex::Package &package_in,
    const annex::CellularRadio &cr_in) {
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }
  annex::Package *package = transfer->mutable_package();
  package->set_correlation_id(package_in.message_id());
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_CELLULAR_RADIO);
  annex::CellularRadio *cr = attr->mutable_cellular_radio();

  if (!cr_in.has_label()) {
    delete transfer;
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "cellular_radio.label expected");
    return;
  }

  if (strcmp(cr_in.label().c_str(), "cell0")) {
    delete transfer;
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_FOUND,
        "cellular_radio.label (%s) not found", cr_in.label().c_str());
    return;
  }

  cr->set_label(cr_in.label());
  CellularRadioFillFromLog(cr);

  TransferStart(transfer, 1000, false);
}

void DeliveryAgent::SendCellularRadio(void)
{
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_CELLULAR_RADIO);
  annex::CellularRadio *cr = attr->mutable_cellular_radio();

  cr->set_label("cell0");
  CellularRadioFillFromLog(cr);

  TransferStart(transfer, 1000, false);
}

void DeliveryAgent::HandleGetAnnexVersion(
    const annex::Package &package_in,
    const annex::AnnexVersion &annex_ver) {

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }

  annex::Package *package = transfer->mutable_package();
  package->set_correlation_id(package_in.message_id());
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_ANNEX_VERSION);
  annex::AnnexVersion *av = attr->mutable_annex_version();

  av->mutable_version();

  TransferStart(transfer, 1000, false);
}

void DeliveryAgent::HandleSetClientConfig(
    const annex::Package &package_in,
    const annex::ClientConfig &client_config) {

  bool restart_courier = false;

  if (client_config.has_rpd_interval()) {
    if (client_config.rpd_interval() == 0) {
      StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "Invalid RPD Interval - Client Configuration NOT Updated");
      return;
    }
    courier()->set_rpd_interval(client_config.rpd_interval());
  }
  if (client_config.has_gps_interval()) {
    courier()->set_gps_interval(client_config.gps_interval());
  }
  if (client_config.has_stats_interval()) {
    courier()->set_stats_interval(client_config.stats_interval());
  }
  if (client_config.has_cell_interval()) {
    courier()->set_cell_interval(client_config.cell_interval());
  }
  if (client_config.has_trigger_interval()) {
    courier()->set_trigger_interval(client_config.trigger_interval());
  }

  // FIXME: need to add code for supporting SSL options

  if (client_config.has_dm_host_name()) {
    if (courier()->host().compare(client_config.dm_host_name())) {
      courier()->set_host(client_config.dm_host_name());
      restart_courier = true;
    }
  }
  if (client_config.has_dm_host_port()) {
    if (courier()->port() != client_config.dm_host_port()) {
      courier()->set_port(client_config.dm_host_port());
      restart_courier = true;
    }
  }

  if (restart_courier)
    courier()->restart_courier();

  StatusReply(
      1000,
      package_in.message_id(),
      annex::Package::STATUS_CODE_OK,
      "Client Configuration Updated");
}

bool DeliveryAgent::SendUpgradeStatus(
    int sendq_timeout_msec,
    annex::SoftwareUpgradeStatus_Code status_code,
    annex::File *file,
    const char *status_msg) {

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return false;
  }

  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_SOFTWARE_UPGRADE_STATUS);
  annex::SoftwareUpgradeStatus *sw_status = attr->mutable_software_upgrade_status();

  sw_status->set_name(su_name_);
  sw_status->set_uri(su_uri_);
  sw_status->set_code(status_code);

  if (file) {
    annex::File *sw_file = sw_status->mutable_file();
    *sw_file = *file;
  }

  if (status_msg)
    sw_status->set_description(status_msg);

  TransferStart(transfer, sendq_timeout_msec, false);

  return true;
}

void DeliveryAgent::HandleDoSoftwareUpgrade(
    const annex::Package &package_in,
    const annex::SoftwareUpgrade &su_in) {

  size_t start = 0;
  size_t end;
  size_t count;
  char *tarball;
  char *binfile;

  std::string uri = su_in.uri();
  std::string scheme;
  std::string host;
  std::string port;
  std::string path;

printf ("In Software Upgrade\n");

  if (su_in.has_abort()) {
    if (su_in.abort()) {
      // cancel the current upgrade parameters and send OK
      su_uri_ = "";
      su_md5sum_ = "";

      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_ACCEPTED,
          NULL);

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ABORTED,
        NULL,
        "Software Upgrade Aborted");
printf ("Software Upgrade Aborted\n");
      return;
    }
  }

  su_rcell_ = su_rcell_config_ = su_cdp_ = 0;
  if (! strcmp(su_in.name().c_str(), "rcell-firmware")) {

    //first check if the configuration allows firmware upgrade
    if (courier()->firmware_upgrade() == false) {
      log_err("Software Upgrade is not allowed in Configuration");
      StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_UNKNOWN,
        "Software Upgrade is not allowed in Configuration");

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
        NULL,
        "Software Upgrade not allowed in Configuration");
      return;
    }

    su_name_ = su_in.name();
    su_rcell_ = 1;
  }
  else if (! strcmp(su_in.name().c_str(), "rcell-config")) {

    //first check if the configuration allows firmware upgrade
    if (courier()->config_upgrade() == false) {
      log_err("Configuration Upgrade is not allowed in Configuration");
      StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_UNKNOWN,
        "Configuration Upgrade is not allowed in Configuration");

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
        NULL,
        "Configuration Upgrade not allowed in Configuration");
      return;
    }

    su_name_ = su_in.name();
    su_rcell_config_ = 1;
  }
  else if (! strcmp(su_in.name().c_str(), "cdp-firmware")) {
    su_name_ = su_in.name();
    su_cdp_ = 1;
  }
  else {
    log_err("software_upgrade.name (%s) unknown", su_in.name().c_str());
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_UNKNOWN,
        "software_upgrade.name (%s) unknown", su_in.name().c_str());

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
        NULL,
        "Software Upgrade Name Invalid");
printf ("Software Upgrade name Unknown - Aborting...\n");
    return;
  }

  //Check the model info and the filename being downloaded
  //If there is a mismatch send error and ignore the software upgrade.
  if (su_cdp_ && strstr(courier()->product_id().c_str(), "CDP")) {
  }
  else if ((su_rcell_ || su_rcell_config_) && strstr(courier()->product_id().c_str(), "EN")) {
  }
  else {
    log_err("software_upgrade (%s) doesn't match the model (%s) ",
      su_in.name().c_str(), courier()->product_id().c_str());
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_UNKNOWN,
        "software_upgrade (%s) doesn't match the model (%s)",
        su_in.name().c_str(), courier()->product_id().c_str());

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
        NULL,
        "Software Upgrade is not for the right model");
printf ("Software Upgrade filename(%s) doesn't match - Aborting...\n", su_in.name().c_str());
    return;
  }

  if (su_in.has_md5sum() == false) {
    log_err("software_upgrade (%s) doesn't specify md5sum", su_in.name().c_str());
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_UNKNOWN,
        "software_upgrade (%s) requires md5sum", su_in.name().c_str());

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
        NULL,
        "Software Upgrade Needs to specify md5sum");
printf ("Software Upgrade doesn't specify md5sum - Aborting...\n");
    return;
  }

  end = uri.find("://", start);
  if (end == std::string::npos || end == 0) {
    log_err("find scheme failed");
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "software_upgrade.uri invalid");
printf ("Software Upgrade URI(%s) is bad - Aborting...\n", uri.c_str());
    goto bad_uri;
  }
  count = end - start;
  scheme = uri.substr(start, count);

  start = start + count + 3;
  end = uri.find("/", start);
  if (end == std::string::npos || end == 0) {
    log_err("find host failed");
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "software_upgrade.uri invalid");
printf ("Software Upgrade URI(%s) is invalid - Aborting...\n", uri.c_str());
    goto bad_uri;
  }
  count = end - start;
  host = uri.substr(start, count);

  start = start + count;
  path = uri.substr(start);
  if (path.empty()) {
    log_err("path is empty");
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "software_upgrade.uri invalid");
printf ("Software Upgrade URI(%s) path is empty - Aborting...\n", uri.c_str());
    goto bad_uri;
  }

  if (su_cdp_ || su_rcell_config_) {
    tarball = basename_safe(path.c_str());
    if (!*tarball || (!str_ends_with(tarball, ".tar.gz") && !str_ends_with(tarball, ".tgz"))) {
      log_err("path is not a tarball");
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_INVALID_FIELD,
          "software_upgrade.uri invalid");
      goto bad_uri;
    }
    if (su_rcell_config_) {
      if (strcmp(tarball, "config.tar.gz")) {
        log_err("Invalid filename for configuration");
        StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_INVALID_FIELD,
          "software_upgrade.uri invalid");
        goto bad_uri;
      }
    }
  }
  else if (su_rcell_) {
    binfile = basename_safe(path.c_str());
    if (!*binfile || (!str_ends_with(binfile, ".bin") )) {
      log_err("path is not a binfile");
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_INVALID_FIELD,
          "software_upgrade.uri invalid");
printf ("Software Upgrade URI(%s) is not a BIN file - Aborting...\n", uri.c_str());
      goto bad_uri;
    }
  }

  start = 0;
  end = host.find(":", start);
  if (end != std::string::npos) {
    port = host.substr(start + end + 1);
    host = host.substr(start, end);
    goto bad_uri;
  }

  if (host.empty()) {
    log_err("host is empty");
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "software_upgrade.uri invalid");

bad_uri:
    SendUpgradeStatus(
      3000,
      annex::SoftwareUpgradeStatus_Code_UPGRADE_INVALID_FILE_NAME,
      NULL,
      "Software Upgrade URI is invalid");
printf ("Software Upgrade URI(%s) is BAD - Aborting...\n", uri.c_str());
    return;
  }

  if (!strcmp(scheme.c_str(), "annex")) {
    // FIXME: Assuming host and port are the same as this session.

    // The protocol can specify a different host and port for downloading the file.
    // In this version we are assuming the annex protocol uses the same session 
    // parameters for software upgrade process.

    su_uri_ = su_in.uri();
    su_scheme_ = scheme;
    su_host_ = host;
    su_port_ = port;
    su_path_ = path;
    su_md5sum_ = su_in.md5sum();

    log_info ("Software Upgarde: URI received = %s\n", su_uri_.c_str());
    log_info ("Software Upgarde: MD5 Checksum received = %s\n", su_md5sum_.c_str());

    // check if the URI and the md5sum match the previous ones
    // if so, if the file already exists, resume the upgrade process
    if ((! su_uri_.compare(su_prev_uri_)) && (! su_md5sum_.compare(su_prev_md5sum_))) {
      su_resume_ = true;
      log_info ("Software Upgarde: URI and MD5 Checksum match previous Upgarde. Will try to resume if file exists\n");
    }
    else {
      su_prev_uri_ = su_uri_;
      su_prev_md5sum_ = su_md5sum_;
      su_resume_ = false;
    }

printf ("Software Upgrade will start NOW...\n");
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_ACCEPTED,
        NULL);

    SendUpgradeStatus(
      3000,
      annex::SoftwareUpgradeStatus_Code_UPGRADE_ACCEPTED,
      NULL,
      "Software Upgrade URI is valid");
  } else {
    log_err("su_scheme_ invalid");
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_UNSUPPORTED,
        "software_upgrade.uri uses unsupported scheme (%s)", su_scheme_.c_str());

    SendUpgradeStatus(
      3000,
      annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
      NULL,
      "Software Upgrade URI scheme is not supported");
  }
}

void DeliveryAgent::HandleDoReboot(
    const annex::Package &package_in,
    const annex::Reboot &reboot_in) {

  switch (reboot_in.cmd()) {
  case annex::Reboot::CMD_RESTART:
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_ACCEPTED,
        NULL);
    restart_ = true;
    break;
  case annex::Reboot::CMD_HALT:
  case annex::Reboot::CMD_POWER_OFF:
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_UNSUPPORTED,
        "reboot.cmd (%d) unsupported", reboot_in.cmd());
    return;
  default:
    return;
  }
}

void DeliveryAgent::HandleAction(
    const annex::Package &package_in,
    const annex::Action &action_in) {
  const annex::Attribute &attr_in = action_in.attribute();
  const annex::Function &func_in = action_in.function();

  switch (action_in.type()) {
  case annex::Action::TYPE_GET:
    switch (attr_in.type()) {
    case annex::Attribute::TYPE_NETWORK_INTERFACE:
      HandleGetNetworkInterface(package_in, attr_in.network_interface());
      return;
    case annex::Attribute::TYPE_CELLULAR_RADIO:
      HandleGetCellularRadio(package_in, attr_in.cellular_radio());
      return;
    case annex::Attribute::TYPE_ANNEX_VERSION:
      HandleGetAnnexVersion(package_in, attr_in.annex_version());
      return;
    default:
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_UNIMPLEMENTED,
          "action.type (%d) attribute.type (%d) unimplemented",
              action_in.type(), attr_in.type());
      return;
    }
  case annex::Action::TYPE_SET:
    switch (attr_in.type()) {
    case annex::Attribute::TYPE_CLIENT_CONFIG:
      HandleSetClientConfig(package_in, attr_in.client_config());
      return;
    case annex::Attribute::TYPE_NETWORK_INTERFACE:
    default:
      // FIXME: Add rCell and CDP network config changes
      // Do not allow any changing of the network interface
      // parameters in this version. One can only do GET operations.
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_UNIMPLEMENTED,
          "action.type (%d) attribute.type (%d) unimplemented",
              action_in.type(), attr_in.type());
      return;
    }
  case annex::Action::TYPE_DO:
    switch (func_in.type()) {
    case annex::Function::TYPE_SOFTWARE_UPGRADE:
      HandleDoSoftwareUpgrade(package_in, func_in.software_upgrade());
      return;
    case annex::Function::TYPE_REBOOT:
      HandleDoReboot(package_in, func_in.reboot());
      return;
    default:
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_UNIMPLEMENTED,
          "function.type (%d) unimplemented", func_in.type());
      return;
    }
  default:
    return;
  }
}

void DeliveryAgent::HandlePackage(const annex::Package &package_in) {
  if (Courier::PackageExpired(package_in)) {
    return;
  }

  if (package_in.has_content()) {
    const annex::Content &content_in = package_in.content();

    switch(content_in.type()) {
    case annex::Content::TYPE_NOTIFICATION:
      if (package_in.request_ack()) {
        StatusReply(
            1000,
            package_in.message_id(),
            annex::Package::STATUS_CODE_OK,
            NULL);
      }
      log_info("Notification: %s", content_in.notification().DebugString().c_str());
      return;
    case annex::Content::TYPE_ACTION:
      HandleAction(package_in, content_in.action());
      return;
    default:
      return;
    }
  }
}

bool DeliveryAgent::RequestPackageDelivery() {
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return false;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_REQUEST_PACKAGE_DELIVERY);
  content->mutable_request_package_delivery();

  TransferStart(transfer, 1000, true);
  TransferWaitComplete(transfer, -1);
  if (transfer->events() & TRANSFER_EVENT_SENDQ_TIMEOUT) {
    log_err("sendq timeout");
    DeleteTransfer(transfer);
    return false;
  }

  int package_no = 0;
  int handled;
  AnnexTransfer *reply = NULL;

  while (true) {
    handled = false;

    if (package_no == 0) {
      reply = TransferWaitReply(transfer, 30000);
      if (!reply) {
        log_err("TransferWaitReply status failed");
        DeleteTransfer(transfer);
        return false;
      }
    } else {
      reply = TransferWaitReply(transfer, 200);
      if (!reply) {
        DeleteTransfer(transfer);
        return true;
      }
    }

    package = reply->mutable_package();

    if (package->has_status_code() && package_no) {
      StatusReply(
          1000,
          package->message_id(),
          annex::Package::STATUS_CODE_UNEXPECTED_FIELD,
          "package.status_code unexpected");
      goto next;
    }
    if (!package->has_content()) {
      if (package_no) {
        StatusReply(
            1000,
            package->message_id(),
            annex::Package::STATUS_CODE_EXPECTED_FIELD,
            "package.content expected");
      }
      goto next;
    }

    package->clear_correlation_id();
    package->clear_status_code();
    package->clear_status_message();
    handled = courier_->FireTransferEvent(reply, TRANSFER_EVENT_RECV);

next:
    if (!handled) {
      delete reply;
    }

    ++package_no;
  }
}

bool DeliveryAgent::Md5sumCheck(std::string &file, std::string &md5sum_expected) {
  char buf[1024];
  std::string cmd = "md5sum " + file;

  FILE *stream = popen(cmd.c_str(), "r");
  if (!stream) {
    log_err("popen %s failed: %m", cmd.c_str());
    return false;
  }

  char *line = fgets(buf, sizeof(buf), stream);
  pclose(stream);
  if (!line) {
    log_err("%s invalid output", cmd.c_str());
    return false;
  }

  char *cp = strchr(line, ' ');
  if (!cp) {
    log_err("%s invalid output", cmd.c_str());
    return false;
  }
  *cp = '\0';

  int rc = !strcmp(line, md5sum_expected.c_str());
  if (!rc) {
    log_err("md5sum check of %s failed: got %s expected %s",
        file.c_str(), line, md5sum_expected.c_str());
    return false;
  }

  return rc;
}

bool DeliveryAgent::MediaCardAvailable() {
  char buf[1024];
  std::string cmd = "mount | grep media/card";

  FILE *stream = popen(cmd.c_str(), "r");
  if (!stream) {
    log_err("popen %s failed: %m", cmd.c_str());
    return false;
  }

  char *line = fgets(buf, sizeof(buf), stream);
  pclose(stream);
  if (!line) {
    log_err("%s : no output", cmd.c_str());
    return false;
  }

  char *cp = strstr(line, "media/card");
  if (!cp) {
    log_info("No Media card available");
    return false;
  }

  log_info("Media card found");
  return true;
}

void DeliveryAgent::PerformSoftwareUpgrade() {
  int tmp;
  int rc;
  AnnexTransfer *note;
  std::string cmd;
  FILE *stream = NULL;
  annex::File *file = NULL;
  struct stat dir_stat;
  struct stat download_file_stat;
  std::string flash_root;
  std::string flash_upgrade_dir;
  std::string download_file_name;

  if (su_cdp_) {
    if (MediaCardAvailable() == true) {
      flash_root = "/media/card";
      log_info("Using /media/card for software upgrade.");
printf("Using /media/card for software upgrade.");
    }
    else {
      flash_root = "/var/volatile";
      log_info("Using /var/volatile for software upgrade.");
printf("Using /var/volatile for software upgrade.");
      log_info("This will be insufficient if you are doing full upgrade of Kernel and FileSystem.");
      log_info("Insert a flash card of at least 512MB for safe software upgrades.");
    }
  }
  else if (su_rcell_) {
    flash_root = "/var/tmp";
  }
  else if (su_rcell_config_) {
    flash_root = "/var/tmp";
  }
  flash_upgrade_dir = flash_root + "/" + "flash-upgrade";
  download_file_name = basename_safe(su_path_.c_str());
  su_download_file_path_ = flash_upgrade_dir + "/" + download_file_name;

  // if the download_file already exists and the resume flag is set,
  // open the file and seek to the end of the file and resume from there.
  if (! stat(su_download_file_path_.c_str(), &download_file_stat)) {
    if (su_resume_) {
printf ("The download_file path - %s exists. Resuming Software Upgrade\n", su_download_file_path_.c_str());
      log_info ("The download_file path - %s exists. Resuming Software Upgrade\n", su_download_file_path_.c_str());

      stream = fopen(su_download_file_path_.c_str(), "a");
      if (!stream) {
        log_err("opening %s failed: %m", su_download_file_path_.c_str());
printf("opening %s failed: %m", su_download_file_path_.c_str());
        return;
      }

      file = AnnexFileOpen(su_path_.c_str(), "r");
      if (!file) {
        log_err("AnnexFileOpen %s failed: %m", su_path_.c_str());
printf("AnnexFileOpen %s failed: %m", su_path_.c_str());
        return;
      }

      AnnexFileSeek (file, download_file_stat.st_size);

      note = NotificationMessageStart(
          5000,
          annex::Notification::LEVEL_INFO,
          "Resuming software upgrade.");
      if (note) {
        TransferWaitComplete(note, -1);
        DeleteTransfer(note);
      }

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_RESUMED,
        NULL,
        "Resuming Software Upgrade");

      goto DoSoftwareUpgrade;

    }
  }

  if (! stat(flash_upgrade_dir.c_str(), &dir_stat)) {
    log_info ("The flash-upgrade path - %s exists. Clean up.\n", flash_upgrade_dir.c_str());
printf ("The flash-upgrade path - %s exists. Clean up.\n", flash_upgrade_dir.c_str());
    cmd = "rm -fr " + flash_upgrade_dir;
    tmp = system(cmd.c_str());
    if (tmp != 0) {
      log_err("%s failed", cmd.c_str());
printf("%s failed", cmd.c_str());
      goto file_error;
    }
  }

  cmd = "mkdir -p " + flash_upgrade_dir;
  tmp = system(cmd.c_str());
  if (tmp != 0) {
    log_err("%s failed", cmd.c_str());
printf("%s failed", cmd.c_str());
    goto file_error;
  }

  stream = fopen(su_download_file_path_.c_str(), "w");
  if (!stream) {
    log_err("opening %s failed: %m", su_download_file_path_.c_str());
printf("opening %s failed: %m", su_download_file_path_.c_str());
    goto file_error;
  }

  file = AnnexFileOpen(su_path_.c_str(), "r");
  if (!file) {
    log_err("AnnexFileOpen %s failed: %m", su_path_.c_str());
printf("AnnexFileOpen %s failed: %m", su_path_.c_str());

file_error:
    SendUpgradeStatus(
      3000,
      annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
      NULL,
      "Error during file operations");
    return;
  }

  SendUpgradeStatus(
    3000,
    annex::SoftwareUpgradeStatus_Code_UPGRADE_IN_PROGRESS,
    NULL,
    "Software Upgrade is in Progress");

  note = NotificationMessageStart(
      5000,
      annex::Notification::LEVEL_INFO,
      "Starting software upgrade.");
  if (note) {
    TransferWaitComplete(note, -1);
    DeleteTransfer(note);
  }

DoSoftwareUpgrade:

  annex::FileBuffer buffer;
  while (true) {
    rc = AnnexFileRead(file, &buffer, ContainerBuffer::kSerializedMax - 1024);
    if (!rc) {
      log_err("AnnexFileRead failed");
printf("AnnexFileRead failed");
      goto fail;
    }
    if (buffer.has_abort()) {
      if(buffer.abort()) {
        log_info ("Software Upgrade Aborted - Request from server\n");
printf ("Software Upgrade Aborted - Request from server\n");
        SendUpgradeStatus(
          3000,
          annex::SoftwareUpgradeStatus_Code_UPGRADE_ABORTED,
          NULL,
          "Software Upgrade Aborted by Server");
        goto fail;
      }
    }
    if (buffer.has_eof()) {
        if(buffer.eof() == true) {
           if (buffer.has_buffer()) {
               log_info ("Some data in the buffer when EOF reached\n");
               fwrite(buffer.buffer().c_str(), buffer.buffer().size(), 1, stream);
           }
           log_info("buffer EOF reached, close the file");
printf("buffer EOF reached, close the file");
           break;
        }
    }

    if (!buffer.has_buffer()) {
      log_info("buffer not set: must be eof? pos: %ld", ftell(stream));
printf("buffer not set: must be eof? pos: %ld", ftell(stream));
      break;
    }
    else {
      fwrite(buffer.buffer().c_str(), buffer.buffer().size(), 1, stream);
      log_info("wrote chunk: %ld", ftell(stream));
printf("wrote chunk: %ld", ftell(stream));
    }
  }

  rc = AnnexFileClose(file);
  if (!rc) {
    log_err("AnnexFileClose %s failed", su_path_.c_str());
printf("AnnexFileClose %s failed", su_path_.c_str());
  }
  file = NULL;

  fclose(stream);
  stream = NULL;

  rc = Md5sumCheck(su_download_file_path_, su_md5sum_);
  if (!rc) {
printf ("MD5sumCheck failed - Upgrade fail\n");
    SendUpgradeStatus(
      3000,
      annex::SoftwareUpgradeStatus_Code_UPGRADE_CHKSUM_FAILURE,
      NULL,
      "Software Upgrade Failed - MD5SUM doesn't match");
    goto fail;
  }

  if (su_cdp_) {
    cmd = "tar -C " + flash_upgrade_dir + " -zxf " + su_download_file_path_;
    tmp = system(cmd.c_str());
    if (tmp != 0) {
      log_err("%s failed", cmd.c_str());
      goto fail;
    }
    cmd = "mv " + flash_upgrade_dir + "/*/* " + flash_upgrade_dir;
    tmp = system(cmd.c_str());
    if (tmp != 0) {
      log_err("%s failed", cmd.c_str());

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
        NULL,
        "Software Upgrade Failed - Unable to extract or copy the files");
      goto fail;
    }
  }
  else if (su_rcell_config_) {
    //FIXME: Check the version of this config file and see if it is correct
    // If version mismatch or not supported, report error and cleanup.

    cmd = "mv " + flash_upgrade_dir + "/* /var";
    tmp = system(cmd.c_str());
    if (tmp != 0) {
      log_err("%s failed", cmd.c_str());

      SendUpgradeStatus(
        3000,
        annex::SoftwareUpgradeStatus_Code_UPGRADE_ERROR,
        NULL,
        "Configuration Upgrade Failed - Unable to extract or copy the files");
      goto fail;
    }
  }

  SendUpgradeStatus(
    3000,
    annex::SoftwareUpgradeStatus_Code_UPGRADE_COMPLETE,
    NULL,
    "Software Upgrade Complete");

  note = NotificationMessageStart(
      5000,
      annex::Notification::LEVEL_INFO,
      "Software upgrade complete.");
printf ("Software Upgrade Complete\n");
  if (note) {
    TransferWaitComplete(note, -1);
    DeleteTransfer(note);
  }

  su_uri_ = "";
  if (su_cdp_ || su_rcell_ || su_rcell_config_) {
    restart_ = true;
  }

  return;

fail:
  if (file) {
    rc = AnnexFileClose(file);
    if (!rc) {
      log_err("AnnexFileClose %s failed", su_path_.c_str());
    }
    file = NULL;
  }

  if (stream) {
    fclose(stream);
    stream = NULL;
  }

  note = NotificationMessageStart(
      5000,
      annex::Notification::LEVEL_NOTICE,
      "Software upgrade failed.");
  if (note) {
    TransferWaitComplete(note, -1);
    DeleteTransfer(note);
  }

  su_uri_ = "";
}

bool DeliveryAgent::Start() {
  int rc;

  rc = Runnable::Start();
  if (rc) {
    StartListening();
  }

  return rc;
}

bool DeliveryAgent::Stop() {
  int rc;

  StopListening();
  rc = Runnable::Stop();

  return rc;
}

#define LOOP_WAIT_TIME 100

void DeliveryAgent::RunnableCore() {
  int tmp;
  AnnexTransfer *transfer;
  int rpd_loop_count;
  int cell_loop_count;
  int stats_loop_count;
  std::string cmd;

  if (!courier()) {
    log_err("courier must be set");
    return;
  }

  rpd_loop_count = courier()->rpd_interval() / LOOP_WAIT_TIME;
  cell_loop_count = courier()->cell_interval() / LOOP_WAIT_TIME;
  stats_loop_count = courier()->stats_interval() / LOOP_WAIT_TIME;
 
  while (!runnable_stop_) {
    if (synchronous_) {

      if (courier()->cell_interval()) {
        if (cell_loop_count)
          cell_loop_count --;
        else {
          SendCellularRadio();
          cell_loop_count = courier()->cell_interval() / LOOP_WAIT_TIME;
        }
      }

      if (courier()->stats_interval()) {
        if (stats_loop_count)
          stats_loop_count --;
        else {
          SendNetworkStats();
          stats_loop_count = courier()->stats_interval() / LOOP_WAIT_TIME;
        }
      }

      if (courier()->rpd_interval()) {
        if (rpd_loop_count)
          rpd_loop_count --;
        else {
          RequestPackageDelivery();
          rpd_loop_count = courier()->rpd_interval() / LOOP_WAIT_TIME;
        }
      }

      while ((transfer = TransferWaitRecv(LOOP_WAIT_TIME))) {
        HandlePackage(transfer->package());
        delete transfer;
      }
    }
    else {
      if (courier()->cell_interval()) {
        if (cell_loop_count)
          cell_loop_count --;
        else {
          SendCellularRadio();
          cell_loop_count = courier()->cell_interval() / LOOP_WAIT_TIME;
        }
      }

      if (courier()->stats_interval()) {
        if (stats_loop_count)
          stats_loop_count --;
        else {
          SendNetworkStats();
          stats_loop_count = courier()->stats_interval() / LOOP_WAIT_TIME;
        }
      }

      transfer = TransferWaitRecv(LOOP_WAIT_TIME);
      if (transfer) {
        HandlePackage(transfer->package());
        delete transfer;
      }
    }

    if (!su_uri_.empty()) {
      PerformSoftwareUpgrade();
    }

    if (restart_) {
      transfer = NotificationMessageStart(
          5000,
          annex::Notification::LEVEL_NOTICE,
          "Rebooting in %d seconds.", 10);
      if (transfer) {
        TransferWaitComplete(transfer, -1);
        DeleteTransfer(transfer);
      }

      sleep_msec(10 * 1000);

      if (su_cdp_) {
        cmd = "shutdown -r now";
        tmp = system(cmd.c_str());
        if (tmp != 0) {
          log_err("%s failed", cmd.c_str());
        } else {
          while (true) {
            sleep_msec(1000);
            log_info("waiting for reboot");
          }
        }
        restart_ = false;
        su_cdp_ = 0;
      }
      else if (su_rcell_) {
        if (strstr(su_download_file_path_.c_str(), "MTCBA-x-EN2-o-")) {
          log_info("In overwrite firmware download...");
          system("touch /var/run/bino");
        }

        cmd = "netflash " + su_download_file_path_ + " -un -r /dev/mtd0";
        tmp = system(cmd.c_str());
        if (tmp != 0) {
          log_err("%s failed", cmd.c_str());
        } else {
          while (true) {
            sleep_msec(1000);
            log_info("waiting for reboot");
          }
        }
        restart_ = false;
        su_rcell_ = 0;
      }
      else if (su_rcell_config_) {
#if 0
        cmd = "/sbin/kill_processes";
        tmp = system(cmd.c_str());
        if (tmp != 0)
          log_err("%s failed", cmd.c_str());
#endif

        cmd = "/sbin/do_config_untar &";
        tmp = system(cmd.c_str());
        if (tmp != 0)
          log_err("%s failed", cmd.c_str());
        sleep_msec(1000);

#if 0
        cmd = "/bin/umount -a -r";
        tmp = system(cmd.c_str());
        if (tmp != 0)
          log_err("%s failed", cmd.c_str());
#endif

        reboot(0x01234567); // This is the magic ID for the rCell

        restart_ = false;
        su_rcell_config_ = 0;
      }
      else {
        if (strstr(courier()->product_id().c_str(), "CDP")) {
          cmd = "shutdown -r now";
          tmp = system(cmd.c_str());
          if (tmp != 0) {
            log_err("%s failed", cmd.c_str());
          } else {
            while (true) {
              sleep_msec(1000);
              log_info("waiting for reboot");
            }
          }
          restart_ = false;
        }
        else if (strstr(courier()->product_id().c_str(), "EN")) {
          reboot(0x01234567); // This is the magic ID for the rCell
          restart_ = false;
        }
      }
    }
  }
}

void DeliveryAgent::set_synchronous(bool value) {
  synchronous_ = value;
}

bool DeliveryAgent::synchronous() const {
  return synchronous_;
}

}
