/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>

#include <courier.h>
#include <gpio.h>

#define MTCDP_GPIO 1

namespace mts {

const char *Gpio::kSysfsPlaformDir = "/sys/devices/platform/mtcdp";

Gpio::Gpio() {
  SysfsPlatformGpio *gpio;
  SysfsPlatformSensor *sensor;

#if MTCDP_GPIO
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_ANALOG_TO_DIGITAL, "adc0");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_ANALOG_TO_DIGITAL, "adc1");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_ANALOG_TO_DIGITAL, "adc2");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_ANALOG_TO_DIGITAL, "adc3");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din0");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din1");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din2");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din3");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din4");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din5");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din6");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "din7");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout0");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout1");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout2");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout3");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout4");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout5");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout6");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "dout7");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "led-dtr");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "led-ls");
#endif
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "extserial-dcd");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "extserial-dsr");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_INPUT, "extserial-dtr");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "extserial-ri");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "led-cd");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "led-sig1");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "led-sig2");
  gpio = SysfsPlatformGpioAdd(annex::Gpio::TYPE_DIGITAL_OUTPUT, "led-sig3");

  sensor = SysfsPlatformSensorAdd("board-temperature", annex::Scalar::TYPE_INT32);
}

Gpio::~Gpio() {
  Stop();
}

bool Gpio::TransferEventRecv(AnnexTransfer *transfer) {
  const annex::Package &package = transfer->package();

  if (package.has_content() && package.content().type() == annex::Content::TYPE_ACTION) {
    const annex::Action &action = package.content().action();
    if (action.type() == annex::Action::TYPE_GET ||
        action.type() == annex::Action::TYPE_SET) {
      const annex::Attribute &attr = action.attribute();
      if (attr.type() == annex::Attribute::TYPE_GPIO ||
          attr.type() == annex::Attribute::TYPE_SENSOR) {
        AddTransfer(transfer);
        return true;
      }
    }
  }

  return false;
}

bool Gpio::Start() {
  int rc;

  rc = Runnable::Start();
  if (rc) {
    StartListening();
  }

  return rc;
}

bool Gpio::Stop() {
  int rc;

  StopListening();
  rc = Runnable::Stop();

  return rc;
}

Gpio::SysfsPlatformGpio *Gpio::SysfsPlatformGpioAdd(
    annex::Gpio_Type type,
    const std::string &label) {
  SysfsPlatformGpio empty;
  sysfs_platform_gpios_.push_back(empty);
  SysfsPlatformGpio *entry = &sysfs_platform_gpios_.back();

  entry->sysfs_path = std::string(kSysfsPlaformDir) + "/" + label;
  entry->annex_gpio.set_type(type);
  entry->annex_gpio.set_label(label);
  entry->annex_gpio.set_dio_trigger(annex::Gpio::DIO_TRIGGER_NONE);

  return entry;
}

Gpio::SysfsPlatformSensor *Gpio::SysfsPlatformSensorAdd(
    const std::string &name,
    annex::Scalar_Type value_type) {
  SysfsPlatformSensor empty;
  sysfs_platform_sensors_.push_back(empty);
  SysfsPlatformSensor *entry = &sysfs_platform_sensors_.back();

  entry->sysfs_path = std::string(kSysfsPlaformDir) + "/" + name;
  entry->annex_sensor.set_name(name);
  annex::Scalar *value = entry->annex_sensor.mutable_value();
  value->set_type(value_type);

  return entry;
}

Gpio::SysfsPlatformGpio *Gpio::SysfsPlatformGpioGet(const std::string &label) {
  SysfsPlatformGpio *gpio = NULL;

  for (std::vector<SysfsPlatformGpio>::iterator iter = sysfs_platform_gpios_.begin();
       iter != sysfs_platform_gpios_.end();
       ++iter) {
    if (!iter->annex_gpio.label().compare(label)) {
      gpio = &(*iter);
      break;
    }
  }

  return gpio;
}

Gpio::SysfsPlatformSensor *Gpio::SysfsPlatformSensorGet(const std::string &name) {
  SysfsPlatformSensor *sensor = NULL;

  for (std::vector<SysfsPlatformSensor>::iterator iter = sysfs_platform_sensors_.begin();
       iter != sysfs_platform_sensors_.end();
       ++iter) {
    if (!iter->annex_sensor.name().compare(name)) {
      sensor = &(*iter);
      break;
    }
  }

  return sensor;
}

void Gpio::HandleGetGpio(
    const annex::Package &package_in,
    const annex::Gpio &gpio_in) {
  SysfsPlatformGpio *sys_gpio;

  if (!gpio_in.has_label()) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "gpio.label expected");
    return;
  }

  sys_gpio = SysfsPlatformGpioGet(gpio_in.label());
  if (!sys_gpio) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_FOUND,
        "gpio.label (%s) not found", gpio_in.label().c_str());
    return;
  }

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }
  annex::Package *package = transfer->mutable_package();
  package->set_correlation_id(package_in.message_id());
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_GPIO);
  annex::Gpio *gpio = attr->mutable_gpio();

  unsigned uint_value;
  int tmp;

  *gpio = sys_gpio->annex_gpio;

  tmp = path_scanf(sys_gpio->sysfs_path.c_str(), "%u", &uint_value);
  if (tmp != 1) {
    log_err("error reading %s", sys_gpio->sysfs_path.c_str());
    delete transfer;
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_INTERNAL_ERROR,
        "reading gpio.label (%s) failed", gpio_in.label().c_str());
    return;
  } else {
    gpio->set_value(uint_value);
    sys_gpio->last_value = uint_value;
  }

  TransferStart(transfer, 1000, false);
}

void Gpio::HandleSetGpio(
    const annex::Package &package_in,
    const annex::Gpio &gpio_in) {
  int tmp;
  int cur_value;
  SysfsPlatformGpio *sys_gpio;

  log_debug ("In Set GPIO - label = %s\n", gpio_in.label().c_str());

  if (!gpio_in.has_label()) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "gpio.label expected");
    return;
  }

  if (!gpio_in.has_value() && !gpio_in.has_dio_trigger()) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "gpio.value expected");
    return;
  }

  sys_gpio = SysfsPlatformGpioGet(gpio_in.label());
  if (!sys_gpio) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_FOUND,
        "gpio.label (%s) not found", gpio_in.label().c_str());
    return;
  }

  if (gpio_in.has_value() && (sys_gpio->annex_gpio.type() != annex::Gpio::TYPE_DIGITAL_OUTPUT)) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "gpio.label (%s) is not of type (TYPE_DIGITAL_OUTPUT)", gpio_in.label().c_str());
    return;
  }
  else {
    if (sys_gpio->annex_gpio.type() == annex::Gpio::TYPE_DIGITAL_OUTPUT) {
      tmp = path_printf(sys_gpio->sysfs_path.c_str(), "%u", gpio_in.value());
      if (tmp <= 0) {
        log_err("error writing %s", sys_gpio->sysfs_path.c_str());
        StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_INTERNAL_ERROR,
          "writing gpio.label (%s) failed", gpio_in.label().c_str());
        return;
      }
    }

    sys_gpio->annex_gpio.set_value(gpio_in.value());
    sys_gpio->last_value = gpio_in.value();
  }

  if (gpio_in.has_dio_trigger() && (sys_gpio->annex_gpio.type() != annex::Gpio::TYPE_DIGITAL_INPUT)) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "cannot set trigger, gpio.label (%s) is not of type (TYPE_DIGITAL_INPUT)", gpio_in.label().c_str());
    return;
  }
  else {
    if (sys_gpio->annex_gpio.type() == annex::Gpio::TYPE_DIGITAL_INPUT) {
      log_debug ("Set GPIO Trigger\n");
      sys_gpio->annex_gpio.set_dio_trigger(gpio_in.dio_trigger());

      tmp = path_scanf(sys_gpio->sysfs_path.c_str(), "%d", &cur_value);
      if (tmp != 1) {
        log_err("error reading %s", sys_gpio->sysfs_path.c_str());
      } else {
        sys_gpio->last_value = cur_value;
      }
    }
  }

  StatusReply(
      1000,
      package_in.message_id(),
      annex::Package::STATUS_CODE_OK,
      NULL);
}

void Gpio::MonitorTriggers() {

  int tmp, cur_value;

  for (std::vector<SysfsPlatformGpio>::iterator iter = sysfs_platform_gpios_.begin();
       iter != sysfs_platform_gpios_.end();
       ++iter) {

    // check the trigger setting
    if (iter->annex_gpio.has_dio_trigger()) {

      // read the current value
      tmp = path_scanf(iter->sysfs_path.c_str(), "%d", &cur_value);

      if (tmp != 1) {
          log_err("error reading %s", iter->sysfs_path.c_str());
          continue;
      }

      switch (iter->annex_gpio.dio_trigger()) {
        case annex::Gpio::DIO_TRIGGER_NONE:
          break;
        case annex::Gpio::DIO_TRIGGER_EDGE_RISING:
          if (!iter->last_value && cur_value) {
            goto send_trigger;
          }
          break;
        case annex::Gpio::DIO_TRIGGER_EDGE_FALLING:
          if (iter->last_value && !cur_value) {
            goto send_trigger;
          }
          break;
        case annex::Gpio::DIO_TRIGGER_EDGE_BOTH:
          if (iter->last_value != cur_value) {
            goto send_trigger;
          }
          break;
        case annex::Gpio::DIO_TRIGGER_LEVEL_HIGH:
          if (cur_value) {
            goto send_trigger;
          }
          break;
        case annex::Gpio::DIO_TRIGGER_LEVEL_LOW:
          if (!cur_value) {
send_trigger:
            log_debug ("Sending Trigger - last_value = %d, cur_value = %d\n", iter->last_value, cur_value);
            AnnexTransfer *transfer = Courier::NewTransfer();
            if (!transfer) {
              log_err("NewTransfer failed");
              continue;
            }
            annex::Package *package = transfer->mutable_package();
            annex::Content *content = package->mutable_content();
            content->set_type(annex::Content::TYPE_ATTRIBUTE);
            annex::Attribute *attr = content->mutable_attribute();
            attr->set_type(annex::Attribute::TYPE_GPIO);
            annex::Gpio *gpio = attr->mutable_gpio();

            *gpio = iter->annex_gpio;

            gpio->set_value(cur_value);

            TransferStart(transfer, 1000, false);
          }
          break;
      }

      iter->last_value = cur_value;
    }
  }
}


bool Gpio::SysfsReadScalar(
    const std::string &path,
    annex::Scalar *scalar) {
  int tmp;

  if (!scalar->has_type()) {
    log_err("scalar.type is not set");
    return false;
  }

  switch (scalar->type()) {
  case annex::Scalar::TYPE_INT32:
    {
      int value;

      tmp = path_scanf(path.c_str(), "%d", &value);
      if (tmp != 1) {
        log_err("error reading %s", path.c_str());
        return false;
      }

      scalar->set_int32(value);
      return true;
    }
  case annex::Scalar::TYPE_UINT32:
  case annex::Scalar::TYPE_SINT32:
  case annex::Scalar::TYPE_FIXED32:
  case annex::Scalar::TYPE_SFIXED32:
  case annex::Scalar::TYPE_BOOL:
  case annex::Scalar::TYPE_STRING:
  case annex::Scalar::TYPE_BYTES:
  case annex::Scalar::TYPE_DOUBLE:
  case annex::Scalar::TYPE_FLOAT:
  case annex::Scalar::TYPE_INT64:
  case annex::Scalar::TYPE_UINT64:
  case annex::Scalar::TYPE_SINT64:
  case annex::Scalar::TYPE_FIXED64:
  case annex::Scalar::TYPE_SFIXED64:
  default:
    return false;
  }
}

void Gpio::HandleGetSensor(
    const annex::Package &package_in,
    const annex::Sensor &sensor_in) {
  SysfsPlatformSensor *sys_sensor;

  if (!sensor_in.has_name()) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "sensor.name expected");
    return;
  }

  sys_sensor = SysfsPlatformSensorGet(sensor_in.name());
  if (!sys_sensor) {
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_NOT_FOUND,
        "sensor.name (%s) not found", sensor_in.name().c_str());
    return;
  }

  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }
  annex::Package *package = transfer->mutable_package();
  package->set_correlation_id(package_in.message_id());
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_SENSOR);
  annex::Sensor *sensor = attr->mutable_sensor();

  *sensor = sys_sensor->annex_sensor;
  annex::Scalar *value = sensor->mutable_value();
  value->set_type(annex::Scalar::TYPE_INT32);

  bool rc = SysfsReadScalar(sys_sensor->sysfs_path, value);
  if (!rc) {
    delete transfer;
    StatusReply(
        1000,
        package_in.message_id(),
        annex::Package::STATUS_CODE_INTERNAL_ERROR,
        "reading sensor.name (%s) failed", sensor->name().c_str());
    return;
  }

  TransferStart(transfer, 1000, false);
}

void Gpio::HandlePackage(const annex::Package &package_in) {
  if (Courier::PackageExpired(package_in)) {
    return;
  }

  const annex::Action &action_in = package_in.content().action();
  const annex::Attribute &attr_in = action_in.attribute();

  switch (action_in.type()) {
  case annex::Action::TYPE_GET:
    switch (attr_in.type()) {
    case annex::Attribute::TYPE_GPIO:
      HandleGetGpio(package_in, attr_in.gpio());
      return;
    case annex::Attribute::TYPE_SENSOR:
      HandleGetSensor(package_in, attr_in.sensor());
      return;
    default:
      return;
    }
  case annex::Action::TYPE_SET:
    switch (attr_in.type()) {
    case annex::Attribute::TYPE_GPIO:
      HandleSetGpio(package_in, attr_in.gpio());
      return;
    case annex::Attribute::TYPE_SENSOR:
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_UNIMPLEMENTED,
          "action.type (TYPE_SET) attribute.sensor unimplemented");
      return;
    default:
      return;
    }
    return;
  default:
    return;
  }
}

void Gpio::RunnableCore() {
  int trigger_loop_count = courier()->trigger_interval() / 50;

  while (!runnable_stop_) {
    AnnexTransfer *transfer = TransferWaitRecv(50);
    if (transfer) {
      HandlePackage(transfer->package());
      delete transfer;
    }

    // Trigger monitoring - Checks after the loop count expires (each loop approx 20 msec)
#if MTCDP_GPIO
    if (courier()->trigger_interval()) {
      if (trigger_loop_count)
        trigger_loop_count--;
      else {
        MonitorTriggers();
        trigger_loop_count = courier()->trigger_interval() / 50;
      }
    }
#endif

  }
}

}
