/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>

#include <courier.h>
#include <gps_receiver.h>

namespace mts {

GpsReceiver::GpsReceiver() {
  bind_addr_ = "127.0.0.1";
  port_ = kPort;

  pthread_mutex_init(&msg_mutex_, NULL);

  msg_.Clear();
}

GpsReceiver::~GpsReceiver() {
  Stop();
  pthread_mutex_destroy(&msg_mutex_);
}

bool GpsReceiver::TransferEventRecv(AnnexTransfer *transfer) {
  const annex::Package &package = transfer->package();

  if (package.has_content() && package.content().type() == annex::Content::TYPE_ACTION) {
    const annex::Action &action = package.content().action();
    if (action.type() == annex::Action::TYPE_GET ||
        action.type() == annex::Action::TYPE_SET) {
      const annex::Attribute &attr = action.attribute();
      if (attr.type() == annex::Attribute::TYPE_GPS_RECEIVER ||
          attr.type() == annex::Attribute::TYPE_GPS_NMEA) {
        AddTransfer(transfer);
        return true;
      }
    }
  }

  return false;
}

AnnexTransfer *GpsReceiver::GpsNmeaTransferCreate(const GpsMessage &msg) {
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return false;
  }
  annex::Package *package = transfer->mutable_package();
  annex::Content * content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_GPS_NMEA);
  annex::GpsNmea *gps_nmea = attr->mutable_gps_nmea();

#if 0
  log_debug("gpgga: %s", msg.gpgga().c_str());
  log_debug("gpgsa: %s", msg.gpgsa().c_str());
  log_debug("gpgsv[0]: %s", msg.gpgsv(0).c_str());
  log_debug("gpgsv[1]: %s", msg.gpgsv(1).c_str());
  log_debug("gpgsv[2]: %s", msg.gpgsv(2).c_str());
  log_debug("gpgll: %s", msg.gpgll().c_str());
  log_debug("gprmc: %s", msg.gprmc().c_str());
  log_debug("gpvtg: %s", msg.gpvtg().c_str());
#endif

  gps_nmea->set_gpgga(msg.gpgga());
  gps_nmea->set_gpgsa(msg.gpgsa());
  for (int i = 0; i < GpsMessage::kGpgsvMax; ++i) {
    if (msg.gpgsv(i).empty()) {
      break;
    }
    gps_nmea->add_gpgsv(msg.gpgsv(i));
  }
  gps_nmea->set_gpgll(msg.gpgll());
  gps_nmea->set_gprmc(msg.gprmc());
  gps_nmea->set_gpvtg(msg.gpvtg());

  return transfer;
}

bool GpsReceiver::GpsNmeaReply(const GpsMessage &msg, uint32_t correlation_id) {
  AnnexTransfer *transfer = GpsNmeaTransferCreate(msg);
  if (!transfer) {
    return false;
  }

  annex::Package *package = transfer->mutable_package();

  if(correlation_id)
    package->set_correlation_id(correlation_id);
  package->set_status_code(annex::Package::STATUS_CODE_OK);

  TransferStart(transfer, 1000, false);

  return true;
}

bool GpsReceiver::GpsNmeaPush(const GpsMessage &msg) {
  AnnexTransfer *transfer = GpsNmeaTransferCreate(msg);
  if (!transfer) {
    return false;
  }

  TransferStart(transfer, 1000, false);

  return true;
}

bool GpsReceiver::Start() {
  int rc;

  rc = Runnable::Start();
  if (rc) {
    StartListening();
  }

  return rc;
}

bool GpsReceiver::Stop() {
  int rc;

  StopListening();
  rc = Runnable::Stop();

  return rc;
}

void GpsReceiver::RecvGpsNmea(int sd) {
  int tmp;
  struct sockaddr_in cli_addr;
  socklen_t cli_addr_len = sizeof(cli_addr);
  char buf[1024];

  tmp = recvfrom(sd, buf, sizeof(buf) - 1, MSG_DONTWAIT, (struct sockaddr *) &cli_addr, &cli_addr_len);
  if (tmp <= 0) {
    if (errno != EAGAIN) {
      log_err("recvfrom failed: %m");
    }
    return;
  }
  buf[tmp] = '\0';

#if 0
  char cli_addr_str[INET_ADDRSTRLEN];
  char *cp;

  cp = (char *) inet_ntop(AF_INET, &cli_addr.sin_addr, cli_addr_str, sizeof(cli_addr_str));
  if (!cp) {
    log_err("inet_ntop cli_addr failed: %m");
  } else {
    log_debug("remote %s:%d", cli_addr_str, ntohs(cli_addr.sin_port));
  }
#endif

  GpsMessage msg;

  if (!msg.Load(buf)) {
    log_err("failed to load gps message");
    return;
  }

  mutex_lock(&msg_mutex_);

  msg_ = msg;

  mutex_unlock(&msg_mutex_);

  // FIXME: Send to server if configured to do so here
}

void GpsReceiver::HandleGetGpsReceiver(
    const annex::Package &package_in,
    const annex::GpsReceiver &gr_in) {
  AnnexTransfer *transfer = Courier::NewTransfer();
  if (!transfer) {
    log_err("NewTransfer failed");
    return;
  }
  annex::Package *package = transfer->mutable_package();
  package->set_correlation_id(package_in.message_id());
  annex::Content *content = package->mutable_content();
  content->set_type(annex::Content::TYPE_ATTRIBUTE);
  annex::Attribute *attr = content->mutable_attribute();
  attr->set_type(annex::Attribute::TYPE_GPS_RECEIVER);
  annex::GpsReceiver *gr = attr->mutable_gps_receiver();

  // FIXME: Load from rCell and CDP config
  gr->set_gpgga(1);
  gr->set_gpgsa(1);
  gr->set_gpgsv(1);
  gr->set_gpgll(1);
  gr->set_gprmc(1);
  gr->set_gpvtg(1);
  gr->set_message_interval(1);

  TransferStart(transfer, 1000, false);
}

void GpsReceiver::HandlePackage(const annex::Package &package_in) {
  if (Courier::PackageExpired(package_in)) {
    return;
  }

  const annex::Action &action_in = package_in.content().action();
  const annex::Attribute &attr_in = action_in.attribute();

  switch (action_in.type()) {
  case annex::Action::TYPE_GET:
    switch (attr_in.type()) {
    case annex::Attribute::TYPE_GPS_NMEA: {
      GpsMessage msg;
      Message(&msg);
      GpsNmeaReply(msg, package_in.message_id());
      return;
      }
    case annex::Attribute::TYPE_GPS_RECEIVER:
      HandleGetGpsReceiver(package_in, attr_in.gps_receiver());
      return;
    default:
      return;
    }
  case annex::Action::TYPE_SET:
    switch (attr_in.type()) {
    case annex::Attribute::TYPE_GPS_NMEA:
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_NOT_ALLOWED,
          "action.type (TYPE_SET) attribute.gps_nmea not allowed");
      return;
    case annex::Attribute::TYPE_GPS_RECEIVER:
      // FIXME: set rCell and CDP config
      StatusReply(
          1000,
          package_in.message_id(),
          annex::Package::STATUS_CODE_UNIMPLEMENTED,
          "action.type (TYPE_SET) attribute.gps_receiver unimplemented");
      return;
    default:
      return;
    }
    return;
  default:
    return;
  }
}

void GpsReceiver::RunnableCore() {
  int tmp;
  int sd = -1;
  int reuse = 1;
  struct sockaddr_in serv_addr;
  int gps_loop_count = courier()->gps_interval() / 50;

  if (!courier()) {
    log_err("courier must be set");
    return;
  }

  sd = socket(PF_INET, SOCK_DGRAM, 0);
  if (sd < 0) {
    log_err("socket failed: %m");
    goto error;
  }

  memset(&serv_addr, 0, sizeof(serv_addr));
  serv_addr.sin_family = AF_INET;
  tmp = inet_pton(AF_INET, bind_addr_.c_str(), &serv_addr.sin_addr);
  if (!tmp) {
    log_err("inet_pton failed for %s: %m", bind_addr_.c_str());
    goto error;
  }
  serv_addr.sin_port = htons(port_);

  tmp = setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse));
  if (tmp < 0) {
    log_err("setsockopt failed: %m");
    goto error;
  }

  tmp = bind(sd, (struct sockaddr *) &serv_addr, sizeof(serv_addr));
  if (tmp < 0) {
    log_err("bind failed: %m");
    goto error;
  }

  while (!runnable_stop_) {
    RecvGpsNmea(sd);

    AnnexTransfer *transfer = TransferWaitRecv(50);
    if (transfer) {
      HandlePackage(transfer->package());
      delete transfer;
    }

    if (courier()->gps_interval()) {
      if (gps_loop_count) 
        gps_loop_count --;
      else {
        GpsMessage msg;
        Message(&msg);
        GpsNmeaReply(msg, 0);

        gps_loop_count = courier()->gps_interval() / 50;
      }
    }
  }

error:

  if (sd >= 0) {
    close(sd);
  }
}

void GpsReceiver::Message(GpsMessage *msg) {
  mutex_lock(&msg_mutex_);

  *msg = msg_;

  mutex_unlock(&msg_mutex_);
}

bool GpsReceiver::set_bind_addr(const char *value) {
  bind_addr_ = value;

  return true;
}

bool GpsReceiver::set_bind_addr(const std::string &value) {
  bind_addr_ = value;

  return true;
}

const std::string &GpsReceiver::bind_addr() const {
  return bind_addr_;
}

bool GpsReceiver::set_port(uint16_t value) {
  port_ = value;

  return true;
}

uint16_t GpsReceiver::port() const {
  return port_;
}

}
