/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#ifndef GPS_RECEIVER_H_
#define GPS_RECEIVER_H_

#include <pthread.h>
#include <string>
#include <stdint.h>

#include <annex_common.h>
#include <runnable.h>
#include <annex_transfer_agent.h>
#include <annex_transfer.h>

namespace mts {

class Courier;

class GpsMessage {
 public:
  static const uint16_t kGpgsvMax = 3;

  GpsMessage() {
    Clear();
  }
  ~GpsMessage() {}

  void Clear() {
    gpgga_ = "";
    gpgsa_ = "";
    for (int i = 0; i < kGpgsvMax; ++i) {
      gpgsv_[i] = "";
    }
    gpgll_ = "";
    gprmc_ = "";
    gpvtg_ = "";
  }

  const std::string &gpgga() const { return gpgga_; };
  const std::string &gpgsa() const { return gpgsa_; };
  const std::string &gpgsv(size_t index) const {
    return gpgsv_[index];
  };
  const std::string &gpgll() const { return gpgll_; };
  const std::string &gprmc() const { return gprmc_; };
  const std::string &gpvtg() const { return gpvtg_; };

  bool Load(const char *buf) {
    const char *start;
    const char *end;

    Clear();

    start = buf;

    while (1) {
      if (!*start) {
        return true;
      }
      if (*start != '$') {
        return false;
      }
      end = strchr(start, '\n');

      if (!end) {
        return false;
      }

      if (!strncmp(start, "$GPGGA", 6)) {
        gpgga_.assign(start, end - start + 1);
      } else if (!strncmp(start, "$GPGSA", 6)) {
        gpgsa_.assign(start, end - start + 1);
      } else if (!strncmp(start, "$GPGSV", 6)) {
        if (start[9] == '1') {
          gpgsv_[0].assign(start, end - start + 1);
        } else if (start[9] == '2') {
          gpgsv_[1].assign(start, end - start + 1);
        } else if (start[9] == '3') {
          gpgsv_[2].assign(start, end - start + 1);
        }
      } else if (!strncmp(start, "$GPGLL", 6)) {
        gpgll_.assign(start, end - start + 1);
      } else if (!strncmp(start, "$GPRMC", 6)) {
        gprmc_.assign(start, end - start + 1);
      } else if (!strncmp(start, "$GPVTG", 6)) {
        gpvtg_.assign(start, end - start + 1);
      }

      start = end + 1;
    }
  }

private:

	std::string gpgga_;
	std::string gpgsa_;
	std::string gpgsv_[kGpgsvMax];
	std::string gpgll_;
	std::string gprmc_;
	std::string gpvtg_;
};

class GpsReceiver : public Runnable, public AnnexTransferAgent {
 public:
  static const uint16_t kPort = 5446;

  GpsReceiver();
  ~GpsReceiver();

  void Message(GpsMessage *msg);

  bool set_bind_addr(const char *value);
  bool set_bind_addr(const std::string &value);
  const std::string &bind_addr() const;

  bool set_port(uint16_t value);
  uint16_t port() const;

  bool Start();
  bool Stop();
 private:
  void RunnableCore();

  AnnexTransfer *GpsNmeaTransferCreate(const GpsMessage &msg);
  bool GpsNmeaReply(const GpsMessage &msg, uint32_t correlation_id);
  bool GpsNmeaPush(const GpsMessage &msg);
  bool TransferGpsNmea(const GpsMessage &msg);
  void RecvGpsNmea(int sd);

  bool TransferEventRecv(AnnexTransfer *transfer);
  void HandleGetGpsReceiver(
      const annex::Package &package_in,
      const annex::GpsReceiver &gr_in);
  void HandlePackage(const annex::Package &package_in);

  pthread_mutex_t msg_mutex_;
  GpsMessage msg_;

  std::string bind_addr_;
  uint16_t port_;

  DISALLOW_COPY_AND_ASSIGN(GpsReceiver);
};

}
#endif  // GPS_RECEIVER_H_
