/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#define _MAIN_CC_  1

#include <pthread.h>
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <sasl/sasl.h>

#include <annex_common.h>
#include <courier.h>
#include <gpio.h>
#include <gps_receiver.h>
#include <delivery_agent.h>

#define PROGRAM_NAME "annexcd"

#define SSL_SUPPORT 1
#define GPIO_SUPPORT 0

#if HAVE_MTS_IO_H

#include <mts_io.h>

#define ID_EEPROM_PATH "/sys/devices/platform/i2c-gpio/i2c-0/0-0056/eeprom"

static int id_eeprom_exists() {
  struct stat st;

  if (stat(ID_EEPROM_PATH, &st) < 0) {
    return 0;
  } else {
    return 1;
  }
}

static int load_eeprom(struct mts_id_eeprom_layout *id_eeprom) {
  int tmp;
  int fd;

  memset(id_eeprom, 0, sizeof(*id_eeprom));

  fd = open(ID_EEPROM_PATH, O_RDONLY);
  if (fd < 0) {
    log_err("could not open eeprom file: %m");
    return 0;
  }

  tmp = read(fd, (char *) id_eeprom, sizeof(*id_eeprom));
  if (tmp < 0) {
    log_err("reading eeprom failed: %m");
    return 0;
  } else if (tmp != sizeof(id_eeprom)) {
    log_err("only read %d bytes from eeprom", tmp);
    return 0;
  }

  close(fd);

  id_eeprom->vendor_id[sizeof(id_eeprom->vendor_id) - 1] = '\0';
  id_eeprom->product_id[sizeof(id_eeprom->product_id) - 1] = '\0';
  id_eeprom->device_id[sizeof(id_eeprom->device_id) - 1] = '\0';
  id_eeprom->hw_version[sizeof(id_eeprom->hw_version) - 1] = '\0';
  id_eeprom->imei[sizeof(id_eeprom->imei) - 1] = '\0';

  return 1;
}
#endif

static void print_version(const char *name) {
  printf("%s (" PACKAGE ") " VERSION " (" __DATE__ " " __TIME__ ")\n", name);
  printf("Copyright (C) 2010 by Multi-Tech Systems\n");
  printf(
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License version 2 or (at your option) any later version.\n"
"This program has absolutely no warranty.\n");
}

static void usage(FILE *out) {
  fprintf(out, "Usage: " PROGRAM_NAME " [ OPTIONS ... ]\n");
  fprintf(out, "where  OPTIONS := { \n");
  fprintf(out, "         --host <host> |\n");
  fprintf(out, "         --port <port> |\n");
  fprintf(out, "         --vendor-id <vendor-id> |\n");
  fprintf(out, "         --product-id <product-id> |\n");
  fprintf(out, "         --device-id <device-id> |\n");
  fprintf(out, "         --log-upto <0-7> |\n");
  fprintf(out, "         --rpd-interval <(msecs - default 1000)> |\n");
  fprintf(out, "         --gps-interval <(msecs - default 0)> |\n");
  fprintf(out, "         --net-interval <(msecs - default 0)> |\n");
  fprintf(out, "         --cell-interval <(msecs - default 0)> |\n");
  fprintf(out, "         --when-ppp-up <on/off> |\n");
  fprintf(out, "         --bring-ppp-up <on/off> |\n");
  fprintf(out, "         --daemonize |\n");
#if SSL_SUPPORT
  fprintf(out, "         --sasl-username <username> |\n");
  fprintf(out, "         --sasl-passwd <passwd> |\n");
  fprintf(out, "         --sasl-realm <realm> |\n");
  fprintf(out, "         --sasl-force-mech <mech> |\n");
  fprintf(out, "         --ssl-method {none | ssl | tls} |\n");
#endif
  fprintf(out, "         --gps-host <host> |\n");
  fprintf(out, "         --gps-port <port> |\n");
  fprintf(out, "         --version\n");
  fprintf(out, "         --help\n");
  fprintf(out, "       }\n");
  fprintf(out, "\n");
}

enum cmd_opt_long_only {
  CMD_OPT_VERSION = 128,
  CMD_OPT_HELP,

  CMD_OPT_HOST,
  CMD_OPT_PORT,
  CMD_OPT_VENDOR_ID,
  CMD_OPT_PRODUCT_ID,
  CMD_OPT_DEVICE_ID,
  CMD_OPT_LOG_UPTO,
  CMD_OPT_DAEMONIZE,
#if SSL_SUPPORT
  CMD_OPT_SSL_METHOD,
  CMD_OPT_SASL_USERNAME,
  CMD_OPT_SASL_PASSWD,
  CMD_OPT_SASL_REALM,
  CMD_OPT_SASL_FORCE_MECH,
#endif
  CMD_OPT_RPD_INTERVAL,
  CMD_OPT_GPS_INTERVAL,
  CMD_OPT_NET_INTERVAL,
  CMD_OPT_CELL_INTERVAL,
  CMD_OPT_WHEN_PPP_UP,
  CMD_OPT_BRING_PPP_UP,
  CMD_OPT_FIRMWARE_UPGRADE,
  CMD_OPT_CONFIG_UPGRADE,
  CMD_OPT_GPS_HOST,
  CMD_OPT_GPS_PORT,
};

static const char *short_options = "";
static struct option long_options[] = {
  {"host", 1, NULL, CMD_OPT_HOST},
  {"port", 1, NULL, CMD_OPT_PORT},
  {"vendor-id", 1, NULL, CMD_OPT_VENDOR_ID},
  {"product-id", 1, NULL, CMD_OPT_PRODUCT_ID},
  {"device-id", 1, NULL, CMD_OPT_DEVICE_ID},
  {"daemonize", 0, NULL, CMD_OPT_DAEMONIZE},
  {"log-upto", 1, NULL, CMD_OPT_LOG_UPTO},
#if SSL_SUPPORT
  {"ssl-method", 1, NULL, CMD_OPT_SSL_METHOD},
  {"sasl-username", 1, NULL, CMD_OPT_SASL_USERNAME},
  {"sasl-passwd", 1, NULL, CMD_OPT_SASL_PASSWD},
  {"sasl-realm", 1, NULL, CMD_OPT_SASL_REALM},
  {"sasl-force-mech", 1, NULL, CMD_OPT_SASL_FORCE_MECH},
#endif
  {"rpd-interval", 1, NULL, CMD_OPT_RPD_INTERVAL},
  {"gps-interval", 1, NULL, CMD_OPT_GPS_INTERVAL},
  {"net-interval", 1, NULL, CMD_OPT_NET_INTERVAL},
  {"cell-interval", 1, NULL, CMD_OPT_CELL_INTERVAL},
  {"when-ppp-up", 1, NULL, CMD_OPT_WHEN_PPP_UP},
  {"bring-ppp-up", 1, NULL, CMD_OPT_BRING_PPP_UP},
  {"firmware-upgrade", 1, NULL, CMD_OPT_FIRMWARE_UPGRADE},
  {"config-upgrade", 1, NULL, CMD_OPT_CONFIG_UPGRADE},
  {"gps-host", 1, NULL, CMD_OPT_GPS_HOST},
  {"gps-port", 1, NULL, CMD_OPT_GPS_PORT},
  {"version", 0, NULL, CMD_OPT_VERSION},
  {"help", 0, NULL, CMD_OPT_HELP},
  {0, 0, 0, 0},
};

int main(int argc, char *argv[]) {
  int i;
  int option_index;
  int tmp;

  int daemonize = 0;

  mts::Courier courier;
  mts::GpsReceiver gps_receiver;

  openlog(PROGRAM_NAME, LOG_NDELAY | LOG_CONS | LOG_PERROR, LOG_DAEMON);
  //openlog(PROGRAM_NAME, LOG_NDELAY | LOG_CONS, LOG_DAEMON);
  setlogmask(LOG_UPTO(LOG_INFO));

  while ((i = getopt_long(argc, argv, short_options, long_options, &option_index)) >= 0) {
    switch (i) {
    case 0:
      break;

    case CMD_OPT_DAEMONIZE:
      daemonize = 1;
      break;

    case CMD_OPT_LOG_UPTO:
      courier.set_log_level(atoi(optarg));
      setlogmask(LOG_UPTO(atoi(optarg)));
      break;

    case CMD_OPT_HOST:
      courier.set_host(optarg);
      break;

    case CMD_OPT_VENDOR_ID:
      courier.set_vendor_id(optarg);
      break;

    case CMD_OPT_PRODUCT_ID:
      courier.set_product_id(optarg);
      break;

    case CMD_OPT_DEVICE_ID:
      courier.set_device_id(optarg);
      break;

    case CMD_OPT_PORT:
      courier.set_port(atoi(optarg));
      break;

#if SSL_SUPPORT
    case CMD_OPT_SSL_METHOD:
      courier.set_ssl_method(optarg);
      break;

    case CMD_OPT_SASL_USERNAME:
      courier.set_sasl_username(optarg);
      break;

    case CMD_OPT_SASL_PASSWD:
      courier.set_sasl_passwd(optarg);
      break;

    case CMD_OPT_SASL_REALM:
      courier.set_sasl_realm(optarg);
      break;

    case CMD_OPT_SASL_FORCE_MECH:
      courier.set_sasl_force_mech(optarg);
      break;
#endif

    case CMD_OPT_RPD_INTERVAL:
      courier.set_rpd_interval(atoi(optarg));
      break;

    case CMD_OPT_GPS_INTERVAL:
      courier.set_gps_interval(atoi(optarg));
      break;

    case CMD_OPT_NET_INTERVAL:
      courier.set_stats_interval(atoi(optarg));
      break;

    case CMD_OPT_CELL_INTERVAL:
      courier.set_cell_interval(atoi(optarg));
      break;

    case CMD_OPT_WHEN_PPP_UP:
      if (!strcmp(optarg, "on"))
        courier.set_when_ppp_up(true);
      else
        courier.set_when_ppp_up(false);
      break;

    case CMD_OPT_BRING_PPP_UP:
      if (!strcmp(optarg, "on"))
        courier.set_bring_ppp_up(true);
      else
        courier.set_bring_ppp_up(false);
      break;

    case CMD_OPT_FIRMWARE_UPGRADE:
      courier.set_firmware_upgrade(atoi(optarg));
      break;

    case CMD_OPT_CONFIG_UPGRADE:
      courier.set_config_upgrade(atoi(optarg));
      break;

    case CMD_OPT_GPS_HOST:
      gps_receiver.set_bind_addr(optarg);
      break;

    case CMD_OPT_GPS_PORT:
      gps_receiver.set_port(atoi(optarg));
      break;

    case CMD_OPT_VERSION:
      print_version(PROGRAM_NAME);
      exit(0);
      break;

    case CMD_OPT_HELP:
      usage(stdout);
      exit(0);
      break;

    default:
      log_err("unknown option");
      usage(stderr);
      exit(1);
    }
  }

  if (optind != argc) {
    log_err("expected 0 non-options");
    usage(stderr);
    exit(1);
  }
  argc -= optind;
  argv += optind;

  optind = 1;

  if (courier.host().empty()) {
    log_err("host required");
    usage(stderr);
    exit(1);
  }

#if SSL_SUPPORT
  if (courier.sasl_username().empty() || courier.sasl_passwd().empty()) {
    log_err("sasl username and passwd are required");
    usage(stderr);
    exit(1);
  }

  tmp = sasl_client_init(NULL);
  if (tmp != SASL_OK) {
    log_err("sasl_client_init failed: %d", tmp);
    exit(1);
  }

  SSL_load_error_strings();
  SSL_library_init();
  ERR_load_BIO_strings();
  ERR_load_SSL_strings();
  OpenSSL_add_all_algorithms();
#endif

  if (daemonize) {
    tmp = daemon(0, 0);
    if (tmp < 0) {
      log_err("daemon failed: %m");
      exit(1);
    }
  }

  sigset_t sigset;
  sigfillset(&sigset);
  pthread_sigmask(SIG_BLOCK, &sigset, NULL);

#if GPIO_SUPPORT
  mts::Gpio gpio;
#endif
  mts::DeliveryAgent delivery_agent;

  courier.set_synchronous(true);
#if HAVE_MTS_IO_H
  if (id_eeprom_exists()) {
    struct mts_id_eeprom_layout id_eeprom;
    load_eeprom(&id_eeprom);
    courier.set_vendor_id(id_eeprom.vendor_id);
    courier.set_product_id(id_eeprom.product_id);
    courier.set_device_id(id_eeprom.device_id);
    log_notice("courier.set_hw_version unimplemented");
    log_notice("courier.set_sw_version unimplemented");
  } else {
    log_notice("rCell courier.set_* unimplemented");
  }
#else
  courier.set_vendor_id("Multi-Tech Systems");

  std::string product_id_file = "/var/config/device_type";
  char product_id[128];
  if (file_is_reg(product_id_file.c_str())) {
    path_scanf(product_id_file.c_str(), "%127s", product_id);
    courier.set_product_id(product_id);
  } else {
//    courier.set_product_id("MTCDP-E1-DK");
    courier.set_product_id("MTCBA-E1-EN2");
  }

  std::string device_id_file = "/var/config/device_id";
  char device_id[128];
  if (file_is_reg(device_id_file.c_str())) {
    path_scanf(device_id_file.c_str(), "%127s", device_id);
    courier.set_device_id(device_id);
  } else {
    courier.set_device_id("12345678");
  }
#endif
  courier.Start();

#if GPIO_SUPPORT
  gpio.set_courier(&courier);
  gpio.Start();
#endif

  gps_receiver.set_courier(&courier);
  gps_receiver.Start();

  delivery_agent.set_courier(&courier);
  delivery_agent.set_synchronous(true);
  // DeliveryAgent must be started last so it is the last event listener
  delivery_agent.Start();

  while (1) {
    siginfo_t info;
    struct timespec timeout;

    timeout.tv_sec = 1;
    timeout.tv_nsec = 0;

    tmp = sigtimedwait(&sigset, &info, &timeout);
    switch (tmp) {
    case -1:
      if (errno != EAGAIN) {
        log_err("sigtimedwait failed: %m");
      }
      break;
    case SIGINT:
      log_notice("SIGINT caught");
      exit(0);
      break;
    case SIGTERM:
      log_notice("SIGTERM caught");
      exit(0);
      break;
    case SIGPIPE:
      log_notice("SIGPIPE caught");
      break;
    default:
      //log_err("unhandled signal %d caught", tmp);
      break;
    }

    if (!courier.Running()) {
      log_err("courier died");
      exit(1);
    }
  }

  return 0;
}

/*
 * ./annexcd --host "204.26.122.94" --log-up 7 --sasl-username "uip" --sasl-passwd "passwd" --ssl-method tls
 * ./annexcd --host "192.168.2.2" --log-up 7 --sasl-username "uip" --sasl-passwd "passwd"
 */
