/* vim: set sw=2 ts=2 expandtab:
 *
 * Copyright (C) 2010 by Multi-Tech Systems
 *
 * Author: James Maki <jmaki@multitech.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <pthread.h>

#include <validator.h>

#define SSL_SUPPORT 1

namespace mts {

bool Validator::ValidateNotification(const annex::Package &package, const annex::Notification &note) {
  if (!note.has_level()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "notification.level expected");
    return false;
  }

  if (note.has_attribute()) {
    if (!ValidateAttribute(package, note.attribute())) {
      return false;
    }
  }

  return true;
}

bool Validator::ValidateGpio(const annex::Package &package, const annex::Gpio &gpio) {
#if 0
  if (!gpio.has_type()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "gpio.type expected");
    return false;
  }
#endif

  return true;
}

bool Validator::ValidateSensor(const annex::Package &package, const annex::Sensor &sensor) {
  if (!sensor.has_name()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "sensor.name expected");
    return false;
  }

  return true;
}

bool Validator::ValidateAction(const annex::Package &package, const annex::Action &action) {
  if (!action.has_type()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "action.type expected");
    return false;
  }

  switch (action.type()) {
  case annex::Action::TYPE_GET:
  case annex::Action::TYPE_SET:
    if (action.has_attribute()) {
      return ValidateAttribute(package, action.attribute());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "action.attribute expected");
      return false;
    }
  case annex::Action::TYPE_DO:
    if (action.has_function()) {
      return ValidateFunction(package, action.function());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "action.function expected");
      return false;
    }
  default:
    return false;
  }
}

bool Validator::ValidateAttribute(const annex::Package &package, const annex::Attribute &attr) {
  switch (attr.type()) {
  case annex::Attribute::TYPE_KEY_VALUE_PAIR:
    if (attr.has_key_value_pair()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.key_value_pair expected");
      return false;
    }
  case annex::Attribute::TYPE_GPS_NMEA:
    if (attr.has_gps_nmea()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.gps_nmea expected");
      return false;
    }
  case annex::Attribute::TYPE_GPIO:
    if (attr.has_gpio()) {
      return ValidateGpio(package, attr.gpio());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.gpio expected");
      return false;
    }
  case annex::Attribute::TYPE_SENSOR:
    if (attr.has_sensor()) {
      return ValidateSensor(package, attr.sensor());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.sensor expected");
      return false;
    }
  case annex::Attribute::TYPE_ANNEX_VERSION:
    if (attr.has_annex_version()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.annex_version expected");
      return false;
    }
#if SSL_SUPPORT
  case annex::Attribute::TYPE_SASL_TOKEN:
    if (attr.has_sasl_token()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.sasl_token expected");
      return false;
    }
  case annex::Attribute::TYPE_SASL_SERVER_MECH_LIST:
    if (attr.has_sasl_server_mech_list()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.sasl_server_mech_list expected");
      return false;
    }
#endif
  case annex::Attribute::TYPE_AW_CELLULAR_PORT:
    if (attr.has_aw_cellular_port()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.aw_cellular_port expected");
      return false;
    }
  case annex::Attribute::TYPE_AW_ANALOG_PORT:
    if (attr.has_aw_analog_port()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.aw_analog_port expected");
      return false;
    }
  case annex::Attribute::TYPE_FILE_BUFFER:
    if (attr.has_file_buffer()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.file_buffer expected");
      return false;
    }
  case annex::Attribute::TYPE_FILE_POS:
    if (attr.has_file_pos()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.file_pos expected");
      return false;
    }
  case annex::Attribute::TYPE_FILE:
    if (attr.has_file()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.file expected");
      return false;
    }
  case annex::Attribute::TYPE_SERIAL_PORT:
    if (attr.has_serial_port()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.serial_port expected");
      return false;
    }
  case annex::Attribute::TYPE_GPS_RECEIVER:
    if (attr.has_gps_receiver()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.gps_receiver expected");
      return false;
    }
  case annex::Attribute::TYPE_CELLULAR_RADIO:
    if (attr.has_cellular_radio()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.cellular_radio expected");
      return false;
    }
  case annex::Attribute::TYPE_NETWORK_INTERFACE:
    if (attr.has_network_interface()) {
      return true;
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "attribute.network_interface expected");
      return false;
    }
  case annex::Attribute::TYPE_CLIENT_CONFIG:
      return true;
  default:
    return false;
  }
}

bool Validator::ValidateSoftwareUpgrade(const annex::Package &package, const annex::SoftwareUpgrade &su) {
  if (!su.has_name()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "software_upgrade.name expected");
    return false;
  }
  if (su.name().empty()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "software_upgrade.name invalid");
    return false;
  }

  if (!su.has_uri()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "software_upgrade.uri expected");
    return false;
  }
  if (su.uri().empty()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "software_upgrade.uri invalid");
    return false;
  }

  if (!su.has_md5sum()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "software_upgrade.md5sum expected");
    return false;
  }
  if (su.md5sum().empty()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_INVALID_FIELD,
        "software_upgrade.md5sum invalid");
    return false;
  }

  return true;
}

bool Validator::ValidateReboot(const annex::Package &package, const annex::Reboot &reboot) {
  if (!reboot.has_cmd()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "reboot.cmd expected");
    return false;
  }

  return true;
}

bool Validator::ValidateFunction(const annex::Package &package, const annex::Function &func) {
  if (!func.has_type()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "function.type expected");
    return false;
  }

  switch (func.type()) {
  case annex::Function::TYPE_SOFTWARE_UPGRADE:
    if (func.has_software_upgrade()) {
      return ValidateSoftwareUpgrade(package, func.software_upgrade());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "function.software_upgrade expected");
      return false;
    }
  case annex::Function::TYPE_FILE_OPEN:
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "function.type (TYPE_FILE_OPEN) not allowed");
    return false;
  case annex::Function::TYPE_FILE_CLOSE:
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "function.type (TYPE_FILE_CLOSE) not allowed");
    return false;
  case annex::Function::TYPE_FILE_WRITE:
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "function.type (TYPE_FILE_WRITE) not allowed");
    return false;
  case annex::Function::TYPE_FILE_GET_POS:
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "function.type (TYPE_FILE_GET_POS) not allowed");
    return false;
  case annex::Function::TYPE_FILE_SET_POS:
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "function.type (TYPE_FILE_SET_POS) not allowed");
    return false;
  case annex::Function::TYPE_REBOOT:
    if (func.has_reboot()) {
      return ValidateReboot(package, func.reboot());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "function.reboot expected");
      return false;
    }
  default:
    return false;
  }
}

bool Validator::ValidateContent(const annex::Package &package, const annex::Content &content) {
  if (!content.has_type()) {
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_EXPECTED_FIELD,
        "content.type expected");
    return false;
  }

  switch (content.type()) {
  case annex::Content::TYPE_SESSION_INIT:
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "content.type (TYPE_SESSION_INIT) not allowed");
    return false;
  case annex::Content::TYPE_REQUEST_PACKAGE_DELIVERY:
    StatusReply(
        1000,
        package.message_id(),
        annex::Package::STATUS_CODE_NOT_ALLOWED,
        "content.type (TYPE_REQUEST_PACKAGE_DELIVERY) not allowed");
    return false;
  case annex::Content::TYPE_NOTIFICATION:
    if (content.has_notification()) {
      return ValidateNotification(package, content.notification());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "content.notification expected");
      return false;
    }
  case annex::Content::TYPE_ACTION:
    if (content.has_action()) {
      return ValidateAction(package, content.action());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "content.action expected");
      return false;
    }
  case annex::Content::TYPE_ATTRIBUTE:
    if (!package.has_correlation_id()) {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_UNSUPPORTED,
          "content.type (TYPE_ATTRIBUTE) unsupported outside of reply");
      return false;
    }
    if (content.has_attribute()) {
      return ValidateAttribute(package, content.attribute());
    } else {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "content.attribute expected");
      return false;
    }
  default:
    return false;
  }
}

bool Validator::Validate(const annex::Package &package) {
  if (!package.has_message_id()) {
    log_notice("no message_id");
    return false;
  }

  if (!package.has_correlation_id()) {
    if (package.has_status_code()) {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_UNEXPECTED_FIELD,
          "package.status_code unexpected");
      return false;
    }
    if (!package.has_content()) {
      StatusReply(
          1000,
          package.message_id(),
          annex::Package::STATUS_CODE_EXPECTED_FIELD,
          "package.content expected");
      return false;
    }
  }

  if (package.has_content()) {
    if (!ValidateContent(package, package.content())) {
      return false;
    }
  }

  return true;
}

bool Validator::TransferEventReply(AnnexTransfer *transfer) {
    bool valid = Validate(transfer->package());
    if (!valid) {
      delete transfer;
      return true;
    } else {
      return false;
    }
}

bool Validator::TransferEventRecv(AnnexTransfer *transfer) {
    bool valid = Validate(transfer->package());
    if (!valid) {
      delete transfer;
      return true;
    } else {
      return false;
    }
}

}
