#!/usr/bin/env ruby
#
# vim: set sw=2 ts=2 expandtab:
#

###############################################################################################
# YAML file structure for defining the simulation
#
# group[1..65536]:
#   type: [iCell|rCell|CDP|A2W]
#   instances: [1..65536]
#   gpsdata: filename, interval, NMEA type[GPGGA, GPVTG, GPGSA, GPSSV, GPRMC, GPGLL]
#   messages: [SessionInit, RequestPackageDelivery, Notification, Action, GPIO, KVP, Cellular, Network]
#   download: filename (the file to be downloaded should be in files folder for server to access)
#   repeat: [1..65536] | NONE
#
# Notes:
# - one can define multiple groups - each group with unique number
# - all time intervals are in seconds
# - the repeat flag repeats for every instance in the group the specified interval
# - you can only specify one NMEA type with the gpsdata line.
# - messages specified will be delivered in the order listed
# - each instance of the device will be created with unique ID
###############################################################################################

require 'yaml'

require 'annex_common'
require 'annex_client'
require 'annex_server'

Thread.abort_on_exception = true

class AnnexSimulator

  def setup(platform_ip = "127.0.0.1")
    @client = AnnexClient.new(platform_ip)
  end

  def teardown
    @client.close()
  end

 def request_package_delivery(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_REQUEST_PACKAGE_DELIVERY
    request_package_delivery = Annex::RequestPackageDelivery.new
    content.request_package_delivery = request_package_delivery

    @client.send(container)
    recv = @client.recv()
  end

  def send_notification(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content
    package.request_ack = true

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_NOTIFICATION
    notification = Annex::Notification.new
    notification.level = Annex::Notification::Level::LEVEL_ALERT
    notification.message = "Defcon 1"

    content.notification = notification

    @client.send(container)
    ack = @client.recv()
  end

  def send_action(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ACTION
    action = Annex::Action.new
    action.type = Annex::Action::Type::TYPE_DO
    content.action = action
    action.function = Annex::Function.new
    action.function.type = Annex::Function::Type::TYPE_EXTENSION

    @client.send(container)
  end

  def send_attr_gps_nmea_full(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_GPS_NMEA
    gps_nmea = Annex::GpsNmea.new
    gps_nmea.gpgga = "$GPGGA,171331.676,4505.9701,N,09311.7487,W,1,07,1.2,273.0,M,-29.0,M,,0000*66\r\n"
    gps_nmea.gpgll = "$GPGLL,4505.9701,N,09311.7487,W,171331.676,A,A*47\r\n"
    gps_nmea.gpgsa = "$GPGSA,A,3,31,29,13,20,32,16,23,,,,,,2.5,1.2,2.2*38\r\n"
    gps_nmea.gpgsv << "$GPGSV,3,1,11,16,68,144,37,01,67,093,39,23,53,307,40,20,43,233,33*77\r\n"
    gps_nmea.gpgsv << "$GPGSV,3,2,11,32,39,202,36,31,33,067,34,13,26,306,33,29,12,044,28*73\r\n"
    gps_nmea.gpgsv << "$GPGSV,3,3,11,06,10,147,23,03,04,162,26,04,01,309,*44\r\n"
    gps_nmea.gprmc = "$GPRMC,171331.676,A,4505.9701,N,09311.7487,W,000.0,242.1,130710,,,A*71\r\n"
    gps_nmea.gpvtg = "$GPVTG,242.1,T,,M,000.0,N,000.0,K,A*08\r\n"
    content.attribute.gps_nmea = gps_nmea

    @client.send(container)
  end

  def send_attr_gpio(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_GPIO
    gpio = Annex::Gpio.new
    gpio.type = Annex::Gpio::Type::TYPE_DIGITAL_INPUT
    gpio.label = "tc_gpio_label"
    gpio.line = 1
    gpio.value = 1
    content.attribute.gpio = gpio

    @client.send(container)
  end

  def send_attr_key_value_pair_float(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_KEY_VALUE_PAIR
    kvp = Annex::KeyValuePair.new
    kvp.key = "tc_key_fuel"
    value = Annex::KeyValuePair::Value.new
    value.type = Annex::KeyValuePair::Value::Type::TYPE_SCALAR
    value.scalar = Annex::Scalar.new
    value.scalar.type = Annex::Scalar::Type::TYPE_FLOAT
    value.scalar.float = 25.1
    kvp.value << value
    content.attribute.key_value_pair = kvp

    @client.send(container)
  end

  def send_gps_nmea_type(index, gpsdata, nmea_type)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_GPS_NMEA
    gps_nmea = Annex::GpsNmea.new

    case nmea_type 
      when "GPGGA" then gps_nmea.gpgga = gpsdata
      when "GPGLL" then gps_nmea.gpgll = gpsdata
      when "GPGSA" then gps_nmea.gpgsa = gpsdata
      when "GPGSV" then gps_nmea.gpgsv = gpsdata
      when "GPRMC" then gps_nmea.gprmc = gpsdata
      when "GPVTG" then gps_nmea.gpvtg = gpsdata
    end

    content.attribute.gps_nmea = gps_nmea

    @client.send(container)
  end

  def send_attr_cellular_radio(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_CELLULAR_RADIO

    cell_radio = Annex::CellularRadio.new
    sim_info = Annex::Sim.new

    sim_info.cpin = "abcd"
    sim_info.imsi = "123456789"

    cell_radio.sim = sim_info 
    cell_radio.rssi = 14
    cell_radio.imei = "*010644000112667*"
    cell_radio.phone_number = "7632661747"

    content.attribute.cellular_radio = cell_radio

    @client.send(container)
  end

  def send_attr_network_interface(index)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_NETWORK_INTERFACE
    network_interface = Annex::NetworkInterface.new

    flags = Annex::NetworkInterface::Flags.new
    flags.up = true
    flags.lower_up = true
    flags.loopback = false
    flags.broadcast = true
    flags.pointtopoint = false
    flags.multicast = false
    flags.dynamic = false
    flags.noarp = false
    flags.allmulti = false
    flags.promisc = false
    
    network_interface.name = "eth0"
    network_interface.link_type = Annex::NetworkInterface::LinkType::LINK_TYPE_ETHER
    network_interface.flags = flags
    network_interface.addr = "0080aabbccdd"
    network_interface.mtu = 1500

    inet_settings = Annex::InetSettings.new
    ip_addr = Annex::IpAddrNet.new

    ip_addr.addr = 0x34450011
    ip_addr.network_prefix_length = 8

    inet_settings.method = Annex::InetSettings::Method::METHOD_MANUAL
    inet_settings.ip_addrs << ip_addr
    inet_settings.default_route = 0x34450001
    inet_settings.name_servers << 0x34567809

    network_interface.inet_settings = inet_settings

    link_stats = Annex::LinkStatistics.new
    link_stats.rx_bytes = 10000
    link_stats.rx_packets = 200
    link_stats.rx_errors = 5
    link_stats.rx_dropped = 4
    link_stats.rx_overrun = 1
    link_stats.rx_multicast = 0
    link_stats.tx_bytes = 20000
    link_stats.tx_packets = 400
    link_stats.tx_errors = 10
    link_stats.tx_dropped = 8
    link_stats.tx_carrier = 0
    link_stats.tx_collisions = 2

    network_interface.link_statistics = link_stats

    content.attribute.network_interface = network_interface

    @client.send(container)
  end

  def send_gps_data_from_file(index, datafile, interval, nmea_type)

    File.open(datafile) do |file|
      while gpsline = file.gets
        if (gpsline.match(nmea_type) != nil) then
          send_gps_nmea_type(index, gpsline, nmea_type)
          if interval.to_i > 0 then
            sleep(interval.to_i)
          end
        end
      end
    end

  end

  def download_file(index, filename)
    send_session_init(index)

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ACTION
    action = Annex::Action.new
    action.type = Annex::Action::Type::TYPE_DO
    function = Annex::Function.new
    function.type = Annex::Function::Type::TYPE_FILE_OPEN
    file_open = Annex::FileOpen.new
    file_open.path = filename
    file_open.mode = "r"
    function.file_open = file_open
    action.function = function
    content.action = action

    @client.send(container)
    recv = @client.recv()

    file = recv.packages.first.content.attribute.file

    firmware = File.open("tmp000000", "w")
    offset = 0

    loop {
      container = Annex::Container.new
      package = Annex::Package.new
      content = Annex::Content.new
      container.packages << package
      package.content = content

      package.message_id = @client.message_id()

      content.type = Annex::Content::Type::TYPE_ACTION
      action = Annex::Action.new
      action.type = Annex::Action::Type::TYPE_DO
      function = Annex::Function.new
      function.type = Annex::Function::Type::TYPE_FILE_READ
      file_read = Annex::FileRead.new
      file_read.file = file
      file_read.length = 8192
      function.file_read = file_read
      action.function = function
      content.action = action

      @client.send(container)
      recv = @client.recv()

      buf = recv.packages.first.content.attribute.file_buffer
      firmware.write(buf.buffer)

      offset += buf.buffer.length

      Inform.debug "read #{offset}"
      if buf.eof
        Inform.debug "eof at offset #{offset}"
        break
      end
    }

    firmware.close()

    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_ACTION
    action = Annex::Action.new
    action.type = Annex::Action::Type::TYPE_DO
    function = Annex::Function.new
    function.type = Annex::Function::Type::TYPE_FILE_CLOSE
    file_close = Annex::FileClose.new
    file_close.file = file
    function.file_close = file_close
    action.function = function
    content.action = action

    @client.send(container)
    recv = @client.recv()

    Inform.debug "recv_total: #{@client.recv_total}"
    Inform.debug "send_total #{@client.send_total}"

    cksum_recv = `cksum tmp000000 | cut -d ' ' -f 1`
    puts "File downloaded - Chksum = "+cksum_recv.to_s+"\n"
    
#    File.unlink("tmp000000")
  end

  def do_job(index, gps, messages, platform_ip, filename, interval)
    puts "Start job " + index.to_s + "\n"
    setup(platform_ip)

    loop {
      if (gps != nil) then
        gpslist = gps.split(",")
        send_gps_data_from_file(index, gpslist[0].strip, gpslist[1].strip, gpslist[2].strip)
        # send_attr_gps_nmea_full(index)
      end

      if (messages != nil) then
        msglist = messages.split(",")
        
        for m in 0..msglist.length-1
          if ("sessioninit".casecmp(msglist[m].strip) == 0) 
            send_session_init (index)
          elsif ("requestpackagedelivery".casecmp(msglist[m].strip) == 0) 
            request_package_delivery(index)
          elsif ("notification".casecmp(msglist[m].strip) == 0) 
            send_notification(index)
          elsif ("action".casecmp(msglist[m].strip) == 0) 
            send_action(index)
          elsif ("gpio".casecmp(msglist[m].strip) == 0) 
            send_attr_gpio(index)
          elsif ("kvp".casecmp(msglist[m].strip) == 0) 
            send_attr_key_value_pair_float (index)
          elsif ("cellular".casecmp(msglist[m].strip) == 0) 
            send_attr_cellular_radio (index)
          elsif ("network".casecmp(msglist[m].strip) == 0) 
            send_attr_network_interface (index)
          end
        end

      end

      if (filename != nil) then
        download_file(index, filename)
      end

      if (interval == 'NONE') then
        break;
      else
        sleep(interval)
      end
    }

    teardown()
    puts "End job " + index.to_s + "\n"
  end

end

class AnnexICellSimulator < AnnexSimulator

  def send_session_init(serial_num)
    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_SESSION_INIT
    session_init = Annex::SessionInit.new

    session_init.hostname = "Simulator"
    session_init.vendor_id = "MultiTech"
    session_init.product_id = "MTCMR-E1"
    session_init.device_id = "50105502M" + serial_num.to_s
    session_init.sw_version = "1.0"
    session_init.hw_version = "rev E"
    session_init.synchronized = true
    session_init.container_buffer_max = 8192

    content.session_init = session_init

    @client.send(container)
    recv = @client.recv
  end

end

class AnnexRCellSimulator < AnnexSimulator
  def send_session_init (serial_num)
    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_SESSION_INIT
    session_init = Annex::SessionInit.new

    session_init.hostname = "Simulator"
    session_init.vendor_id = "MultiTech"
    session_init.product_id = "MTCBA-E1-EN2"
    session_init.device_id = "50105503N" + serial_num.to_s
    session_init.sw_version = "2.60"
    session_init.hw_version = "rev C"
    session_init.synchronized = true
    session_init.container_buffer_max = 8192

    content.session_init = session_init

    @client.send(container)
    recv = @client.recv
  end
end

class AnnexCDPSimulator < AnnexSimulator
  def send_session_init (serial_num)
    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_SESSION_INIT
    session_init = Annex::SessionInit.new

    session_init.hostname = "Simulator"
    session_init.vendor_id = "MultiTech"
    session_init.product_id = "MTCDP-E1-0.0"
    session_init.device_id = "50105504C" + serial_num
    session_init.sw_version = "1.0.0"
    session_init.hw_version = "rev 0"
    session_init.synchronized = true
    session_init.container_buffer_max = 8192

    content.session_init = session_init

    @client.send(container)
    recv = @client.recv
  end
end

class AnnexA2WSimulator < AnnexSimulator
  def send_session_init (serial_num)
    container = Annex::Container.new
    package = Annex::Package.new
    content = Annex::Content.new
    container.packages << package
    package.content = content

    package.message_id = @client.message_id()

    content.type = Annex::Content::Type::TYPE_SESSION_INIT
    session_init = Annex::SessionInit.new

    session_init.hostname = "Simulator"
    session_init.vendor_id = "MultiTech"
    session_init.product_id = "MT100A2W-G"
    session_init.device_id = "50105505H" + serial_num
    session_init.sw_version = "2.60"
    session_init.hw_version = "rev C"
    session_init.synchronized = true
    session_init.container_buffer_max = 8192

    content.session_init = session_init

    @client.send(container)
    recv = @client.recv
  end
end

if $0 == __FILE__

  if (ARGV[0] == nil) then
    puts "Open default file - sim.yaml\n"
    sim_config = YAML::load_file('sim.yaml')
  else
    puts "Open yaml file - " + ARGV[0] + "\n"
    sim_config = YAML::load_file(ARGV[0])
  end

  threads = []

  for group in 1..sim_config.length do
    group_index = "group" + group.to_s

    if sim_config[group_index] != nil then

       type = sim_config[group_index]["type"]

       case type 
         when 'iCell' then
           num_icell = sim_config[group_index]["instances"]

           for index in 1..num_icell
             threads << Thread.new {
               icell = AnnexICellSimulator.new()

               icell.do_job(index,
                 sim_config[group_index]["gpsdata"],
                 sim_config[group_index]["messages"],
		 sim_config[group_index]["platform_ip"],
                 sim_config[group_index]["download"],
                 sim_config[group_index]["repeat"])
             }
           end

         when 'rCell' then
           num_rcell = sim_config[group_index]["instances"]

           for index in 1..num_rcell
             threads << Thread.new {
               rcell = AnnexRCellSimulator.new()

               rcell.do_job(index,
                 sim_config[group_index]["gpsdata"],
                 sim_config[group_index]["messages"],
		 sim_config[group_index]["platform_ip"],
                 sim_config[group_index]["download"],
                 sim_config[group_index]["repeat"])
             }
           end

         when 'CDP' then
         when 'A2W' then
         else
       end

    end

  end

  # wait for all threads to finish and then exit
  threads.each {|t| t.join}

end
