# vim: set sw=2 ts=2 expandtab:
#
# Copyright (C) 2010 by James Maki
#
# Author: James Maki <jamescmaki@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

require 'syslog'

#  Inform.syslog = false
#  if Inform.syslog
#    Syslog.open("smsws", Syslog::LOG_NDELAY, Syslog::LOG_LOCAL7)
#  else
#    Inform.level = Inform::LOG_ERR
#    Inform.file = $stdout
#  end

module Inform
  include Syslog

  LEVELS = {
    LOG_EMERG => :emerg,
    LOG_ALERT => :alert,
    LOG_CRIT => :crit,
    LOG_ERR => :err,
    LOG_WARNING => :warning,
    LOG_NOTICE => :notice,
    LOG_INFO => :info,
    LOG_DEBUG => :debug,
  }

  LEVELS.each_pair do |key, value|
    module_eval <<-EOS
      def #{value}(msg)
        inform(#{key}, msg, caller[0])
      end
    EOS

    module_function value
  end

  module_function

  def level=(level)
    raise ArgumentError, "bad level" unless level >= LOG_EMERG && level <= LOG_DEBUG

    @level = level
  end

  def level
    @level
  end

  def file=(file)
    @file = file
  end

  def file
    @file
  end

  def syslog=(syslog)
    @syslog = syslog
  end

  def syslog
    @syslog
  end

  def inform(level, msg, from = caller[0])
    time = Time.now.strftime('%Y-%m-%d %H:%M:%S')

    from =~ /(.+)\:(\d+)(?:\:in \`(\S+)\')?/
    file = File.basename($1)
    line = $2
    func = $3
    func ||= 'main'

    str = "#{time} [#{LEVELS[level].to_s.upcase}] #{file}:#{func}:#{line}: "

    if msg.is_a?(Exception)
      str << "#{msg.class} #{msg}"
    else
      str << "#{msg}"
    end

    log(level, str)

    if msg.is_a?(Exception)
      msg.backtrace.each do |line|
        log(level, ">  #{line}")
      end
    end
  end

  def escape_spec(str)
    new = ""
    str.each_byte { |c|
      #if c == "%".ord
      if c == ?%
        new << "%%"
      else
        new << c
      end
    }

    return new
  end

  def log(level, msg)
    if @syslog
      Syslog.log(level, escape_spec(msg))
    elsif @level && @file
      @file.puts(msg) if level <= @level
    end
  end
end
