#!/usr/bin/env ruby
#
# vim: set sw=2 ts=2 expandtab:
#

require 'annex_common'
require 'annex_client'
require 'annex_server'

require 'test/unit'

Thread.abort_on_exception = true
Inform.level = Inform::LOG_DEBUG
Inform.file = $stdout

SERVER_HOSTNAME = "204.26.122.94"
#SERVER_HOSTNAME = "127.0.0.1"

class TC_AnnexClient < Test::Unit::TestCase
  def setup
    @client = AnnexClient.new(SERVER_HOSTNAME)
  end

  def teardown
    @client.close() if @client
  end

  def test_000000_empty_container
    container = Annex::Container.new
    @client.send(container)
  end

  def test_000010_empty_package
    container = Annex::Container.new
    package = Annex::Package.new
    container.packages << package
    @client.send(container)
  end

  def test_000020_empty_content
    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content
    content.type = Annex::Content::Type::TYPE_NOTIFICATION
    @client.send(container)
  end

  def test_000030_session_init_empty
    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_SESSION_INIT
    session_init = Annex::SessionInit.new
    content.session_init = session_init

    @client.send(container)
    recv = @client.recv()
  end

  def test_000040_session_init_normal
    send_session_init()
  end

  def test_000050_request_package_delivery
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_REQUEST_PACKAGE_DELIVERY
    request_package_delivery = Annex::RequestPackageDelivery.new
    content.request_package_delivery = request_package_delivery

    @client.send(container)
    recv = @client.recv()
  end



  def test_000060_notification
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_NOTIFICATION
    notification = Annex::Notification.new
    notification.level = Annex::Notification::Level::LEVEL_INFO
    notification.message = "Defcon 5"

    content.notification = notification

    @client.send(container)
  end

  def test_000070_notification_with_ack
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content
    package.request_ack = true

    content.type = Annex::Content::Type::TYPE_NOTIFICATION
    notification = Annex::Notification.new
    notification.level = Annex::Notification::Level::LEVEL_ALERT
    notification.message = "Defcon 1"

    content.notification = notification

    @client.send(container)
    ack = @client.recv()
    assert_not_nil(ack, "I would like an ack with my Defcon 1 please")
    assert_equal(package.message_id, ack.packages[0].correlation_id)
  end

  def test_000080_action
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ACTION
    action = Annex::Action.new
    action.type = Annex::Action::Type::TYPE_DO
    content.action = action
    action.function = Annex::Function.new
    action.function.type = Annex::Function::Type::TYPE_EXTENSION

    @client.send(container)
  end

  def test_000090_attr_gps_nmea_empty
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_GPS_NMEA
    gps_nmea = Annex::GpsNmea.new
    content.attribute.gps_nmea = gps_nmea

    @client.send(container)
  end

  def test_000100_attr_gps_nmea_full
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_GPS_NMEA
    gps_nmea = Annex::GpsNmea.new
    gps_nmea.gpgga = "$GPGGA,171331.676,4505.9701,N,09311.7487,W,1,07,1.2,273.0,M,-29.0,M,,0000*66\r\n"
    gps_nmea.gpgll = "$GPGLL,4505.9701,N,09311.7487,W,171331.676,A,A*47\r\n"
    gps_nmea.gpgsa = "$GPGSA,A,3,31,29,13,20,32,16,23,,,,,,2.5,1.2,2.2*38\r\n"
    gps_nmea.gpgsv << "$GPGSV,3,1,11,16,68,144,37,01,67,093,39,23,53,307,40,20,43,233,33*77\r\n"
    gps_nmea.gpgsv << "$GPGSV,3,2,11,32,39,202,36,31,33,067,34,13,26,306,33,29,12,044,28*73\r\n"
    gps_nmea.gpgsv << "$GPGSV,3,3,11,06,10,147,23,03,04,162,26,04,01,309,*44\r\n"
    gps_nmea.gprmc = "$GPRMC,171331.676,A,4505.9701,N,09311.7487,W,000.0,242.1,130710,,,A*71\r\n"
    gps_nmea.gpvtg = "$GPVTG,242.1,T,,M,000.0,N,000.0,K,A*08\r\n"
    content.attribute.gps_nmea = gps_nmea

    @client.send(container)
  end

  def test_000110_attr_gpio
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_GPIO
    gpio = Annex::Gpio.new
    gpio.label = "tc_gpio_label"
    gpio.value = 1
    content.attribute.gpio = gpio

    @client.send(container)
  end

  def test_000120_attr_sensor_temp_float
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_SENSOR
    sensor = Annex::Sensor.new
    sensor.name = "tc_sensor_temp"
    sensor.value = Annex::Scalar.new
    sensor.value.type = Annex::Scalar::Type::TYPE_FLOAT
    sensor.value.float = 36.3
    content.attribute.sensor = sensor

    @client.send(container)
  end

  def test_000130_attr_key_value_pair_float
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ATTRIBUTE

    content.attribute = Annex::Attribute.new
    content.attribute.type = Annex::Attribute::Type::TYPE_KEY_VALUE_PAIR
    kvp = Annex::KeyValuePair.new
    kvp.key = "tc_key_fuel"
    value = Annex::KeyValuePair::Value.new
    value.type = Annex::KeyValuePair::Value::Type::TYPE_SCALAR
    value.scalar = Annex::Scalar.new
    value.scalar.type = Annex::Scalar::Type::TYPE_FLOAT
    value.scalar.float = 25.1
    kvp.value << value
    content.attribute.key_value_pair = kvp

    @client.send(container)
  end

  def no_test_000140_file_read_8192_blocks
    send_session_init()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ACTION
    action = Annex::Action.new
    action.type = Annex::Action::Type::TYPE_DO
    function = Annex::Function.new
    function.type = Annex::Function::Type::TYPE_FILE_OPEN
    file_open = Annex::FileOpen.new
    file_open.path = "UIP_021Y_05D922EE.BIN"
    file_open.mode = "r"
    function.file_open = file_open
    action.function = function
    content.action = action

    @client.send(container)
    recv = @client.recv()

    file = recv.packages.first.content.attribute.file

    Dir.mkdir("tmp") rescue 0
    firmware = File.open("tmp/recv.bin", "w")
    offset = 0

    loop {
      container = Annex::Container.new
      package = Annex::Package.new
      package.message_id = @client.message_id()
      content = Annex::Content.new
      container.packages << package
      package.content = content

      content.type = Annex::Content::Type::TYPE_ACTION
      action = Annex::Action.new
      action.type = Annex::Action::Type::TYPE_DO
      function = Annex::Function.new
      function.type = Annex::Function::Type::TYPE_FILE_READ
      file_read = Annex::FileRead.new
      file_read.file = file
      file_read.length = 8192
      function.file_read = file_read
      action.function = function
      content.action = action

      @client.send(container)
      recv = @client.recv()

      buf = recv.packages.first.content.attribute.file_buffer
      firmware.write(buf.buffer)

      offset += buf.buffer.length

      Inform.debug "read #{offset}"
      if buf.eof
        Inform.debug "eof at offset #{offset}"
        break
      end
    }

    firmware.close()

    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_ACTION
    action = Annex::Action.new
    action.type = Annex::Action::Type::TYPE_DO
    function = Annex::Function.new
    function.type = Annex::Function::Type::TYPE_FILE_CLOSE
    file_close = Annex::FileClose.new
    file_close.file = file
    function.file_close = file_close
    action.function = function
    content.action = action

    @client.send(container)
    recv = @client.recv()

    Inform.debug "recv_total: #{@client.recv_total}"
    Inform.debug "send_total #{@client.send_total}"

    cksum_recv = `cksum tmp/recv.bin | cut -d ' ' -f 1`
    cksum_uip = `cksum files/UIP_021Y_05D922EE.BIN | cut -d ' ' -f 1`

    assert_equal(cksum_uip, cksum_recv)
  end

  def send_session_init
    container = Annex::Container.new
    package = Annex::Package.new
    package.message_id = @client.message_id()
    content = Annex::Content.new
    container.packages << package
    package.content = content

    content.type = Annex::Content::Type::TYPE_SESSION_INIT
    session_init = Annex::SessionInit.new

    session_init.hostname = "tc_hostname"
    session_init.vendor_id = "tc_vendor_id"
    session_init.product_id = "tc_product_id"
    session_init.device_id = "tc_device_id"
    session_init.sw_version = "tc_sw_version"
    session_init.hw_version = "tc_hw_version"
    session_init.synchronized = true
    session_init.container_buffer_max = 8192

    content.session_init = session_init

    @client.send(container)
    recv = @client.recv()
  end

end
